; ---------------------------------------------------------------------------
; SC_MSG.SC                SCREEN MESSAGE PROCEDURES
;
; Three procedures for producing centered messages on screen, with choice of
; line, attributes, sleep time, wait for keypress, etc.  Msg() does the
; basics, Msg2() allows for selected time of blinking then steady, BDMsg()
; puts three blinking dots at the end (for instance, for a "WORKING ..."
; message where the dots blink).
;
; Steve Caple, SpellBound Services
; 09/02/90
;
; ----------------------------------MSG()------------------------------------
; Produces message centered in line
;
; Parameter Msg       (string, max 80, trimmed if over, message sent).
;           MLine     (row number to centerit on)
;           MClear    (clear msg line if true)
;           MStyle    (numeric style attribute
;           MDuration (time to hold - ensures minimum time visible)
;           MWait     (message  "Press any key.. " if true)
;
; Example proc call:    Msg("WORKING",0,false,107,1200,false)
;
; Returns no value.
; ---------------------------------------------------------------------------
proc Msg(Msg,MLine,MClear,MStyle,MDuration,MWait)
  private x
  CurrentProc = "Msg"
  if len(Msg) > 80 then
    message "Proc Msg(): message trimmed to fit screen..."  sleep 500
    Msg = substr(Msg,1,80)
  endif

  ;-------------------------------------------------------------------------
  ; check for numeric attribute, make sure it's in range 0-255
  ;-------------------------------------------------------------------------
  if MStyle > 255 then
    MStyle = mod(MStyle,256)
  endif
  style attribute MStyle

  ; -------------------------------------------------------------------------
  ; if MClear is set to true, clear the msg line, otherwise just place the
  ; msg in the middle of the line
  ; -------------------------------------------------------------------------
  canvas off
  if MClear then
    @ MLine, 0 ?? spaces(80)          ; writes spaces in MStyle attribute
  endif
  @ MLine, 40-int(len(Msg)/2) ?? Msg  ; center message om screen
  canvas on
  style
  if MDuration > 0 then sleep MDuration endif
  if MWait then
    message "Press any key to continue"
    x = getchar()
  endif
endproc

writelib LibName Msg
release procs Msg
? "Msg()"
ColumnSet()




; ---------------------------------MSG2()------------------------------------
; Produces message centered in line - optional blinking first - selectable
; duration for blinking and steady
;
; Parameter Msg       (string, max 80, trimmed if over, message sent).
;           MLine     (row number to centerit on)
;           MClear    (clear msg line if true)
;           MStyle    (numeric style attribute)
;           BDuration (blink duration)
;           SDuration (steady duration)
;           MWait     ('message  "Press any key.. "' if true)
;
; Example proc call: Msg2("WORKING",24,false,107,900,1200,false)
;
; Returns no value.
; ---------------------------------------------------------------------------
proc Msg2(Msg,MLine,MClear,MStyle,BDuration,SDuration,MWait)
  private x, BStyle
  CurrentProc = "Msg2"

  if len(Msg) > 80 then
    message "Proc Msg2(): message trimmed to fit screen..."  sleep 500
    Msg = substr(Msg,1,80)
  endif

  ;-------------------------------------------------------------------------
  ; check numeric attribute, make sure it's non-blinking & in valid range
  ;-------------------------------------------------------------------------
  if MStyle > 127 then
    MStyle = mod(MStyle,128)
  endif
  BStyle = MStyle+128

  ;-------------------------------------------------------------------------
  ; if there's any blink time, ?? msg and hold BDuration
  ;-------------------------------------------------------------------------
  if BDuration > 0 then
    style attribute BStyle
    ; ----------------------------------------------------------------------
    ; if MClear is set to true, clear the msg line
    ; ----------------------------------------------------------------------
    canvas off
    if MClear then
      @ MLine, 0 ?? spaces(80)          ; writes spaces in MStyle attribute
    endif
    @ MLine, 40-int(len(Msg)/2) ?? Msg  ; center message on screen
    canvas on
    sleep BDuration
  endif

  ;-------------------------------------------------------------------------
  ; if there's ANY steady time, or both are 0, then ?? without adding blink
  ; and hold SDuration (straight steady message)
  ;-------------------------------------------------------------------------
  style attribute MStyle
  @ MLine, 40-int(len(Msg)/2) ?? Msg  ; center message on screen
  if SDuration > 0 then
    sleep SDuration
  endif

  ;-------------------------------------------------------------------------
  ; if MWait set true, message "Press any key ... " and hold for getchar()
  ; adding 'blink' and hold SDuration
  ;-------------------------------------------------------------------------
  if MWait then
    message "Press any key to continue"
    x = getchar()
  endif
  style
endproc

writelib LibName Msg2
release procs Msg2
? "Msg2()"
ColumnSet()




; ---------------------------------BDMsg()-----------------------------------
; Produces message centered in line with 3 blinking dots at end
;
; Parameter Msg       (string, max 76, trimmed if over, message sent).
;           MLine     (row number to center it on)
;           MClear    (clear msg line if true)
;           MStyle    (numeric style attribute)
;           MDuration (time to hold - ensures minimum time visible)
;           MWait     (message  "Press any key.. " & wait if true)
;
; Returns no value.
;
; Example proc call:  BDMsg("WORKING",24,false,107,1200,false)
; ---------------------------------------------------------------------------
proc BDMsg(Msg,MLine,MClear,MStyle,MDuration,MWait)
  private x
  CurrentProc = "BDMsg"

  if len(Msg) > 76 then       ; 76 = 80 - space + three dots
    message "Proc BDMsg(): message trimmed to fit screen..."  sleep 500
    Msg = substr(Msg,1,76)
  endif

  ;-------------------------------------------------------------------------
  ; check attribute, make sure it's non-blinking & in valid range
  ;-------------------------------------------------------------------------
  if MStyle > 127 then
    MStyle = mod(MStyle,128)
  endif
  BStyle = Mstyle + 128
  style attribute MStyle
  canvas off

  ; -------------------------------------------------------------------------
  ; if MClear is set to true, clear the msg line, otherwise just place the
  ; msg in the middle of the line
  ; -------------------------------------------------------------------------
  if MClear then
    @ MLine, 0 ?? spaces(80)          ; writes spaces in MStyle attribute
  endif
  @ MLine, 38-int(len(Msg)/2) ?? Msg  ; center message on screen

  ;-------------------------------------------------------------------------
  ; set blinking attribute, put dots at end of msg
  ;-------------------------------------------------------------------------
  style attribute BStyle
  ?? " ..."
  canvas on

  ;-------------------------------------------------------------------------
  ; if MDuration > 0, then sleep before proceding
  ;-------------------------------------------------------------------------
  if MDuration > 0 then
    sleep MDuration
  endif

  ;-------------------------------------------------------------------------
  ; if MWait set true, message "Press any key ... " and hold for getchar()
  ;-------------------------------------------------------------------------
  if MWait then
    message "Press any key to continue"
    x = getchar()
  endif

  style

endproc

writelib LibName BDMsg
release procs BDMsg
? "BDMsg()"
ColumnSet()




