{$S-,R-,V-,I-,B-,F+,X+}
{$M 8192,0,655360}

{*********************************************************}
{*                    ANALYZE.PAS 1.00                   *}
{*        Copyright (c) TurboPower Software 1992.        *}
{*                 All rights reserved.                  *}
{*********************************************************}

program Analyze;
  {-Adds compressed file sizes to a fileinfo file}

uses
  {--- RTL ---}
  Dos,
  Crt,
  {--- APRO --}
  ApMisc,
  OoArchiv,
  OoLzh;

const
  {Miscellaneous constants}
  PgmVersion = 'Build Analyzer. Copyright (c) 1992 TurboPower Software. Version 1.00.';
  HaltSoon   : Boolean = False;
  TempLzh    : PathStr = '$$$.LZH';

  {Program limits}
  MaxSrcDirs = 10;
  MaxFiles = 1000;

  {Runtime options}
  AnalyzePass  : Boolean = False;           {True to analyze files}
  Build36Pass  : Boolean = False;           {True to build 360K images}
  Build72Pass  : Boolean = False;           {True to build 720K images}
  Build12Pass  : Boolean = False;           {True to build 1200K images}
  Build14Pass  : Boolean = False;           {True to build 1440K images}
  GenerateBat  : Boolean = False;           {True to generate backup BAT file}
  InfoFileName : PathStr = 'FILEINFO.TXT';  {Default input file name}
  OutName      : PathStr = 'INSTALL.DAT';   {Default output name}
  RptName      : PathStr = 'INSTALL.RPT';   {Default report name}
  ExcludeName  : PathStr = '';              {No exclude file}
  RootDir      : String  = 'C:\IMAGE';      {Default root directory}
  Version      : Word    = 100;             {Default version number}

  {Disk size limits}
  Disk360Limit = 350000;
  Disk720Limit = 710000;
  Disk1200Limit = 1150000;
  Disk1440Limit = 1300000;

  {Other limits}
  CopyBufSize = 8192;

  {Other miscellaneous}
  CRLF = #13#10;
  ExtLen = 3;

type
  {Media types}
  MediaType = (Disk360, Disk720, Disk1200, Disk1440);

  {Format of install.dat output}
  FileInfoRec = record
    Name       : String[12];
    SourceDir  : Byte;
    DestDir    : Byte;
    Group      : LongInt;
    Size       : LongInt;
    CompSize   : LongInt;
    Disk36     : Byte;
    Disk72     : Byte;
    Disk12     : Byte;
    Disk14     : Byte;
    Archived   : Boolean;
    Disk1      : Boolean;
    Tagged     : Boolean;
    FileVer    : Longint;
  end;
  FilesArray = array[1..MaxFiles] of FileInfoRec;
  PFilesArray = ^FilesArray;

  DT = record
    Time : Word;
    Date : Word;
  end;

  DW = record
    L : Word;
    H : Word;
  end;

var
  FI          : Text;                           {FileInfo file}
  F           : Text;                           {General}
  SourceDirs  : array[1..MaxSrcDirs] of PathStr;{List of source dirs}
  DestDirs    : array[1..MaxSrcDirs] of PathStr;{list of dest dirs}
  Map         : File of FileInfoRec;            {Output file (install.dat)}
  Rpt         : Text;                           {Report file}
  NewSize     : LongInt;                        {Compressed size of file}
  OrigSize    : LongInt;                        {Original size of file}
  Files       : PFilesArray;                    {Internal list of files}
  CopyBuffer  : array[1..CopyBufSize] of Char;  {General copy buffer}
  LzhName     : PathStr;                        {Name of LZH archive}
  L           : Lzh;                            {LZH archive object}
  FileCnt     : Word;                           {Total file count}
  OrigDir     : PathStr;                        {Original directory}
  SaveName    : PathStr;                        {Name of file if compress fails}
  SavedTime   : LongInt;                        {Saved timestamp}
  Building    : Boolean;                        {True if in build pass}
  TimeStamp   : LongInt;                        {New date/time for arced files}
  CheckDups   : FileMaskList;                   {Avoid arc'ing dup files}
  Pivot       : FileInfoRec;                    {Sorting pivot}
  SortMedia   : MediaType;                      {For sorting media}
  ExcludeCnt  : Byte;                           {Number of files in exclude list}
  ExcludeList : array[1..255] of String[12];   {Exclude files}

  procedure Abort(Msg : String);
  begin
    WriteLn(Msg);
    Halt(1);
  end;

  function GetKey : Word;
    {-Get a key}
  var
    Regs : Registers;
  begin
    Regs.AH := 0;
    Intr($16, Regs);
    GetKey := Regs.AX;
  end;

  function WaitForKey : Word;
    {-Wait for a keypress and throw away the result}
  var
    Key : Word;
    Ch  : Char absolute Key;
  begin
    {halt if ^Break or ^C pressed}
    Key := GetKey;
    if (Key = $0000) or (Ch = ^C) then
      Halt;
    WaitForKey := Key;
  end;

  function IsDirectory(FName : String) : Boolean;
    {-Return true if FName is a directory}
  var
    IO : Word;
    CurDir : PathStr;
    CurDestDir : PathStr;
    DiffDrive : Boolean;
  begin
    GetDir(0, CurDir);

    if (Length(FName) >= 2) and (FName[2] = ':') and (FName[1] <> CurDir[1])
    then begin
      {Checking on a different drive}
      DiffDrive := True;
      ChDir(FName[1]+':');
      if IoResult <> 0 then begin
        IsDirectory := False;
        Exit;
      end;
      GetDir(0, CurDestDir);
    end else
      DiffDrive := False;

    ChDir(FName);
    IsDirectory := (IoResult = 0);

    if DiffDrive then begin
      ChDir(CurDestDir);
      IO := IoResult;
    end;

    ChDir(CurDir);
    IO := IoResult;
  end;

  function TodayString : String;
    {-Return a date string like DD/MM/YYYY}
  var
    Year, Month, Day, DOW : Word;
    SYear, SMonth, SDay : String[2];
    S : String;
    I : Word;
  begin
    GetDate(Year, Month, Day, DOW);
    Str(Year:4, SYear);
    Str(Month:2, SMonth);
    Str(Day:2, SDay);
    S := SDay + '/' + SMonth + '/' + SYear;
    for I := 1 to Length(S) do
      if S[I] = ' ' then
        S[I] := '0';
    TodayString := S;
  end;

  function GetFileVerMS(FName : PathStr) : LongInt;
   {-Search for versioninfo resource in FName and return FileVersionMS}
  type
    {Old header snippet}
    OldHeader = record
      Junk1         : array[1..$18] of Char;
      NewHeaderFlag : Byte;
      Junk2         : array[1..$23] of Char;
      NewHeaderOfs  : Word;
    end;

    {New header snippet}
    NewHeader = record
      Junk : array[1..36] of Byte;
      ResTableOfs   : Word;
    end;

    {Resource table entry - used for skipping entries}
    ResourceNameInfo = record
      rnOffset : Word;
      rnLength : Byte;
      rnFlags  : Word;
      rnJunk1  : Byte;
      rnID     : Word;
      rnJunk2  : array[1..4] of Byte;
    end;

    {Fixed file info format of VERINFO resource}
    Tvs_FixedFileInfo = record
      dwStrucVersion     : Longint;
      dwFileVersionMS    : Longint;
      dwFileVersionLS    : Longint;
      dwProductVersionMS : Longint;
      dwProductVersionLS : Longint;
      dwFileFlagsMask    : Longint;
      dwFileFlags        : Longint;
      dwFileOS           : Longint;
      dwFileType         : Longint;
      dwFileSubtype      : Longint;
      dwFileDateMS       : Longint;
      dwFileDateLS       : Longint;
    end;

  const
    VerInfoRes = $8010;

  var
    F : File;
    OH : OldHeader;
    NH : NewHeader;
    RN : ResourceNameInfo;
    ResType : Word;
    Count : Word;
    Root : Tvs_FixedFileInfo;
    VerOfs : Word;
    Finished : Boolean;
    Align : Word;

    function ReadNextType : Boolean;
      {-Read the next resource type record}
    begin
      {Read resource type and count}
      BlockRead(F, ResType, 2);
      BlockRead(F, Count, SizeOf(Count));
      ReadNextType := Lo(ResType) <> 0;
    end;

    procedure SkipNextType;
      {-Skip all nameinfo entries for this resource type}
    var
      I : Word;
      Junk : array[1..5] of Word;
    begin
      BlockRead(F, Junk, 4);
      for I := 1 to Count do
        BlockRead(F, RN, SizeOf(RN));
    end;

    function Power(Exp : Byte) : LongInt;
    var
      L : LongInt;
      I : Word;
    begin
      L := 2;
      for I := 1 to Exp-1 do
        L := L * 2;
      Power := L;
    end;

    function ReadVerResource : LongInt;
      {-Read the VERINFO resourse and return FileVersionMS}
    var
      Junk : array[1..10] of Byte;
      Name : String[15];
      B : Byte;
      Adjust : Longint;
    begin
      BlockRead(F, Junk, 4);
      BlockRead(F, VerOfs, SizeOf(VerOfs));
      Adjust := Power(Align);
      Seek(F, LongInt(VerOfs)*Adjust);

      {Read cbBlock, cbValue}
      BlockRead(F, Junk, 4);

      {Read in the name, must be VS_VERSION_INFO}
      BlockRead(F, Name[1], 15);
      Name[0] := #15;
      if Name = 'VS_VERSION_INFO' then begin
        repeat
          BlockRead(F, B, 1);
        until B <> 0;
        BlockRead(F, Junk, 3);
        BlockRead(F, Root, SizeOf(Root));
        ReadVerResource := Root.dwFileVersionMS;
        Exit;
      end;
    end;

  begin
    {Assume failure}
    GetFileVerMS := 0;

    {Open file}
    Assign(F, FName);
    Reset(F, 1);
    if IoResult <> 0 then
      Exit;

    {Read in old-style header, done if no new style header}
    BlockRead(F, OH, SizeOf(OH));
    if OH.NewHeaderFlag < $40 then
      Exit;

    {Read in new header, seek to start of Resource Table}
    Seek(F, OH.NewHeaderOfs);
    BlockRead(F, NH, SizeOf(NH));
    Seek(F, OH.NewHeaderOfs+NH.ResTableOfs);

    {Read align shift word}
    BlockRead(F, Align, 2);

    {Scan for VERINFO resource}
    while ReadNextType do begin

      {Exit on errors}
      if IoResult <> 0 then begin
        Close(F);
        if IoResult <> 0 then ;
        Exit;
      end;

      {Handle this resource type}
      if ResType <> VerInfoRes then
        SkipNextType
      else
        GetFileVerMS := ReadVerResource;
    end;

    {Close up and exit}
    Close(F);
    if IoResult <> 0 then ;
  end;

  function HasExtension(Name : string; var DotPos : Word) : Boolean;
    {-Return whether and position of extension separator dot in a pathname}
  var
    I : Word;
  begin
    DotPos := 0;
    for I := Length(Name) downto 1 do
      if (Name[I] = '.') and (DotPos = 0) then
        DotPos := I;
    HasExtension := (DotPos > 0) and (Pos('\', Copy(Name, Succ(DotPos), 64)) = 0);
  end;

  function JustExtension(Name : string) : ExtStr;
    {-Return just the extension of a pathname}
  var
    DotPos : Word;
  begin
    if HasExtension(Name, DotPos) then
      JustExtension := Copy(Name, Succ(DotPos), ExtLen)
    else
      JustExtension := '';
  end;

  function CopyFile(SrcPath, DestPath : String;
                    Buffer : Pointer;
                    BufferSize : Word) : Word;
    {-Copy the file specified by SrcPath into DestPath. DestPath must specify
      a complete filename, it may not be the name of a directory without the
      file portion.  This a low level routine, and the input pathnames are not
      checked for validity. Buffer must already be allocated, and must be no
      less than BufferSize.}
  var
    ErrorCode,BytesRead,BytesWritten : Word;
    Time : LongInt;
    Src,Dest : File;
    SaveFileMode : Word;
    SaveFAttr : Word;

    procedure UnDo(CloseAndDeleteDest : Boolean);
    begin
      Close(Src);
      if IoResult <> 0 then ;
      if CloseAndDeleteDest then begin
        Close(Dest);
        if IoResult <> 0 then ;
        Erase(Dest);
        if IoResult <> 0 then ;
      end;
    end;

  begin
    SaveFileMode := FileMode;
    Assign(Src,SrcPath);
    GetFAttr(Src, SaveFAttr);
    if DosError <> 0 then begin
      CopyFile := 1;
      Exit;
    end;

    FileMode := FileMode and $F0;
    Reset(Src,1);
    FileMode := SaveFileMode;
    if IoResult <> 0 then begin
      CopyFile := 1;                   {unable to open SrcPath}
      Exit;
    end;

    Assign(Dest,DestPath);
    Rewrite(Dest,1);
    if IoResult <> 0 then begin
      CopyFile := 2;                   {unable to open DestPath}
      Undo(False);
      Exit;
    end;

    while not EOF(Src) do begin
      BlockRead(Src,Buffer^,BufferSize,BytesRead);
      if IoResult <> 0 then begin
        CopyFile := 3;                 {error reading SrcPath}
        UnDo(True);
        Exit;
      end;
      BlockWrite(Dest,Buffer^,BytesRead,BytesWritten);
      if (IoResult <> 0) or (BytesWritten <> BytesRead) then begin
        CopyFile := 4;                 {error reading SrcPath}
        UnDo(True);                    {error writing DestPath}
        Exit;
      end;
    end;

    GetFTime(Src,Time);
    if DosError <> 0 then begin
      CopyFile := 5;                   {error getting SrcPath's Date/Time}
      UnDo(True);
      Exit;
    end;

    SetFTime(Dest,Time);
    if DosError <> 0 then begin
      CopyFile := 6;                   {error getting DestPath's Date/Time}
      UnDo(True);
      Exit;
    end;

    Close(Dest);
    if IoResult <> 0 then begin
      CopyFile := 7;
      Close(Src);
      if IoResult <> 0 then ;
    end else begin
      Close(Src);
      if IoResult <> 0 then ;
      CopyFile := 0;
    end;
    SetFAttr(Dest, SaveFAttr);
  end;

  function Long2Str(L : LongInt) : string;
    {-Convert a long/word/integer/byte/shortint to a string}
  var
    S : string;
  begin
    Str(L, S);
    Long2Str := S;
  end;

  function Trim(S : string) : string;
    {-Return a string with leading and trailing white space removed}
  var
    I : Word;
    SLen : Byte absolute S;
  begin
    while (SLen > 0) and (S[SLen] <= ' ') do
      Dec(SLen);

    I := 1;
    while (I <= SLen) and (S[I] <= ' ') do
      Inc(I);
    Dec(I);
    if I > 0 then
      Delete(S, 1, I);

    Trim := S;
  end;

  function OkToSetTime(N : String) : Boolean;
  var
    I : Word;
  begin
    OkToSetTime := False;
    N := StUpcase(N);
    for I := 1 to ExcludeCnt do
      if N = ExcludeList[I] then
        Exit;
    OkToSetTime := True;
  end;

  function MyShowProgressFunc(UP : UnLzhPtr;
                              BytesWritten, TotalBytes : LongInt) : Boolean;
    {-Checks for user abort, then calls default function to show progress}
  var
    C : Char;
  begin
    {Handle pending halt requests}
    if HaltSoon then begin
      MyShowProgressFunc := False;
      Exit;
    end;

    {Check for user aborts}
    while KeyPressed do begin
      C := ReadKey;
      if C = #0 then
        C := ReadKey
      else if C = #27 then begin
        MyShowProgressFunc := False;
        Exit;
      end;
    end;

    {Show progress via default progress routine}
    MyShowProgressFunc := DefShowProgressFunc(UP, BytesWritten, TotalBytes);
  end;

  function MyCompressSuccessFunc(LP : LzhPtr;
                                 LH : LzhHeader) : Boolean;
    {-My CompressSuccess function}
  var
    Time : LongInt;
  begin
    {Do default stuff}
    DefCompressSuccessFunc(LP, LH);

    {Note original and compressed size}
    OrigSize := LH.OrigSize;
    NewSize := LH.NewSize;

    {If building, restore file's original date/time stamp}
    if Building then begin
      Assign(F, SaveName);
      Reset(F);
      SetFTime(F, SavedTime);
      Close(F);
    end;
  end;

  function MyOkToCompressFunc(LP : LzhPtr; NewFile : PathStr;
                              LH : LzhHeader) : Boolean;
    {-Change file date/time stamp when building}
  begin
    {Always compress}
    MyOkToCompressFunc := True;

    {Note filename}
    SaveName := NewFile;

    {Note original time stamp}
    if Building then begin
      Assign(F, NewFile);
      Reset(F);
      GetFTime(F, SavedTime);
      if OkToSetTime(JustFileName(NewFile)) then
        SetFTime(F, TimeStamp);
      Close(F);
    end;
  end;

  procedure MyShowNameProc(UP : UnLzhPtr);
    {-Show nothing}
  begin
  end;

  procedure ReadTillInfo;
    {-Skip comments and whitespace until we get to [Info] section}
  var
    S : String;
  begin
    repeat
      ReadLn(FI, S);
      S := StUpcase(S);
      if Pos('[INFO]', S) <> 0 then
        Exit;
    until Eof(FI);
    Abort('No [Info] section');
  end;

  procedure ReadInfo;
    {-Read [Info] section}
  var
    S : String;
    S1 : String;
    Loc : Byte;
    Tag : String;
    Value : String;
    Index : Byte;
    Code : Word;
    I : Integer;
  begin
    repeat
      ReadLn(FI, S);
      S := StUpcase(S);

      {Exit if start of [FILES] section}
      Loc := Pos('[FILES]', S);
      if Loc <> 0 then
        Exit;

      {Process entry}
      if (S[1] <> ';') and (S[1] <> ' ') and (Length(S) <> 0) then begin
        Loc := Pos('=', S);
        if Loc = 0 then
          Abort('Info section format error');
        Tag := Trim(Copy(S, 1, Loc-1));
        Value := Trim(Copy(S, Loc+1, 255));
        if Copy(Tag, 1, 6) = 'SRCDIR' then begin
          S1 := Copy(Tag, 7, 1);
          Val(S1, Index, Code);
          if Code <> 0 then
            Abort('SRCDIR format error');
          if Index > 9 then
            Abort('Max source directories is 9');
          SourceDirs[Index] := Value;
        end;
        if Copy(Tag, 1, 6) = 'DSTDIR' then begin
          S1 := Copy(Tag, 7, 1);
          Val(S1, Index, Code);
          if Code <> 0 then
            Abort('DSTDIR format error');
          if Index > 9 then
            Abort('Max destination directories is 9');
          I := 1;
          while (I < 255) and (Value[I] <> ' ') do
            Inc(I);
          DestDirs[Index] := Copy(Value, 1, I);
        end;
      end;
    until Eof(FI);

    {If we get here we never found a files section}
    Abort('No [Info] section');
  end;

  procedure WriteHelp;
    {-Write help and halt}
  begin
    WriteLn;
    WriteLn('Usage: ANALYZE [options]');
    WriteLn('  /F FileInfoName    [default = FILEINFO.TXT]');
    WriteLn('  /O OutFileName     [default = INSTALL.DAT]');
    WriteLn('  /R RptFileName     [default = INSTALL.RPT]');
    WriteLn('  /D DestDir         Root build directory [default = C:\]');
    WriteLn('  /V nnn             Version number (treated as n.nn)');
    WriteLn('  /X ExcludeFileName Name of file with exclude list');
    WriteLn('  /A                 Perform analyze pass');
    WriteLn('  /3                 Perform 360K disk build');
    WriteLn('  /7                 Perform 720K disk build');
    WriteLn('  /2                 Perform 1200K disk build');
    WriteLn('  /4                 Perform 1440K disk build');
    WriteLn('  /B                 Generate backup COPY.BAT batch file');
    Halt(1);
  end;

  procedure ParseCommandLine;
    {-Gets command line options and sets various parameters.}
  var
    Code : Word;
    Param : String;
    Cnt : Word;
  begin
    if ParamCount < 1 then
      WriteHelp;
    Param := ParamStr(1);
    Cnt := 2;

    while True do begin
      case Param[1] of
        '/', '-' :
          if Length(Param) <> 2 then
            Abort('Error: invalid parameter: '+Param)
          else
            case Upcase(Param[2]) of
              '?' : {Request for help}
                WriteHelp;
              'A' :
                AnalyzePass := True;
              '3' :
                Build36Pass := True;
              '7' :
                Build72Pass := True;
              '2' :
                Build12Pass := True;
              '4' :
                Build14Pass := True;
              'B' :
                GenerateBat := True;
              'X' :
                begin
                  ExcludeName := ParamStr(Cnt);
                  Inc(Cnt);
                end;
              'V' :
                begin
                  Val(ParamStr(Cnt), Version, Code);
                  if Code <> 0 then
                    Version := 100;
                  Inc(Cnt);
                end;
              'D' :
                begin
                  RootDir := ParamStr(Cnt);
                  Inc(Cnt);
                  if not IsDirectory(RootDir) then
                    Abort('Invalid directory: '+Param);
                end;
              'F' :
                begin
                  InfoFileName := ParamStr(Cnt);
                  Inc(Cnt);
                end;
              'O' :
                begin
                  OutName := ParamStr(Cnt);
                  Inc(Cnt);
                end;
              'R' :
                begin
                  RptName := ParamStr(Cnt);
                  Inc(Cnt);
                end;
              '?' :
                WriteHelp;
              else
                Abort('Error: invalid parameter: '+Param);
            end;
      end;

      {Get next parameter}
      if Cnt > ParamCount then
        Exit;
      Param := ParamStr(Cnt);
      Inc(Cnt);
    end;
  end;

  procedure OpenFiles;
  begin
    {Open the Map output file}
    Assign(Map, OutName);
    Rewrite(Map);

    {Open the Rpt output file}
    Assign(Rpt, RptName);
    ReWrite(Rpt);
    WriteLn(Rpt, 'Report created by ANALYZE pass on ', TodayString);
    WriteLn(Rpt, '========================================');
    WriteLn(Rpt);
    WriteLn(Rpt, '        Name    Size   Comp Src Dst    Group D36 D72 D12 D14  Version');
    WriteLn(Rpt, '        ----    ----   ---- --- ---    ----- --- --- --- ---  -------');

    {Read the FileInfo file and build an FML of all files}
    Assign(FI, InfoFileName);
    Reset(FI);
  end;

  procedure CheckFiles;
    {-Make sure all files exist}
  var
    S : String;
    FName : String;
    SrcDir : Byte;
    Code : Integer;
  begin
    WriteLn('Checking files in ', InfoFileName);

    {First read the srcdir tags}
    ReadTillInfo;
    ReadInfo;

    {Read [Files] section}
    repeat
      {Read in the file info record}
      ReadLn(FI, S);
      if Eof(FI) then
        Exit;

      {Skip comments and blank lines}
      if (S[1] <> ';') and (S[1] <> ' ') and (S <> '') then begin
        {Parse name}
        FName := Trim(Copy(S, 1, 12));
        {Parse source directory}
        Val(Copy(S, 15, 1), SrcDir, Code);
        if (Code <> 0) or (SrcDir > 9) then
          Abort('Invalid source directory');
        FName := AddBackSlash(SourceDirs[SrcDir]) + FName;

        {Assure file exists}
        if not ExistFile(FName) then
          Abort(FName + ' does not exist');
      end;
    until Eof(FI);
  end;

  procedure AnalyzeFiles;
  var
    S, S1 : String;
    FName : String;
    SrcDir : Byte;
    DstDir : Byte;
    Grp : LongInt;
    Arc : Boolean;
    Dsk1 : Boolean;
    Code : Word;
    FML  : FileMaskList;
    MapRec : FileInfoRec;
    Ver : Longint;
    Ext : ExtStr;
  begin
    Building := False;

    WriteLn('Analyzing files in ', InfoFileName);

    {First read the srcdir tags}
    ReadTillInfo;
    ReadInfo;

    {Read [Files] section}
    repeat
      {Read in the file info record}
      ReadLn(FI, S);

      {Skip comments and blank lines}
      if (S[1] <> ';') and (S[1] <> ' ') and (S <> '') then begin
        {Parse name}
        FName := Trim(Copy(S, 1, 12));
        {Parse source directory}
        Val(Copy(S, 15, 1), SrcDir, Code);
        if (Code <> 0) or (SrcDir > 9) then
          Abort('Invalid source directory');
        FName := AddBackSlash(SourceDirs[SrcDir]) + FName;

        {Parse destination directory}
        Val(Copy(S, 22, 1), DstDir, Code);
        if (Code <> 0) or (DstDir > 9) then
          Abort('Invalid destination directory');

        {Parse group}
        S1 := '$' + Copy(S, 29, 4);
        Val(S1, Grp, Code);
        if Code <> 0 then
          Abort('Group must be a hex mask');

        {Parse archive flag}
        Arc := S[34] = '1';

        {Parse disk1 flag}
        Dsk1 := S[38] = '1';

        {If file is EXE or DLL, get FileVersionMS longint}
        Ext := StUpcase(JustExtension(FName));
        if (Ext = 'EXE') or (Ext = 'DLL') then
          Ver := GetFileVerMS(FName)
        else
          Ver := 0;

        {Open/create LZH file}
        Assign(F, TempLzh);
        Erase(F);
        if IOResult <> 0 then ;
        L.Create(TempLzh);
        if ArchiveStatus <> ecOk then
          Abort('Failed to create archive');

        {set standard options}
        L.SetShowMethodProc(DefShowMethodProc);
        L.SetShowProgressFunc(MyShowProgressFunc);
        L.SetShowNameProc(MyShowNameProc);

        {set compressing-only options}
        L.SetOkToCompressFunc(MyOkToCompressFunc);
        L.SetCompressSuccessFunc(MyCompressSuccessFunc);
        L.SetProgressWidth(40);

        {Do it}
        L.Compress(FName);

        {Report results}
        case ArchiveStatus of
          ecOk : ;
          ecFileNotFound :
            Abort(CRLF+'File ' + FName + ' not found. Aborting.');
          else
            Abort(CRLF+'Got error ' + Long2Str(ArchiveStatus) + ' while compressing ' + FName + '. Aborting.');
        end;

        {Delete temp archive}
        L.Done;
        Assign(F, TempLzh);
        Erase(F);
        if IOResult <> 0 then ;

        {Write a Map entry}
        with MapRec do begin
          Name := JustFileName(FName);
          SourceDir := SrcDir;
          DestDir := DstDir;
          Group := Grp;
          Size := OrigSize;
          CompSize := NewSize;
          Archived := Arc;
          Disk1 := Dsk1;
          Disk36 := 0;
          Disk72 := 0;
          Disk12 := 0;
          Disk14 := 0;
          Tagged := False;
          FileVer := Ver;
          Write(Map, MapRec);

          {Write a report entry}
          WriteLn(Rpt, JustFileName(FName):12, '  ',
                       Size:6, ' ',
                       CompSize:6, ' ',
                       SourceDir:3, ' ',
                       DestDir:3, ' ',
                       HexL(Group):8, ' ',
                       Disk36:2, ' ',
                       Disk72:3, ' ',
                       Disk12:3, ' ',
                       Disk14:3, ' ',
                       DW(Ver).H:5, '/',
                       DW(Ver).L:3);
         end;
      end;
    until Eof(FI);
  end;

  procedure DeleteAllFiles;
    {-Delete all files in the current directory}
  var
    SRec : SearchRec;

    procedure EraseFile(FName : DirStr);
    begin
      Assign(F, FName);
      Erase(F);
      if IOResult <> 0 then ;
    end;

  begin
    {Scan all the files in the current directory}
    FindFirst('*.*', AnyFile, SRec);
    if DosError = 0 then
      repeat
        with SRec do
          if Attr and (Directory+Volumeid) = 0 then
            EraseFile(Name);
        FindNext(SRec);
      until DosError <> 0;
  end;

  function FileIsDup(Name : String) : Boolean;
    {-Return True if Name is duplicate, else add Name to list}
  begin
    {Special check for empty list}
    if CheckDups.fmlHead = nil then begin
      FileIsDup := False;
      CheckDups.Append(Name);
    end else if CheckDups.Match(Name) then
      FileIsDup := True
    else begin
      FileIsDup := False;
      CheckDups.Append(Name);
    end;
  end;

  procedure BuildImages(Media : MediaType);
  var
    Index : Word;
    CurSize : LongInt;
    CurDisk : Byte;
    Src : PathStr;
    FML : FileMaskList;
    RequiredSpace : LongInt;
    DiskLimit : LongInt;
    DupFile : Text;


    function DirName : String;
      {-Return a dirname like DISK1_3 or DISK1_5}
    begin
      case Media of
        Disk360  : DirName := 'DISK' + Long2Str(CurDisk) + '_36';
        Disk720  : DirName := 'DISK' + Long2Str(CurDisk) + '_72';
        Disk1200 : DirName := 'DISK' + Long2Str(CurDisk) + '_12';
        Disk1440 : DirName := 'DISK' + Long2Str(CurDisk) + '_14';
      end;
    end;

    procedure CollectNotArchived;
      {-Find all files that aren't archived}
    var
      I : Word;
      Src : PathStr;
      Dst : PathStr;
    begin
      for I := 1 to FileCnt do
        with Files^[I] do begin
          if not Archived then begin
            Tagged := True;
            Src := AddBackSlash(SourceDirs[SourceDir]) + Name;
            Dst := AddBackSlash(RootDir) + DirName + '\' + Name;
            CopyFile(Src, Dst, @CopyBuffer, SizeOf(CopyBuffer));
            Inc(CurSize, Size);
            case Media of
              Disk360  : Disk36 := 1;
              Disk720  : Disk72 := 1;
              Disk1200 : Disk12 := 1;
              Disk1440 : Disk14 := 1;
            end;

            if OkToSetTime(Name) then begin
              {Update timestamp of destination file}
              Assign(F, Dst);
              Reset(F);
              SetFTime(F, TimeStamp);
              Close(F);
            end;
          end;
        end;
    end;

    procedure CollectDisk1;
      {-Find all files that must be archived onto DISK1}
    var
      I : Word;
      InstallFML : FileMaskList;
      L : Lzh;
    begin
      InstallFML.Init;

      for I := 1 to FileCnt do
        with Files^[I] do begin
          if Disk1 and Archived then begin
            Tagged := True;
            Inc(CurSize, CompSize);
            case Media of
              Disk360  : Disk36 := 1;
              Disk720  : Disk72 := 1;
              Disk1200 : Disk12 := 1;
              Disk1440 : Disk14 := 1;
            end;
            Src := AddBackSlash(SourceDirs[SourceDir]) + Name;
            if not InstallFML.Append(Src) then
              Abort('Out of memory');
          end;
        end;

      {Make a special INSTALL.LZH archive}
      LzhName := AddBackSlash(RootDir) + DirName + '\' + 'INSTALL.LZH';
      L.Create(LzhName);
      if ArchiveStatus <> 0 then
        Abort('Failed to create new archive');

      {Set standard options}
      L.SetShowMethodProc(DefShowMethodProc);
      L.SetShowProgressFunc(DefShowProgressFunc);
      L.SetShowNameProc(DefShowNameProc);

      {Set compressing-only options}
      L.SetOkToCompressFunc(MyOkToCompressFunc);
      L.SetCompressSuccessFunc(MyCompressSuccessFunc);

      {Don't store path info}
      L.arOptionsOn(arStripPath);

      {Compress all the files}
      L.CompressFileMaskList(InstallFML);

      {Report errors}
      case ArchiveStatus of
        ecOK : ;
        ecFileNotFound :
          Abort(CRLF+'File not found. Aborting.');
        else
          Abort(CRLF+'Got error ' + Long2Str(ArchiveStatus) + ' while compressing. Aborting.');
      end;

      {Close the archive}
      L.Done;
      InstallFML.Done;
    end;

    function FindBestFile(var Index : Word) : Boolean;
      {-Find largest file that fits}
    var
      I : Word;
      FreeSpace : LongInt;
      BestSize : LongInt;
    begin
      BestSize := -1;
      FreeSpace := DiskLimit - CurSize;
      for I := 1 to FileCnt do
        with Files^[I] do begin
          if (not Tagged) and (CompSize < FreeSpace) then
            {It will fit, see if it's the best fit yet}
            if CompSize > BestSize then begin
              {Save new best fit}
              BestSize := CompSize;
              Index := I;
            end;
        end;
      FindBestFile := BestSize <> -1;
    end;

    function NoMoreFiles : Boolean;
    var
      I : Word;
    begin
      NoMoreFiles := False;
      for I := 1 to FileCnt do
        if not Files^[I].Tagged then
          Exit;
      NoMoreFiles := True;
    end;

  begin
    Building := True;

    {Assure we're starting in the original directory}
    ChDir(OrigDir);
    if IoResult <> 0 then ;

    {Show progress}
    WriteLn;
    case Media of
      Disk360  : Writeln('Building 5.25 360K images');
      Disk720  : Writeln('Building 3.50 720K images');
      Disk1200 : Writeln('Building 5.25 1200K images');
      Disk1440 : Writeln('Building 5.25 1440K images');
    end;

    {Make the destination root directory}
    MkDir(RootDir);
    if IOResult <> 0 then ;

    {Read in install.dat}
    Assign(Map, OutName);
    Reset(Map);
    Index := 1;
    repeat
      Read(Map, Files^[Index]);
      Files^[Index].Tagged := False;
      Inc(Index);
    until Eof(Map) or (Index > MaxFiles);
    if Index > MaxFiles then
      Abort('Too many files to build');
    FileCnt := Index-1;

    {Set disk limit}
    case Media of
      Disk360  : DiskLimit := Disk360Limit;
      Disk720  : DiskLimit := Disk720Limit;
      Disk1200 : DiskLimit := Disk1200Limit;
      Disk1440 : DiskLimit := Disk1440Limit;
    end;

    {Keep track of file names to avoid arc'ing duplicate files}
    CheckDups.Init;

    {Collect and archive files until disk1 is full}
    CurDisk := 1;
    repeat
      {Make the subdirectory for the disk image}
      MkDir(AddBackSlash(RootDir) + DirName);
      if IOResult <> 0 then ;
      ChDir(AddBackSlash(RootDir) + DirName);
      if IOResult <> 0 then
        Abort('Failed to locate image directory');

      {Delete all existing files in this directory}
      DeleteAllFiles;

      {Make a disk name file in this directory}
      Assign(F, AddBackSlash(RootDir) + DirName + '\' + DirName);
      ReWrite(F);
      WriteLn(F);
      Close(F);
      Reset(F);
      SetFTime(F, TimeStamp);
      Close(F);

      {No files yet}
      CurSize := 0;

      {Make a file mask list}
      FML.Init;

      {Collect all the files that must go on disk1}
      if CurDisk = 1 then begin
        {Guess room for INSTALL.DAT}
        Inc(CurSize, 10000);

        {Collect all non-archived files}
        CollectNotArchived;

        {Collect the archived files that must go on disk 1}
        CollectDisk1;
      end;

      {Collect files until disk is full}
      while (CurSize < DiskLimit) and not NoMoreFiles do begin
        if FindBestFile(Index) then begin
          with Files^[Index] do begin
            Tagged := True;
            Inc(CurSize, CompSize);
            case Media of
              Disk360  : Disk36 := CurDisk;
              Disk720  : Disk72 := CurDisk;
              Disk1200 : Disk12 := CurDisk;
              Disk1440 : Disk14 := CurDisk;
            end;

            if not FileIsDup(Name) then begin
              Src := AddBackSlash(SourceDirs[SourceDir]) + Name;
              if not FML.Append(Src) then
                Abort('Out of memory');
            end;
          end;
        end else
          {No more files fit, force exit}
          CurSize := 9999999;
      end;

      {Create a new archive for this disk}
      LzhName := AddBackSlash(RootDir) + DirName + '\' + 'FILES' + Long2Str(CurDisk) + '.LZH';
      L.Create(LzhName);
      if ArchiveStatus <> 0 then
        Abort('Failed to create new archive');

      {Set standard options}
      L.SetShowMethodProc(DefShowMethodProc);
      L.SetShowProgressFunc(DefShowProgressFunc);
      L.SetShowNameProc(DefShowNameProc);

      {Set compressing-only options}
      L.SetOkToCompressFunc(MyOkToCompressFunc);
      L.SetCompressSuccessFunc(MyCompressSuccessFunc);

      {Don't store path info}
      L.arOptionsOn(arStripPath);

      {Compress all the files}
      L.CompressFileMaskList(FML);

      {Report errors}
      case ArchiveStatus of
        ecOK : ;
        ecFileNotFound :
          Abort(CRLF+'File not found. Aborting.');
        else
          Abort(CRLF+'Got error ' + Long2Str(ArchiveStatus) + ' while compressing. Aborting.');
      end;

      {Close the archive}
      L.Done;
      FML.Done;
      Inc(CurDisk);

      {Set the timestamp of the archive}
      Assign(F, LzhName);
      Reset(F);
      SetFTime(F, TimeStamp);
      Close(F);
    until NoMoreFiles;

  end;

  procedure BuildInstallDat;
    {-Write a new INSTALL.DAT file}
  var
    I : Word;
  begin
    Assign(Map, OutName);
    Rewrite(Map);

    Assign(Rpt, RptName);
    Rewrite(Rpt);
    WriteLn(Rpt, 'Report created by BUILD pass on ', TodayString);
    WriteLn(Rpt, '========================================');
    WriteLn(Rpt);
    WriteLn(Rpt, '        Name    Size   Comp Src Dst    Group D36 D72 D12 D14  Version');
    WriteLn(Rpt, '        ----    ----   ---- --- ---    ----- --- --- --- ---  -------');

    for I := 1 to FileCnt do begin
      Write(Map, Files^[I]);

      {Write a report entry}
      with Files^[I] do
        {Write a report entry}
        WriteLn(Rpt, Name:12, '  ',
                     Size:6, ' ',
                     CompSize:6, ' ',
                     SourceDir:3, ' ',
                     DestDir:3, ' ',
                     HexL(Group):8, ' ',
                     Disk36:2, ' ',
                     Disk72:3, ' ',
                     Disk12:3, ' ',
                     Disk14:3, ' ',
                     DW(FileVer).H:5, '/',
                     DW(FileVer).L:3);
    end;

    Close(Map);
    Close(Rpt);

    if IOResult <> 0 then
      Abort('Failed to update INSTALL.DAT');
  end;

  procedure UpdateInstallLzh(Media : MediaType);
    {-Freshen INSTALL.DAT in INSTALL.LZH}
  var
    FML : FileMaskList;

    function DirName : String;
      {-Return a dirname like DISK1_3 or DISK1_5}
    begin
      case Media of
        Disk360  : DirName := 'DISK1_36';
        Disk720  : DirName := 'DISK1_72';
        Disk1200 : DirName := 'DISK1_12';
        Disk1440 : DirName := 'DISK1_14';
      end;
    end;

  begin
    WriteLn;

    FML.Init;
    if not FML.Append('INSTALL.DAT') then
      Abort('Out of memory');

    {Open the INSTALL.LZH archive}
    LzhName := AddBackSlash(RootDir) + DirName + '\' + 'INSTALL.LZH';
    L.Init(LzhName);
    if ArchiveStatus <> 0 then
      Abort('Failed to open archive');

    {Set standard options}
    L.SetShowMethodProc(DefShowMethodProc);
    L.SetShowProgressFunc(DefShowProgressFunc);
    L.SetShowNameProc(DefShowNameProc);

    {Set compressing-only options}
    L.SetOkToCompressFunc(DefOkToCompressFunc);
    L.SetCompressSuccessFunc(DefCompressSuccessFunc);

    {Don't store path info}
    L.arOptionsOn(arStripPath);

    {Compress all the files}
    L.CompressFileMaskList(FML);

    {Close the archive}
    L.Done;
    FML.Done;

    {Set the timestamp of the archive}
    Assign(F, LzhName);
    Reset(F);
    SetFTime(F, TimeStamp);
    Close(F);
  end;

  procedure SetDateTimeStamp;
    {-Set the global variable TimeStamp to the new date/time}
  var
    Current : DateTime;
    Junk : Word;
  begin
    with Current do begin
      GetDate(Year, Month, Day, Junk);
      Hour := Version div 100;
      Min := Version mod 100;
      Sec := 0;
    end;
    PackTime(Current, TimeStamp);
  end;

  function Less(X, Y : FileInfoRec) : Boolean;
    {-Return True if X is less than Y}
  var
    XDisk, YDisk : Byte;
  begin
    {Get the right media type}
    case SortMedia of
      Disk360 : begin
                  XDisk := X.Disk36;
                  YDisk := Y.Disk36;
                end;
      Disk720 : begin
                  XDisk := X.Disk72;
                  YDisk := Y.Disk72;
                end;
      Disk1200 : begin
                  XDisk := X.Disk12;
                  YDisk := Y.Disk12;
                end;
      Disk1440 : begin
                  XDisk := X.Disk14;
                  YDisk := Y.Disk14;
                end;
    end;

    Less := (XDisk < YDisk) or
            ((XDisk = YDisk) and (X.DestDir < Y.DestDir)) or
            ((XDisk = YDisk) and (X.DestDir = Y.DestDir) and (X.Name < Y.Name));
  end;

  procedure ExchangeFiles(var X, Y : FileInfoRec);
  var
    Temp : FileInfoRec;
  begin
    Move(X, Temp, SizeOf(X));
    Move(Y, X, SizeOf(X));
    Move(Temp, Y, SizeOf(X));
  end;

  {$S+}
  procedure QuickSort(L, R : Word);
    {-General recursive quicksort}
  var
    I : Word;
    J : Word;
  begin
    I := L;
    J := R;

    {Save pivot}
    Pivot := Files^[(L+R) shr 1];
    repeat
      while Less(Files^[I], Pivot) do
        Inc(I);
      while Less(Pivot, Files^[J]) do
        Dec(J);
      if I <= J then begin
        {Swap elements I and J}
        ExchangeFiles(Files^[I], Files^[J]);
        Inc(I);
        Dec(J);
      end;
    until I > J;
    if L < J then
      QuickSort(L, J);
    if I < R then
      QuickSort(I, R);
  end;
  {$S-}

  procedure CreateCopyBat(Media : MediaType);
    {-Create COPY.BAT, a backup method for installing files (assumes 1.2)}
  var
    I, J : Word;
    Copy : Text;
    CurDisk : Byte;
    CurDir : Byte;
    Disk : Byte;
    S : String;
    FName : String;

    procedure EmitLine;
      {-Emit an lharc line}
    begin
      if J <> 0 then begin
        WriteLn(Copy, 'lharc x %1:\FILES', CurDisk, ' ', S);
        S := '';
        J := 0;
      end;
    end;

  begin
    {Open INSTALL.DAT}
    Assign(Map, 'INSTALL.DAT');
    Reset(Map);
    if IoResult <> 0 then
      Abort('Error while creating COPY.BAT, failed to open INSTALL.DAT');

    {Read data from install.dat to Files^}
    if Files = nil then
      Abort('Out of memory');
    I := 0;
    repeat
      Inc(I);
      Read(Map, Files^[I]);
    until Eof(Map);
    FileCnt := I;

    {Sort by diskette/destdir/name for requested media}
    SortMedia := Media;
    QuickSort(1, FileCnt);

    {Process sorted Files^, creating batch file}
    case Media of
      Disk360  : FName := 'COPY36.BAT';
      Disk720  : FName := 'COPY72.BAT';
      Disk1200 : FName := 'COPY12.BAT';
      Disk1440 : FName := 'COPY14.BAT';
    end;

    Assign(Copy, FName);
    Rewrite(Copy);
    WriteLn(Copy, 'rem  This batch file can be used to install all distributed files if,');
    WriteLn(Copy, 'rem  for some reason, you don''t want to use the INSTALL program. Before');
    WriteLn(Copy, 'rem  running this batch file you''ll need to manually copy LHARC.EXE and');
    WriteLn(Copy, 'rem  this batch file from our first distribution diskette to a directory');
    WriteLn(Copy, 'rem  on your DOS path.');
    WriteLn(Copy, 'rem');
    WriteLn(Copy, 'rem  %1 is "from" drive (a or b)');
    WriteLn(Copy, 'rem  %2 is "to" drive (c, d, etc.)');
    WriteLn(Copy);
    WriteLn(Copy, '%2:');
    WriteLn(Copy);

    CurDisk := 1;
    CurDir := 99;
    J := 0;
    S := '';
    for I := 1 to FileCnt do begin
      with Files^[I] do begin
        if DestDir <> 0 then begin

          {Check for diskette change}
          case Media of
            Disk360  : Disk := Disk36;
            Disk720  : Disk := Disk72;
            Disk1200 : Disk := Disk12;
            Disk1440 : Disk := Disk14;
          end;
          if Disk <> CurDisk then begin
            EmitLine;
            CurDisk := Disk;
            WriteLn(Copy);
            WriteLn(Copy, 'pause Insert diskette ', Disk);
            WriteLn(Copy);
          end;

          {Check for destdir change}
          if CurDir <> DestDir then begin
            EmitLine;
            CurDir := DestDir;
            WriteLn(Copy);
            WriteLn(Copy, 'md ', DestDirs[DestDir]);
            WriteLn(Copy, 'cd ', DestDirs[DestDir]);
          end;

          if Archived then begin
            {Append new file name}
            S := S + ' ' + Name;
            Inc(J);
          end else
            {Issue copy command now}
            WriteLn(Copy, 'copy %1:\', Name);

          {Emit line every 5 file names}
          if J >= 5 then
            EmitLine;
        end;
      end;
    end;
    EmitLine;
    Close(Copy);
  end;

procedure ReadExcludeList;
  {-Read in a list of files to exclude from date/time stamping}
var
  T : Text;
  S : String;
  Result : Word;
begin
  Assign(T, ExcludeName);
  Reset(T);
  Result := IoResult;
  if Result <> 0 then
    Abort('Failed to open exclude file, got DOS error: ' + Long2Str(Result));

  ExcludeCnt := 0;
  repeat
    ReadLn(T, S);
    if Length(S) > 12 then
      Abort('File name in exclude list too long (limit 12 chars)');
    if IoResult <> 0 then
      Abort('I/O error reading exclude file');
    Inc(ExcludeCnt);
    ExcludeList[ExcludeCnt] := StUpcase(S);
  until Eof(T);

  Close(T);
  if IoResult <> 0 then ;
end;

begin
  {Display copyright and stuff}
  WriteLn(PgmVersion);
  WriteLn;

  {Get command line options}
  ParseCommandLine;

  {Get exclude list if an exclude file was specified}
  if ExcludeName <> '' then
    ReadExcludeList;

  {Analyze files if requested}
  if AnalyzePass then begin
    {Open files}
    OpenFiles;

    {Make sure all files are where they are supposed to by}
    CheckFiles;

    {Close files}
    Close(FI);
    Close(Map);
    Close(Rpt);

    {Open files}
    OpenFiles;

    {Process file info}
    AnalyzeFiles;

    {Close files}
    Close(FI);
    Close(Map);
    Close(Rpt);
  end;

  {Build images if requested}
  if Build36Pass or Build72Pass or Build12Pass or Build14Pass then begin
    if not AnalyzePass then begin
      Assign(FI, InfoFileName);
      Reset(FI);
      ReadTillInfo;
      ReadInfo;
      Close(FI);
    end;

    {Allocate a buffer for files}
    GetMem(Files, SizeOf(FileInfoRec)*MaxFiles);

    {Note original directory}
    GetDir(0, OrigDir);

    {Note new date/time stamp for files}
    SetDateTimeStamp;

    {Build the 360K images}
    if Build36Pass then begin
      BuildImages(Disk360);
      ChDir(OrigDir);
      BuildInstallDat;
      UpdateInstallLzh(Disk360);
    end;

    {Build the 720K images}
    if Build72Pass then begin
      BuildImages(Disk720);
      ChDir(OrigDir);
      BuildInstallDat;
      UpdateInstallLzh(Disk720);
    end;

    {Build the 1200K images}
    if Build12Pass then begin
      BuildImages(Disk1200);
      ChDir(OrigDir);
      BuildInstallDat;
      UpdateInstallLzh(Disk1200);
    end;

    {Build the 1440K images}
    if Build14Pass then begin
      BuildImages(Disk1440);
      ChDir(OrigDir);
      BuildInstallDat;
      UpdateInstallLzh(Disk1440);
    end;

    {Build COPY??.BAT if requested}
    if GenerateBat then begin
      if Build36Pass then
        CreateCopyBat(Disk360);
      if Build72Pass then
        CreateCopyBat(Disk720);
      if Build12Pass then
        CreateCopyBat(Disk1200);
      if Build14Pass then
        CreateCopyBat(Disk1440);
    end;

    {Finished with buffer}
    FreeMem(Files, SizeOf(FileInfoRec)*MaxFiles);
  end;

end.
