(*----------------------------------------------------------------------*)
(*    Color_Screen_Active --- Determine if color or mono screen         *)
(*----------------------------------------------------------------------*)

FUNCTION Color_Screen_Active : BOOLEAN;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Function:   Color_Screen_Active                                  *)
(*                                                                      *)
(*     Purpose:    Determines if color or mono screen active            *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Color_Active := Color_Screen_Active : BOOLEAN;                *)
(*                                                                      *)
(*           Color_Active --- set to TRUE if the color screen is        *)
(*                            active, FALSE if the mono screen is       *)
(*                            active.                                   *)
(*                                                                      *)
(*     Calls:   INTR                                                    *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

VAR
   Regs : Registers;

BEGIN  (* Color_Screen_Active *)

   Regs.Ax := 15 SHL 8;

   INTR( $10 , Regs );

   Color_Screen_Active := ( Regs.Al <> 7 );

END    (* Color_Screen_Active *);

(*----------------------------------------------------------------------*)
(*     Current_Video_Mode --- Determine current video mode setting      *)
(*----------------------------------------------------------------------*)

FUNCTION Current_Video_Mode: INTEGER;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Function:   Current_Video_Mode                                   *)
(*                                                                      *)
(*     Purpose:    Gets current video mode setting from system          *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Current_Mode := Current_Video_Mode : INTEGER;                 *)
(*                                                                      *)
(*           Current_Mode --- set to integer representing current       *)
(*                            video mode inherited from system.         *)
(*                                                                      *)
(*     Calls:   INTR                                                    *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

VAR
   Regs : Registers;

BEGIN  (* Current_Video_Mode *)

   Regs.Ax := 15 SHL 8;

   INTR( $10 , Regs );

   Current_Video_Mode := Regs.Al;

END    (* Current_Video_Mode *);

(*----------------------------------------------------------------------*)
(*        Get_Screen_Address --- Get address of current screen          *)
(*----------------------------------------------------------------------*)

PROCEDURE Get_Screen_Address( VAR Actual_Screen : Screen_Ptr );

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Procedure:  Get_Screen_Address                                   *)
(*                                                                      *)
(*     Purpose:    Gets screen address for current type of display      *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Get_Screen_Address( VAR Actual_Screen : Screen_Ptr );         *)
(*                                                                      *)
(*           Actual_Screen --- pointer whose value receives the         *)
(*                             current screen address.                  *)
(*                                                                      *)
(*     Calls:   Color_Screen_Active                                     *)
(*              PTR                                                     *)
(*              Get_Virtual_Screen_Address                              *)
(*              TimeSharingActive                                       *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

VAR
   Regs: Registers;

BEGIN  (* Get_Screen_Address *)
                                   (* Check if timesharing active.   *)
   IF TimeSharingActive THEN
      CASE MultiTasker OF
         DoubleDos:  BEGIN
                        Regs.Ax := $EC00;
                        MsDos( Regs );
                        Actual_Screen := PTR( Regs.Es, 0 );
                     END;
         TaskView,
         TopView,
         MSWindows,
         DesqView:   CASE Current_Video_Mode OF
                        HiRes_GraphMode : Actual_Screen := PTR( Color_Screen_Address , 0 );
                        EGA_GraphMode   : Actual_Screen := PTR( EGA_Screen_Address , 0 );
                        ELSE              Actual_Screen := Virtual_Screen;
                     END (* CASE *);
         ELSE;
      END
   ELSE
      IF Color_Screen_Active THEN
         Actual_Screen := PTR( Color_Screen_Address , 0 )
      ELSE
         Actual_Screen := PTR( Mono_Screen_Address , 0 );

END    (* Get_Screen_Address *);
