{ This unit implements a TPhoneColl, which is a collection of TPhone
  objects. It has methods which allow one to view the collection via a list
  box, and to add, edit or delete records. A TPhone is simply a record
  containing a name and phone number, along with appropriate Load and Store
  methods. TPhone can easily be modified to accomodate additional data.      }

{$X+}
unit PhoneLst;

interface

uses
  Objects,Drivers,Views,Dialogs,App;

const
  cmAdd    = 104;  { Add a new TPhone }
  cmEdit   = 105;  { Edit the current TPhone }
  cmRemove = 106;  { Remove the current TPhone }

type
  NameStr = String[40];
  NumberStr = String[12];

  PPhone = ^TPhone;
  TPhone = object (TObject)
    Name: NameStr;
    Number: NumberStr;
    constructor Init (AName: NameStr; ANumber: NumberStr);
    constructor Load (var S: TStream);
    procedure Store (var S: TStream);
    end;

{ TPhoneListBox is a simple descendant of TListBox, with a specialized
  GetText method to display the TPhone objects in the list box.              }

  PPhoneListBox = ^TPhoneListBox;
  TPhoneListBox = object (TListBox)
    function GetText (Item: Integer; MaxLen: Integer): String; virtual;
    end;

{ TPhoneColl is implemented as a descendant of TCollection. It may be more
  appropriate to implement it as a descendant of a TPhoneCollection, so that
  the TPhones can be sorted in alphabetical order by name, for example. The
  Show method opens up a dialog box which allows viewing and editing of the
  TPhones.                                                                   }

  PPhoneColl = ^TPhoneColl;
  TPhoneColl = object (TCollection)
    function Show: Word;
    end;

{ TViewDialog is a descendant of TDialog which is used to display the TPhone
  information and allow for editing. PhoneColl points to the associated
  TPhoneColl object, and L points to the list box that displays the TPhones. }

  PViewDialog = ^TViewDialog;
  TViewDialog = object (TDialog)
    PhoneColl: PPhoneColl;
    L: PPhoneListBox;
    constructor Init (APhoneColl: PPhoneColl);
    procedure HandleEvent (var Event: TEvent); virtual;
    end;

{ The RegisterPhone procedure takes care of registering the newly defined
  object types so that they can be written to or read from a stream. Only
  those object types which are actually expected to be stored are
  registered.                                                                }

procedure RegisterPhone;

implementation

{ TPhone methods }

constructor TPhone.Init (AName: NameStr; ANumber: NumberStr);

begin
TObject.Init;
Name := AName;
Number := ANumber;
end;

constructor TPhone.Load (var S: TStream);

begin
S.Read (Name,SizeOf (NameStr));
S.Read (Number,SizeOf (NumberStr));
end;

procedure TPhone.Store (var S: TStream);

begin
S.Write (Name,SizeOf (NameStr));
S.Write (Number,SizeOf (NumberStr));
end;

{ TPhoneListBox methods }

{ TPhoneListBox.GetText returns a composite string containing both the name
  and number fields of the appropriate TPhone object.                        }

function TPhoneListBox.GetText (Item: Integer; MaxLen: Integer): String;

var
  S: String;

begin
S := '                                                     ';
Move (PPhone (List^.At (Item))^.Name[1],S[1],
  Length (PPhone (List^.At (Item))^.Name));
Move (PPhone (List^.At (Item))^.Number[1],S[42],
  Length (PPhone (List^.At (Item))^.Number));
GetText := S;
end;

{ TPhoneColl methods }

{ TPhoneColl.Show ExecViews a TViewDialog, and returns the result of that
  ExecView as its own result.                                                }

function TPhoneColl.Show: Word;

begin
Show := DeskTop^.ExecView (New (PViewDialog,Init (@Self)));
end;

{ ModifyRecord instantiates a dialog box which is used for adding a new
  TPhone record or editing an existing one. In the case of adding a record,
  the calling routine passes empty strings as the values of Name and Number
  in the Phone parameter; upon return, Phone contains the new values of Name
  and Number. In the case of editing, the calling routine passes the
  existing values of Name and Number in the Phone parameter, and they are
  replaced by the new values upon return. ModifyRecord returns a value equal
  to the result of ExecViewing the dialog; if the dialog was cancelled by
  the user, the Phone parameter is returned unaltered.                       }

function ModifyRecord (Phone: PPhone; Title: TTitleStr): Word;

var
  R: TRect;
  D: PDialog;
  N,P: PInputLine;
  Result: Word;

begin
R.Assign (27,11,73,21);
D := New (PDialog,Init (R,Title + ' a record'));
R.Assign (2,2,44,3);
N := New (PInputLine,Init (R,40));
N^.SetData (Phone^.Name);
D^.Insert (N);
R.Assign (2,1,44,2);
D^.Insert (New (PLabel,Init (R,'~N~ame',N)));
R.Assign (2,5,16,6);
P := New (PInputLine,Init (R,12));
P^.SetData (Phone^.Number);
D^.Insert (P);
R.Assign (2,4,44,5);
D^.Insert (New (PLabel,Init (R,'~P~hone',N)));
R.Assign (5,7,15,9);
D^.Insert (New (PButton,Init (R,'~O~K',cmOK,bfDefault)));
R.Assign (20,7,30,9);
D^.Insert (New (PButton,Init (R,'Cancel',cmCancel,bfNormal)));
D^.SelectNext (False);
Result := DeskTop^.ExecView (D);
if Result <> cmCancel then
  begin
  N^.GetData (Phone^.Name);
  P^.GetData (Phone^.Number);
  end;
Dispose (D,Done);
ModifyRecord := Result;
end;

{ TViewDialog methods }

{ TViewDialog.Init is a basic dialog box constructor; nothing fancy here.    }

constructor TViewDialog.Init (APhoneColl: PPhoneColl);

var
  R: TRect;
  SB: PScrollBar;

begin
R.Assign (10,5,70,16);
TDialog.Init (R,'Phone List');
PhoneColl := APhoneColl;
R.Assign (57,2,58,7);
SB := New (PScrollBar,Init (R));
Insert (SB);
R.Assign (2,2,57,7);
L := New (PPhoneListBox,Init (R,1,SB));
L^.NewList (PhoneColl);
Insert (L);
R.Assign (2,1,57,2);
Insert (New (PStaticText,Init (R,
  ' Name                                     Phone #')));
R.Assign (2,8,12,10);
Insert (New (PButton,Init (R,'~A~dd',cmAdd,bfNormal)));
R.Assign (13,8,23,10);
Insert (New (PButton,Init (R,'~E~dit',cmEdit,bfNormal)));
R.Assign (24,8,34,10);
Insert (New (PButton,Init (R,'~R~emove',cmRemove,bfNormal)));
R.Assign (37,8,47,10);
Insert (New (PButton,Init (R,'~S~ave',cmOK,bfDefault)));
R.Assign (48,8,58,10);
Insert (New (PButton,Init (R,'Cancel',cmCancel,bfNormal)));
SelectNext (False);
end;

{ TViewDialog.HandleEvent takes care of the special commands (cmAdd,cmEdit,
  and cmRemove) used by TViewDialog. It also updates the list box display as
  required, and disables cmEdit and cmRemove commands if the TPhoneColl is
  empty.                                                                     }

procedure TViewDialog.HandleEvent (var Event: TEvent);

var
  P: PPhone;

begin
TDialog.HandleEvent (Event);
if Event.What = evCommand then
  begin
  case Event.Command of
    cmAdd: begin
      P := New (PPhone,Init ('',''));
      if ModifyRecord (P,'Add') <> cmCancel then PhoneColl^.Insert (P)
      else Dispose (P,Done);
      end;
    cmEdit: ModifyRecord (PPhone (PhoneColl^.At (L^.Focused)),'Edit');
    cmRemove: PhoneColl^.AtDelete (L^.Focused);
    end;
  L^.SetRange (L^.List^.Count);
  L^.DrawView;
  end;
if PhoneColl^.Count >= 1 then EnableCommands ([cmRemove,cmEdit])
else DisableCommands ([cmRemove,cmEdit]);
end;

{ stream registration records }

const
  srPhone     = 10001;
  srPhoneColl = 10002;

  RPhone: TStreamRec = (
    ObjType: srPhone;
    VMTLink: Ofs (TypeOf (TPhone)^);
    Load: @TPhone.Load;
    Store: @TPhone.Store
  );

  RPhoneColl: TStreamRec = (
    ObjType: srPhoneColl;
    VMTLink: Ofs (TypeOf (TPhoneColl)^);
    Load: @TPhoneColl.Load;
    Store: @TPhoneColl.Store
  );

procedure RegisterPhone;

begin
RegisterType (RPhone);
RegisterType (RPhoneColl);
end;

end.
