{===========================================================================}
{                                                                           }
{ Calendar is a unit that defines a date display object, a calendar dis-    }
{ play object and a date selector object.                                   }
{                                                                           }
{===========================================================================}

                           { Optimize compiler. }

                      {$A+,B-,D-,F+,I-,L-,O+,R-,S-,V-}

                         { Allow extended syntax. }

                                    {$X+}

Unit Calendar;

Interface

Uses

 App,
 Dates,
 Views,
 Drivers,
 Dialogs,
 Objects;

Const

 cmDateChanged = 1451; { The cmDateChanged command is broadcast by the      }
                       { TDateInputView object whenever it's date is        }
                       { changed by the user.  This allows other objects    }
                       { dependent on this date to update themselves.  The  }
                       { value 1451 is pseudo-random (my date of birth is   }
                       { 1/4/51).  It is not likely to be selected as a     }
                       { command constant by another programmer, therefore, }
                       { the probability of conflict is minimized.          }

                       { The help context constants below allow the         }
                       { programmer to create context sensitive help for    }
                       { their respective views. The variable HelpCtx is    }
                       { set to these values in their respective views.     }

 hcMonthYearView    = cmDateChanged + 1;
 hcDeskCalendar     = cmDateChanged + 2;
 hcSetDateCalendar  = cmDateChanged + 3;
 hcDateInputView    = cmDateChanged + 4;

Type

 MagicBuffer       = Record
                      character,
                      Attribute:  byte;
                     end;

 BufferWords       = Record
                      Case Boolean of
                        TRUE:  (element:   MagicBuffer);
                       FALSE:  (compound:  word);
                     end;

 PDrawBuffer       = ^DrawBuffer;
 DrawBuffer        = Array[1..168] of BufferWords;

 HighLow           = Record
                      normal,
                      highlight:     byte;
                     end;

 MagicColor        = Record
                      Case Boolean of
                        TRUE:  (color:   HighLow);
                       FALSE:  (colors:  word);
                     end;

 DateInputString   = String[36];

 PGenericDateView  = ^TGenericDateView;
 TGenericDateView  = Object(TView)
                      The_Date:     Date;
                      Date_String:  DateInputString;
                      Date_Buffer:  TDrawBuffer;
                      Attribute:    MagicColor;
                      Right:        Boolean;
                      Constructor Init(Bounds: TRect);
                      Function    DataSize:  Word;                   Virtual;
                      Procedure   SetData(Var  Rec);                 Virtual;
                      Procedure   SetState(AState:  Word;
                                           Enable:  Boolean);        Virtual;
                      Procedure   GetData(Var  Rec);                 Virtual;
                      Function    GetPalette:  PPalette;             Virtual;
                      Procedure   Draw;                              Virtual;
                      Procedure   Update(ADate:  Date);              Virtual;
                     end;

 PDateView         = ^TDateView;
 TDateView         = Object(TGenericDateView)
                      Constructor Init(ARect:  TRect);
                      Function    GetPalette:  PPalette;  Virtual;
                     end;

 PMonthYearView    = ^TMonthYearView;
 TMonthYearView    = Object(TDateView)
                      Constructor Init(ARect:  TRect);
                      Procedure   Toggle;
                      Procedure   UnToggle;
                      Procedure   Draw;                              Virtual;
                      Procedure   HandleEvent(Var  Event:  TEvent);  Virtual;
                     end;

 PScrollCalendar   = ^TScrollCalendar;
 TScrollCalendar   = Object(TScroller)
                      The_Date:         Date;
                      calendar_buffer:  PDrawBuffer;
                      Attribute:        MagicColor;
                      date_offset,
                      buffer_offset:    byte;
                      Compare:          TPoint;
                      View_Only,
                      Get_Colors,
                      New_Pointer,
                      New_Calendar:     Boolean;
                      Constructor Init(ADate:        Date;
                                       AHScrollBar,
                                       AVScrollBar:  PScrollBar;
                                       View:         Boolean);
                      Procedure   Update_Pointer;
                      Procedure   Create_Calendar_Buffer;
                      Function    DataSize:  Word;                   Virtual;
                      Procedure   SetData(Var  Rec);                 Virtual;
                      Procedure   GetData(Var  Rec);                 Virtual;
                      Function    GetPalette:  PPalette;             Virtual;
                      Procedure   Draw;                              Virtual;
                      Procedure   HandleBroadcast;                   Virtual;
                      Procedure   HandleEvent(Var  Event:  TEvent);  Virtual;
                      Destructor  Done;                              Virtual;
                     end;

 PCalendarIcons    = ^TCalendarIcons;
 TCalendarIcons    = Object(TView)
                      View_Only:  Boolean;
                      Icons:      Array[0..19] of BufferWords;
                      Attribute:  MagicColor;
                      Constructor Init(ARect:  TRect;  AView_Only:  Boolean);
                      Function    GetPalette:  PPalette;             Virtual;
                      Procedure   Draw;                              Virtual;
                      Procedure   HandleEvent(Var  Event:  TEvent);  Virtual;
                      Destructor  Done;                              Virtual;
                     end;

 PCalendar         = ^TCalendar;
 TCalendar         = Object(TDialog)
                      Calendar:   PScrollCalendar;
                      Icons:      PCalendarIcons;
                      Constructor Init(ARect:       TRect;
                                       ADate:       Date;
                                       AView_Only:  Boolean);
                      Procedure   HandleEvent(Var  Event:  TEvent);  Virtual;
                      Destructor  Done;                              Virtual;
                     end;

 PDateViewIcon     = ^TDateViewIcon;
 TDateViewIcon     = Object(TView)
                      Icon:       Array[0..3] of BufferWords;
                      Attribute:  MagicColor;
                      Constructor Init(ARect:  TRect);
                      Function    GetPalette:  PPalette;             Virtual;
                      Procedure   Draw;                              Virtual;
                      Procedure   HandleEvent(Var  Event:  TEvent);  Virtual;
                      Destructor  Done;                              Virtual;
                     end;

 PDateInputView    = ^TDateInputView;
 TDateInputView    = Object(TGroup)
                      The_Date:  Date;
                      DateView:  PDateView;
                      Icon:      PDateViewIcon;
                      Constructor Init(ARect:  TRect);
                      Function    DataSize:  Word;                   Virtual;
                      Procedure   SetData(Var  Rec);                 Virtual;
                      Procedure   SetState(AState:  Word;
                                           Enable:  Boolean);        Virtual;
                      Procedure   GetData(Var  Rec);                 Virtual;
                      Procedure   Draw;                              Virtual;
                      Procedure   HandleEvent(Var  Event:  TEvent);  Virtual;
                     end;

 PJulianDays       = ^TJulianDays;
 TJulianDays       = Object(TStaticText)
                        Target_Date,
                        Reference_Date:  PDate;
                        Constructor Init(Var ABox:  TRect);
                        Procedure   Point_To_Target(ATarget:  PDate);
                        Procedure   Point_To_Reference(AReference:  PDate);
                        Procedure   Update;
                        Procedure   Draw;                              Virtual;
                        Procedure   HandleEvent(Var  Event:  TEvent);  Virtual;
                       end;

 PAgeDisplay       = ^TAgeDisplay;
 TAgeDisplay       = Object(TJulianDays)
                      Procedure   Draw;  Virtual;
                     end;

 PGestDisplay      = ^TGestDisplay;
 TGestDisplay      = Object(TJulianDays)
                      Use_EDC:         Boolean;
                      Constructor Init(Var ABox:  TRect);
                      Procedure   Set_Use_EDC(ABoolean:  Boolean);
                      Procedure   Draw;  Virtual;
                     end;

 PDiffDisplay      = ^TDiffDisplay;
 TDiffDisplay      = Object(TJulianDays)
                      Procedure   Draw;  Virtual;
                     end;

 PEDCDisplay       = ^TEDCDisplay;
 TEDCDisplay       = Object(TJulianDays)
                      Procedure   Draw;  Virtual;
                     end;

Procedure View_Calendar(ARect:  TRect;  ADate:  Date);
Procedure Pick_Date(ARect:  TRect;  Var ADate:  Date);
Function  Age_String(Reference_Date, Target_Date:  Date):  date_strings;
Function  Gestation_String(Reference_Date, Target_Date:  Date; Use_EDC:  Boolean):  date_strings;
Function  Difference_String(LMP_Date, Sonogram_Date:  Date):  date_strings;
Function  EDC_String(LMP_Date:  Date):  date_strings;

Implementation

Const

 HMin           =   0;
 HMax           = 104;
 HArPlu         =  15;
 HArMin         = -15;
 HPgPlu         =   1;
 HPgMin         =  -1;
 HCol           =   5;
 HOffset        =  HArPlu DIV 2;

 VMin           =   0;
 VMax           =  29;
 VArPlu         =   5;
 VArMin         =  -5;
 VPgPlu         =   1;
 VPgMin         =  -1;
 VCol           =   5;
 VOffset        =  VArPlu DIV 2;

 cmDecadeMinus  = 0;
 cmCenturyMinus = 1;
 cmCenturyPlus  = 2;
 cmDecadePlus   = 3;

 Eighteen       = '                  ';

Type

 ninety_three  = string[93];

Var

 calendar_string:   ^string;
 calendar_numbers:  ^ninety_three;
 count:             byte;
 number:            string[3];
 Local_Box:         TRect;

{===========================================================================}
{                                                                           }
{ View_Calendar is a stand-alone procedure that generates a desk calendar   }
{ type object that allows the user to view a calendar for any month in any  }
{ year.                                                                     }
{                                                                           }
{===========================================================================}

Procedure View_Calendar(ARect:  TRect;  ADate:  Date);

Var

 DateDialog:  PCalendar;

Begin
 DateDialog := New(PCalendar,Init(ARect,ADate,TRUE));
 DeskTop^.ExecView(DateDialog);
 Dispose(DateDialog,Done);
end;

{===========================================================================}
{                                                                           }
{ Pick_Date is a stand-alone procedure that generates a desk calendar type  }
{ object that allows the user to view a calendar for any month in any month }
{ in any year and set a date by clicking on that date.                      }
{                                                                           }
{===========================================================================}

Procedure Pick_Date(ARect:  TRect;  Var ADate:  Date);

Var

 DateDialog:  PCalendar;

Begin
 DateDialog := New(PCalendar,Init(ARect,ADate,FALSE));
 If (DeskTop^.ExecView(DateDialog) = cmOK) then DateDialog^.GetData(ADate);
 Dispose(DateDialog,Done);
end;

{===========================================================================}
{                                                                           }
{ The Age_String function accepts a reference date, usually a date of birth,}
{ and a target date, usually today's date and returns a string representing }
{ the age on the target date.                                               }
{                                                                           }
{===========================================================================}

Function  Age_String(Reference_Date, Target_Date:  Date):  date_strings;

Var

 age:     real;
 StrAge:  date_strings;

Begin
 If NOT(Valid_Date(Reference_Date)) OR NOT(Valid_Date(Target_Date))
 then StrAge := 'Unknown             '
 else begin
  age := (Julian(Target_Date) - Julian(Reference_Date))/365.25;
  If (age <= 0.0)
   then StrAge := 'Unknown             '
   else begin
    Str(age:5:1,StrAge);
    While StrAge[1] = ' ' do Delete(StrAge,1,1);
    StrAge := StrAge + ' Years';
   end;
 end;
 Age_String := StrAge;
end;

{===========================================================================}
{                                                                           }
{ Gestation_String takes a reference date, usually the first day of a       }
{ woman's last menstrual period, and a target date, usually today's date,   }
{ and returns a string that represents the gestation in weeks and days for  }
{ the woman's pregnancy.                                                    }
{                                                                           }
{===========================================================================}

Function  Gestation_String(Reference_Date, Target_Date:  Date; Use_EDC:  Boolean):  date_strings;

Var

 Reference_Julian,
 Target_Julian:     Scaliger;
 Temporary:         date_strings;
 StrGest:           date_strings;

Begin
 If NOT(Valid_Date(Reference_Date)) OR NOT(Valid_Date(Target_Date))
  then begin
   Reference_Julian := 0;
   Target_Julian    := 0;
  end
  else begin
   Reference_Julian := Julian(Reference_Date);
   If Use_EDC then System.Dec(Reference_Julian,280);
   Target_Julian := Julian(Target_Date);
  end;
 If (Target_Julian <= Reference_Julian)
  then StrGest := 'Unknown'
  else begin
   Str(((Target_Julian - Reference_Julian) DIV 7),temporary);
   If (Length(temporary) = 1) then temporary := '0' + temporary;
   If (temporary = ' 1')
    then StrGest := temporary + ' Week  '
    else StrGest := temporary + ' Weeks ';
   Str(((Target_Julian - Reference_Julian) MOD 7),temporary);
   If (temporary = '1')
    then StrGest := StrGest + temporary + ' Day '
    else StrGest := StrGest + temporary + ' Days';
  end;
 Gestation_String := StrGest;
end;

{===========================================================================}
{                                                                           }
{ Difference_String takes a "due date" based on a last menstrual period and }
{ a "due date" based on an ultrasound and returns a string that represents  }
{ the difference in weeks and days between the two dates.                   }
{                                                                           }
{===========================================================================}

Function  Difference_String(LMP_Date, Sonogram_Date:  Date):  date_strings;

Var

 LMP_Julian,
 Sonogram_Julian:  Scaliger;
 Temporary:        date_strings;
 StrGest:          date_strings;

Begin
 Difference_String := 'Unknown';
 If NOT(Valid_Date(LMP_Date)) OR NOT(Valid_Date(Sonogram_Date))
  then Exit
  else begin
   LMP_Julian := Julian(LMP_Date);
   Sonogram_Julian := Julian(Sonogram_Date);
   System.Dec(Sonogram_Julian,280);
  end;
  Str((Abs(LMP_Julian - Sonogram_Julian) DIV 7),temporary);
  If (Length(temporary) = 1) then temporary := '0' + temporary;
  If (temporary = ' 1')
   then StrGest := temporary + ' Week  '
   else StrGest := temporary + ' Weeks ';
  Str((Abs(LMP_Julian - Sonogram_Julian) MOD 7),Temporary);
  If (Temporary = '1')
   then StrGest := StrGest + temporary + ' Day '
   else StrGest := StrGest + temporary + ' Days';
 Difference_String := StrGest;
end;

{===========================================================================}
{                                                                           }
{ EDC_String takes a last menstrual period date and returns a string that   }
{ represents her "due date." A "due date" is calculated by adding 280 days  }
{ to the first day of the last menstrual period.                            }
{                                                                           }
{===========================================================================}

Function  EDC_String(LMP_Date:  Date):  date_strings;

Begin
 Standard(Julian(LMP_Date) + 280,LMP_Date);
 EDC_String := String_Date(LMP_Date,FALSE,FALSE,FALSE);
end;

{===========================================================================}
{                                                                           }
{ Define methods for TGenericDateView. TGenericDateView is an object that   }
{ draws a date. The date that is drawn is the date represented by the       }
{ TGenericDateView variable "The_Date." If The_Date is not set to a date,   }
{ i.e., it is set to the constant "Null_Date," then it will display         }
{ "Unknown." TGenericDateView is the ancestor of all date views.            }
{                                                                           }
{===========================================================================}


Constructor TGenericDateView.Init(Bounds: TRect);

Begin
 Inherited Init(Bounds);
 Options  := Options AND NOT(ofSelectable);
 The_Date := Null_Date;
 Right    := TRUE;
end;

Function    TGenericDateView.DataSize:  Word;

Begin
 DataSize := SizeOf(Date);
end;

Procedure   TGenericDateView.SetData(Var  Rec);

Begin
 Move(Rec,The_Date,DataSize);
 Inherited SetData(Rec);
end;

Procedure   TGenericDateView.SetState(AState:  Word;  Enable:  Boolean);

Begin
 Inherited SetState(AState,Enable);
 If (AState AND sfSelected) = sfSelected then DrawView;
end;

Procedure   TGenericDateView.GetData(Var  Rec);

Begin
 Inherited GetData(Rec);
 Move(The_Date,Rec,DataSize);
end;

Function    TGenericDateView.GetPalette:  PPalette;

Const

 CDateView = #2#2;
 PDateView:  string[2] = CDateView;

Begin
 GetPalette := @PDateView;
end;

Procedure   TGenericDateView.Draw;

Begin
 Attribute.color.normal    := GetColor(1);
 Attribute.color.highlight := GetColor(2);
 If (The_Date.month = error)
  then Date_String := 'Unknown'
  else Date_String := String_Date(The_Date,FALSE,FALSE,FALSE);
 If Right then begin
  Date_String := Eighteen + Date_String;
  Date_String := Copy(Date_String,Length(Date_String) - 17,18);
 end;
 If ((State AND sfSelected) = sfSelected)
  then Date_String := ' ~' + Date_String + '~'
  else Date_String := ' '  + Date_String;
 MoveChar(Date_Buffer,' ',Attribute.color.normal,Size.X);
 MoveCStr(Date_Buffer,Date_String,Attribute.colors);
 WriteBuf(0,0,Size.X,1,Date_Buffer);
end;

Procedure   TGenericDateView.Update(ADate:  Date);

Begin
 The_Date := ADate;
 DrawView;
end;

{===========================================================================}
{                                                                           }
{ Define methods for TDateView.                                             }
{                                                                           }
{===========================================================================}

Constructor TDateView.Init(ARect:  TRect);

Begin
 Inherited Init(ARect);
 Right := FALSE;
end;

Function    TDateView.GetPalette:  PPalette;

Const

 CDateView = #19#20;
 PDateView:  string[2] = CDateView;

Begin
 GetPalette := @PDateView;
end;

{===========================================================================}
{                                                                           }
{ Define methods for TMonthYearView. TMonthYearView is a special version of }
{ TGenericDateView that displays only the month and year part of The_Date.  }
{                                                                           }
{===========================================================================}

Constructor TMonthYearView.Init(ARect:  TRect);

Begin
 TDateView.Init(ARect);
 Options := Options OR ofSelectable;
 HelpCtx := hcMonthYearView;
end;

{===========================================================================}
{                                                                           }
{ Toggle is a method that updates The_Date to the next month.               }
{                                                                           }
{===========================================================================}

Procedure   TMonthYearView.Toggle;

Begin
 If (The_Date.month = error) then The_Date := Todays_Date;
 The_Date.day := Days_Per_Month(The_Date.month,The_Date.year);
 If (The_Date.month = Dec)
  then begin
   Inc(The_Date.year);
   The_Date.month := Jan;
   The_Date.day   := 31;
  end
  else begin
   Inc(The_Date.month);
   The_Date.day := Days_Per_Month(The_Date.month,The_Date.year);
  end;
 DrawView;
end;

{===========================================================================}
{                                                                           }
{ UnToggle is a method that updates The_Date to the previous month.         }
{                                                                           }
{===========================================================================}

Procedure   TMonthYearView.UnToggle;

Begin
 If (The_Date.month = error) then The_Date := Todays_Date;
 The_Date.day := Days_Per_Month(The_Date.month,The_Date.year);
 If (The_Date.month = Jan)
  then begin
   System.Dec(The_Date.year);
   The_Date.month := Dec;
   The_Date.day   := 31;
  end
  else begin
   System.Dec(The_Date.month);
   The_Date.day := Days_Per_Month(The_Date.month,The_Date.year);
  end;
 DrawView;
end;

Procedure   TMonthYearView.Draw;

Begin
 Attribute.color.normal    := GetColor(1);
 Attribute.color.highlight := GetColor(2);
 If (The_Date.month = error)
  then Date_String := 'Unknown'
  else begin
   Date_String := String_Date(The_Date,TRUE,FALSE,FALSE);
   Delete(Date_String,1,3);
  end;
 If ((State AND sfSelected) = sfSelected)
  then Date_String := ' ~' + Date_String + '~'
  else Date_String := ' '  + Date_String;
 MoveChar(Date_Buffer,' ',Attribute.color.normal,Size.X);
 MoveCStr(Date_Buffer,Date_String,Attribute.colors);
 WriteBuf(0,0,Size.X,1,Date_Buffer);
end;

Procedure   TMonthYearView.HandleEvent(Var  Event:  TEvent);

Var

 Offset:  Integer;

Begin
  If (Event.What = evKeyDown) then begin

   { Interpret a keyboard digit as an instruction }
   { to add that many months to The_Date.         }

   If (Event.CharCode IN ['0'..'9']) then begin
    If (The_Date.month = error) then begin
     The_Date := Todays_Date;
     The_Date.day := Days_Per_Month(The_Date.month,The_Date.year);
    end;
    Offset := (Ord(Event.CharCode) - Ord('0')) + Ord(The_Date.month);
    If (Offset > 12) then begin
     Offset := Offset MOD 12;
     Inc(The_Date.year);
    end;
    The_Date.month := Months(Offset);
    The_Date.day   := Days_Per_Month(The_Date.month,The_Date.year);
    DrawView;
    ClearEvent(Event);
   end;
  Case Event.KeyCode of
   kbDel:   begin { Make The_Date a Null_Date. }
             The_Date := Null_Date;
             DrawView;
             ClearEvent(Event);
            end;
   kbUp,
   kbPgUp:  begin { Make The_Date the previous month. }
             UnToggle;
             ClearEvent(Event);
            end;
   kbDown,
   kbPgDn:  begin { Make The_Date the next month. }
             Toggle;
             ClearEvent(Event);
            end;
   kbHome:  begin { Make The_Date Todays_date. }
             The_Date := Todays_Date;
             The_Date.day := Days_Per_Month(The_Date.month,The_Date.year);
             DrawView;
             ClearEvent(Event);
            end;
  end;
 end;
 If (Event.What = evMouseDown) then begin

  { Interpret mouse actions. }

  If NOT((State AND sfSelected) = sfSelected) then begin { Select me. }
   Select;
   DrawView;
   ClearEvent(Event);
   Exit;
  end;
  Case Event.Buttons of
   mbLeftButton:   Toggle;    { Update the display. }
   mbRightButton:  UnToggle;
  end;
  ClearEvent(Event);
 end;
 Inherited HandleEvent(Event);
end;

{===========================================================================}
{                                                                           }
{ Define methods for TScrollCalendar. TScrollCalendar is a generic scrolling}
{ calendar object that draws the calendar for any month in any year and can }
{ be "scrolled" to display the calendar for any other month and year. If    }
{ the Boolean argument "View" is TRUE, then the TScrollCalendar can only be }
{ used to display and view calendars. If "View" is FALSE, then it gains     }
{ additional features that allow the user to select a date by pointing and  }
{ clicking the date.                                                        }
{                                                                           }
{===========================================================================}

Constructor TScrollCalendar.Init(ADate:        Date;
                                 AHScrollBar,
                                 AVScrollBar:  PScrollBar;
                                 View:         Boolean);

Begin
 Local_Box.Assign(0,0,21,8);
 Local_Box.Move(2,1);
 TScroller.Init(Local_Box,AHScrollBar,AVScrollBar);
 View_Only := View;
 New(calendar_buffer);
 If Valid_Date(ADate)
  then The_Date := ADate
  else The_Date := todays_date;
 Get_Colors   := TRUE;
 New_Pointer  := TRUE;
 New_Calendar := TRUE;
end;

Procedure   TScrollCalendar.Update_Pointer;

Var

 offset:  integer;
 test:    Days;

Begin
 If View_Only
  then begin
   The_Date.day := 17;
   test := Day_of_Week(The_Date);
   offset := 2 - Ord(test);
   If (test = Sun) then offset := 3;
   The_Date.day := The_Date.day + offset;
   date_offset  := The_Date.day;
   If (The_Date.year = 1752) AND (The_Date.month = Sep) AND (The_Date.day > 2)
    then date_offset := date_offset - 11;
  end
  else begin
   offset := buffer_offset + ((date_offset - 1) * 3);
   calendar_buffer^[offset + 1].element.Attribute := Attribute.color.normal;
   calendar_buffer^[offset + 2].element.Attribute := Attribute.color.normal;
   date_offset := The_Date.day;
   If (The_Date.year = 1752) AND (The_Date.month = Sep) AND (The_Date.day > 2)
    then date_offset := date_offset - 11;
   offset := buffer_offset + ((date_offset - 1) * 3);
   calendar_buffer^[offset + 1].element.Attribute := Attribute.color.highlight;
   calendar_buffer^[offset + 2].element.Attribute := Attribute.color.highlight;
  end;
 offset := buffer_offset - 43;
 Compare.Y := (((offset + ((date_offset - 1) * 3)) DIV 21) * VCol) + VOffset;
 Compare.X := (((offset + ((date_offset - 1) * 3)) MOD 21) * HCol) + HOffset;
 Delta.Y   := Compare.Y;
 Delta.X   := Compare.X;
 VScrollBar^.Value := Compare.Y;
 VScrollBar^.DrawView;
 HScrollBar^.Value := Compare.X;
 HScrollBar^.DrawView;
 New_Pointer := FALSE;
end;

Procedure   TScrollCalendar.Create_Calendar_Buffer;

Const

 blanks    = '                     ';
 weekdays  = ' ~Su Mo Tu We Th Fr Sa~';

Var

 local_date:      Date;          { Local date variable.                }
 padding:         string[21];    { Blanks for offsetting strings.      }
 month_and_year:  date_strings;  { A string for the calendar title.    }
 offset:          byte;          { For calculating offsets.            }
 the_day:         Days;          { 1st of the month falls on this day. }

Begin

 { Retrieve the date and force it to the first of the month. }

 local_date     := The_Date;
 local_date.day := 1;

 { Create a new dynamic string. }

 New(calendar_string);
 calendar_string^ := '';

 { Copy the numbers into the string. }

 With local_date do
  calendar_string^ := Copy(calendar_numbers^,1,(Days_per_Month(month,year)*3));

 { Adjust the string for the Gregorian Calendar Reform. }

 If (local_date.year = 1752) AND (local_date.month = Sep) then
  Delete(calendar_string^,7,33);

 { Find out on what day of the week the month starts. }

 the_day := Day_of_Week(local_date);

 { Construct blank padding to offset for the day of the week. }

 offset := 3 + (Ord(the_day) * 3);
 If (the_day = Sun)
  then padding := ''
  else padding := Copy(blanks,1,offset);

 { Set the positional values. }

 offset        := Length(padding);
 buffer_offset := 43 + offset;         
 date_offset   := The_Date.day;
 If (The_Date.year = 1752) AND (The_Date.month = Sep) AND (The_Date.day > 2)
  then date_offset := date_offset - 11;

 { Offset the number string. }

 calendar_string^ := padding + calendar_string^;

 { Construct a title for the calendar. }

 month_and_year := String_Date(local_date,TRUE,FALSE,FALSE);
 Delete(month_and_year,1,3);

 { Offset the title with blanks to center it over the calendar. }

 offset         := 11 - (Length(month_and_year) DIV 2);
 month_and_year := Copy(blanks,1,offset) + month_and_year + blanks;
 month_and_year := Copy(month_and_year,1,21);

 calendar_string^ := month_and_year + weekdays + calendar_string^;
 MoveChar(calendar_buffer^,' ',Attribute.color.normal,168);
 MoveCStr(calendar_buffer^,calendar_string^,Attribute.colors);
 Dispose(calendar_string);
 New_Calendar := FALSE;
 Update_Pointer;
end;

Function    TScrollCalendar.DataSize:  Word;

Begin
 DataSize := 5;
end;

Procedure   TScrollCalendar.SetData(Var  Rec);

Begin
 Move(Rec,The_Date,DataSize);
end;

Procedure   TScrollCalendar.GetData(Var  Rec);

Begin
 Move(The_Date,Rec,DataSize);
end;

Function    TScrollCalendar.GetPalette:  PPalette;

Const

 CScrollCalendar = #7#8;
 PScrollCalendar:  string[2] = CScrollCalendar;

Begin
 GetPalette := @PScrollCalendar;
end;

Procedure   TScrollCalendar.Draw;

Begin
 If Get_Colors then begin
  Attribute.color.normal    := GetColor(1);
  Attribute.color.highlight := GetColor(2);
  Get_Colors := FALSE;
 end;
 If New_Calendar then Create_Calendar_Buffer;
 If New_Pointer  then Update_Pointer;
 WriteBuf(0,0,21,8,calendar_buffer^);
end;

Procedure   TScrollCalendar.HandleBroadcast;

Const

 cmNone        =  0;
 cmLeft        =  1;
 cmRight       =  2;
 cmUp          =  3;
 cmDown        =  4;
 cmLastMonth   =  5;
 cmNextMonth   =  6;
 cmLastYear    =  7;
 cmNextYear    =  8;

Var

 change,
 end_of_month:  integer;
 Weekday:       Days;
 Command:       word;

Begin
 Command := cmNone;
 end_of_month := Days_per_Month(The_Date.month,The_Date.year);
 If NOT(Delta.X = Compare.X) then begin
  change := Delta.X - Compare.X;
  If (Delta.X = HMax) then change := HArPlu;
  If (Delta.X = HMin) then change := HArMin;
  Case change of
   HArPlu:  Command := cmRight;
   HArMin:  Command := cmLeft;
   HPgPlu:  Command := cmNextYear;
   HPgMin:  Command := cmLastYear;
  else      begin
             New_Pointer := TRUE;
             DrawView;
            end;
  end;
 end;
 Case Command of
        cmNone:  { Don't look further. };
        cmLeft:  With The_Date do begin
                  day := Pred(day);
                  If (day = 0) then day := end_of_month;
                  If (year = 1752) AND (month = Sep) AND (day = 13)
                   then day := 2;
                  New_Pointer := TRUE;
                  DrawView;
                 end;
       cmRight:  With The_Date do begin
                  day := Succ(day);
                  If (day > end_of_month) then day := 1;
                  If (year = 1752) AND (month = Sep) AND (day = 3)
                   then day := 14;
                  New_Pointer := TRUE;
                  DrawView;
                 end;
    cmLastYear:  With The_Date do begin
                  year := Pred(year);
                  end_of_month := Days_per_Month(month,year);
                  If (year = 1752) AND (month = Sep) AND (day IN [3..13])
                   then day := 14;
                  If (day > end_of_month) then day := end_of_month;
                  New_Calendar := TRUE;
                  DrawView;
                 end;
    cmNextYear:  With The_Date do begin
                  year := Succ(year);
                  end_of_month := Days_per_Month(month,year);
                  If (year = 1752) AND (month = Sep) AND (day IN [3..13])
                   then day := 14;
                  If (day > end_of_month) then day := end_of_month;
                  New_Calendar := TRUE;
                  DrawView;
                 end;
 end;
 Command  := cmNone;
 If (Command = cmNone) AND NOT(Delta.Y = Compare.Y) then begin
  change := Delta.Y - Compare.Y;
  If (Delta.Y = VMax) then change := VArPlu;
  If (Delta.Y = VMin) then change := VArMin;
  Case change of
   VArPlu:  Command := cmDown;
   VArMin:  Command := cmUp;
   VPgPlu:  Command := cmNextMonth;
   VPgMin:  Command := cmLastMonth;
  else      begin
             New_Pointer := TRUE;
             DrawView;
            end;

  end;
 end;
 Case Command of
          cmNone:  { Don't look further. };
          cmDown:  With The_Date do begin
                    If (year = 1752) AND (month = Sep)
                     then Case day of
                              1,2:  day := day + 18;
                            24,25:  day := day -  7;
                            26,27:  day := day - 25;
                           28..30:  day := day - 14;
                          else      day := day +  7;
                          end
                     else day := day + 7;
                    If (day > end_of_month) then day := day MOD 7;
                    If (day = 0) then day := 7;
                    New_Pointer := TRUE;
                    DrawView;
                   end;
            cmUp:  With The_Date do begin
                    If (year = 1752) AND (month = Sep)
                     then Case day of
                              1,2:  day := day + 25;
                           14..16:  day := day + 14;
                            17,18:  day := day +  7;
                            19,20:  day := day - 18;
                          else      day := day -  7;
                          end
                     else day := day - 7;
                    If (day < 1) then day := day + 42;
                    While (day > end_of_month) do day := day - 7;
                    New_Pointer := TRUE;
                    DrawView;
                   end;
     cmLastMonth:  With The_Date do begin
                    month := Pred(month);
                    If (month = error) then begin
                     month := Dec;
                     year  := Pred(year);
                    end;
                    end_of_month := Days_per_Month(month,year);
                    If (year = 1752) AND (month = Sep) AND (day IN [3..13])
                     then day := 14;
                    If (day > end_of_month) then day := end_of_month;
                    New_Calendar := TRUE;
                    DrawView;
                   end;
     cmNextMonth:  With The_Date do begin
                    If (month = Dec) then begin
                     month := error;
                     year  := Succ(year);
                    end;
                    month := Succ(month);
                    end_of_month := Days_per_Month(month,year);
                    If (year = 1752) AND (month = Sep) AND (day IN [3..13])
                     then day := 14;
                    If (day > end_of_month) then day := end_of_month;
                    New_Calendar := TRUE;
                    DrawView;
                   end;
 end;
end;

Procedure   TScrollCalendar.HandleEvent(Var  Event:  TEvent);

Var

 end_of_month:  integer;
 Weekday:       Days;
 Mouse:         TPoint;

Begin         
 If (Event.What = evKeyDown) then begin
  Case Event.KeyCode of
       kbHome:  begin
                 The_Date := todays_date;
                 New_Calendar := TRUE;
                 DrawView;
                 ClearEvent(Event);
                 Exit;
                end;
   kbCtrlPgUp:  begin
                 The_Date.month := Jan;
                 New_Calendar := TRUE;
                 DrawView;
                 ClearEvent(Event);
                 Exit;
                end;
   kbCtrlPgDn:  begin
                 The_Date.month := Dec;
                 New_Calendar := TRUE;
                 DrawView;
                 ClearEvent(Event);
                 Exit;
                end;
  end;
  Case Event.CharCode of
   ',',
   '<':  begin
          Event.What    := evCommand;
          Event.Command := cmDecadeMinus;
         end;
   '.',
   '>':  With The_Date do begin
          Event.What    := evCommand;
          Event.Command := cmDecadePlus;
         end;
   '-',
   '_':  With The_Date do begin
          Event.What    := evCommand;
          Event.Command := cmCenturyMinus;
         end;
   '+',
   '=':  With The_Date do begin
          Event.What    := evCommand;
          Event.Command := cmCenturyPlus;
         end;
  end;
 end;
 If (Event.What = evCommand) then begin
  Case Event.Command of
    cmDecadeMinus:  With The_Date do begin
                     year := year - 10;
                     end_of_month := Days_per_Month(month,year);
                     If (year = 1752) AND (month = Sep) AND (day IN [3..13])
                      then day := 14;
                     If (day > end_of_month) then day := end_of_month;
                     New_Calendar := TRUE;
                     DrawView;
                     ClearEvent(Event);
                     Exit;
                    end;
   cmCenturyMinus:  With The_Date do begin
                     year := year - 100;
                     end_of_month := Days_per_Month(month,year);
                     If (year = 1752) AND (month = Sep) AND (day IN [3..13])
                      then day := 14;
                     If (day > end_of_month) then day := end_of_month;
                     New_Calendar := TRUE;
                     DrawView;
                     ClearEvent(Event);
                     Exit;
                    end;
     cmDecadePlus:  With The_Date do begin
                     year := year + 10;
                     end_of_month := Days_per_Month(month,year);
                     If (year = 1752) AND (month = Sep) AND (day IN [3..13])
                      then day := 14;
                     If (day > end_of_month) then day := end_of_month;
                     New_Calendar := TRUE;
                     DrawView;
                     ClearEvent(Event);
                     Exit;
                    end;
    cmCenturyPlus:  With The_Date do begin
                     year := year + 100;
                     end_of_month := Days_per_Month(month,year);
                     If (year = 1752) AND (month = Sep) AND (day IN [3..13])
                      then day := 14;
                     If (day > end_of_month) then day := end_of_month;
                     New_Calendar := TRUE;
                     DrawView;
                     ClearEvent(Event);
                     Exit;
                    end;
  end;
 end;
 If (Event.What = evMouseDown) then begin
  MakeLocal(Event.Where,Mouse);
  ClearEvent(Event);
  If View_Only then Exit;
  If (Mouse.X > Size.X) then Exit;
  If (Mouse.Y > Size.Y) then Exit;
  If (Mouse.X < 0) then Exit;
  If (Mouse.Y < 2) then Exit;
  With The_Date do begin
   day          := 1;
   Weekday      := Day_of_Week(The_Date);
   end_of_month := Days_per_Month(month,year);
   day := ((Mouse.Y - 2) * 7) + (Mouse.X DIV 3);
   If (Weekday IN [Mon..Sat])
    then day := day - Ord(Weekday)
    else Inc(day);
   If (year = 1752) AND (month = Sep) AND (day > 2)
    then day := day + 11;
   If (day < 1) then day := 1;
   If (day > end_of_month) then day := end_of_month;
   New_Pointer := TRUE;
   DrawView;
  end;
 end;
 TScroller.HandleEvent(Event);
 If (Event.What = evBroadcast) then begin
  HandleBroadcast;
  ClearEvent(Event);
 end;
end;

Destructor  TScrollCalendar.Done;

Begin
 Dispose(calendar_buffer);
 TScroller.Done;
end;

{===========================================================================}
{                                                                           }
{ Define methods for TCalendarIcons. The calendar icons allow the user to   }
{ control the calendar by clicking various icons.                           }
{                                                                           }
{===========================================================================}

Constructor TCalendarIcons.Init(ARect:  TRect;  AView_Only:  Boolean);

Begin
 TView.Init(ARect);
 View_Only := AView_Only;
end;

Function    TCalendarIcons.GetPalette:  PPalette;

Const

 CCalendarIcons = #2#3;
 PCalendarIcons:  string[2] = CCalendarIcons;

Begin
 GetPalette := @PCalendarIcons;
end;

Procedure   TCalendarIcons.Draw;

Const

 icon_string0 =  '[~-'#60#04#62'+~]';
 icon_string1 =  ' Calendar ';
 icon_string2 =  '[~=~] Set Date ';

Begin
 Attribute.color.normal    := GetColor(1);
 Attribute.color.highlight := GetColor(2);
 If View_Only
  then MoveCStr(Icons,icon_string1 + icon_string0,Attribute.colors)
  else MoveCStr(Icons,icon_string2 + icon_string0,Attribute.colors);
 WriteBuf(0,0,20,1,Icons);
end;

Procedure   TCalendarIcons.HandleEvent(Var  Event:  TEvent);

Var

 Mouse:  TPoint;

Begin
 If (Event.What = evMouseDown) then begin
  MakeLocal(Event.Where,Mouse);
  Event.What := evCommand;
  Case Mouse.X of
    1:  If View_Only
         then ClearEvent(Event)
         else Event.Command := cmOk;
   14:  Event.Command := cmCenturyMinus;
   15:  Event.Command := cmDecadeMinus;
   16:  begin
         Event.What    := evKeyDown;
         Event.KeyCode := kbHome;
        end;
   17:  Event.Command := cmDecadePlus;
   18:  Event.Command := cmCenturyPlus;
  else  ClearEvent(Event);
  end;
  PutEvent(Event);
  ClearEvent(Event);
 end;
 TView.HandleEvent(Event);
end;

Destructor  TCalendarIcons.Done;

Begin
 TView.Done;
end;

{===========================================================================}
{                                                                           }
{ Define methods for TCalendar. A TCalendar assembles all the calendar      }
{ views into a working calendar dialog box.                                 }
{                                                                           }
{===========================================================================}

Constructor TCalendar.Init(ARect:       TRect;
                           ADate:       Date;
                           AView_Only:  Boolean);

Var

 HScrollBar,
 VScrollBar:  PScrollBar;
 Bar:         TRect;

Begin
 If (ARect.A.X < 0) OR (ARect.A.Y < 0)
  then begin
   ARect.Assign(0,0,26,10);
   TDialog.Init(ARect,'');
   Options := (Options OR ofCentered)
  end
  else begin
   ARect.B.X := ARect.A.X + 26;
   ARect.B.Y := ARect.A.Y + 10;
   TDialog.Init(ARect,'');
  end;
 Bar.Assign(0,0,23,1);
 Bar.Move(1,9);
 HScrollBar := New(PScrollBar,Init(Bar));
 HScrollBar^.Min     := HMin;
 HScrollBar^.Max     := HMax;
 HScrollBar^.PgStep  := HPgPlu;
 HScrollBar^.ArStep  := HArPlu;
 HScrollBar^.Options := (HScrollBar^.Options OR ofPreProcess);
 Bar.Assign(0,0,1,8);
 Bar.Move(25,2);
 VScrollBar := New(PScrollBar,Init(Bar));
 VScrollBar^.Min     := VMin;
 VScrollBar^.Max     := VMax;
 VScrollBar^.PgStep  := VPgPlu;
 VScrollBar^.ArStep  := VArPlu;
 VScrollBar^.Options := (VScrollBar^.Options OR ofPreProcess);
 Calendar := New(PScrollCalendar,Init(ADate,HScrollBar,VScrollBar,AView_Only));
 Bar.Assign(0,0,20,1);
 Bar.Move(5,0);                         
 Icons := New(PCalendarIcons,Init(Bar,AView_Only));
 If AView_Only
  then begin
   HelpCtx := hcDeskCalendar;
   Calendar^.HelpCtx := hcDeskCalendar;
  end
  else begin
   HelpCtx := hcSetDateCalendar;
   Calendar^.HelpCtx := hcSetDateCalendar;
  end;
 Insert(Icons);
 Insert(HScrollBar);
 Insert(VScrollBar);
 Insert(Calendar);
end;

Procedure   TCalendar.HandleEvent(Var  Event:  TEvent);

Begin
 If (Event.What = evKeyDown) then
  Case Event.KeyCode of
       kbHome,
   kbCtrlPgUp,
   kbCtrlPgDn:  Calendar^.HandleEvent(Event);
      kbEnter:  begin
                 EndModal(cmOK);
                 ClearEvent(Event);
                end;
  end;
 TDialog.HandleEvent(Event);
end;

Destructor  TCalendar.Done;

Begin
 TDialog.Done;
end;

{===========================================================================}
{                                                                           }
{ Define methods for TDateViewIcon. Creates a down-arrow icon for a         }
{ TDateInputView.                                                           }
{                                                                           }
{===========================================================================}


Constructor TDateViewIcon.Init(ARect:  TRect);

Begin
 Inherited Init(ARect);
end;

Function    TDateViewIcon.GetPalette:  PPalette;

Const

 CDateViewIcon = #2#3;
 PDateViewIcon:  string[2] = CDateViewIcon;

Begin
 GetPalette := @PDateViewIcon;
end;

Procedure   TDateViewIcon.Draw;

Const

 Icon_String =  '~'#25'~';

Begin
 Attribute.color.normal    := GetColor(1);
 Attribute.color.highlight := GetColor(2);
 MoveCStr(Icon,Icon_String,Attribute.colors);
 WriteBuf(0,0,3,1,Icon);
end;

Procedure   TDateViewIcon.HandleEvent(Var  Event:  TEvent);

Var

 Mouse:  TPoint;

Begin
 If (Event.What = evMouseDown) then begin
  MakeLocal(Event.Where,Mouse);
  If (Mouse.X IN [0..2]) then begin
   Event.What    := evKeyDown;
   Event.KeyCode := kbDown;
   PutEvent(Event);
   ClearEvent(Event);
  end;
 end;
 Inherited HandleEvent(Event);
end;

Destructor  TDateViewIcon.Done;

Begin
 Inherited Done;
end;

{===========================================================================}
{                                                                           }
{ Define methods for TDateInputView. This view allows the user to input a   }
{ date. The date is input by invoking a calendar and pointing and clicking  }
{ the desired date.                                                         }
{                                                                           }
{===========================================================================}

Constructor TDateInputView.Init(ARect:  TRect);

Begin
 Inherited Init(ARect);
 Options := Options OR ofSelectable;
 HelpCtx := hcDateInputView;
 The_Date := Null_Date;
 Local_Box.Assign(0,0,19,1);
 DateView := New(PDateView,Init(Local_Box));
 DateView^.Options  := DateView^.Options OR ofSelectable;
 Local_Box.Assign(0,0,3,1);
 Local_Box.Move(19,0);
 Icon := New(PDateViewIcon,Init(Local_Box));
 Insert(DateView);
 Insert(Icon);
end;

Function    TDateInputView.DataSize:  Word;

Begin
 DataSize := SizeOf(Date);
end;

Procedure   TDateInputView.SetData(Var  Rec);

Begin
 DateView^.SetData(Rec);
 The_Date := DateView^.The_Date;
 ReDraw;
end;

Procedure   TDateInputView.SetState(AState:  Word;  Enable:  Boolean);

Begin
 Inherited SetState(AState,Enable);
 If (AState AND sfSelected) = sfSelected then begin
  DateView^.SetState(AState,Enable);
  ReDraw;
 end;
end;

Procedure   TDateInputView.GetData(Var  Rec);

Begin
 DateView^.The_Date := The_Date;
 DateView^.GetData(Rec);
end;

Procedure   TDateInputView.Draw;

Begin
 Inherited Draw;
end;

Procedure   TDateInputView.HandleEvent(Var  Event:  TEvent);

Var

 DateDialog:  PCalendar;
 Location:    TPoint;
 Mouse:       TPoint;

{===========================================================================}
{                                                                           }
{ Find uses a recursive method to find the absolute location of a date      }
{ input view on the screen.                                                 }
{                                                                           }
{===========================================================================}


Procedure Find(Var Location:  TPoint;  AOwner:  PGroup);

Begin
 If (AOwner = NIL) then Exit;
 If (Pointer(AOwner) = Pointer(DeskTop)) then Exit;
 Location.X := Location.X + Owner^.Origin.X;
 Location.Y := Location.Y + Owner^.Origin.Y;
 Find(Location,AOwner^.Owner);
end;

Begin
 If (Event.What = evMouseDown) then begin
  If NOT((State AND sfSelected) = sfSelected) then Select;
  MakeLocal(Event.Where,Mouse);
  If (Mouse.X IN [19..21]) then Icon^.HandleEvent(Event);
 end;
 If (Event.What = evKeyDown) then begin
  If (Event.CharCode IN [' '..'~']) then ClearEvent(Event);
  Case Event.KeyCode of
   kbDown:  begin
             Location := Origin;
             Find(Location,Owner);
             Local_Box.Assign(0,0,26,10);
             Local_Box.Move(Location.X,Location.Y);
             While (Local_Box.B.X > DeskTop^.Size.X) do Local_Box.Move(-1,0);
             While (Local_Box.B.Y > DeskTop^.Size.Y) do Local_Box.Move(0,-1);
             DateDialog := New(PCalendar,Init(Local_Box,The_Date,FALSE));
             If (DeskTop^.ExecView(DateDialog) = cmOK) then begin
              DateDialog^.GetData(The_Date);
              DateView^.Update(The_Date);
              Message(Owner,evCommand,cmDateChanged,@Self);
             end;
             Dispose(DateDialog,Done);
             ClearEvent(Event);
            end;
    kbDel:  begin
             The_Date := Null_Date;
             DateView^.Update(The_Date);
             Message(Owner,evCommand,cmDateChanged,@Self);
             ClearEvent(Event);
            end;
  end;
 end;
 Inherited HandleEvent(Event);
end;

{===========================================================================}
{                                                                           }
{ Define methods for TJulianDays. Displays the number of days between a     }
{ Target_Date and a Reference_Date.                                         }
{                                                                           }
{===========================================================================}

Constructor TJulianDays.Init(Var ABox:  TRect);

Begin
 TStaticText.Init(ABox,'                    ');
 Options := Options OR ofPreProcess;
 Target_Date    := NIL;
 Reference_Date := NIL;
end;

{===========================================================================}
{                                                                           }
{ Point_To_Reference is used to point the pointer variable "Reference_Date" }
{ towards the first date that TJulianDays uses to calculate the difference  }
{ between two dates that it displays.                                       }
{                                                                           }
{===========================================================================}

Procedure   TJulianDays.Point_To_Reference(AReference:  PDate);

Begin
 Reference_Date := AReference;
end;

{===========================================================================}
{                                                                           }
{ Point_To_Target is used to point the pointer variable "Target_Date"       }
{ towards the second date that TJulianDays uses to calculate the difference }
{ between the two dates that it displays.                                   }
{                                                                           }
{===========================================================================}

Procedure   TJulianDays.Point_To_Target(ATarget:  PDate);

Begin
 Target_Date := ATarget;
end;

Procedure   TJulianDays.Update;

Begin
 DrawView;
end;

Procedure   TJulianDays.Draw;

Begin
 If NOT(Valid_Date(Reference_Date^)) OR NOT(Valid_Date(Target_Date^))
  then Text^ := 'Unknown             '
  else begin
   Str((Julian(Target_Date^) - Julian(Reference_Date^)),Text^);
   Text^ := Text^ + ' Days';
  end;
 TStaticText.Draw;
end;

Procedure   TJulianDays.HandleEvent(Var  Event:  TEvent);

Begin
 If ((Event.What = evCommand) AND (Event.Command = cmDateChanged))
  then DrawView;
 TStaticText.HandleEvent(Event);
end;

{===========================================================================}
{                                                                           }
{ Define methods for TAgeDisplay. Displays an "age" in years between a      }
{ Reference_Date, assumed to be a date of birth and a Target_Date, assumed  }
{ to be a date of interest, such as today's date.                           }
{                                                                           }
{===========================================================================}

Procedure   TAgeDisplay.Draw;

Begin
 Text^ := Age_String(Reference_Date^,Target_Date^);
 TStaticText.Draw;
end;

{===========================================================================}
{                                                                           }
{ Define methods for TGestDisplay. Displays the Gestation of a pregnancy in }
{ weeks and days based on today's date and either a first day of last       }
{ menstrual period or an ultrasound due date.                               }
{                                                                           }
{===========================================================================}

Constructor TGestDisplay.Init(Var ABox:  TRect);

Begin
 TJulianDays.Init(ABox);
 Use_EDC := FALSE;
end;

{===========================================================================}
{                                                                           }
{ Set_Use_EDC is used to set the Boolean variable "Use_EDC." EDC means      }
{ Estimated Date of Confinement which means "due date." The gestation can   }
{ be calculated either "forwards" from the last menstrual period (LMP) or   }
{ "backwards" from the EDC. TGestDisplay assumes that you are calculating   }
{ forewards from the LMP. To force it to calculate backwards from the EDC,  }
{ Use the Set_Use_EDC procedure to set the Boolean variable "Use_EDC" to    }
{ to TRUE.
{                                                                           }
{===========================================================================}

Procedure   TGestDisplay.Set_Use_EDC(ABoolean:  Boolean);

Begin
 Use_EDC := ABoolean;
end;

Procedure   TGestDisplay.Draw;

Begin
 Text^ := Gestation_String(Reference_Date^,Target_Date^,Use_EDC);
 TStaticText.Draw;
end;

{===========================================================================}
{                                                                           }
{ Define methods for TDiffDisplay. Displays difference in weeks and days    }
{ between a due date based on an LMP and a due date based on an ultrasound  }
{ EDC. Menstrual Period (LMP). Point the Reference_Date to the EDC and the  }
{ Target_Date to the ultrasound LPM.                                        }
{                                                                           }
{===========================================================================}

Procedure   TDiffDisplay.Draw;

Begin
 If NOT(Valid_Date(Reference_Date^)) OR NOT(Valid_Date(Target_Date^))
  then Text^ := 'Unknown'
  else Text^ := Difference_String(Reference_Date^,Target_Date^);
 TStaticText.Draw;
end;

{===========================================================================}
{                                                                           }
{ Define methods for TEDCDisplay. Displays the Estimated Date of            }
{ Confinement ("due Date") for a pregnancy calculated from the first day of }
{ Last Menstrual Period (LMP). Use the inherited Point_To_Reference         }
{ method to define what date to use for the EDC calculation.                }
{                                                                           }
{===========================================================================}

Procedure   TEDCDisplay.Draw;

Begin
 If NOT(Valid_Date(Reference_Date^))
 then Text^ := 'Unknown'
 else Text^ := EDC_String(Reference_Date^);
 TStaticText.Draw;
end;

Begin
 New(calendar_numbers);
 calendar_numbers^ := '';
 For count := 1 to 31 do begin
  Str(count:3,number);
  calendar_numbers^ := calendar_numbers^ + number;
 end;
end.