*****************************************************************************
*  Program  : cdd_comp.prg
*****************************************************************************
*
*  Purpose  : A simple compare program for CDD data dictionaries.
*             CDD_COMP is a mono-directional comparison, therefore
*             always compare the original DD to the revised DD.
*             Type "CDD_COMP" with no parameters for help.
*
*             Generally tname is the master (original) data dict. and
*             sname is the copy of the data dict. that you altered.
*
*  Syntax   : CDD_COMP <TNAME> <SPATH> <SNAME> [<OUTPUT>]
*
*  Arguments: TNAME original DD in current directory
*             SPATH is the path where the source DD is located (ends in : or \)
*             SNAME is the DD name in the source DD directory
*             OUTPUT is the optional output device. E.g, a DOS file, "LPT1"...
*
*  Author   : Vick Perry,   Mon  11-30-1992.  This file may be freely
*             distributed.
*
*  Notes    : CDD_COMP contains very little error checking or recovery.
*             Type the command line correctly!
*
*             To compile:
*
*                CLIPPER CDD_COMP /A /W
*
*  Revisions:
*
*****************************************************************************

#xcommand DEFAULT <p> TO <val> [,<pn> TO <valn>] =>;
         <p> := if(<p> = nil, <val>, <p>);
         [;<pn> := if(<pn> = nil, <valn>, <pn>)]
#xtranslate COMPILE(<v>) =>   &("{||" + <v> + "}")   // compile a code block

static hitlist_ := {}             // list of function ids that are changed
static o_output := ""

parameters tname, spath, sname, output
external transform               // a CDD index needs this function

if empty(tname) .or. upper(tname)=="-H"  .or. upper(tname)=="/H"
   ?
   ? "CDD_COMP.EXE is a CDD data dictionary comparison utility designed to"
   ? "compare a revised DD to its original DD.  Generally <old_sys> is the"
   ? "original DD and <new_sys> is your revised version of the original DD."
   ?
   ? "CDD_COMP <old_sys> <new_sys_path> <new_sys> [<output>]"
   ?
   ? "   old_sys - name of original DD in the current directory"
   ? "   new_sys_path - path for revised DD (ends in : or \)"
   ? "   new_sys - name of revised DD in new_sys_path directory"
   ? '   output - optional DOS output device or filename. LPT1, MYFILE.TXT, etc'
   ?
   ?
   quit
endif

default output to ""
set deleted on

* Turn on output routing.  In case of printing to LPT1, LPT2, or PRN
* print to tempfile first, then dump to printer - this is due to
* print spooler timeout on a network chopping up the report.
if !empty(output)
   * use temp file
   if left(upper(output),3) $ "LPT,PRN,AUX,COM"
      o_output := output
      output := "CDD_COMP.TMP"
   endif
   set alternate to (output)
   set alternate on
endif

? "CDD_COMP - CDD Data Dictionary File Compare  " + dtoc(date()) + "  "+time()

* dbf records
?
? "Data aliases of changed data files, fields, or indexes:"
hitlist_ := {}
dd_comp(tname+"D",tname+"D1",spath+sname+"D",spath+sname+"D1","dalias","dalias",@hitlist_)
dd_comp(tname+"F",tname+"F2",spath+sname+"F",spath+sname+"F2","dalias+fname","dalias",@hitlist_)
dd_comp(tname+"I",tname+"I1",spath+sname+"I",spath+sname+"I1","dalias+ialias","dalias",@hitlist_)
show_hits(hitlist_)

* menus
? "Menu IDs of changed menus or menu items:"
hitlist_ := {}
dd_comp(tname+"M",tname+"M1",spath+sname+"M",spath+sname+"M1","id","id",@hitlist_)
dd_comp(tname+"N",tname+"N1",spath+sname+"N",spath+sname+"N1","id+trans(pos,'999')","id",@hitlist_)
show_hits(hitlist_)

* browses
? "Browse IDs of changed browses, browse columns, or browse keys:"
hitlist_ := {}
dd_comp(tname+"B",tname+"B1",spath+sname+"B",spath+sname+"B1","id","id",@hitlist_)
dd_comp(tname+"C",tname+"C1",spath+sname+"C",spath+sname+"C1","id+trans(pos,'999')","id",@hitlist_)
dd_comp(tname+"K",tname+"K1",spath+sname+"K",spath+sname+"K1","id+trans(pos,'999')","id",@hitlist_)
show_hits(hitlist_)

* screens
? "Screen IDs of changed screens, screen fields, or screen keys:"
hitlist_ := {}
dd_comp(tname+"T",tname+"T1",spath+sname+"T",spath+sname+"T1","id","id",@hitlist_)
dd_comp(tname+"U",tname+"U1",spath+sname+"U",spath+sname+"U1","id+trans(pos,'999')","id",@hitlist_)
dd_comp(tname+"V",tname+"V1",spath+sname+"V",spath+sname+"V1","id+trans(pos,'999')","id",@hitlist_)
show_hits(hitlist_)

* finds
? "Find IDs of changed finds or find items:"
hitlist_ := {}
dd_comp(tname+"O",tname+"O1",spath+sname+"O",spath+sname+"O1","id","id",@hitlist_)
dd_comp(tname+"P",tname+"P1",spath+sname+"P",spath+sname+"P1","id+ialias","id",@hitlist_)
show_hits(hitlist_)

* sequences
? "Sequence IDs of changed sequences:"
hitlist_ := {}
dd_comp(tname+"X",tname+"X1",spath+sname+"X",spath+sname+"X1","id","id",@hitlist_)
show_hits(hitlist_)

* compare system config files
? "Startup/Configuration:"
if sys_comp(tname+"S",spath+sname+"S")
   ? space(5)+"Startup/Configuration differs!"
endif
?

* turn off output
if !empty(output)
   set alternate to
   set alternate off
   close alternate

   * if temp file used, copy to proper device and delete temp file.
   if !empty(o_output)
      copy file (output) to (o_output)
      ferase(output)
   endif
endif

***************************** End of Main Program **************************

*****************************************************************************
*  Function : DD_COMP()
*****************************************************************************
*
*  Purpose  : Compare two CDD DD files.  DD function id's listed in
*             hitlist_ array will not be compared again.
*
*  Syntax   : dd_comp(tdbf,tntx,sdbf,sntx,linkexp,hitlist_)
*
*  Arguments: tdbf is the target file in the current directory (no extension)
*             tntx is the target index in the current directory (no extension)
*             sdbf is the source file (no extension)
*             sntx is the source index (no extension)
*             linkexp is the expression to eval that returns a string
*                used to seek to a record in the target file.
*             hitexp is the expression to eval that returns a string
*                used in the array hitlist_.
*             hitlist_ is the stop list array - passed by reference.
*
*  Returns  : NIL
*
*  Author   : Vick Perry,   Mon  11-30-1992
*
*  Notes    : All files are closed at the beginning and at the end of dd_comp()
*
*  Revisions:
*
*  Examples :
*
*****************************************************************************
function dd_comp(tdbf,tntx,sdbf,sntx,linkexp,hitexp,hitlist_)
   local bLink       // code block for link expression
   local bHit        // code block for hitlist expression
   local seekstr := ""
   local hitstr := ""
   local was_hit := .f.
   local nflds := 0
   local i := 0
   dbcloseall()

   select 0
   net_use(tdbf,.f.,2,"tdbf")
   set index to (tntx)

   select 0
   net_use(sdbf,.f.,2,"sdbf")
   set index to (sntx)

   * define get block for the link expression in sdbf
   bLink := compile(linkexp)

   * define get block for the hit expr in sdbf
   bHit := compile(hitexp)

   * loop and compare
   select sdbf
   do while !sdbf->(eof())
      was_hit := .f.

      * NOTE: YOU MUST BE IN THE SDBF WORKAREA AT THIS POINT!!!!

      * get next seek string value from the linkexp block from sdbf
      seekstr := upper(eval(bLink))

      * get next hit string value from the hitexp block from sdbf
      hitstr := upper(eval(bHit))

      if !ascan(hitlist_,hitstr) > 0
         select tdbf
         dbseek(seekstr)
         if !found()
            * record not found, add to hitlist_
            aadd(hitlist_,hitstr)
         else
            * matching record WAS found, compare fields
            nflds := fcount()
            for i = 1 to nflds
               * if a field value changed, add to hitlist_
               if !sdbf->(fieldget(i)) == tdbf->(fieldget(i))
                  aadd(hitlist_,hitstr)
                  exit
               endif
            next
         endif
      endif

      select sdbf
      dbskip()
   enddo

   * check for those records deleted from source, but still found in
   * target.  Ignore records that are already known as hits.
   select tdbf
   do while !tdbf->(eof())
      * NOTE: YOU MUST BE IN THE TDBF WORKAREA AT THIS POINT!!!!

      * get next seek string value from the linkexp block from sdbf
      seekstr := upper(eval(bLink))

      * get next hit string value from the hitexp block from sdbf
      hitstr := upper(eval(bHit))

      if ascan(hitlist_,hitstr) == 0
         select sdbf
         dbseek(seekstr)
         if !found()
            * record not found, therefore it was deleted from
            * the source DD - add it to hitlist_
            aadd(hitlist_,hitstr)
         endif
      endif

      select tdbf
      dbskip()
   enddo

   dbcloseall()
return nil


*****************************************************************************
*  Function : show_hits()
*****************************************************************************
*
*  Purpose  : display the current hit list array
*
*  Syntax   : show_hits(a_)
*
*  Arguments: a_ is a one-dimensional string array to display
*
*  Returns  : NIL
*
*  Author   : Vick Perry,   Mon  11-30-1992
*
*  Notes    :
*
*  Revisions:
*
*  Examples :
*
*****************************************************************************
function show_hits(a_)
   local alen := len(a_)
   local i := 0
   for i = 1 to alen
      ? space(5) + a_[i]
   next
   ?
return nil


*****************************************************************************
*  Function : sys_comp()
*****************************************************************************
*
*  Purpose  : compare the single record system file
*
*  Syntax   : sys_comp(tdbf,sdbf)
*
*  Arguments: target file, source file
*
*  Returns  : NIL
*
*  Author   : Vick Perry,   Mon  11-30-1992
*
*  Notes    :
*
*  Revisions:
*
*  Examples :
*
*****************************************************************************
function sys_comp(tdbf,sdbf)
   local nflds := 0
   local i := 0
   local l := .f.
   dbcloseall()

   select 0
   net_use(tdbf,.f.,2,"tdbf")
   select 0
   net_use(sdbf,.f.,2,"sdbf")
   nflds := fcount()
   for i = 1 to nflds
      * if a field value changed
      if !sdbf->(fieldget(i)) == tdbf->(fieldget(i))
         l := .t.
         exit
      endif
   next
   dbcloseall()
return l


****
*  NET_USE function with alias naming function
*
*  Trys to open a file for exclusive or shared use.
*  SET INDEXes in calling procedure if successful.
*  Pass the following parameters
*    1. Character - name of the .DBF file to open
*    2. Logical - mode of open (exclusive/.NOT. exclusive)
*    3. Numeric - seconds to wait (0 = wait forever)
*    4. Char - Optional alias name (default is datafile name)
*
*  Example:
*    IF NET_USE("Accounts", .T., 5, "ZZZZZ")
*       SET INDEX TO Name
*    ELSE
*       ? "Account file not available"
*    ENDIF

FUNCTION NET_USE
PARAMETERS file, ex_use, wait, the_alias
PRIVATE forever
private is_alias

forever = (wait = 0)
is_alias = (pcount() = 4)
DO WHILE (forever .OR. wait > 0)

   if is_alias
      IF ex_use
         USE &file EXCLUSIVE alias &the_alias
      ELSE
         USE &file alias &the_alias
      ENDIF
   else
      IF ex_use
         USE &file EXCLUSIVE
      ELSE
         USE &file
      ENDIF
   endif

   IF .NOT. NETERR()           && USE succeeds
      RETURN (.T.)
   ENDIF
   
   INKEY(1) 	          	&& wait 1 second
   wait = wait - 1
ENDDO
RETURN (.F.)			&& USE fails
* End - NET_USE

