///////////////////////////////////////////////
//
// File:    CBASE.CPP
// Author:  Patrick T. Hurley
// Date:    02-06-92 08:22:55pm
// Version: 1.0
//
// Notes:   Implement the cbase classes.
//

#include "CBASE.H"
#include <assert.h>
#include <dir.h>


unsigned extern _stklen = 15000;


CODE4  CodeBaseRoot::root;
CODE4* CodeBaseRoot::cb         = 0;
int    CodeBaseRoot::ref_count  = 0;


// This is a support function from another lib
Boolean fexist (char* fname)
{
	return Boolean (access (fname, 0) == 0);
}


extern "C" void e4hook (CODE4*, int err_code, char* desc1, char* desc2, char* desc3)
{
	putchar ('\a');
	cerr << "CBASE Error code (" << err_code << "): " << e4text (err_code) << endl;
	if (errno || _doserrno)
		cerr << "Compiler error codes: " << errno << ", " << _doserrno << endl;
	if (desc1) cerr << desc1;
	if (desc2) cerr << desc2;
	if (desc3) cerr << desc3;
	if (desc1 || desc2 || desc3) cerr << endl;
	cerr << "Press any key to continue..." << endl;
	abort ();
// 	_cget_chr ();
}


#pragma argsused
int file4lock_hook (CODE4 *cb, char *fname, long offset, long num_bytes, int num_tries )
{
#ifdef LOCK_CLICK
	putchar ('\a');
#endif
	if (num_tries <= cb->lock_attempts)
		return  0;
	else
		return -1;
}


void CodeBaseRoot::undo() {
   if (!ref_count && cb)
	{
		d4init_undo(cb);
		cb = NULL;
	}
}


CodeBaseRoot::CodeBaseRoot() {
   if (!ref_count++) {
      cb = &root;
      d4init (cb);

      // My favorite Setups
      auto_open (True);
      create_error (True);
#if defined(NETWORK)
		exclusive (False);
#else
		exclusive (True);
#endif
		expr_error (True);
      field_name_error (True);
      go_error (True);
		lock_attempts (5);			// retry 5 times before quitting
      open_error (True);
      read_lock (False);
		read_only (False);
		relate_error (True);
      safety (False);
      skip_error (True);
      tag_name_error (True);
   }
}


CodeBaseRoot::~CodeBaseRoot() {
   ref_count--;
	undo ();
}


cfield::cfield (char* alias, char* fldName)
{
	DATA4* dbf = d4data (CodeBaseRoot::cb, alias);
	assert (dbf);
   f = d4field (dbf, fldName);
	assert (f);
}


void cfield::init (cbase& cb, char* fld)
{
   f = d4field (cb.dbf, fld);
	assert (f);
}


Boolean cfield::empty ()
{
	int length = len ();
	for ( char* data=f4assign_ptr(f); length && *data == ' '; data++, length--) ;

	return Boolean (!length);
}


ostream& operator<<(ostream& os, const cfield& f)
{
   switch ( f.type() )
   {
      case 'C':
         os.write (f4ptr(f.f), f.len());
         break;
      case 'M':
         os.write (f4memo_ptr(f.f), f.len());
         break;
      case 'D':
         char* date = f4ptr(f.f);
         os << date4cdow (date) << " " <<
               date4cmonth (date) << " " <<
               date4day (date) << ", " <<
               date4year (date);
         break;
      case 'L':
         os << char(f);
         break;
      case 'N':
         os.width ( f.len ());
         os.precision ( f.decimals ());
         os << double (f);
         break;
      default:
         os << "Unknown Field Type!!!";
         break;
   }

   return os;
}


int yrsbtwn (char S4PTR* bdate, char S4PTR* tdate)
{
	char dbuffer [8];
	if (!tdate)
	{
		tdate = dbuffer;
		date4today (tdate);
	}

	int tyear  = date4year (tdate);
	int tmonth = date4month (tdate);
	int tday   = date4day (tdate);

	int byear  = date4year (bdate);
	int bmonth = date4month (bdate);
	int bday   = date4day (bdate);

	int years  = tyear - byear - 1;
	if ((tmonth > bmonth) || ((tmonth == bmonth) && (tday >= bday)))
		years++;

	return years;
}


int mnthsbtwn (char S4PTR* bdate, char S4PTR* tdate)
{
	char dbuffer [8];
	if (!tdate)
	{
		tdate = dbuffer;
		date4today (tdate);
	}

	int tyear  = date4year (tdate);
	int tmonth = date4month (tdate);
	int tday   = date4day (tdate);

	int byear  = date4year (bdate);
	int bmonth = date4month (bdate);
	int bday   = date4day (bdate);

	int years  = tyear - byear - 1;
	if ((tmonth > bmonth) || ((tmonth == bmonth) && (tday >= bday)))
		years++;

	if ((tmonth < bmonth) || ((tmonth == bmonth) && (tday < bday)))
		tmonth += 12;
	int months = tmonth - bmonth - 1;
	if (tday >= bday || months < 0)
		months++;

	return months + (years * 12);
}


void cbase::init() {
   opened = False;
   dbf = 0;
   name = "";
}


cbase::cbase() {
   init ();
}


cbase::cbase(char* fname) {
   init();
   open(fname);
}


cbase::cbase(char* fname, FIELD4INFO* finfo, TAG4INFO* tinfo) {
   init();
   if (fexist (fname))
      open (fname, finfo, tinfo);
   else
      create(fname, finfo, tinfo);
}


cbase::~cbase() {
   if (opened)
		close();
}


Boolean cbase::chkStruct (FIELD4INFO* st1)
{
	Boolean retval = True;

	if (st1)
	{
		// can only be called on an open dbf
		assert (opened);

		FIELD4INFO* st2    = dbStruct ();
		if (st2)
		{
			FIELD4INFO* freeMe = st2;
//    		char S4PTR  *name ;
//    		char         type ;
//    		unsigned int len ;
//    		unsigned int dec ;
			while ( retval && st1->name && st2->name )
			{
				retval = retval   & (!stricmp (st1->name, st2->name))
										& (st1->type == st2->type)
										& (st1->len  == st2->len)
										& (st1->dec  == st2->dec);
				st1++; st2++;
			}

			// check that the field count is the same...
			retval = retval & (!st1->name) & (!st2->name);
			u4free (freeMe);
		}
	}

	return retval;
}


Boolean cbase::chkIndex (TAG4INFO* st1)
{
	Boolean retval = True;

	if (st1)
	{
		// can only be called on an open dbf
		assert (opened);

		TAG4INFO* st2 = tagInfo ();
		if (st2)
		{
				TAG4INFO* freeMe = st2;
//    			char S4PTR  *name ;
//    			char S4PTR  *expression ;
//    			char S4PTR  *filter ;
//    			int unique ;
//    			unsigned int descending ;
			while ( retval && st1->name && st2->name )
			{
				retval = retval   & (!stricmp (st1->name, st2->name))
										& (!strcmp (st1->expression, st2->expression))
										& (!strcmp (st1->filter, st2->filter))
										& (st1->unique == st2->unique)
										& (st1->descending == st2->descending);
				st1++; st2++;
			}

			// check that the tag count is the same...
			retval = retval & (!st1->name) & (!st2->name);
			u4free (freeMe);
		}
	}

	return retval;
}


void cbase::open (char* fname)
{
   if (opened)
		close();

   dbf = d4open (cb, fname);
   if (dbf)
   {
      opened = True;
      name = new char [strlen (fname) + 1];
      strcpy (name, fname);
		idx = d4index (dbf, name);
   }
}


void cbase::open (char* fname, FIELD4INFO* finfo, TAG4INFO* tinfo) {
	if (opened) close ();

   char dbfname [MAXPATH];
   strcpy (dbfname, fname);
   u4name_ext (dbfname, MAXPATH, "DBF", False);

   char cdxname [MAXPATH];
   strcpy (cdxname, fname);
   u4name_ext (cdxname, MAXPATH, "CDX", False);
   u4name_ext (cdxname, MAXPATH, "CDX", True);

   if (!fexist (dbfname))             // check if file exists
   {
		status (Creating);
      create (fname, finfo, tinfo);
   }
   else if (!fexist (cdxname) && tinfo)        // check if index exists
   {
      cb->auto_open = False;
      open (dbfname);
      index (cdxname, tinfo);
      cb->auto_open = True;
   }
   else                                      // open the file
   {
      open (fname);
   }

	// Verify Structure of dbf file
	if (!chkStruct (finfo))
	{
		status (InvalidDBF);
#ifndef NDEBUG
		cerr << "Invalid DBF Structure" << endl;
#endif
		exit (1);
	}

	// Verify Index of dbf file
	if (tinfo && !chkIndex (tinfo))
	{
		status (InvalidCDX);
		status (Reindexing);
   	index (cdxname, tinfo);
	}
}



Boolean cbase::create (char* fname, FIELD4INFO* finfo, TAG4INFO* tinfo) {
   if ( opened ) {
      close();
   }

   dbf = d4create (cb, fname, finfo, tinfo);
   if (dbf)
   {
      opened = True;
      name = new char [strlen (fname) + 1];
      strcpy (name, fname);
		idx = d4index (dbf, name);
   }

   return opened;
}


Boolean cbase::link (char* alias) {
	if (opened)
		close();

   dbf = d4data (cb, alias);
	idx = d4index (dbf, d4alias (dbf));
	return (dbf) ? True : False;
}


Boolean cbase::close() {
   if (opened)
   {
      flush_all ();
      if (d4close (dbf) < 0)
         return False;
      delete name;
      opened = False;
   }
   return True;
}


Boolean cbase::iclose ()
{
   INDEX4* idx = (INDEX4*) l4first (&(dbf->indexes));
   while ( idx )
   {
      i4close (idx);
      idx = (INDEX4*) l4first (&(dbf->indexes));
   }

   return True;
}


cfield cbase::field (char* fld)
{
   FIELD4* f = d4field (dbf, fld);

   return cfield(f);
}


Boolean cbase::tag (char* tag_name) {
   TAG4* newtag = d4tag(dbf, tag_name);
   if ( newtag )
   {
      d4tag_select (dbf, newtag);
   }
	return (newtag) ? True : False;
}


int cbase::tagCount ()
{
   TAG4* t = 0;
   int tc = 0;
   while ((t = d4tag_next (dbf, t)) != 0) tc++;
   return tc;
}


Boolean cbase::tag (int tag_num) {
   TAG4* t = 0;
   while (tag_num--)
      if ((t = d4tag_next (dbf, t)) == 0) return False;

   d4tag_select (dbf, t);
   return True;
}


Boolean cbase::isTag (char* tag_name)
{
	int old_tag_error = cb->tag_name_error;
	cb->tag_name_error = False;
	Boolean retval = Boolean (d4tag (dbf, tag_name));
	cb->tag_name_error = old_tag_error;
	return retval;
}


int cbase::tagPos (char* tag) {
   TAG4* t = 0;
   int tag_num = 0;

   do
   {
      t = d4tag_next (dbf, t);
      tag_num++;
   } while (t && stricmp (t->alias, tag));
   if (!t) tag_num = 0;

   return tag_num;
}


char* cbase::operator[] (char* fldname) {
   cfield fld (*this, fldname);
   return (char*) fld;
}


Boolean cbase::index (char* name, TAG4INFO* t)
{
	// close index if already open
	idx = d4index (dbf, name);
	if (idx) i4close (idx);

   idx = i4create (dbf, name, t);

   return Boolean (idx);
}


Boolean cbase::index (char* name, char* expr, char* filter, int unique, int desc)
{
   TAG4INFO tinfo[] = { { name, expr, filter, unique, desc }, {0,0,0,0} };
   char fname [MAXPATH];
   strcpy (fname, name);
   u4name_ext (fname, MAXPATH, "CDX", True);
   unlink (fname);
   idx = i4create (dbf, name, tinfo);
   if (idx) tag (name);

   return Boolean (idx);
}


char* cbase::key (char* buf, int bufsize)
{
   char* text;
   int len = expr4vary (d4tag_default (dbf)->expr, &text);

   if (buf)
   {
      if (bufsize == -1) bufsize = len + 1;
      else if (len >= bufsize) len = bufsize - 1;
   } else {
      buf = new char [len+1];
   }
   strncpy (buf, text, len);
   *(buf + len) = NULL;

   return buf;
}


#ifdef PH4STATUS
#include <ph4all.h>

#pragma argsused
void ph4status (TAG4* t4, ph4state status, long count, long i_rec) {}
#pragma argsused
void ph4status (INDEX4* i4, ph4state status, long count, long i_rec) {}
#pragma argsused
void ph4status (long count, long i_rec) {}
#endif



//  End of CBASE.CPP
