 /**\t******************************************************************
   Creation Date .......  Tue  02-14-1995  06:57:07
   Filename  ...........  timer.cpp
   Project .............  High Resolution Timer Demo
   Author  .............  Matthew J. W. Ratcliff
   Language  ...........  C++
   Operating System  ...  Windows
   Prefix ..............  TIA
   Function:    Demonstrate the TimerCount function for high
        resolution (1ms) timing in Windows.  Support comes
        from toolhelp.dll.  Include <toolhelp.h> and you
        can call the TOOLHELP support functions.  TOOLHELP.DLL
        must be on your system (it is with any Standard Windows 3.1
        installation).

 **   NO-copyright, Demo software by Matthew J. W. Ratcliff           **
 **\t******************************************************************/

/*\r********************** Revision History ****************************
   Date    By           Change Description
dd-mmm-yy  nnn          text
---------  ----         -----------------------------------------------

**\r*/

/*\i******************** System Include Files *************************/
#ifndef NOSTRICT
#define STRICT // for MSVC users
#endif

#include <windows.h>
#include <ctl3d.h>
#include <toolhelp.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/********************** Constant Include Files ************************/
#include "resource.h"
#include "liidSys.h"    /* System portability typedefs & constants    */
/***************** External Variable Include Files ********************/

/***************** External Procedure Include Files *******************/
#include "wnuutil.h"
/*\i*/

/*\m********************** Module Constants ***************************/
// Initialization file
#define TIM_DEFAULT_FILE        "WINTIMER.INI"
#define TIM_SLEN                240

#define TIM_TIMER_ID            1
#define TIM_100_MS              100u
#define TIM_1_S                 1000u
#define TIM_1_M                 60000u

/************************* Module Variables ***************************/
LOCAL const char *TIM_PROC_NAME="WINTIMER";
LOCAL const char *TIM_ICON_NAME="WINTIMERICO";

LOCAL char       *timMsg;
LOCAL BOOLEAN     timIsNew = FALSE_;

LOCAL WORD        timDelayTime = TIM_100_MS;

LOCAL TIMERINFO   timHAStartTime; // High accuracy start time
LOCAL TIMERINFO   timHAStopTime;  // High accuracy stop  time
LOCAL DWORD       timHApgStart;   // High accuracy program start time

LOCAL DWORD       timLAStartTime; // Low  accuracy start time
LOCAL DWORD       timLAStopTime;  // Low  accuracy stop  time
LOCAL DWORD       timLApgStart;   // Low  accuracy program start time

LOCAL BOOL        timTimerInUse;
LOCAL HWND        timHdlg;

/************************* Module Procedures **************************/
long FAR PASCAL WndProc
  ( HWND pasHwindow,
    WORD pasMessage,
    WORD pasWparam,
    LONG pasLparam );

long FAR PASCAL WndCommand
  ( HWND pasHwindow,
    WORD pasMessage,
    WORD pasWparam,
    LONG pasLparam );

BOOL tiaGetTimer
  ( TIMERINFO *refTimerInfo );

void tiaReportElapsedTime( const char *pasTimeType,
                           HWND pasHdlg );

/*\m*/
void tiaReportElapsedTime( const char *pasTimeType,
                           HWND pasHdlg )
{
/*********** Local Constant & Variable Declarations *******************/
HWND tilHlist;
WORD tilLast;
long tilHAt;
long tilLAt;

/************************* Procedure Body *****************************/
//"TC/WM - TC Time - WM Time - Delta",
tilHAt = timHAStopTime.dwmsThisVM - timHAStartTime.dwmsThisVM;
tilLAt = timLAStopTime-timLAStartTime;

sprintf( timMsg,"%-4s  - %5ldms - %5ldms - %5ldms",
         pasTimeType,
         tilHAt,
         tilLAt,
         tilHAt - tilLAt );
tilHlist = GetDlgItem( pasHdlg, IDC_TIMELIST );
SendMessage( tilHlist, LB_INSERTSTRING, -1, (LPARAM)timMsg );
tilLast = SendMessage( tilHlist, LB_GETCOUNT, 0, 0);
SendMessage( tilHlist, LB_SETCURSEL, tilLast-1, 0 );
} /* tiaReportElapsedTime end */

/*\p********************************************************************
**                                                                    **
    NAME:  tiaGetTimer

    PURPOSE:  to get a reading of the time, accurate to the
      nearest MILLISECOND the time since windows was started
      (dwmsSinceStart) and since this application was
      fired up (dwmsThisVM).

      dwmSinceStart is accurate to the nearest ms
      It is "about" the same as the value returned by
      GetCurrentTime and GetTickCount functions, but
      they are accurate only to the nearest 55ms (1/18Hz).


** By Matthew J. W. Ratcliff                                          **
**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   tilResult          BOOL, TRUE if read Ok, FALSE if failed to read**
**\p*******************************************************************/
BOOL tiaGetTimer( TIMERINFO *refTimerInfo )
{
/*********** Local Constant & Variable Declarations *******************/
BOOL    tilResult;
/************************* Procedure Body *****************************/
tilResult = FALSE;
if (refTimerInfo) // parameter validation
  {
  // Tell windows size of the data structure
  refTimerInfo->dwSize = sizeof(TIMERINFO);
  // Use TOOLHELP.DLL function to get
  // elapsed time to the nearest millisecond resolution.
  tilResult = TimerCount( refTimerInfo );
  if (!tilResult)
    {
    MessageBox( NULL, "Failed to get Timer Count",
                NULL, MB_OK | MB_TASKMODAL );
   }
 }
return(tilResult);
} /* tiaGetTimer end */

/*\p********************************************************************
**                                                                    **
    NAME:  WinMain

    PURPOSE:  to fire up the main application for the WINTIMER
        application.

** By Matthew J. W. Ratcliff                                          **
**\p*******************************************************************/

int PASCAL WinMain( HINSTANCE pasInstance,
                    HINSTANCE pasPrevInstance,
                    LPSTR     pasCmdLine,
                    int       pasCmdShow )
{
/*********** Local Constant & Variable Declarations *******************/
int             tilOpenOk;
WNDCLASS        tilWndclass;
BOOLEAN         tilHwnd;
FARPROC         tilDlgProc;
STAT_TYPE       tilErr;
char           *tilBlank;
/************************* Procedure Body *****************************/
tilHwnd = NULL;
if (!pasPrevInstance)
  {
  tilErr = SUCCEEDED_;
  }
else
  {
  MessageBox( NULL, "Already Running", TIM_PROC_NAME,
              MB_OK | MB_ICONASTERISK | MB_TASKMODAL );
  tilErr    = FAILED_; // Already running, bring to front
  tilOpenOk = FALSE_;
  MessageBeep(0);
  BringWindowToTop(FindWindow(TIM_PROC_NAME,NULL));
  }

if (tilErr == SUCCEEDED_)
  {
  tilWndclass.style           = CS_HREDRAW | CS_VREDRAW;
  tilWndclass.lpfnWndProc     = (WNDPROC)WndProc;
  tilWndclass.cbClsExtra      = 0;
  tilWndclass.cbWndExtra      = DLGWINDOWEXTRA;
  tilWndclass.hInstance       = pasInstance;
  tilWndclass.hIcon           = LoadIcon(pasInstance, TIM_ICON_NAME );
  tilWndclass.hCursor         = LoadCursor( NULL, IDC_ARROW );
  tilWndclass.hbrBackground   = (HBRUSH)GetStockObject(LTGRAY_BRUSH);
  tilWndclass.lpszMenuName    = NULL; // No menu
  tilWndclass.lpszClassName   = TIM_PROC_NAME;
  tilHwnd                     = RegisterClass( &tilWndclass );
  }

// Enable 3d controls too.  Cool, eh?
if (tilHwnd)
  {
  timHdlg = NULL;
  Ctl3dRegister( pasInstance );
  Ctl3dAutoSubclass( pasInstance );
  tilDlgProc = MakeProcInstance( (FARPROC)WndProc, pasInstance );
  DialogBox( pasInstance, TIM_PROC_NAME, NULL, (DLGPROC)tilDlgProc );
  tilOpenOk = TRUE_;
  Ctl3dUnregister( pasInstance );
  }
else
  {
  tilOpenOk = FALSE_;
  MessageBox( NULL,
    "Failed to Iniz Dialog App. Correct Class Name?",
    TIM_PROC_NAME, MB_OK | MB_ICONASTERISK | MB_TASKMODAL );
  }
return(tilOpenOk);
} /* WinMain end */

/*\p********************************************************************
**                                                                    **
    NAME:  exitProgram

    PURPOSE:  to free storage and exit the program

** By Matthew J. W. Ratcliff                                          **
**\p*******************************************************************/

void exitProgram()
{

if (timTimerInUse)
  {
  KillTimer( timHdlg, TIM_TIMER_ID );
  }
LIMDELA(timMsg);
PostQuitMessage(0);
}


/*\p********************************************************************
**                                                                    **
    NAME:  WndProc

    PURPOSE:  to handle window message processing for this
      dialog box application

** By Matthew J. W. Ratcliff                                          **
**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   tilReturnVal       long, return value                            **
**\p*******************************************************************/

long FAR PASCAL WndProc
  ( HWND pasHwindow,
    WORD pasMessage,
    WORD pasWparam,
    LONG pasLparam )
{
/*********** Local Constant & Variable Declarations *******************/

static long           tilReturnVal      = NULL;
static HINSTANCE      tilHinstance      = NULL;
HMENU                 tilHm;

/************************* Procedure Body *****************************/
tilReturnVal = 0;

switch ( pasMessage )
  {
  case WM_CREATE:
    tilHinstance = (HINSTANCE)((LPCREATESTRUCT)pasLparam)->hInstance;
    timMsg       = new char[TIM_SLEN];
    break;

  case WM_SYSCOLORCHANGE:
    Ctl3dColorChange();
    break;

  case WM_INITDIALOG:
    WNUbusy(); // Busy cursor
    // setup dialog with initial values here, if needed
    // also process timArgv0File here if needed.
    // Initialize program start times
    tiaGetTimer( &timHAStartTime );
    timHApgStart = timHAStartTime.dwmsSinceStart;
    timLApgStart = GetCurrentTime();
    tilHm = GetMenu( pasHwindow );
    CheckMenuItem( tilHm, IDM_FILE_10MS, MF_CHECKED );
    timHdlg       = pasHwindow;
    timTimerInUse = FALSE;
    Ctl3dSubclassDlg(pasHwindow, CTL3D_ALL);
    tilReturnVal  = TRUE_;
    WNUdone(); // Normal cursor
    break;

  case WM_CTLCOLOR:
    tilReturnVal = (long)(void far *)Ctl3dCtlColorEx( pasMessage,
                                                      pasWparam,
                                                      pasLparam );
    if (tilReturnVal == NULL)
      {
      tilReturnVal = DefWindowProc( pasHwindow,
                                    pasMessage,
                                    pasWparam,
                                    pasLparam );
      }
    break;
      
  case WM_TIMER:
    tiaGetTimer( &timHAStopTime );
    timLAStopTime    = GetCurrentTime();
    SetDlgItemText(pasHwindow,IDC_MESSAGE,"WM Timer Done");
    KillTimer( pasHwindow, TIM_TIMER_ID );
    timTimerInUse = FALSE;
    tiaReportElapsedTime( "WM", // WM_TIMER, low accuracy timing
                          pasHwindow );
    break;

  case WM_COMMAND:
    tilReturnVal = WndCommand( pasHwindow,
                               pasMessage,
                               pasWparam,
                               pasLparam );
    break;

  case WM_DESTROY:
    exitProgram();
    break;

  default:
    tilReturnVal = DefWindowProc( pasHwindow,
                                  pasMessage,
                                  pasWparam,
                                  pasLparam );
    break;
  }

return(tilReturnVal);
} /* WndProc end */

/*\p********************************************************************
**                                                                    **
    NAME:  WndCommand

    PURPOSE:  to handle WM_COMMAND processing.

** By Matthew J. W. Ratcliff                                          **
**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   tilReturnVal       long, return value                            **
**\p*******************************************************************/

long FAR PASCAL WndCommand
  ( HWND pasHwindow,
    WORD pasMessage,
    WORD pasWparam,
    LONG pasLparam )
{
/*********** Local Constant & Variable Declarations *******************/
long    tilReturnVal;
HMENU   tilHm;
long    tilLAt;
long    tilHAt;
/************************* Procedure Body *****************************/
tilReturnVal = 0l;


switch( pasWparam )
  {
  case IDOK:
    // save changes and exit, possibly?
    break;

  case IDC_WAIT1MS: // Wait 100ms using conventional timer
    if (!SetTimer( pasHwindow,
                   TIM_TIMER_ID,
                   timDelayTime,
                   NULL )) // No callback func, we'll just
                           // handle the WM_TIMER message ourselves.
      {
      MessageBox( pasHwindow,
                  "Too Many Clocks or Timers",NULL,
                  MB_ICONASTERISK | MB_OK );
      }
    else
      {
      sprintf(timMsg,"WM Timer, duration %ums ...",
              timDelayTime );
      SetDlgItemText(pasHwindow,IDC_MESSAGE,timMsg);
      timTimerInUse = TRUE;
      timLAStartTime = GetCurrentTime();
      tiaGetTimer( &timHAStartTime );
      }
    break;
  case IDC_TCWAIT1MS: // Wait 100ms using TOOLHELP's hi accuracy timer
    WNUbusy(); // I'm busy and getting accurate timing
    sprintf(timMsg,"TC Timer, duration %ums ...",
            timDelayTime );
    SetDlgItemText(pasHwindow,IDC_MESSAGE,timMsg);
    timLAStartTime = GetCurrentTime();
    tiaGetTimer( &timHAStartTime );
    do
      {
      tiaGetTimer( &timHAStopTime );
      }
    while ((timHAStopTime.dwmsThisVM -
             timHAStartTime.dwmsThisVM) < timDelayTime);
    timLAStopTime = GetCurrentTime();
    tiaReportElapsedTime( "TC", // TOOLHELP counter, hi accuracy
                          pasHwindow );
    SetDlgItemText(pasHwindow,IDC_MESSAGE,"TC Timer Done");
    WNUdone(); // I'm not busy now
    break;
  case IDC_GETTIME:
    tiaGetTimer( &timHAStopTime );
    timLAStopTime = GetCurrentTime();
    tilHAt = timHAStopTime.dwmsSinceStart-timHApgStart;
    tilLAt = timLAStopTime - timLApgStart; 
    sprintf( timMsg,
      "WC: %-7ldms  TC: %-7ldms Delta: %-5ldms",
             tilLAt,
             tilHAt,
             tilHAt-tilLAt );             
    SetDlgItemText(pasHwindow,IDC_MESSAGE,timMsg);
    break;

  case IDM_FILE_10MS:
    timDelayTime = TIM_100_MS;
    tilHm = GetMenu( pasHwindow );
    CheckMenuItem( tilHm, IDM_FILE_10MS, MF_CHECKED );
    CheckMenuItem( tilHm, IDM_FILE_1S,   MF_UNCHECKED );
    CheckMenuItem( tilHm, IDM_FILE_1M,   MF_UNCHECKED );
    SetDlgItemText( pasHwindow, IDC_WAIT1MS,
                    "WM_TIMER Pause 100 milliseconds");
    SetDlgItemText( pasHwindow, IDC_TCWAIT1MS,
                    "TimerCount Pause 100 Milliseconds");
    break;
  case IDM_FILE_HELP:
    MessageBox( pasHwindow,
"High Resolution Timer Test/Demo\r\n\
By Matthew J. W. Ratcliff\r\n\
GEnie: Mat.Rat\r\n\
CIS: 76711,1443\r\n\
LearnWare, unCopyRight Feb, 1995",
                "HiRez Timer",
                MB_OK | MB_ICONEXCLAMATION );
    break;

  case IDM_FILE_1S:
    timDelayTime = TIM_1_S;
    tilHm = GetMenu( pasHwindow );
    CheckMenuItem( tilHm, IDM_FILE_10MS, MF_UNCHECKED );
    CheckMenuItem( tilHm, IDM_FILE_1S,   MF_CHECKED );
    CheckMenuItem( tilHm, IDM_FILE_1M,   MF_UNCHECKED );
    SetDlgItemText( pasHwindow, IDC_WAIT1MS,
                    "WM_TIMER Pause 1 second");
    SetDlgItemText( pasHwindow, IDC_TCWAIT1MS,
                    "TimerCount Pause 1 second");
    break;
  case IDM_FILE_1M:
    timDelayTime = TIM_1_M;
    tilHm = GetMenu( pasHwindow );
    CheckMenuItem( tilHm, IDM_FILE_10MS, MF_UNCHECKED );
    CheckMenuItem( tilHm, IDM_FILE_1S,   MF_UNCHECKED );
    CheckMenuItem( tilHm, IDM_FILE_1M,   MF_CHECKED );
    SetDlgItemText( pasHwindow, IDC_WAIT1MS,
                    "WM_TIMER Pause 1 minute");
    SetDlgItemText( pasHwindow, IDC_TCWAIT1MS,
                    "TimerCount Pause 1 minute");
    break;
  case IDM_FILE_EXIT:
  case IDC_EXIT:
  case IDCANCEL: // exit program
    // Query to cancel changes before exiting, possibly
    EndDialog( pasHwindow, FALSE_);
    exitProgram();
    tilReturnVal = TRUE_;
    break;

  default:
    tilReturnVal = DefWindowProc( pasHwindow,
                                  pasMessage,
                                  pasWparam,
                                  pasLparam );
    break;
  }

return(tilReturnVal);
} /* WndCommand end */

