/****************************************************************************
*****************************************************************************
***                                                                       ***
***  TFormatLine - A formatted derivative of TInputLine                   ***
***                                                                       ***
***  Formatting Specifiers:                                               ***
***    U = Uppercase Alpha/Numeric                                        ***
***    u = Alpha/Numeric                                                  ***
***    A = Uppercase Alpha                                                ***
***    a = Alpha                                                          ***
***    N = Numeric                                                        ***
***                                                                       ***
***  Any other character passed in the format string will be treated as   ***
***  a literal character and will be displayed in the string              ***
***                                                                       ***
*** Copyright 1991                                                        ***
*** Golden E. Murray                                                      ***
*** Spanish Fork, UT                                                      ***
*** (801) 798-0223                                                        ***
*****************************************************************************
****************************************************************************/
#include <ctype.h>
#include <string.h>
#include "formatln.hpp"

const int CONTROL_Y  = 25;

TFormatLine::TFormatLine( const TRect& bounds, const char *aFormat, int aDataLen):
	TInputLine(bounds, strlen(aFormat) + 1), dataLen(aDataLen)
{
	format = new char[maxLen];
	strcpy(format,aFormat);
	//*** put format into data
	for (int n = 0; n < maxLen; n++)
	{
		if (isInputPos(n))
			data[n] = ' ';
		else
			data[n] = format[n];
	}
	data[n] = '\0';
	out 	 = new char[dataLen];
	memset(out, 0, dataLen);
}

TFormatLine::~TFormatLine()
{
	if (format)
		delete format;
	if (out)
		delete out;
}

void TFormatLine::getData( void *rec )
{
	//*** Copy the formatted string to the out buffer
	//*** without the formatting characters
	int outPos = 0;
	for (int n = 0; n < maxLen; n++)
	{
		switch (format[n])
		{
			case 'U':
			case 'u':
			case 'A':
			case 'a':
			case 'N':
			case 'n':
				out[outPos++] = data[n];
				break;
		}
	}
	out[outPos] = EOS;

	//*** Now copy the out buffer into the rec parameter
	memcpy( rec, out, dataLen );
}

void TFormatLine::handleEvent( TEvent& event )
{
	TView::handleEvent(event);
	if ( (state & sfSelected) != 0 )
	{
		switch( event.what )
		{
			case evKeyDown:
				switch( ctrlToArrow(event.keyDown.keyCode) )
				{
					case kbLeft:
						CursorBack();
						break;

					case kbRight:
						CursorForward();
						break;

					case kbHome:
						curPos = -1;
						CursorForward();
						break;

					case kbEnd:
						CursorEnd();
						break;

					case kbBack:
						CursorBack();
						if( curPos >= 0 )
						{
							data[curPos] = ' ';
							if( firstPos > 0 )
								firstPos--;
						}
						break;

					case kbDel:
						atDelete(curPos);
						break;

					default:
						if( event.keyDown.charScan.charCode >= ' ' )
						{
							deleteSelect();
							if( !isInputPos(curPos) )
								CursorForward();
							if( !isInputPos(curPos) )
							{
								clearEvent( event );
								return;
							}
							//*** Check for valid input for position
							char thisChar = event.keyDown.charScan.charCode;
							switch (format[curPos])
							{
								case 'A':
									if ( !isalpha(thisChar) &&
										(thisChar != ' '))
										return;
								case 'U':	thisChar = toupper(thisChar);
									break;
								case 'a':
									if ( !isalpha(thisChar) &&
										(thisChar != ' '))
										return;
									break;
								case 'N':
								case 'n': if ( !isdigit(thisChar) ) return;
							}
							//*** If we have gotton this far the character is valid
							data[curPos] = thisChar;
							int oldPos = curPos;
							CursorForward();
							if (oldPos == curPos)
							{
								curPos = maxLen;
								CursorBack();
								curPos++;
							}
						}
						else if( event.keyDown.charScan.charCode == CONTROL_Y )
						{
							strcpy(data,format);
							CursorForward();
						}
						else
						{
							TInputLine::handleEvent(event);
							return;
						}
				}
				selStart = 0;
				selEnd = 0;
				if( firstPos > curPos )
					firstPos = curPos;
				int i = curPos - size.x + 3;
				if( firstPos < i )
					firstPos = i;
				drawView();
				clearEvent( event );
				break;

			default:
				TInputLine::handleEvent(event);
		}
	}
}

void TFormatLine::setData( void *rec )
{
	//*** Copy the rec buffer to the data buffer
	//*** expanding it to include the formatting characters
	int recPos = 0;
	int endRecFound = False;
	for (int n = 0; n < maxLen; n++)
	{
		if( isInputPos(n) )
		{
			if (((char *)rec)[recPos] == '\0')
				endRecFound = True;
			if (endRecFound)
				data[n] = ' ';
			else
				data[n] = ((char *)rec)[recPos++];
		}
		else
			data[n] = format[n];
	}
	data[maxLen] = EOS;
	selectAll( True );
}

int TFormatLine::CursorBack()
{
	int oldPos = curPos;
	int tempPos = oldPos;
	if( tempPos > 0 )
	{
		int foundPlace = False;
		while( tempPos > 0 && !foundPlace )
		{
			if( isInputPos(--tempPos) )
			{
				foundPlace = True;
				curPos = tempPos;
			}
		}
	}
	if (oldPos == curPos)
		return False;
	return True;
}

int TFormatLine::CursorForward()
{
	int oldPos = curPos;
	int tempPos = oldPos;
	if( tempPos < maxLen )
	{
		int foundPlace = False;
		while( tempPos < maxLen && !foundPlace )
		{
			if( isInputPos(++tempPos) )
			{
				foundPlace = True;
				curPos = tempPos;
			}
		}
	}
	if (oldPos == curPos)
	{
		CursorEnd();
		return False;
	}
	return True;
}

int TFormatLine::isInputPos(int Position)
{
	switch( format[Position] )
	{
		case 'U':
		case 'u':
		case 'A':
		case 'a':
		case 'N':
		case 'n':
			return True;
		default:
			return False;
	}
}

void TFormatLine::atDelete(int deletePos)
{
	//*** make sure that we are on an input character position ***
	if( !isInputPos(deletePos) )
	{
		CursorForward();
		deletePos = curPos;
	}
	if( !isInputPos(deletePos) )
		return;

	//*** Now move the input position characters down the the previous space ***
	int lastPos = deletePos;
	for (int n = lastPos; n < maxLen; n++)
	{
		if ( isInputPos(n) )
		{
			data[lastPos] = data[n];
			lastPos = n;
			data[n] = ' ';
		}
	}
}

void TFormatLine::deleteSelect()
{
	if( selStart < selEnd )
	{
		//*** Find out how many format positions are in the select range
		int selected = 0;
		int selFirst = -1;
		for(int n = selStart; n <= selEnd; n++ )
		{
			if (isInputPos(n))
			{
				if (selFirst < 0)
					selFirst = n;
				selected++;
			}
		}

		//*** Now delete the number of selected items ***
		for(n = 0; n++ < selected;)
			atDelete(selFirst);

		if (selFirst >= 0)
			curPos = selFirst;
	}
}

void TFormatLine::selectAll( Boolean enable )
{
	int charFound = False;
	int curPos = -1;
	selEnd = selStart = 0;
	if ( enable )
	{
		//*** Check to see if any of the format positions are not space
		CursorForward();
		while ( CursorForward() && !charFound )
		{
			if ( data[curPos] != ' ' )
				charFound = True;
		}
		if (charFound)
			selEnd = curPos;
	}
	else
	{
		curPos = selEnd = 0;
	}
	firstPos = max( 0, curPos-size.x+3 );
	drawView();
}

void TFormatLine::CursorEnd()
{
	curPos = maxLen;
	while ( CursorBack() && data[curPos] == ' ') {}
	if (data[curPos] != ' ') curPos++;
}