/***************************************************************

	sfs_pr.c        Program Interpreter for
			Space Flight Simulator

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023
			Duke Station
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not
	be used for illegal activities.

***************************************************************/

#include "stdio.h"
#include "time.h"
#include "ctype.h"
#include "bw.h"
#include "gr.h"
#include "kb.h"
#include "ui.h"
#include "as.h"
#include "sfs.h"

#ifdef  __STDC__
#include "malloc.h"
#include "stdlib.h"
#else
extern  char * malloc();
#define size_t   int
#define time_t   long
#endif

#define FDFILESIZE 128
#define FILE_DELIMITER  ";---------------------------------------\n"

extern   double atof();
extern   int    atoi();
extern   long   atol();
extern   struct as_focus * pr_getfd();
extern char *fntolc();

#define MAXARGS    4

static   char   arguments[ MAXARGS ][ 64 ];

sfs_pr( filename, pr_name, sorbit_array, n_orbits )
   char *filename;                      /* name of program file */
   char *pr_name;                       /* buffer for program name */
   struct sfs_orbit **sorbit_array;     /* pointer to array of sfs orbit structures */
   int n_orbits;                        /* number of orbit structures in arrays */
   {
   FILE *data;
   static char buffer[ 128 ];
   register int orbit;
   static char fdfile[ FDFILESIZE ];

   /* set up a default focal data file */

   strcpy( fdfile, DEFAULTFD );        /* default focal data file */

   /* deallocate any allocated orbit memory */

   for ( orbit = 0; orbit < n_orbits; ++orbit )
      {
      if ( sorbit_array[ orbit ] != NULL )
	 {

#ifdef  OLD_DEBUG
	 sprintf( bw_ebuf, "Freeing memory for orbit %d", orbit );
	 bw_debug( bw_ebuf );
#endif

	 free( sorbit_array[ orbit ]->aorbit->focus->name );
	 free( sorbit_array[ orbit ]->aorbit->focus->adjective );
	 free( sorbit_array[ orbit ]->aorbit->focus->fdfile );
	 free( sorbit_array[ orbit ]->aorbit->focus );
	 free( sorbit_array[ orbit ]->aorbit );
	 free( sorbit_array[ orbit ]->name );
	 free( sorbit_array[ orbit ] );
	 sorbit_array[ orbit ] = NULL;
	 }
      }

   /* Open the program file */

   if ( ( data = fopen( filename, "rb" )) == NULL )
      {
      sprintf( bw_ebuf, PRERR_OPEN, filename );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }

   /* Get the program name -- always first line of file */

   as_fgets( pr_name, 127, data );
   pr_stripcr( pr_name );

   /* Inform the viewer */

   sprintf( bw_ebuf, PR_NOTICE,
      filename, pr_name );
   bw_message( bw_ebuf );

   /* Parse each line of the file */

   while( feof( data ) == FALSE )
      {
      as_fgets( buffer, 127, data );
      if ( feof( data ) == FALSE )
	 {
	 pr_parse( buffer, fdfile, sorbit_array, n_orbits );
	 }
      }

   fclose( data );

   /* Initialize all orbits utilizing data entered */

   for ( orbit = 0; orbit < ORBITS; ++orbit )
      {
      if ( sorbit_array[ orbit ] != NULL )
	 {

#ifdef  OLD_DEBUG
	 sprintf( bw_ebuf, "[pr:] sfs_pr(): reinitializing orbit %d",
	    orbit );
	 bw_debug( bw_ebuf );
#endif

	 or_init( sorbit_array[ orbit ]->aorbit,
	    sorbit_array[ orbit ]->aorbit->periapsis,
	    sorbit_array[ orbit ]->aorbit->apoapsis );
	 }
      }

   return TRUE;
   }

pr_parse( buffer, fdfile, sorbit_array, n_orbits )
   char *buffer;
   char *fdfile;
   struct sfs_orbit **sorbit_array;
   int n_orbits;
   {
   int n;        /* number of arguments  */
   int i;
   long l;
   double d;
   static char x_name[] = "";

   n = pr_getargs( buffer );       /* get arguments   */
   if ( n == FALSE )
      {
      return;
      }

   if ( strcmp( arguments[ 0 ], "FOCUS" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading focus" );
#endif
      i = atoi( arguments[ 1 ] );
      strncpy( fdfile, fntolc( arguments[ 2 ] ), (size_t) FDFILESIZE );
      if ( sorbit_array[ i - 1 ] == NULL )
	 {
	 pr_default( sorbit_array, i - 1, n_orbits, fdfile );
	 }
      else
	 {
	 sorbit_array[ i - 1 ]->aorbit->focus
	    = pr_getfd( sorbit_array, i - 1, n_orbits, fdfile );
	 }
      }

   else if ( strcmp( arguments[ 0 ], "ORB" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading orb" );
#endif
      i = atoi( arguments[ 1 ] );
      if ( sorbit_array[ i - 1 ] == NULL )
	 {
	 pr_default( sorbit_array, i - 1, n_orbits, fdfile );
	 }
      strncpy( sorbit_array[ i - 1 ]->orbfile,
	 fntolc( arguments[ 2 ] ), (size_t) SFS_FILESIZE - 1 );
      }

   else if ( strcmp( arguments[ 0 ], "GRID" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading grid" );
#endif
      i = atoi( arguments[ 1 ] );
      if ( sorbit_array[ i - 1 ] == NULL )
	 {
	 pr_default( sorbit_array, i - 1, n_orbits, fdfile );
	 }
      strncpy( sorbit_array[ i - 1 ]->gridfile,
	 fntolc( arguments[ 2 ] ), (size_t) SFS_FILESIZE - 1 );
      }

   else if ( strcmp( arguments[ 0 ], "SURFACE" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading surface" );
#endif
      i = atoi( arguments[ 1 ] );
      if ( sorbit_array[ i - 1 ] == NULL )
	 {
	 pr_default( sorbit_array, i - 1, n_orbits, fdfile );
	 }
      strncpy( sorbit_array[ i - 1 ]->surfile,
	 fntolc( arguments[ 2 ] ), (size_t) SFS_FILESIZE - 1 );
      }

   else if ( strcmp( arguments[ 0 ], "PERIAPSIS" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading periapsis" );
#endif
      i = atoi( arguments[ 1 ] );
      d = atof( arguments[ 2 ] );
      if ( sorbit_array[ i - 1 ] == NULL )
	 {
	 pr_default( sorbit_array, i - 1, n_orbits, DEFAULTFD );
	 }
      if ( d < 0 )
	 {
	 bw_error( PRERR_PRI );
	 }
      else
    {
    sorbit_array[ i - 1 ]->aorbit->periapsis = d;
    }
      }

   else if ( strcmp( arguments[ 0 ], "APOAPSIS" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading apoapsis" );
#endif
      i = atoi( arguments[ 1 ] );
      d = atof( arguments[ 2 ] );

      if ( sorbit_array[ i - 1 ] == NULL )
    {
    pr_default( sorbit_array, i - 1, n_orbits, DEFAULTFD );
    }
      if ( ( d < 0 ) || ( d < sorbit_array[ i - 1 ]->aorbit->periapsis ))
    {
    bw_error( PRERR_APO );
    }
      else
    {
    sorbit_array[ i - 1 ]->aorbit->apoapsis = d;
    }
      }

   else if ( strcmp( arguments[ 0 ], "INCLINATION" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading inclination" );
#endif
      i = atoi( arguments[ 1 ] );
      d = atof( arguments[ 2 ] );
      if ( sorbit_array[ i - 1 ] == NULL )
    {
    pr_default( sorbit_array, i - 1, n_orbits, DEFAULTFD );
    }
      if ( ( d < OR_INC_MIN ) || ( d > OR_INC_MAX ) )
    {
    sprintf( bw_ebuf, PRERR_INC, OR_INC_MIN, OR_INC_MAX );
    bw_error( bw_ebuf );
    }
      else
    {
    sorbit_array[ i - 1 ]->aorbit->inclination = d;
    }
      }

   else if ( strcmp( arguments[ 0 ], "LONAN" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading lon asc. node" );
#endif
      i = atoi( arguments[ 1 ] );
      d = atof( arguments[ 2 ] );
      if ( sorbit_array[ i - 1 ] == NULL )
    {
    pr_default( sorbit_array, i - 1, n_orbits, DEFAULTFD );
    }
      if ( ( d < OR_LAN_MIN ) || ( d > OR_LAN_MAX ) )
    {
    sprintf( bw_ebuf, PRERR_LAN, OR_LAN_MIN, OR_LAN_MAX );
    bw_error( bw_ebuf );
    }
      else
    {
    sorbit_array[ i - 1 ]->aorbit->lon_an = d;
    }
      }

   else if ( strcmp( arguments[ 0 ], "ARGPER" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading arg. perigee" );
#endif
      i = atoi( arguments[ 1 ] );
      d = atof( arguments[ 2 ] );
      if ( sorbit_array[ i - 1 ] == NULL )
    {
    pr_default( sorbit_array, i - 1, n_orbits, DEFAULTFD );
    }
      if ( ( d < OR_ARP_MIN ) || ( d > OR_ARP_MAX ) )
    {
    sprintf( bw_ebuf, PRERR_ARP, OR_ARP_MIN, OR_ARP_MAX );
    bw_error( bw_ebuf );
    }
      else
    {
    sorbit_array[ i - 1 ]->aorbit->arg_per = d;
    }
      }

   else if ( strcmp( arguments[ 0 ], "NAME" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading name" );
#endif
      i = atoi( arguments[ 1 ] );
      if ( sorbit_array[ i - 1 ] == NULL )
	 {
	 pr_default( sorbit_array, i - 1, n_orbits, DEFAULTFD );
	 }
      if ( ( sorbit_array[ i - 1 ]->name
	 = malloc( (size_t) strlen( arguments[ 2 ] ) + 2 )) == NULL )
	 {
	 bw_error( PRMEMERR_NAME );
	 sfs_exit();
	 exit(1);
	 }

      pr_stripcr( arguments[ 2 ] );
      strcpy( sorbit_array[ i - 1 ]->name, arguments[ 2 ] );
      }

   else if ( strcmp( arguments[ 0 ], "TFACTOR" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading tfactor" );
#endif
      l = atol( arguments[ 1 ] );
      if ( l < 1 )
    {
    bw_error( PRERR_TF );
    }
      else
    {
    sfs_tfactor = l;
    }
      }

   else if ( strcmp( arguments[ 0 ], "UPDATE" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading update time" );
#endif
      l = atol( arguments[ 1 ] );
      if ( l < 1 )
    {
    bw_error( PRERR_UI );
    }
      else
    {
    sfs_inc = l;
    }
      }

   else if ( strcmp( arguments[ 0 ], "TRIG" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading trig level" );
#endif
      l = atol( arguments[ 1 ] );
      if ( ( l < 1 ) || ( l > 2 ) )
    {
    bw_error( PRERR_TL );
    }
      else
    {
    vpt_level = l;
    }
      }

   else if ( strcmp( arguments[ 0 ], "INSERTION" ) == 0 )
      {
#ifdef  OLD_DEBUG
      bw_message( "reading insertion" );
#endif
      l = atol( arguments[ 1 ] );
      if ( l < 0 )
    {
    sprintf( bw_ebuf, PRERR_INS );
    bw_error( bw_ebuf );
    }
      else
    {
    sfs_insertion = l;
    }
      }

#ifdef  USEGO
   else if ( strcmp( arguments[ 0 ], "GO" ) == 0 )
      {
      sfs_go = 1;
      }
#endif

   else
      {
      sprintf( bw_ebuf, PRERR_UNKNOWN, arguments[ 0 ] );
      bw_error( bw_ebuf );
      }
   }

pr_getargs( buffer )
   char *buffer;
   {
   int     y;            /* yes continue to loop    */
   int     n;            /* number of agument       */
   int     p;            /* position in argument string  */
   register char *c;          /* character position in buffer */

   c = buffer;
   y = TRUE;
   n = 0;

   /* check for null line */

   if ( ( ( *c & 255 ) == 0xa ) || ( ( *c & 255 ) == 0xc ) )
      {
      return 0;
      }

   /* loop to get each argument */

   while( y == TRUE )
      {
      p = 0;
      arguments[ n ][ p ] = 0;
      while( ( isspace( *c ) == 0 ) && ( *c != 0 ))
    {
    if ( n == 0 )
       {
       arguments[ n ][ p ] = toupper( *c );
       }
    else
       {
       arguments[ n ][ p ] = *c;
       }
    arguments[ n ][ p + 1 ] = 0;
    ++c;
    ++p;
    }

      /* Eat up any remaining whitespace before next argument */

      while( isspace( *c ) != 0 )
    {
    ++c;
    }

      /* if *c == 0, then end of line has been reached */

      if ( *c == 0 )
    {
    y = FALSE;
    }
      else
    {
    ++n;
    }
      if ( ( n + 1 ) == MAXARGS )
    {
    y = FALSE;
    }

      }
   return n + 1;
   }

pr_default( sorbit_array, orbit, n_orbits, fdfile )
   struct sfs_orbit **sorbit_array;
   int orbit, n_orbits;
   char *fdfile;
   {
   static char x_name[] = "";

   /* Allocate memory for orbit structure */

   if ( ( sorbit_array[ orbit ] = (struct sfs_orbit *) malloc( sizeof( struct sfs_orbit ) )) == NULL )
      {
      bw_error( PRMEMERR_SORBIT );
      sfs_exit();
      exit(1);
      }

   if ( ( sorbit_array[ orbit ]->aorbit = (struct as_orbit *)
      malloc( sizeof( struct as_orbit ) )) == NULL )
      {
      bw_error( PRMEMERR_AORBIT );
      sfs_exit();
      exit(1);
      }

   sorbit_array[ orbit ]->aorbit->focus
      = pr_getfd( sorbit_array, orbit, n_orbits, fdfile );

   /* Now fill in defaults */

   sorbit_array[ orbit ]->name                 = x_name;
   sorbit_array[ orbit ]->aorbit->lon_an       = 0.0;
   sorbit_array[ orbit ]->aorbit->arg_per      = 0.0;
   sorbit_array[ orbit ]->aorbit->inclination  = 0.0;
   sorbit_array[ orbit ]->aorbit->periapsis    = sorbit_array[ orbit ]->aorbit->focus->radius * 0.1;
   sorbit_array[ orbit ]->aorbit->apoapsis     = sorbit_array[ orbit ]->aorbit->focus->radius * 4;

   strncpy ( sorbit_array[ orbit ]->orbfile, DEFAULTORB, SFS_FILESIZE - 1 );
   strncpy ( sorbit_array[ orbit ]->gridfile, DEFAULTGRID, SFS_FILESIZE - 1 );
   strncpy ( sorbit_array[ orbit ]->surfile, DEFAULTSD, SFS_FILESIZE - 1 );
#ifdef  USEPOINTS
   strncpy ( sorbit_array[ orbit ]->pdfile, DEFAULTPT, SFS_FILESIZE - 1 );
#endif

   or_init( sorbit_array[ orbit ]->aorbit,
      sorbit_array[ orbit ]->aorbit->periapsis,
      sorbit_array[ orbit ]->aorbit->apoapsis );

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "[pr:] pr_default(): orbit %d, focus %s, mass %le",
      orbit, sorbit_array[ orbit ]->aorbit->focus->name,
      sorbit_array[ orbit ]->aorbit->focus->mass );
   bw_debug( bw_ebuf );
#endif

   }

pr_save( filename, sorbit_array, n_orbits, pr_name )
   char *filename;
   struct sfs_orbit **sorbit_array;
   int n_orbits;
   char *pr_name;
   {
   FILE *data;
   register int orbit;
   static time_t ticks;
   static struct tm *tp;

   /* get the date and time */

   ticks = time( NULL );
   tp = localtime( &ticks );

   /* Open the program file */

   if ( ( data = fopen( filename, "w" )) == NULL )
      {
      sprintf( bw_ebuf, PRERR_WRITE,
         filename );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }

   pr_stripcr( pr_name );
   fprintf( data, FILE_DELIMITER );
   fprintf( data, ";\t%s \n", filename );
   fprintf( data, ";\t%s %s", PR_CREATED, asctime( tp ) );
   fprintf( data, FILE_DELIMITER );
   fprintf( data, "%s \n", pr_name );
   fprintf( data, "tfactor\t\t%ld\n", sfs_tfactor );
   fprintf( data, "update\t\t%ld\n", sfs_inc );
   fprintf( data, "trig\t\t%d\n", vpt_level );
   fprintf( data, "insertion\t%ld\n", sfs_insertion );

   for ( orbit = 0; orbit < n_orbits; ++orbit )
      {
      if ( sorbit_array[ orbit ] != NULL )
         {
	 fprintf( data, FILE_DELIMITER );
	 fprintf( data, ";\t%s %d\n", PR_PARAMS, orbit + 1 );
	 fprintf( data, FILE_DELIMITER );
         fprintf( data, "name\t\t%d\t%s\n", orbit + 1,
            sorbit_array[ orbit ]->name );
         fprintf( data, "focus\t\t%d\t%s\n", orbit + 1,
	    sorbit_array[ orbit ]->aorbit->focus->fdfile );
         fprintf( data, "periapsis\t%d\t%lf\n", orbit + 1,
            sorbit_array[ orbit ]->aorbit->periapsis );
         fprintf( data, "apoapsis\t%d\t%lf\n", orbit + 1,
            sorbit_array[ orbit ]->aorbit->apoapsis );
         fprintf( data, "inclination\t%d\t%lf\n", orbit + 1,
	    sorbit_array[ orbit ]->aorbit->inclination );
         fprintf( data, "argper\t\t%d\t%lf\n", orbit + 1,
	    sorbit_array[ orbit ]->aorbit->arg_per );
         fprintf( data, "lonan\t\t%d\t%lf\n", orbit + 1,
	    sorbit_array[ orbit ]->aorbit->lon_an );
	 fprintf( data, "orb\t\t%d\t%s\n", orbit + 1,
	    sorbit_array[ orbit ]->orbfile );
	 fprintf( data, "grid\t\t%d\t%s\n", orbit + 1,
	    sorbit_array[ orbit ]->gridfile );
	 fprintf( data, "surface\t\t%d\t%s\n", orbit + 1,
	    sorbit_array[ orbit ]->surfile );
         }
      }

   fprintf( data, FILE_DELIMITER );
   fprintf( data, ";\t%s %s \n", PR_ENDOF, filename );
   fprintf( data, FILE_DELIMITER );
   fclose( data );
   }

pr_stripcr( s )
   char *s;
   {
   char *p;

   p = s;
   while ( *p != 0 )
      {
      switch( *p )
	 {
	 case 0x0d:
	 case 0x0a:
	    *p = 0;
	    break;
	 }
      ++p;
      }
   *p = 0;
   }

struct as_focus *
pr_getfd( sorbit_array, orbit_n, n_orbits, fdfile )
   struct sfs_orbit **sorbit_array;
   int orbit_n, n_orbits;
   char *fdfile;
   {
   register int orbit;
   struct as_focus *newfocus;

   /* first cycle through orbits to see if this focus has already
      been initialized */

   for ( orbit = 0; orbit < n_orbits; ++orbit )
      {
      if ( ( orbit != orbit_n ) && ( sorbit_array[ orbit ] != NULL ))
	 {
	 if ( strcmp( sorbit_array[ orbit ]->aorbit->focus->fdfile,
	    fdfile ) == 0 )
	    {
#ifdef  OLD_DEBUG
	    sprintf( bw_ebuf, "Reinit %s for orbit %d from orbit %d", 
               fdfile, orbit_n, orbit );
	    bw_debug( bw_ebuf );
#endif
	    return sorbit_array[ orbit ]->aorbit->focus;
	    }
	 }
      }

   /* the focus has not been initialized; initialize it now */

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "Init %s for orbit %d", fdfile, orbit_n );
   bw_debug( bw_ebuf );
#endif
   if ( ( newfocus = (struct as_focus *)
      malloc( sizeof( struct as_focus ) )) == NULL )
      {
      bw_error( PRMEMERR_FOCUS );
      sfs_exit();
      exit(1);
      }

   /* read in focal data */

   as_readfd( fdfile, newfocus );
   return newfocus;

   }

char *
fntolc( s )
   char *s;
   {
   char *p;

   p = s;
   while( *p != 0 )
      {
      *p = tolower( *p );
      ++p;
      }
   return s;
   }
