// nscroll.cpp - implementation of NumberScroller class
//
//

#include <owl.h>
#include <edit.h>
#include "nscroll.h"

// define values for status (which 'button' is pressed)
#define NORMAL	0
#define TOP		1
#define BOTTOM	2
// define values for timer intervals
#define SLOW 300
#define FAST 50

// static members:
//
// The HBITMAP is loaded from resource when the first NumberScroller is
// instantiated, and the GDI object is deleted when the last NumberScroller
// is destroyed.
//
int NumberScroller::nInstanceCount = 0;
HBITMAP	NumberScroller::hBM = (HBITMAP)NULL;

//
// constructor #1 - use this if there is an OWL object representing
// the control to which the number scroller is attatched (in a window
// or a dialog).
//
NumberScroller::NumberScroller(	PTStatic	TheStatic,
								int			nMaxValue,
								int			nMinValue,
								PTModule 	AModule) :
								TWindow(TheStatic->Parent, "", AModule),
                                idControl(TheStatic->Attr.Id),
								pTheStatic(TheStatic),
                                nMax(nMaxValue),
								nMin(nMinValue)
{
	Init(AModule);
}

//
// constructor #2 - for use when no OWL object exists to represent
// the relevant control (in a dialog represented by a TDialog (or
// derived) object)
//
NumberScroller::NumberScroller( int			ResourceID,
								PTDialog	TheDialog,
								int			nMaxValue,
								int			nMinValue,
								PTModule 	AModule) :
                                TWindow(TheDialog, "", AModule),
                                idControl(ResourceID),
								pTheStatic(NULL),
                                nMax(nMaxValue),
								nMin(nMinValue)
{
	Init(AModule);
}

//
// NumberScroller::Init() - protected function to perform initialisation
// of a NumberScroller object.  Called by both constructors, passing a
// pointer to the OWL TModule representing the module from which the
// button bitmap may be loaded.
//
void NumberScroller::Init(PTModule AModule)
{
	Attr.Style = WS_CHILD | WS_VISIBLE;
	if(nInstanceCount == 0) // load bitmap first time only
		hBM = LoadBitmap(
				(AModule == NULL ? GetModule() : AModule)->hInstance,
				"BMP_NSCROLL" );
	BITMAP bm;
	GetObject(hBM, sizeof(bm), &bm);
	Attr.W = bm.bmWidth / 2;
	Attr.H = bm.bmHeight;
	bCapturingMouse = FALSE;
	nStatus = NORMAL;
    nInstanceCount ++;
}

//
// destructor - decrement the instance counter & delete the bitmap if no
// other NumberScrollers exist
//
NumberScroller::~NumberScroller()
{
    nInstanceCount --;
	if(nInstanceCount == 0 && hBM) DeleteObject(hBM);

}

//
//	NumberScroller::SetupWindow
//	Position the NumberScroller 'buttons' 1 pixel to the right of, and
//	vertically centred on the associated control & ensure that the contents
//	of the control are within the permitted range.
//
void NumberScroller::SetupWindow()
{
	RECT	rc;
	POINT	pt;

	TWindow::SetupWindow();

	if(pTheStatic) {
		rc.left = pTheStatic->Attr.W + pTheStatic->Attr.X + 1;
		rc.top = pTheStatic->Attr.Y + (pTheStatic->Attr.H - Attr.H) / 2;
    	hwndControl = pTheStatic->HWindow;
	}
	else
	{
		hwndControl = ((PTDialog)Parent)->GetItemHandle(idControl);
		GetWindowRect(hwndControl, &rc);
		pt.x = rc.left;
		pt.y = rc.top;
		rc.right -= rc.left;
		rc.bottom -= rc.top; 
		ScreenToClient(Parent->HWindow, &pt);
		rc.left = pt.x + rc.right + 1;
        rc.top = pt.y + (rc.bottom - Attr.H) / 2;
	}

	MoveWindow(	HWindow, rc.left, rc.top, Attr.W, Attr.H, FALSE);
	SetMaxValue(nMax);
    SetMinValue(nMin);
}


void NumberScroller::WMLButtonDown(RTMessage Msg)
{
	SetCapture(HWindow);
    nTimerId = 1;
	SetTimer(HWindow, nTimerId, SLOW, NULL);
    nTimerTicks = 0;
	bCapturingMouse = TRUE;
	nStatus = Msg.LP.Hi > (Attr.H >> 1) ? BOTTOM : TOP;
	InvalidateRect(HWindow,NULL,FALSE);
	// pretend a timer event has occurred, to avoid waiting for the
	// initial action
    WMTimer(Msg);
    Msg.Result = 0l;
}

void NumberScroller::WMLButtonUp(RTMessage Msg)
{
	if(bCapturingMouse) {
    	KillTimer(HWindow, nTimerId);
		ReleaseCapture();
		nStatus = NORMAL;
		InvalidateRect(HWindow,NULL,FALSE);
		Msg.Result = 0l;
        
    }
}

void NumberScroller::WMTimer(RTMessage)
{
	int nValue = GetValue() + (nStatus == TOP ? 1 : -1);

	if(nValue >= nMin && nValue <= nMax)
    	SetValue(nValue);
	else
		MessageBeep(-1);

	if(nTimerTicks == 5) {
		KillTimer(HWindow, nTimerId);
        SetTimer(HWindow, nTimerId, FAST, NULL);
	}
	else
		if(nTimerTicks < 6)
        	nTimerTicks ++;  
}

void NumberScroller::Paint(HDC hDC, PAINTSTRUCT _FAR &)
{
	HDC	MemDC = CreateCompatibleDC( hDC );
	HBITMAP hBMOld = (HBITMAP)SelectObject( MemDC, hBM );
	int X,Y,H;
	Y = nStatus == BOTTOM ? Attr.H / 2 : 0;
	H = nStatus == NORMAL ? Attr.H : Attr.H / 2;
	X = nStatus == NORMAL ? 0 : Attr.W;
	BitBlt( hDC, 0,Y,Attr.W, H, MemDC, X, Y, SRCCOPY);
    SelectObject( MemDC, hBMOld );
	DeleteDC(MemDC);
}

int NumberScroller::SetMaxValue(int nMaxValue)
{
	int nValue = GetValue();
	if(nValue > nMaxValue)
		SetValue(nMaxValue);
	nValue = nMax;
	nMax = nMaxValue;
    return nValue;
}

int NumberScroller::SetMinValue(int nMinValue)
{
	int nValue = GetValue();
	if(nValue < nMinValue)
		SetValue(nMinValue);
	nValue = nMin;
	nMin = nMinValue;
    return nValue;
}

int NumberScroller::GetValue()
{
	char szBuff[10] = "0";
	SendMessage(hwndControl, WM_GETTEXT, (WPARAM)sizeof(szBuff),
								(LPARAM)(LPSTR)szBuff);
	return atoi(szBuff);
}

void NumberScroller::SetValue(int nValue)
{
	char szBuff[10];
	wsprintf(szBuff,"%d",nValue);
	SendMessage(hwndControl, WM_SETTEXT, 0, (LPARAM)(LPSTR)szBuff);
}