#include "internat.h"
#include <windows.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>

//****************************************************************************
// Language class members
//****************************************************************************

int LanguageRecord::operator == (const char *s)
{
  int x = 0;
  while (tags[x] && lstrcmpi(s, tags[x])) ++x;
  return 0 != tags[x];
}

//****************************************************************************
// LanguageList class members
//****************************************************************************

// String resources for month/day names located in main.cpl
// Both sets start with full names followed by abbreviations

#define IDSDAYS		260	// start of day names in main.cpl
#define	IDSMONTHS	276	// start of month names in main.cpl

// English month/day names

const char *EnglishNames[38] =
{
  "Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday",
  "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat",
  "January", "February", "March", "April", "May", "June",
  "July", "August", "September", "October", "November", "December",
  "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

const char *DefaultNames[38];

const char *EnglishTags[] = { "enu", "eng", NULL };
const char *DefaultTags[] = { "default", NULL };

class LanguageList
{
  protected:
    LanguageRecord *list, *tail;
    LanguageRecord *def;
  public:
    LanguageList();
    ~LanguageList();
    LanguageRecord &operator +=(LanguageRecord &);
    LanguageRecord &operator [](const char *);
};

const char *LoadName(HINSTANCE hInst, UINT id)
{
  char buffer[256];
  LoadString(hInst, id, buffer, sizeof(buffer));
  return strdup(buffer);
}

LanguageList::LanguageList() : list(0), tail(0), def(0)
{
  UINT em = SetErrorMode(SEM_NOOPENFILEERRORBOX);
  HINSTANCE hInst = LoadLibrary("main.cpl");
  SetErrorMode(em);
  if (hInst > HINSTANCE_ERROR)
  {
    int x;
    const char **p = DefaultNames;
    for (x = IDSDAYS; x < IDSDAYS+14; x++) *p++ = LoadName(hInst, x);
    for (x = IDSMONTHS; x < IDSMONTHS+24; x++) *p++ = LoadName(hInst, x);
    def = new LanguageRecord(DefaultTags, DefaultNames);
    FreeLibrary(hInst);
  } else {
    def = new LanguageRecord(DefaultTags, EnglishNames);
  }
  *this += LanguageRecord(EnglishTags, EnglishNames);
}

LanguageList::~LanguageList()
{
  LanguageRecord *curr = list, *next;
  while(curr)
  {
    next = curr->next;
    delete curr;
    curr = next;
  }
  delete def;
}

LanguageRecord &LanguageList::operator +=(LanguageRecord &l)
{
  LanguageRecord *l1 = new LanguageRecord(l);
  l1->next = list;
  list = l1;
  return l;
}

LanguageRecord &LanguageList::operator[](const char *s)
{
  LanguageRecord *curr = list;
  while(curr)
  {
    if (*curr == s) break;
    curr = curr->next;
  }
  return (curr) ? *curr : *def;
}

//****************************************************************************
// Profile class members
//****************************************************************************

// read a profile string

CString &operator << (CString &s, Profile &p)
{
  char buffer[256];
  GetProfileString(p.section, p.entry, p.def, buffer, sizeof(buffer));
  return s = buffer;
}

// read a profile integer

int &operator << (int &i, Profile &p)
{
  i = GetProfileInt(p.section, p.entry, p.idef);
  return i;
}

//****************************************************************************
// International class members
//****************************************************************************

// static data members

CString International::sLanguage;
CString International::sCountry;
int International::iCountry;
int International::iDate;
int International::iTime;
int International::iTLZero;
int International::iCurrency;
int International::iCurrDigits;
int International::iNegCurr;
int International::iLzero;
int International::iDigits;
int International::iMeasure;
CString International::s1159;
CString International::s2359;
CString International::sCurrency;
CString International::sThousand;
CString International::sDecimal;
CString International::sDate;
CString International::sTime;
CString International::sList;
CString International::sShortDate;
CString International::sLongDate;
International International::TheInfo;
LanguageList International::Languages;

// constructor

International::International()
{
  if (this == &TheInfo) ReadInfo();
}

// read international information from win.ini

void International::ReadInfo()
{
  sLanguage << Profile("intl", "sLanguage", "enu");
  sCountry << Profile("intl", "sCountry", "United States");
  iCountry << Profile("intl", "iCountry", 1);
  iDate << Profile("intl", "iDate", 0);
  iTime << Profile("intl", "iTime", 0);
  iTLZero << Profile("intl", "iTLZero", 0);
  iCurrency << Profile("intl", "iCurrency", 0);
  iCurrDigits << Profile("intl", "iCurrDigits", 2);
  iNegCurr << Profile("intl", "iNegCurr", 0);
  iLzero << Profile("intl", "iLzero", 1);
  iDigits << Profile("intl", "iDigits", 2);
  iMeasure << Profile("intl", "iMeasure", 1);
  s1159 << Profile("intl", "s1159", "AM");
  s2359 << Profile("intl", "s2359", "PM");
  sCurrency << Profile("intl", "sCurrency", "$");
  sThousand << Profile("intl", "sThousand", ",");
  sDecimal << Profile("intl", "sDecimal", ".");
  sDate << Profile("intl", "sDate", "/");
  sTime << Profile("intl", "sTime", ":");
  sList << Profile("intl", "sList", ",");
  sShortDate << Profile("intl", "sShortDate", "M/d/yy");
  sLongDate << Profile("intl", "sLongDate", "dddd, MMMM dd, yyyy");
}

// Add a language

void International::AddLanguage(LanguageRecord &l)
{
  Languages += l;
}

// get the name of a month

const char *International::MonthName(int month, int format)
{
  return Languages[Language()].MonthName(month, format);
}

// get the name of a day

const char *International::DayName(int day, int format)
{
  return Languages[Language()].DayName(day, format);
}

//****************************************************************************
// TimeString class members
//****************************************************************************

// build a string representing the time
 
void TimeString::MakeString(struct tm *t, BOOL seconds)
{
  char buffer[256], *dst = buffer;
  int hour = t->tm_hour % TimeHours();
  if (!hour && !TimeFormat()) hour = 12;
  dst += sprintf(dst, "%0*d%c%02d", TimeDigits(), hour, TimeSeparator(), t->tm_min);
  if (seconds) dst += sprintf(dst, "%c%02d", TimeSeparator(), t->tm_sec);
  sprintf(dst, " %s", TimeFormat() ? PM() : t->tm_hour < 12 ? AM() : PM());
  s = buffer;
}

//****************************************************************************
// DateString class members
//****************************************************************************

// build a string representing the date

void DateString::MakeString(struct tm *t, const char *fmt)
{
  char buffer[256], *dst = buffer;
  int n;
  while (*fmt)
  {
    switch(toupper(*fmt))
    {
      case '\'' :
      case '\"' :
      {
	char quote = *fmt++;
	while (*fmt && *fmt != quote) *dst++ = *fmt++;
	if (*fmt) ++fmt;
	break;
      }

      case 'M' :
      {
	n = 1;
	while ('M' == toupper(*++fmt)) ++ n;
	switch(n)
	{
	  case 1 : // short number
	  case 2 : // long number
	    dst += sprintf(dst, "%0*d", n, t->tm_mon+1);
	    break;

	  case 3 : // short name
	  case 4 : // long name
	    dst += sprintf(dst, "%s", MonthName(t->tm_mon, n-3));
	}
        break;
      }

      case 'D' :
	n = 1;
	while ('D' == toupper(*++fmt)) ++ n;
	switch(n)
	{
	  case 1 : // short number
	  case 2 : // long number
	    dst += sprintf(dst, "%0*d", n, t->tm_mday);
	    break;

	  case 3 : // short name
	  case 4 : // long name
	    dst += sprintf(dst, "%s", DayName(t->tm_wday, n-3));
	}
	break;

      case 'Y' :
	n = 1;
	while ('Y' == toupper(*++fmt)) ++ n;
	switch(n)
	{
	  case 2 : // 2 digit
	    dst += sprintf(dst, "%02d", t->tm_year % 100);
	    break;
	     
	  case 4 : // 4 digit
	    dst += sprintf(dst, "%04d", t->tm_year + 1900);
	    break;
	}
	break;

      default:
        *dst++ = *fmt++;
    }
  }
  *dst = 0;
  s = buffer;
}

//****************************************************************************
// NumberString class members
//****************************************************************************

// build a string representing a number

NumberString::NumberString(long double d, int prec)
{
  char buffer[256], buffer1[256], *dst = buffer, *src = buffer1;
  int index;
  sprintf(buffer1, "%.*Lf", prec, d);
  if (*src == '-') *dst++ = *src++;
  if (*src == '0') { if (LeadZero()) *dst++ = '0'; ++src; }
  index = (int) (strchr(src, '.') - src);
  while (*src)
  {
    if (*src != '.')
    {
      *dst++ = *src++;
      --index;
      if (*src && index % 3 == 0)
        *dst++ = (index) ? ThousandsSeparator() : DecimalPoint();
    } else
      ++src;
  }
  *dst = 0;
  s = buffer;
}

//****************************************************************************
// CurrencyString class members
//****************************************************************************

// build a string representing a currency amount

CurrencyString::CurrencyString(long double d)
{
  static char *formats[] =
  {
    // positive values
    "SN", "NS", "S N", "N S",
    // negative values
    "(SN)", "-SN", "S-N", "SN-", "(NS)", "-NS", "N-S",
    "NS-", "-N S", "-S N", "S N-"
  };
  CString number(NumberString(fabsl(d), CurrencyDigits()));
  char buffer[256], *dst = buffer, *fmt;
  fmt = ( d < 0.0) ? formats[CurrencyFormat(TRUE) + 4] : formats[CurrencyFormat(FALSE)];
  while (*fmt)
  {
    switch(*fmt)
    {
      case 'S' :
	dst += sprintf(dst, "%s", CurrencySymbol());
	break;

      case 'N' :
	dst += sprintf(dst, "%s", (const char *) number);
	break;

      default :
       *dst++ = *fmt;
    }
    ++fmt;
  }
  *dst = 0;
  s = buffer;
}
