/**************************************************************************
These C++ classes are copyright 1989, 1990, 1991 by William Herrera.
I hereby release this source code for free distrubution and use.
If you modify it and distribute it, please indicate any changes you
make as your own and the code as copyrighted above.
**************************************************************************/

// file gcstring.hpp
// this class demonstrates the use of the gcobject and PointerHeap classes
// to implement a fast executing string class.

#if !defined(GCSTRING_HPP)
#define GCSTRING_HPP

#include <fstream.h>

#include "gcobject.hpp"
#include "pointerh.hpp"


class gcstring
{
public:
	// intialize buffer.
	static void Initialize(unsigned int bufsize=10000, 
		int maxstrings = 4000, int gc_interval=10000);

	 // constructors
	gcstring();	// gcstring s; default constructor
	gcstring(const char *);	// gcstring s("Hello");
	gcstring(const gcstring &);	  // copy-initializer
	gcstring(const char * p1, const char * p2);

	 // destructor
	~gcstring();

	// assignment
	gcstring & operator =(const gcstring &); // gcstring s = gcstring st;
	gcstring & operator =(const char *); // gcstring s = char * s;

	// reference a char (can potentially be an lvalue and modify string!)
	const char & operator [](const int i); // reference a char in string.

	// functions and operators ...
	int length();							// length of string
	gcstring operator +(const gcstring & st);   // concatenate strings

	// this returns a raw pointer to internal representation.
	// the only protection is that it's const.
	operator const char * () const;

	int operator ==(const gcstring & s) const;
	int operator !=(const gcstring & s) const;
	int compare(const gcstring & s) const;
	int compare(const char * p) const;

	gcstring uppercase();	// return string changed to uppercase
	gcstring lowercase();	// ... or lowercase


	friend gcstring operator +(char * p, gcstring & g);
	friend gcstring operator +(gcstring & g, char * p);
	friend istream & operator >>(istream & ist, gcstring & s);
	friend ostream & operator <<(ostream & ost, const gcstring & s);

protected:
	char * GetPtr() const;

private:
	data_record ** str;		// pointer to data record in gcobject memory.
	static gcobject * gc;	// static gcobject allocation buffer.
	static PointerHeap * po;	// static pointer allocation heap.
};

inline gcstring::operator const char * () const
{ 
	return (char *)((*str)->data); 
}

inline int gcstring::operator ==(const gcstring & s) const 
{ 
	return !compare(s); 
}

inline int gcstring::operator !=(const gcstring & s) const 
{ 
	return compare(s); 
}

inline char * gcstring::GetPtr() const 
{ 
	return (char *)((*str)->data); 
}

gcstring operator +(char * p, gcstring & g);	// concatenate

gcstring operator +(gcstring & g, char * p);	// concatenate

istream & operator >>(istream & ist, gcstring & s);
ostream & operator <<(ostream & ost, const gcstring & s);


#endif

// end of file gcstring.hpp
