/**************************************************************************
These C++ classes are copyright 1989, 1990, 1991 by William Herrera.
I hereby release this source code for free distrubution and use.
If you modify it and distribute it, please indicate any changes you
make as your own and the code as copyrighted above.
**************************************************************************/

#include <string.h>

#include "gcstring.hpp"
#include "error.hpp"

gcobject * gcstring::gc = new gcobject;
PointerHeap * gcstring::po = new PointerHeap;

void gcstring::Initialize(size_t bufsize, int maxstrings, int gc_interval)
{
	if(gc != NULL)
		delete gc;
	if(po != NULL)
		delete po;
	gc = new gcobject(bufsize, gc_interval);
	po = new PointerHeap(maxstrings);
	if(gc == NULL || po == NULL)
		Error("Cannot allocate memory for gcstring::Initailize");
}

gcstring::gcstring()   // string s; default constructor
{
	str = (data_record **)po->Allocate();
	if(str == NULL)
		Error(handle_alloc_err);
	gc->Allocate(1, str);
	if(*str == NULL)
		Error(gcobject_alloc_err, " gcstring()");
	(*str)->data[0] = 0;
}

gcstring::gcstring(const char * s)
{
	int len = strlen(s) + 1;
	str = (data_record **)po->Allocate();
	if(str == NULL)
		Error(handle_alloc_err);
	gc->Allocate(len, str);
	if(*str == NULL)
		Error(gcobject_alloc_err, " gcstring(const char *)");
	memcpy((*str)->data, s, len);
}

gcstring::gcstring(const gcstring & st)
// copy-initializer avoids dup of char vector.
{
	str = st.str;			// minimizes copying.
	(*str)->ref_count++;	// increment reference count.
}


gcstring::gcstring(const char * p1, const char * p2)
{
	int len = strlen(p1) + strlen(p2) + 1;
	str = (data_record **)po->Allocate();
	if(str == NULL)
		Error(handle_alloc_err);
	gc->Allocate(len, str);
	if(*str == NULL)
		Error(gcobject_alloc_err,
			" gcstring(const char*,const char*)");
	strcpy((*str)->data, p1);
	strcat((*str)->data, p2);
}

gcstring::~gcstring()		   // destructor
{
	int & refcount = (*str)->ref_count;
	--refcount;
	if(refcount < 0)
		Error("\ngcstring::~gcstring() deallocation error: refcount < 0");
	else if(refcount == 0)
		po->Deallocate((void **)str);
	gc->IncDeleteCount();
}

gcstring & gcstring::operator =(const gcstring & st)
{
	int & refcount = (*str)->ref_count;
	--refcount;
	if(refcount < 0)
		Error("\ngcstring::op=(const gcstring &) : refcount < 0");
	else if(refcount == 0)
		po->Deallocate((void **)str);
	str = st.str;
	(*str)->ref_count++;
	return *this;
}

gcstring & gcstring::operator =(const char * s)
{
	gcstring st(s);
	int & refcount = (*str)->ref_count;
	--refcount;
	if(refcount < 0)
		Error("\ngcstring::op=(const char *) : refcount < 0");
	else if(refcount == 0)
		po->Deallocate((void **)str);
	str = st.str;
	(*str)->ref_count++;
	return *this;
}

int gcstring::length()
{
	return strlen(GetPtr());
}


static const char dummychar[] = "\0";
const char & gcstring::operator [](const int i) // reference a char in string.
{
	char * p = GetPtr();
	if(p == NULL || i >= strlen(p) || i < 0)
		return dummychar[0];
	return p[i];
}

gcstring gcstring::operator +(const gcstring & st)
{
	gc->Lock();
	gcstring s(GetPtr(), st.GetPtr());
	gc->Unlock();
	return s;	// return by value will copy s.
}

int gcstring::compare(const gcstring & s) const
{
	return strcmp(GetPtr(), s.GetPtr());
}

int gcstring::compare(const char * p) const
{
	return strcmp(GetPtr(), p);
}

gcstring gcstring::uppercase()
{
	gc->CompactBuffer();
	gc->Lock();
	char * p = GetPtr();
	gcstring temp(p);
	p  = temp.GetPtr();
	strupr(p);			// sneaky but hidden...
	gc->Unlock();
	return temp;
}

gcstring gcstring::lowercase()
{
	gc->CompactBuffer();
	gc->Lock();
	char * p = GetPtr();
	gcstring temp(p);
	p  = temp.GetPtr();
	strlwr(p);			// sneaky but hidden...
	gc->Unlock();
	return temp;
}

gcstring operator +(char * p, gcstring & g)
{
	g.gc->CompactBuffer();
	g.gc->Lock();
	gcstring st(p, g.GetPtr());
	g.gc->Unlock();
	return st;
}

gcstring operator +(gcstring & g, char * p)
{
	g.gc->CompactBuffer();
	g.gc->Lock();
	gcstring st(g.GetPtr(), p);
	g.gc->Unlock();
	return st;
}

istream & operator >>(istream & ist, gcstring & st)
{
	ist >> ws;
	char buf[1024];
	ist.get(buf, 1023, '\n');
	ist >> ws;
	st = buf;
	return ist;
}

ostream & operator <<(ostream & ost, const gcstring & st)
{
	return ost << st.GetPtr();
}

// end of file gcstring.cpp

