/*******************************************************************************
** EXPCONST.CPP - ExpConstant class
** Created 5/16/95 by John Stockton
** CIS: 76041,2534
** INTERNET: jstockton%inova@mcimail.com
**
** Copyright (c) 1995 INOVA Corporation.  May be copied, used and distributed, 
** as long as no charge is made for its use and this copyright notice is not
** altered or removed.
**
** $Header: j:/library/express/rcs/expconst.cpp 1.4 1995/06/29 15:06:23 JOHN Exp $
*******************************************************************************/

#include "express.hpp"
#include <assert.h>
#include <string.h>

/*******************************************************************************
** Public members
*******************************************************************************/

ExpConstant::ExpConstant( INT16 i) :
      ExpElement( ExpElement::ExpCONSTANT, ExpElement::ExpINT16),
      pValue( new INT16( i))
{
   assert( this->pValue);
}


ExpConstant::ExpConstant( UINT16 u) :
      ExpElement( ExpElement::ExpCONSTANT, ExpElement::ExpUINT16),
      pValue( new UINT16( u))
{
   assert( this->pValue);
}


ExpConstant::ExpConstant( INT32 l) :
      ExpElement( ExpElement::ExpCONSTANT, ExpElement::ExpINT32),
      pValue( new INT32( l))
{
   assert( this->pValue);
}


ExpConstant::ExpConstant( UINT32 ul) :
      ExpElement( ExpElement::ExpCONSTANT, ExpElement::ExpUINT32),
      pValue( new UINT32( ul))
{
   assert( this->pValue);
}


ExpConstant::ExpConstant( double d) :
      ExpElement( ExpElement::ExpCONSTANT, ExpElement::ExpDOUBLE),
      pValue( new double( d))
{
   assert( this->pValue);
}


ExpConstant::ExpConstant( char c) :
      ExpElement( ExpElement::ExpCONSTANT, ExpElement::ExpCHAR),
      pValue( new char( c))
{
   assert( this->pValue);
}


ExpConstant::ExpConstant( const char *s) :
      ExpElement( ExpElement::ExpCONSTANT, ExpElement::ExpSTRING),
      pValue( strdup( s))
{
   assert( this->pValue);
}


ExpConstant::ExpConstant( const TDate &date) :
      ExpElement( ExpElement::ExpCONSTANT, ExpElement::ExpDATE),
      pValue( new TDate( date))
{
   assert( this->pValue);
}


ExpConstant::ExpConstant( const TTime &time) :
      ExpElement( ExpElement::ExpCONSTANT, ExpElement::ExpTIME),
      pValue( new TTime( time))
{
   assert( this->pValue);
}


ExpConstant::ExpConstant( const ExpConstant &_c) :
      ExpElement( _c.GetElementType(), _c.GetEvalType()),
      pValue( 0)
{
   switch (GetEvalType())
   {
   case ExpINT16:
      this->pValue = new INT16( _c);
      break;
   case ExpUINT16:
      this->pValue = new UINT16( _c);
      break;
   case ExpINT32:
      this->pValue = new INT32( _c);
      break;
   case ExpUINT32:
      this->pValue = new UINT32( _c);
      break;
   case ExpDOUBLE:
      this->pValue = new double( _c);
      break;
   case ExpCHAR:
      this->pValue = new char( _c);
      break;
   case ExpSTRING:
      this->pValue = strdup( _c);
      break;
   case ExpDATE:
      this->pValue = new TDate( (TDate) _c);
      break;
   case ExpTIME:
      this->pValue = new TTime( (TTime) _c);
      break;
   }

   assert( this->pValue);
}


ExpConstant::~ExpConstant( void)
{
   delete this->pValue;
}


INT16 ExpConstant::EvalINT16( void) const
{
   if (!this->pValue)
   {
      SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpINT16);
      return 0;
   }

   return *(INT16 *) this->pValue;
}


UINT16 ExpConstant::EvalUINT16( void) const
{
   if (!this->pValue)
   {
      SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpUINT16);
      return 0;
   }

   return *(UINT16 *) this->pValue;
}


INT32 ExpConstant::EvalINT32( void) const
{
   if (!this->pValue)
   {
      SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpINT32);
      return 0;
   }

   return *(INT32 *) this->pValue;
}


UINT32 ExpConstant::EvalUINT32( void) const
{
   if (!this->pValue)
   {
      SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpUINT32);
      return 0;
   }

   return *(UINT32 *) this->pValue;
}


double ExpConstant::EvalDOUBLE( void) const
{
   if (!this->pValue)
   {
      SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpDOUBLE);
      return 0;
   }

   return *(double *) this->pValue;
}


char ExpConstant::EvalCHAR( void) const
{
   if (!this->pValue)
   {
      SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpCHAR);
      return 0;
   }

   return *(char *) this->pValue;
}


const char *ExpConstant::EvalSTRING( void) const
{
   if (!this->pValue)
   {
      SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpSTRING);
      return 0;
   }

   return (const char *) this->pValue;
}


BOOL ExpConstant::EvalBOOL( void) const
{
   // Note: boolean constants not supported b/c C does not distinguish them from integers
   SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpBOOL);
   return 0;
}


TDate ExpConstant::EvalDATE( void) const
{
   if (!this->pValue)
   {
      SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpDATE);
      return TDate();
   }

   return *(TDate *) this->pValue;
}


TTime ExpConstant::EvalTIME( void) const
{
   if (!this->pValue)
   {
      SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpTIME);
      return 0;
   }

   return *(TTime *) this->pValue;
}


#if !defined( EXPRESS_NO_IOSTREAM)
void ExpConstant::Output( ostream &_os) const
{
   ExpElement::eEvalErrors eEvalErrorSaved = GetEvalError();
      // Remember original error code

   switch (GetEvalType())
   {
   case ExpConstant::ExpINT16:
      _os << (INT16) *this;
      break;
   case ExpConstant::ExpUINT16:
      _os << (UINT16) *this;
      break;
   case ExpConstant::ExpINT32:
      _os << (INT32) *this;
      break;
   case ExpConstant::ExpUINT32:
      _os << (UINT32) *this;
      break;
   case ExpConstant::ExpDOUBLE:
      _os << (double) *this;
      break;
   case ExpConstant::ExpCHAR:
      _os << (char) *this;
      break;
   case ExpConstant::ExpSTRING:
      _os << (const char *) *this;
      break;
   case ExpConstant::ExpDATE:
      _os << (TDate) *this;
      break;
   case ExpConstant::ExpTIME:
      _os << (TTime) *this;
      break;
   default:
      assert( FALSE);
      break;
   }

   OverrideEvalError( eEvalErrorSaved);
      // Restore original error code
}


ostream &operator<<( ostream &_os, const ExpConstant &_c)
{
   _c.Output( _os);

   return _os;
}
#endif


/******************************************************************************
** IDAPI CANExpr conversion
******************************************************************************/

#if !defined( EXPRESS_NO_IDAPI)
void ExpConstant::BuildCANBuffers(
      OutputBuffer &_bufNode,
      OutputBuffer &_bufLiteral,
      UINT16 &_nNodes,
      UINT16 &_nNodeOffset,
      UINT16 &_nLiteralOffset)
{
   CANConst canConst;
   canConst.nodeClass = nodeCONST;
   canConst.canOp = canCONST;
   canConst.iOffset = _nLiteralOffset;

   switch (GetEvalType())
   {
   case ExpConstant::ExpINT16:
      _bufLiteral << (INT16) *this;
      canConst.iType = fldINT16;
      break;
   case ExpConstant::ExpUINT16:
      _bufLiteral << (UINT16) *this;
      canConst.iType = fldUINT16;
      break;
   case ExpConstant::ExpINT32:
      _bufLiteral << (INT32) *this;
      canConst.iType = fldINT32;
      break;
   case ExpConstant::ExpUINT32:
      _bufLiteral << (UINT32) *this;
      canConst.iType = fldUINT32;
      break;
   case ExpConstant::ExpDOUBLE:
      _bufLiteral << (double) *this;
      canConst.iType = fldFLOAT;
      break;
   case ExpConstant::ExpCHAR:
   case ExpConstant::ExpSTRING:
      _bufLiteral << (const char *) *this;
      canConst.iType = fldZSTRING;
      break;
   case ExpConstant::ExpDATE:
      {
         TDate tDate = *this;
         DATE dbiDate;
         DBIResult rslt = DbiDateEncode( (UINT16) tDate.Month(), (UINT16) tDate.DayOfMonth(), INT16( tDate.Year()), &dbiDate);
         assert( rslt == DBIERR_NONE);
         _bufLiteral << dbiDate;
         canConst.iType = fldDATE;
      }
      break;
   case ExpConstant::ExpTIME:
      assert( FALSE); // No direct IDAPI support for time fields!
      break;
   default:
      assert( FALSE);
      break;
   }

   canConst.iSize = (UINT16) _bufLiteral.GetSize();
   _bufNode << canConst;

   _nNodes++;
   _nNodeOffset += (UINT16) _bufNode.GetSize();
   _nLiteralOffset += (UINT16) _bufLiteral.GetSize();
}
#endif
