/*******************************************************************************
** EXPRESS.CPP - Expression class
** Created 5/14/95 by John Stockton
** CIS: 76041,2534
** INTERNET: jstockton%inova@mcimail.com
**
** Copyright (c) 1995 INOVA Corporation.  May be copied, used and distributed, 
** as long as no charge is made for its use and this copyright notice is not
** altered or removed.
**
** $Header: j:/library/express/rcs/express.cpp 1.3 1995/06/29 15:06:47 JOHN Exp $
*******************************************************************************/

#include "express.hpp"
#include <assert.h>
#include <string.h>

Expression::Expression( const ExpElement &_eqLeft,
      eOperators _eOperator,
      const ExpElement &_eqRight) :
      ExpElement( ExpElement::ExpEXPRESSION),
      eOperator( _eOperator),
      pExpLeft( _eqLeft.Duplicate()),
      pExpRight( _eqRight.Duplicate())
{
   assert( IsBinaryOperator());
      // Two operands requires a binary operator

   if (IsLogicalOperator())
      SetEvalType( ExpElement::ExpBOOL);
   else
   {
      assert( IsArithmeticOperator());
         // If it ain't binary, it must be arithmetic

      eEvalTypes etLeft = this->pExpLeft->GetEvalType();
      eEvalTypes etRight = this->pExpRight->GetEvalType();

      if (etLeft == Expression::ExpTIME || etLeft == Expression::ExpDATE)
         // All operations on times or dates result in:
      {
         if (etRight == Expression::ExpTIME || etRight == Expression::ExpDATE)
            SetEvalType( ExpElement::ExpINT32);
               // ..INT32 if both operands are TIMEs/DATEs (actually, minus is
               // the only valid operator here, but we don't care about that now)
         else
            SetEvalType( etLeft);
               // ..TIME or DATE if only one operand is TIME or DATE
      }
      else if (etRight == Expression::ExpTIME || etRight == Expression::ExpDATE)
         SetEvalType( etRight);
               // ..TIME or DATE if only one operand is TIME or DATE

      else if (etLeft == Expression::ExpDOUBLE || etRight == Expression::ExpDOUBLE ||
            GetOperator() == Expression::ExpDIVIDE)
         // All operations on DOUBLEs and all divisions result in DOUBLE:
         SetEvalType( ExpElement::ExpDOUBLE);
      else
         // Everything else results in INT32:
         SetEvalType( ExpElement::ExpINT32);
   }
}


Expression::Expression( const ExpElement &_eq,
      eOperators _eOperator) :
      ExpElement( Expression::ExpEXPRESSION),
      eOperator( _eOperator),
      pExpLeft( _eq.Duplicate()), pExpRight( 0)
{
   assert( IsUnaryOperator());

   SetEvalType( ExpElement::ExpBOOL);
}


Expression::Expression( const Expression &_eq) :
      ExpElement( _eq),
      eOperator( _eq.GetOperator()),
      pExpLeft( 0), pExpRight( 0)
{
   if (_eq.pExpLeft)
      this->pExpLeft = _eq.pExpLeft->Duplicate();
   if (_eq.pExpRight)
      this->pExpRight = _eq.pExpRight->Duplicate();
}


Expression::~Expression( void)
{
   delete this->pExpLeft;
   delete this->pExpRight;
}


BOOL Expression::IsBinaryOperator( void) const
{
   return BOOL( this->eOperator == Expression::ExpEQUAL ||
         this->eOperator == Expression::ExpLIKE ||
         this->eOperator == Expression::ExpNOT_EQUAL ||
         this->eOperator == Expression::ExpGREATER ||
         this->eOperator == Expression::ExpLESS ||
         this->eOperator == Expression::ExpGREATER_OR_EQUAL ||
         this->eOperator == Expression::ExpLESS_OR_EQUAL ||
         this->eOperator == Expression::ExpAND ||
         this->eOperator == Expression::ExpOR ||
         this->eOperator == Expression::ExpPLUS ||
         this->eOperator == Expression::ExpMINUS ||
         this->eOperator == Expression::ExpMULTIPLY ||
         this->eOperator == Expression::ExpDIVIDE);
}


BOOL Expression::IsUnaryOperator( void) const
{
   return BOOL( this->eOperator == Expression::ExpTRUE ||
         this->eOperator == Expression::ExpFALSE ||
         this->eOperator == Expression::ExpBLANK ||
         this->eOperator == Expression::ExpNOT_BLANK);
}


BOOL Expression::IsLogicalOperator( void) const
{
   return BOOL( this->eOperator == Expression::ExpEQUAL ||
         this->eOperator == Expression::ExpLIKE ||
         this->eOperator == Expression::ExpNOT_EQUAL ||
         this->eOperator == Expression::ExpGREATER ||
         this->eOperator == Expression::ExpLESS ||
         this->eOperator == Expression::ExpGREATER_OR_EQUAL ||
         this->eOperator == Expression::ExpLESS_OR_EQUAL ||
         this->eOperator == Expression::ExpAND ||
         this->eOperator == Expression::ExpOR ||
         this->eOperator == Expression::ExpTRUE ||
         this->eOperator == Expression::ExpFALSE ||
         this->eOperator == Expression::ExpBLANK ||
         this->eOperator == Expression::ExpNOT_BLANK);
}


BOOL Expression::IsArithmeticOperator( void) const
{
   return BOOL( this->eOperator == Expression::ExpPLUS ||
         this->eOperator == Expression::ExpMINUS ||
         this->eOperator == Expression::ExpMULTIPLY ||
         this->eOperator == Expression::ExpDIVIDE);
}


/*******************************************************************************
** Evaluation functions
*******************************************************************************/

#define BINARY_OP_EVAL( _type_, _op_, _exptype_) \
if (this->pExpLeft && this->pExpRight) \
{ \
   _type_ left = *this->pExpLeft; \
   if (GetEvalError() == ExpElement::ExpNONE) \
      return (_type_) (left _op_ (_type_) *this->pExpRight); \
   else \
      return 0; \
} \
else \
{ \
   SetEvalError( ExpElement::ExpERROR, this, _exptype_); \
   return 0; \
}

INT16 Expression::EvalINT16( void) const
{
   switch (GetOperator())
   {
   // Binary relational operators
   case ExpEQUAL:
   case ExpLIKE:
   case ExpNOT_EQUAL:
   case ExpGREATER:
   case ExpLESS:
   case ExpGREATER_OR_EQUAL:
   case ExpLESS_OR_EQUAL:
   case ExpAND:
   case ExpOR:
   // Unary operators (no operand):
   case ExpTRUE:
   case ExpFALSE:
   case ExpBLANK:
   case ExpNOT_BLANK:
      return INT16( EvalBOOL());
   // Binary arithmetic operators
   case ExpPLUS:
      BINARY_OP_EVAL( INT16, +, ExpElement::ExpINT16)
   case ExpMINUS:
      BINARY_OP_EVAL( INT16, -, ExpElement::ExpINT16)
   case ExpMULTIPLY:
      BINARY_OP_EVAL( INT16, *, ExpElement::ExpINT16)
   case ExpDIVIDE:
      BINARY_OP_EVAL( INT16, /, ExpElement::ExpINT16)
   default:
      assert( FALSE);
   }

   return 0;
}


UINT16 Expression::EvalUINT16( void) const
{
   switch (GetOperator())
   {
   // Binary relational operators
   case ExpEQUAL:
   case ExpLIKE:
   case ExpNOT_EQUAL:
   case ExpGREATER:
   case ExpLESS:
   case ExpGREATER_OR_EQUAL:
   case ExpLESS_OR_EQUAL:
   case ExpAND:
   case ExpOR:
   // Unary operators (no operand):
   case ExpTRUE:
   case ExpFALSE:
   case ExpBLANK:
   case ExpNOT_BLANK:
      return (UINT16)( EvalBOOL());
   // Binary arithmetic operators
   case ExpPLUS:
      BINARY_OP_EVAL( UINT16, +, ExpElement::ExpUINT16)
   case ExpMINUS:
      BINARY_OP_EVAL( UINT16, -, ExpElement::ExpUINT16)
   case ExpMULTIPLY:
      BINARY_OP_EVAL( UINT16, *, ExpElement::ExpUINT16)
   case ExpDIVIDE:
      BINARY_OP_EVAL( UINT16, /, ExpElement::ExpUINT16)
   default:
      assert( FALSE);
   }

   return 0;
}


INT32 Expression::EvalINT32( void) const
{
   switch (GetOperator())
   {
   // Binary relational operators
   case ExpEQUAL:
   case ExpLIKE:
   case ExpNOT_EQUAL:
   case ExpGREATER:
   case ExpLESS:
   case ExpGREATER_OR_EQUAL:
   case ExpLESS_OR_EQUAL:
   case ExpAND:
   case ExpOR:
   // Unary operators (no operand):
   case ExpTRUE:
   case ExpFALSE:
   case ExpBLANK:
   case ExpNOT_BLANK:
      return INT32( EvalBOOL());
   // Binary arithmetic operators
   case ExpPLUS:
      BINARY_OP_EVAL( INT32, +, ExpElement::ExpINT32)
   case ExpMINUS:
      if (!this->pExpLeft || !this->pExpRight)
      {
         SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpINT32);
         return 0;
      }
      else
      {
         eEvalTypes etLeft = this->pExpLeft->GetEvalType();
         eEvalTypes etRight = this->pExpRight->GetEvalType();
         if (etLeft == ExpElement::ExpTIME || etRight == ExpElement::ExpTIME)
         {
            TTime left = *this->pExpLeft;
            if (GetEvalError() != ExpElement::ExpNONE)
               return 0;
            TTime right = *this->pExpRight;
            if (GetEvalError() != ExpElement::ExpNONE)
               return 0;
            return INT32( left.Seconds() - right.Seconds());
         }
         else if (etLeft == ExpElement::ExpDATE)
         {
            TDate left = *this->pExpLeft;
            if (GetEvalError() != ExpElement::ExpNONE)
               return 0;
            TDate right = *this->pExpRight;
            if (GetEvalError() != ExpElement::ExpNONE)
               return 0;
            return INT32( TDate::Jday( left.Month(), left.DayOfMonth(), left.Year()) -
               TDate::Jday( right.Month(), right.DayOfMonth(), right.Year()));
                  // TDate has no function that returns the Jday of a given TDate!
         }
         else
         {
            INT32 left = *this->pExpLeft;
            if (GetEvalError() == ExpElement::ExpNONE)
               return (INT32) (left - (INT32) *this->pExpRight);
            else
               return 0;
         }
      }
   case ExpMULTIPLY:
      BINARY_OP_EVAL( INT32, *, ExpElement::ExpINT32)
   case ExpDIVIDE:
      BINARY_OP_EVAL( INT32, /, ExpElement::ExpINT32)
   default:
      assert( FALSE);
   }

   return 0;
}


UINT32 Expression::EvalUINT32( void) const
{
   switch (GetOperator())
   {
   // Binary relational operators
   case ExpEQUAL:
   case ExpLIKE:
   case ExpNOT_EQUAL:
   case ExpGREATER:
   case ExpLESS:
   case ExpGREATER_OR_EQUAL:
   case ExpLESS_OR_EQUAL:
   case ExpAND:
   case ExpOR:
   // Unary operators (no operand):
   case ExpTRUE:
   case ExpFALSE:
   case ExpBLANK:
   case ExpNOT_BLANK:
      return (UINT32)( EvalBOOL());
   // Binary arithmetic operators
   case ExpPLUS:
      BINARY_OP_EVAL( UINT32, +, ExpElement::ExpUINT32)
   case ExpMINUS:
      BINARY_OP_EVAL( UINT32, -, ExpElement::ExpUINT32)
   case ExpMULTIPLY:
      BINARY_OP_EVAL( UINT32, *, ExpElement::ExpUINT32)
   case ExpDIVIDE:
      BINARY_OP_EVAL( UINT32, /, ExpElement::ExpUINT32)
   default:
      assert( FALSE);
   }

   return 0;
}


double Expression::EvalDOUBLE( void) const
{
   switch (GetOperator())
   {
   // Binary relational operators
   case ExpEQUAL:
   case ExpLIKE:
   case ExpNOT_EQUAL:
   case ExpGREATER:
   case ExpLESS:
   case ExpGREATER_OR_EQUAL:
   case ExpLESS_OR_EQUAL:
   case ExpAND:
   case ExpOR:
   // Unary operators (no operand):
   case ExpTRUE:
   case ExpFALSE:
   case ExpBLANK:
   case ExpNOT_BLANK:
      return double( EvalBOOL());
   // Binary arithmetic operators
   case ExpPLUS:
      BINARY_OP_EVAL( double, +, ExpElement::ExpDOUBLE)
   case ExpMINUS:
      BINARY_OP_EVAL( double, -, ExpElement::ExpDOUBLE)
   case ExpMULTIPLY:
      BINARY_OP_EVAL( double, *, ExpElement::ExpDOUBLE)
   case ExpDIVIDE:
      BINARY_OP_EVAL( double, /, ExpElement::ExpDOUBLE)
   default:
      assert( FALSE);
   }

   return 0;
}


char Expression::EvalCHAR( void) const
{
   switch (GetOperator())
   {
   // Binary relational operators
   case ExpEQUAL:
   case ExpLIKE:
   case ExpNOT_EQUAL:
   case ExpGREATER:
   case ExpLESS:
   case ExpGREATER_OR_EQUAL:
   case ExpLESS_OR_EQUAL:
   case ExpAND:
   case ExpOR:
   // Unary operators (no operand):
   case ExpTRUE:
   case ExpFALSE:
   case ExpBLANK:
   case ExpNOT_BLANK:
      return INT16( EvalBOOL());
   // Binary arithmetic operators
   case ExpPLUS:
      BINARY_OP_EVAL( char, +, ExpElement::ExpCHAR)
   case ExpMINUS:
      BINARY_OP_EVAL( char, -, ExpElement::ExpCHAR)
   case ExpMULTIPLY:
      BINARY_OP_EVAL( char, *, ExpElement::ExpCHAR)
   case ExpDIVIDE:
      BINARY_OP_EVAL( char, /, ExpElement::ExpCHAR)
   default:
      assert( FALSE);
   }

   return 0;
}


const char *Expression::EvalSTRING( void) const
{
   switch (GetOperator())
   {
   // Binary relational operators
   case ExpEQUAL:
   case ExpLIKE:
   case ExpNOT_EQUAL:
   case ExpGREATER:
   case ExpLESS:
   case ExpGREATER_OR_EQUAL:
   case ExpLESS_OR_EQUAL:
   case ExpAND:
   case ExpOR:
   // Unary operators (no operand):
   case ExpTRUE:
   case ExpFALSE:
   case ExpBLANK:
   case ExpNOT_BLANK:
   // Binary arithmetic operators
   case ExpPLUS:
   case ExpMINUS:
   case ExpMULTIPLY:
   case ExpDIVIDE:
      SetEvalError( ExpElement::ExpSYNTAX_ERROR, this, ExpElement::ExpSTRING);
      break;
   default:
      assert( FALSE);
   }

   return 0;
}


TDate Expression::EvalDATE( void) const
{
   switch (GetOperator())
   {
   // Binary relational operators
   case ExpEQUAL:
   case ExpLIKE:
   case ExpNOT_EQUAL:
   case ExpGREATER:
   case ExpLESS:
   case ExpGREATER_OR_EQUAL:
   case ExpLESS_OR_EQUAL:
   case ExpAND:
   case ExpOR:
   // Unary operators (no operand):
   case ExpTRUE:
   case ExpFALSE:
   case ExpBLANK:
   case ExpNOT_BLANK:
      SetEvalError( ExpElement::ExpSYNTAX_ERROR, this, ExpElement::ExpDATE);
      break;
   // Binary arithmetic operators
   case ExpPLUS:
      if (this->pExpLeft && this->pExpRight)
      {
         TDate left = *this->pExpLeft;
         if (GetEvalError() == ExpElement::ExpNONE)
            return (TDate)( left + INT16( *this->pExpRight));
         else
            return left;
      }
      else
         SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpDATE);
      break;
   case ExpMINUS:
      if (this->pExpLeft && this->pExpRight)
      {
         TDate left = *this->pExpLeft;
         if (GetEvalError() == ExpElement::ExpNONE)
            return (TDate)( left - INT16( *this->pExpRight));
         else
            return left;
      }
      else
         SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpDATE);
      break;
   case ExpMULTIPLY:
   case ExpDIVIDE:
      SetEvalError( ExpElement::ExpSYNTAX_ERROR, this, ExpElement::ExpDATE);
      break;
   default:
      assert( FALSE);
   }

   return TDate();
}


TTime Expression::EvalTIME( void) const
{
   switch (GetOperator())
   {
   // Binary relational operators
   case ExpEQUAL:
   case ExpLIKE:
   case ExpNOT_EQUAL:
   case ExpGREATER:
   case ExpLESS:
   case ExpGREATER_OR_EQUAL:
   case ExpLESS_OR_EQUAL:
   case ExpAND:
   case ExpOR:
   // Unary operators (no operand):
   case ExpTRUE:
   case ExpFALSE:
   case ExpBLANK:
   case ExpNOT_BLANK:
      SetEvalError( ExpElement::ExpSYNTAX_ERROR, this, ExpElement::ExpTIME);
      break;
   // Binary arithmetic operators
   case ExpPLUS:
      if (this->pExpLeft && this->pExpRight)
      {
         TTime left = *this->pExpLeft;
         if (GetEvalError() == ExpElement::ExpNONE)
            return (TTime)( left + (UINT32)( *this->pExpRight));
         else
            return left;
      }
      else
         SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpTIME);
      break;
   case ExpMINUS:
      if (this->pExpLeft && this->pExpRight)
      {
         TTime left = *this->pExpLeft;
         if (GetEvalError() == ExpElement::ExpNONE)
            return (TTime)( left - (UINT32)( *this->pExpRight));
         else
            return left;
      }
      else
         SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpTIME);
      break;
   case ExpMULTIPLY:
   case ExpDIVIDE:
      SetEvalError( ExpElement::ExpSYNTAX_ERROR, this, ExpElement::ExpTIME);
      break;
   default:
      assert( FALSE);
   }

   return 0;
}


#define COMP_CAST( _op_, _exptype_) \
if (!this->pExpLeft || !this->pExpRight) \
{ \
   SetEvalError( ExpElement::ExpERROR, this, _exptype_); \
   return 0; \
} \
else \
   { \
   eEvalTypes etLeft = this->pExpLeft->GetEvalType(); \
   eEvalTypes etRight = this->pExpRight->GetEvalType(); \
   if (etLeft == ExpElement::ExpDOUBLE || etRight == ExpElement::ExpDOUBLE) \
   { \
      double left = *this->pExpLeft; \
      if (GetEvalError() == ExpElement::ExpNONE) \
         return BOOL( left _op_ double( *this->pExpRight)); \
      else \
         return 0; \
   } \
   else if (etLeft == ExpElement::ExpSTRING || etRight == ExpElement::ExpSTRING) \
   { \
      const char *left = *this->pExpLeft; \
      if (GetEvalError() == ExpElement::ExpNONE) \
         return BOOL( strcmp( left, (const char *) *this->pExpRight) _op_ 0); \
      else \
         return 0; \
   } \
   else if (etLeft == ExpElement::ExpDATE || etLeft == ExpElement::ExpTIME) \
   { \
      TTime left = *this->pExpLeft; \
      if (GetEvalError()== ExpElement::ExpNONE) \
         return BOOL( left _op_ (TTime) *this->pExpRight); \
      else \
         return 0; \
   } \
   else \
   { \
      INT32 left = *this->pExpLeft; \
      if (GetEvalError() == ExpElement::ExpNONE) \
         return BOOL( left _op_ (INT32) *this->pExpRight); \
      else \
         return 0; \
   } \
}

BOOL Expression::EvalBOOL( void) const
{

   switch (GetOperator())
   {
   // Binary relational operators:
   case ExpEQUAL:
      COMP_CAST( ==, ExpElement::ExpBOOL)
   case ExpLIKE:
      if (this->pExpLeft->GetEvalType() == ExpElement::ExpSTRING && this->pExpRight)
      {
         const char *s = (const char *) *this->pExpRight;
         if (GetEvalError() == ExpElement::ExpNONE)
            return BOOL( strncmp( (const char *) *this->pExpLeft, s, strlen( s)) == 0);
         else
            return 0;
      }
      else
      {
         SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpBOOL);
         return 0;
      }
   case ExpNOT_EQUAL:
      COMP_CAST( !=, ExpElement::ExpBOOL)
   case ExpGREATER:
      COMP_CAST( >, ExpElement::ExpBOOL)
   case ExpLESS:
      COMP_CAST( <, ExpElement::ExpBOOL)
   case ExpGREATER_OR_EQUAL:
      COMP_CAST( >=, ExpElement::ExpBOOL)
   case ExpLESS_OR_EQUAL:
      COMP_CAST( <=, ExpElement::ExpBOOL)
   case ExpAND:
      BINARY_OP_EVAL( BOOL, &&, ExpElement::ExpBOOL)
   case ExpOR:
      BINARY_OP_EVAL( BOOL, ||, ExpElement::ExpBOOL)
   // Unary operators (no operand):
   case ExpTRUE:
      return BOOL( *this->pExpLeft);
   case ExpFALSE:
      return BOOL( !BOOL( *this->pExpLeft));
   case ExpBLANK:
      {
         eEvalTypes etLeft = this->pExpLeft->GetEvalType();
         if (etLeft == ExpElement::ExpDOUBLE)
            return BOOL( double( *this->pExpLeft) == 0.0);
         else if (etLeft == ExpElement::ExpSTRING)
            return BOOL( !BOOL( (const char *) *this->pExpLeft));
         else
            return BOOL( INT32( *this->pExpLeft) == 0);
      }
   case ExpNOT_BLANK:
      {
         eEvalTypes etLeft = this->pExpLeft->GetEvalType();
         if (etLeft == ExpElement::ExpDOUBLE)
            return BOOL( double( *this->pExpLeft) != 0.0);
         else if (etLeft == ExpElement::ExpSTRING)
            return BOOL( (const char *) *this->pExpLeft);
         else
            return BOOL( INT32( *this->pExpLeft) != 0);
      }
   // Binary arithmetic operators:
   case ExpPLUS:
   case ExpMINUS:
   case ExpMULTIPLY:
   case ExpDIVIDE:
      SetEvalError( ExpElement::ExpSYNTAX_ERROR, this, ExpElement::ExpBOOL);
      break;
   default:
      assert( FALSE);
   }

   return 0;
}


/*******************************************************************************
** Output stream operator
*******************************************************************************/

#if !defined( EXPRESS_NO_IOSTREAM)
BOOL Expression::bVerbose = FALSE;

void Expression::Output( ostream &_os) const
{
   if (GetOperator() == Expression::ExpFALSE && !Expression::bVerbose)
      _os << "!";

   if (this->pExpLeft->GetElementType() == ExpElement::ExpEXPRESSION &&
         ((Expression *) this->pExpLeft)->IsBinaryOperator() &&
         ((Expression *) this->pExpLeft)->GetOperator() != GetOperator())
   {
      _os << "(";
      this->pExpLeft->Output( _os);
      _os << ")";
   }
   else
      this->pExpLeft->Output( _os);

   switch (GetOperator())
   {
   case Expression::ExpEQUAL:
      _os << (Expression::bVerbose ? " is equal to " : " = ");
      break;
   case Expression::ExpLIKE:
      _os << (Expression::bVerbose ? " is like " : " *= ");
      break;
   case Expression::ExpNOT_EQUAL:
      _os << (Expression::bVerbose ? " is not equal to " : " != ");
      break;
   case Expression::ExpGREATER:
      _os << (Expression::bVerbose ? " is greater than " : " > ");
      break;
   case Expression::ExpLESS:
      _os << (Expression::bVerbose ? " is less than " : " < ");
      break;
   case Expression::ExpGREATER_OR_EQUAL:
      _os << (Expression::bVerbose ? " is greater than or equal to " : " >= ");
      break;
   case Expression::ExpLESS_OR_EQUAL:
      _os << (Expression::bVerbose ? " is less than or equal to " : " <= ");
      break;
   case Expression::ExpAND:
      _os << " and ";
      break;
   case Expression::ExpOR:
      _os << " or ";
      break;
   case Expression::ExpPLUS:
      _os << " + ";
      break;
   case Expression::ExpMINUS:
      _os << " - ";
      break;
   case Expression::ExpMULTIPLY:
      _os << " * ";
      break;
   case Expression::ExpDIVIDE:
      _os << " / ";
      break;
   case Expression::ExpTRUE:
      _os << (Expression::bVerbose ? " is true" : "");
      break;
   case Expression::ExpFALSE:
      _os << (Expression::bVerbose ? " is false" : "");
      break;
   case Expression::ExpBLANK:
      _os << " is blank";
      break;
   case Expression::ExpNOT_BLANK:
      _os << " is not blank";
      break;
   }

   if (IsBinaryOperator())
   {
      if (this->pExpRight->GetElementType() == ExpElement::ExpEXPRESSION &&
            ((Expression *) this->pExpRight)->IsBinaryOperator() &&
            ((Expression *) this->pExpRight)->GetOperator() != GetOperator())
      {
         _os << "(";
         this->pExpRight->Output( _os);
         _os << ")";
      }
      else
         this->pExpRight->Output( _os);
   }
}


ostream &operator<<( ostream &_os, const Expression &_eq)
{
   _eq.Output( _os);

   return _os;
}
#endif


/******************************************************************************
** IDAPI CANExpr conversion
******************************************************************************/

#if !defined( EXPRESS_NO_IDAPI)
void Expression::BuildCANBuffers(
      OutputBuffer &_bufNode,
      OutputBuffer &_bufLiteral,
      UINT16 &_nNodes,
      UINT16 &_nNodeOffset,
      UINT16 &_nLiteralOffset)
{
   if (GetOperator() == Expression::ExpTRUE)
      this->pExpLeft->BuildCANBuffers( _bufNode, _bufLiteral, _nNodes, _nNodeOffset, _nLiteralOffset);
   else if (IsUnaryOperator())
   {
      _nNodes++;
      _nNodeOffset += (UINT16) sizeof( CANUnary);

      CANUnary canUnary;
      canUnary.nodeClass = nodeUNARY;
      canUnary.canOp = GetCANOp();
      canUnary.iOperand1 = _nNodeOffset;

      OutputBuffer lbufNode, lbufLiteral;
      this->pExpLeft->BuildCANBuffers( lbufNode, lbufLiteral, _nNodes, _nNodeOffset, _nLiteralOffset);

      _bufNode << canUnary << lbufNode;
      _bufLiteral << lbufLiteral;
   }
   else
   {
      _nNodes++;
      _nNodeOffset += (UINT16) sizeof( CANBinary);

      CANBinary canBinary;
      canBinary.nodeClass = nodeBINARY;
      canBinary.canOp = GetCANOp();
      canBinary.iOperand1 = _nNodeOffset;

      OutputBuffer lbufNode, lbufLiteral;
      this->pExpLeft->BuildCANBuffers( lbufNode, lbufLiteral, _nNodes, _nNodeOffset, _nLiteralOffset);

      canBinary.iOperand2 = _nNodeOffset;

      OutputBuffer rbufNode, rbufLiteral;
      this->pExpRight->BuildCANBuffers( rbufNode, rbufLiteral, _nNodes, _nNodeOffset, _nLiteralOffset);

      _bufNode << canBinary << lbufNode << rbufNode;
      _bufLiteral << lbufLiteral << rbufLiteral;
   }
}


CANOp Expression::GetCANOp( void)
{
   switch (GetOperator())
   {
   case Expression::ExpEQUAL:
      return canEQ;
   case Expression::ExpLIKE:
      return canLIKE;
   case Expression::ExpNOT_EQUAL:
      return canNE;
   case Expression::ExpGREATER:
      return canGT;
   case Expression::ExpLESS:
      return canLT;
   case Expression::ExpGREATER_OR_EQUAL:
      return canGE;
   case Expression::ExpLESS_OR_EQUAL:
      return canLE;
   case Expression::ExpAND:
      return canAND;
   case Expression::ExpOR:
      return canOR;
   case Expression::ExpPLUS:
      return canADD;
   case Expression::ExpMINUS:
      return canSUB;
   case Expression::ExpMULTIPLY:
      return canMUL;
   case Expression::ExpDIVIDE:
      return canDIV;
   case Expression::ExpFALSE:
      return canNOT;
   case Expression::ExpBLANK:
      return canISBLANK;
   case Expression::ExpNOT_BLANK:
      return canNOTBLANK;
   default:
      assert( FALSE);
   }

   return canNOTDEFINED;
}
#endif
