/*******************************************************************************
** EXPELEM - ExpElement class
** Created 5/22/95 by John Stockton
** CIS: 76041,2534
** INTERNET: jstockton%inova@mcimail.com
**
** Copyright (c) 1995 INOVA Corporation.  May be copied, used and distributed, 
** as long as no charge is made for its use and this copyright notice is not
** altered or removed.
**
** $Header: j:/library/express/rcs/expelem.cpp 1.4 1995/06/29 15:06:35 JOHN Exp $
*******************************************************************************/

#include "express.hpp"
#if !defined( EXPRESS_NO_WINDOWS)
   #include <windows.h>
   #include <strstrea.h>
#endif
#include <assert.h>
#include <string.h>

/*******************************************************************************
** Static members
*******************************************************************************/

ExpElement::eEvalErrors ExpElement::eEvalError = ExpElement::ExpNONE;
BOOL ExpElement::bMessageBoxEnabled = FALSE;

#pragma argsused
void ExpElement::SetEvalError( ExpElement::eEvalErrors _eEvalError, const ExpElement *_elem, eEvalTypes _eEvalType)
{
   if (ExpElement::eEvalError == ExpElement::ExpNONE)
      ExpElement::eEvalError = _eEvalError;
#if !defined( EXPRESS_NO_WINDOWS)
   if (_eEvalError != ExpElement::ExpNONE &&
         ExpElement::bMessageBoxEnabled)
   {
      ExpElement::eEvalErrors eEvalErrorSaved = ExpElement::eEvalError;
         // Remember original error code
      ExpElement::bMessageBoxEnabled = FALSE;
         // Prevent recursive error reporting

      strstream s;
      switch (_eEvalError)
      {
      case ExpDATA_UNAVAILABLE:
         s << "Data unavalable evaluating ";
         break;
      case ExpTYPE_MISMATCH:
         s << "Data type mismatch evaluating ";
         break;
      case ExpOVERFLOW:
         s << "Data overflow evaluating ";
         break;
      case ExpERROR:
         s << "Error evaluating ";
         break;
      case ExpSYNTAX_ERROR:
         s << "Syntax error evaluating ";
         break;
      default:
         s << "Unknown error evaluating ";
      }

      assert( _elem);
      switch (_elem->GetElementType())
      {
      case ExpElement::ExpEXPRESSION:
         s << "expression ";
         break;
      case ExpElement::ExpVARIABLE:
         s << "variable ";
         break;
      case ExpElement::ExpCONSTANT:
         s << "constant ";
         break;
      }

      _elem->Output( s);

      s << " as ";
      switch (_eEvalType)
      {
      case ExpINT16:
         s << " INT16";
         break;
      case ExpUINT16:
         s << " UINT16";
         break;
      case ExpINT32:
         s << " INT32";
         break;
      case ExpUINT32:
         s << " UINT32";
         break;
      case ExpDOUBLE:
         s << " DOUBLE";
         break;
      case ExpCHAR:
         s << " CHAR";
         break;
      case ExpSTRING:
         s << " STRING";
         break;
      case ExpBOOL:
         s << " BOOL";
         break;
      case ExpDATE:
         s << " DATE";
         break;
      case ExpTIME:
         s << " TIME";
         break;
      default:
         s << " UNKNOWN";
      }

      s << '\0';
      MessageBox( 0, s.str(), "Expression Evaluation Error", MB_OK);

      ExpElement::eEvalError = eEvalErrorSaved;
         // Restore original error code (may have been reset during error reporting)
      ExpElement::bMessageBoxEnabled = TRUE;
         // Turn error reporting back on
   }
#endif
}


void ExpElement::ResetEvalError( void)
{
   ExpElement::eEvalError = ExpElement::ExpNONE;
}


void ExpElement::EnableMessageBox( BOOL bEnable)
{
   ExpElement::bMessageBoxEnabled = bEnable;
}


/*******************************************************************************
** Binary expression construction operators
*******************************************************************************/

Expression ExpElement::operator==( const ExpElement &_rhs) const
{
   return Expression( *this, Expression::ExpEQUAL, _rhs);
}


Expression ExpElement::operator*=( const ExpElement &_rhs) const
{
   return Expression( *this, Expression::ExpLIKE, _rhs);
}


Expression ExpElement::operator!=( const ExpElement &_rhs) const
{
   return Expression( *this, Expression::ExpNOT_EQUAL, _rhs);
}


Expression ExpElement::operator>( const ExpElement &_rhs) const
{
   return Expression( *this, Expression::ExpGREATER, _rhs);
}


Expression ExpElement::operator<( const ExpElement &_rhs) const
{
   return Expression( *this, Expression::ExpLESS, _rhs);
}


Expression ExpElement::operator>=( const ExpElement &_rhs) const
{
   return Expression( *this, Expression::ExpGREATER_OR_EQUAL, _rhs);
}


Expression ExpElement::operator<=( const ExpElement &_rhs) const
{
   return Expression( *this, Expression::ExpLESS_OR_EQUAL, _rhs);
}


Expression ExpElement::operator&&( const ExpElement &_rhs) const
{
   return Expression( *this, Expression::ExpAND, _rhs);
}


Expression ExpElement::operator||( const ExpElement &_rhs) const
{
   return Expression( *this, Expression::ExpOR, _rhs);
}


Expression ExpElement::operator+( const ExpElement &_rhs) const
{
   return Expression( *this, Expression::ExpPLUS, _rhs);
}


Expression ExpElement::operator-( const ExpElement &_rhs) const
{
   return Expression( *this, Expression::ExpMINUS, _rhs);
}


Expression ExpElement::operator*( const ExpElement &_rhs) const
{
   return Expression( *this, Expression::ExpMULTIPLY, _rhs);
}


Expression ExpElement::operator/( const ExpElement &_rhs) const
{
   return Expression( *this, Expression::ExpDIVIDE, _rhs);
}


/*******************************************************************************
** Unary expression construction operators
*******************************************************************************/

ExpElement::operator Expression( void) const
{
   return Expression( *this, Expression::ExpTRUE);
}


Expression ExpElement::operator!( void) const
{
   return Expression( *this, Expression::ExpFALSE);
}


Expression ExpElement::IsBlank( void) const
{
   return Expression( *this, Expression::ExpBLANK);
}


Expression ExpElement::IsNotBlank( void) const
{
   return Expression( *this, Expression::ExpNOT_BLANK);
}


/*******************************************************************************
** Evaluation operators
*******************************************************************************/

ExpElement::operator INT16( void) const
{
   INT16 i = 0;

   ResetEvalError();

   switch (GetEvalType())
   {
   case ExpElement::ExpUNKNOWN:
      SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpINT16);
      break;
   case ExpElement::ExpINT16:
      i = EvalINT16();
      break;
   case ExpElement::ExpUINT16:
      i = INT16( EvalUINT16());
      SetEvalError( i >= 0 ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpINT16);
      break;
   case ExpElement::ExpINT32:
      {
      INT32 l = EvalINT32();
      i = INT16( l);
      SetEvalError( INT32( i) == l ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpINT16);
      }
      break;
   case ExpElement::ExpUINT32:
      {
      UINT32 ul = EvalUINT32();
      i = INT16( ul);
      SetEvalError( i >= 0 && (UINT32)( i) == ul ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpINT16);
      }
      break;
   case ExpElement::ExpDOUBLE:
      {
      double d = EvalDOUBLE();
      i = INT16( d);
      SetEvalError( double( i) == d ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpINT16);
      }
      break;
   case ExpElement::ExpCHAR:
      i = INT16( EvalCHAR());
      break;
   case ExpElement::ExpSTRING:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpINT16);
      break;
   case ExpElement::ExpBOOL:
      i = INT16( EvalBOOL());
      break;
   case ExpElement::ExpDATE:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpINT16);
      break;
   case ExpElement::ExpTIME:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpINT16);
      break;
   default:
      assert( FALSE);
   }

   return i;
}


ExpElement::operator UINT16( void) const
{
   UINT16 u = 0;

   ResetEvalError();

   switch (GetEvalType())
   {
   case ExpElement::ExpUNKNOWN:
      SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpUINT16);
      break;
   case ExpElement::ExpINT16:
      {
      INT16 i = EvalINT16();
      u = (UINT16)( i);
      SetEvalError( i >= 0 ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpUINT16);
      }
      break;
   case ExpElement::ExpUINT16:
      u = EvalUINT16();
      break;
   case ExpElement::ExpINT32:
      {
      INT32 l = EvalINT32();
      u = (UINT16)( l);
      SetEvalError( l >= 0 && INT32( u) == l ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpUINT16);
      }
      break;
   case ExpElement::ExpUINT32:
      {
      UINT32 ul = EvalUINT32();
      u = (UINT16)( ul);
      SetEvalError( (UINT32)( u) == ul ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpUINT16);
      }
      break;
   case ExpElement::ExpDOUBLE:
      {
      double d = EvalDOUBLE();
      u = (UINT16)( d);
      SetEvalError( double( u) == d ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpUINT16);
      }
      break;
   case ExpElement::ExpCHAR:
      u = (UINT16)( EvalCHAR());
      break;
   case ExpElement::ExpSTRING:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpUINT16);
      break;
   case ExpElement::ExpBOOL:
      u = (UINT16)( EvalBOOL());
      break;
   case ExpElement::ExpDATE:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpUINT16);
      break;
   case ExpElement::ExpTIME:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpUINT16);
      break;
   default:
      assert( FALSE);
   }

   return u;
}


ExpElement::operator INT32( void) const
{
   INT32 l = 0;

   ResetEvalError();

   switch (GetEvalType())
   {
   case ExpElement::ExpUNKNOWN:
      SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpINT32);
      break;
   case ExpElement::ExpINT16:
      l = INT32( EvalINT16());
      break;
   case ExpElement::ExpUINT16:
      l = INT32( EvalUINT16());
      break;
   case ExpElement::ExpINT32:
      l = EvalINT32();
      break;
   case ExpElement::ExpUINT32:
      l = INT32( EvalUINT32());
      SetEvalError( l >= 0 ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpINT32);
      break;
   case ExpElement::ExpDOUBLE:
      {
      double d = EvalDOUBLE();
      l = INT32( d);
      SetEvalError( double( l) == d ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpINT32);
      }
      break;
   case ExpElement::ExpCHAR:
      l = INT32( EvalCHAR());
      break;
   case ExpElement::ExpSTRING:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpINT32);
      break;
   case ExpElement::ExpBOOL:
      l = INT32( EvalBOOL());
      break;
   case ExpElement::ExpDATE:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpINT32);
      break;
   case ExpElement::ExpTIME:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpINT32);
      break;
   default:
      assert( FALSE);
   }

   return l;
}


ExpElement::operator UINT32( void) const
{
   UINT32 ul = 0;

   ResetEvalError();

   switch (GetEvalType())
   {
   case ExpElement::ExpUNKNOWN:
      SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpUINT32);
      break;
   case ExpElement::ExpINT16:
      {
      int i = EvalINT16();
      ul = (UINT32)( i);
      SetEvalError( i >= 0 ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpUINT32);
      }
      break;
   case ExpElement::ExpUINT16:
      ul = (UINT32)( EvalUINT16());
      break;
   case ExpElement::ExpINT32:
      {
      INT32 l = EvalINT32();
      ul = (UINT32)( l);
      SetEvalError( l >= 0 ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpUINT32);
      }
      break;
   case ExpElement::ExpUINT32:
      ul = EvalUINT32();
      break;
   case ExpElement::ExpDOUBLE:
      {
      double d = EvalDOUBLE();
      ul = (UINT32)( d);
      SetEvalError( double( ul) == d ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpUINT32);
      }
      break;
   case ExpElement::ExpCHAR:
      ul = (UINT32)( EvalCHAR());
      break;
   case ExpElement::ExpSTRING:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpUINT32);
      break;
   case ExpElement::ExpBOOL:
      ul = (UINT32)( EvalBOOL());
      break;
   case ExpElement::ExpDATE:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpUINT32);
      break;
   case ExpElement::ExpTIME:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpUINT32);
      break;
   default:
      assert( FALSE);
   }

   return ul;
}


ExpElement::operator double( void) const
{
   double d = 0;

   ResetEvalError();

   switch (GetEvalType())
   {
   case ExpElement::ExpUNKNOWN:
      SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpDOUBLE);
      break;
   case ExpElement::ExpINT16:
      d = double( EvalINT16());
      break;
   case ExpElement::ExpUINT16:
      d = double( EvalUINT16());
      break;
   case ExpElement::ExpINT32:
      d = double( EvalINT32());
      break;
   case ExpElement::ExpUINT32:
      d = double( EvalUINT32());
      break;
   case ExpElement::ExpDOUBLE:
      d = EvalDOUBLE();
      break;
   case ExpElement::ExpCHAR:
      d = double( EvalCHAR());
      break;
   case ExpElement::ExpSTRING:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDOUBLE);
      break;
   case ExpElement::ExpBOOL:
      d = double( EvalBOOL());
      break;
   case ExpElement::ExpDATE:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDOUBLE);
      break;
   case ExpElement::ExpTIME:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDOUBLE);
      break;
   default:
      assert( FALSE);
   }

   return d;
}


ExpElement::operator char( void) const
{
   char c = 0;

   ResetEvalError();

   switch (GetEvalType())
   {
   case ExpElement::ExpUNKNOWN:
      SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpCHAR);
      break;
   case ExpElement::ExpINT16:
      {
      int i = EvalINT16();
      c = char( i);
      SetEvalError( int( c) == i ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpCHAR);
      }
      break;
   case ExpElement::ExpUINT16:
      {
      UINT16 u = EvalUINT16();
      c = char( u);
      SetEvalError( (UINT16)( c) == u ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpCHAR);
      }
      break;
   case ExpElement::ExpINT32:
      {
      INT32 l = EvalINT32();
      c = char( l);
      SetEvalError( INT32( c) == l ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpCHAR);
      }
      break;
   case ExpElement::ExpUINT32:
      {
      UINT32 ul = EvalUINT32();
      c = char( ul);
      SetEvalError( (UINT32)( c) == ul ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpCHAR);
      }
      break;
   case ExpElement::ExpDOUBLE:
      {
      double d = EvalDOUBLE();
      c = char( d);
      SetEvalError( double( c) == d ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpCHAR);
      }
      break;
   case ExpElement::ExpCHAR:
      c = EvalCHAR();
      break;
   case ExpElement::ExpSTRING:
      {
      const char *s = EvalSTRING();
      if (s)
      {
         c = *s;
         SetEvalError( strlen( s) <= 1 ? ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpCHAR);
      }
      else
         SetEvalError( ExpElement::ExpERROR, this, ExpElement::ExpCHAR);
      }
      break;
   case ExpElement::ExpBOOL:
      c = char( EvalBOOL());
      break;
   case ExpElement::ExpDATE:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpCHAR);
      break;
   case ExpElement::ExpTIME:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpCHAR);
      break;
   default:
      assert( FALSE);
   }

   return c;
}


ExpElement::operator const char *( void) const
{
   const char *s = 0;

   ResetEvalError();

   switch (GetEvalType())
   {
   case ExpElement::ExpUNKNOWN:
      SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpSTRING);
      break;
   case ExpElement::ExpINT16:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpSTRING);
      break;
   case ExpElement::ExpUINT16:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpSTRING);
      break;
   case ExpElement::ExpINT32:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpSTRING);
      break;
   case ExpElement::ExpUINT32:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpSTRING);
      break;
   case ExpElement::ExpDOUBLE:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpSTRING);
      break;
   case ExpElement::ExpCHAR:
      {
         static char sz[2];
         sz[ 0] = EvalCHAR();
         sz[ 1] = '\0';
         s = sz;
      }
      break;
   case ExpElement::ExpSTRING:
      s = EvalSTRING();
      break;
   case ExpElement::ExpBOOL:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpSTRING);
      break;
   case ExpElement::ExpDATE:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpSTRING);
      break;
   case ExpElement::ExpTIME:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpSTRING);
      break;
   default:
      assert( FALSE);
   }

   return s;
}


ExpElement::operator TDate( void) const
{
   TDate date;

   ResetEvalError();

   switch (GetEvalType())
   {
   case ExpElement::ExpUNKNOWN:
      SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpINT16:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpUINT16:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpINT32:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpUINT32:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpDOUBLE:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpCHAR:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpSTRING:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpBOOL:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpDATE:
      date = EvalDATE();
      break;
   case ExpElement::ExpTIME:
      {
         TTime time( EvalTIME());
         date = TDate( time);
         SetEvalError( time.Hour() == 0 && time.Minute() == 0 && time.Second() == 0 ?
            ExpElement::ExpNONE : ExpElement::ExpOVERFLOW, this, ExpElement::ExpDATE);
      }
      break;
   default:
      assert( FALSE);
   }

   return date;
}


ExpElement::operator TTime( void) const
{
   TTime time;

   ResetEvalError();

   switch (GetEvalType())
   {
   case ExpElement::ExpUNKNOWN:
      SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpINT16:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpUINT16:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpINT32:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpUINT32:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpDOUBLE:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpCHAR:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpSTRING:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpBOOL:
      SetEvalError( ExpElement::ExpTYPE_MISMATCH, this, ExpElement::ExpDATE);
      break;
   case ExpElement::ExpDATE:
      time = TTime( EvalDATE());
      break;
   case ExpElement::ExpTIME:
      time = EvalTIME();
      break;
   default:
      assert( FALSE);
   }

   return time;
}


/******************************************************************************
** IDAPI CANExpr conversion
******************************************************************************/

#if !defined( EXPRESS_NO_IDAPI)
ExpElement::operator const pCANExpr( void)
{
   OutputBuffer bufNodes, bufLiterals;
   UINT16 nNodes = 0, nNodeOffset = 0, nLiteralOffset = 0;
   BuildCANBuffers( bufNodes, bufLiterals, nNodes, nNodeOffset, nLiteralOffset);

   CANExpr canExpr = {
      CANEXPRVERSION,
      (UINT16) (sizeof( CANExpr) + bufNodes.GetSize() + bufLiterals.GetSize()),
      nNodes,
      sizeof( CANExpr),
      (UINT16) (sizeof( CANExpr) + bufNodes.GetSize()) };

   this->bufCANExpr.Reset();
   this->bufCANExpr << canExpr << bufNodes << bufLiterals;

   return (pCANExpr)(const BYTE *) this->bufCANExpr;
}
#endif

