
/*  DOCMEM.C  */

/**************************************************************************/
/****  Memory allocation bound checking routines.                      ****/
/**************************************************************************/
/****  Written and released into the public domain 1995 by Brian Reed  ****/
/****  (aka Doc Raster)  CIS 73503,3364 or 73503.3364@compuserve.com   ****/
/**************************************************************************/
/****  Use at your own risk.                                           ****/
/**************************************************************************/

/**************************************************************************/
/****  When reading this code, remember that far pointers cannot be    ****/
/****  compared directly, and are first converted to huge pointers.    ****/
/**************************************************************************/

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <alloc.h>
#include "docmem.h"



/*  The head and tail of our master tracker linked list.  */
/*  Initialized in DOCstart().                            */
DocMem DOChead, DOCtail;



/****************  ERROR ROUTINES.  ****************/


/*  Global variables.  */
static char DOClogfile[]="DOCMEM.LOG";
static int  DOClasterror=DOC_OK;


/*  Set the last error value.  */
void DOCerr_set(int err)
{
  DOClasterror = err;
}


/*  Get the last error value.  */
int DOCerr_last(void)
{
  return DOClasterror;
}


/*  Retrieve string corresponding to an error value.  */
char * DOCerr_string(int err)
{
static char dm_0[] = "DOCMEM(00): No error";
static char dm_1[] = "DOCMEM(01): Pointer to low address";
static char dm_2[] = "DOCMEM(02): DocMem memory corrupted";
static char dm_3[] = "DOCMEM(03): Pointer is invalid";
static char dm_4[] = "DOCMEM(04): Memory overrun";
static char dm_5[] = "DOCMEM(05): Memory underrun";
static char dm_6[] = "DOCMEM(06): Unfreed allocation";
static char dm_7[] = "USRMESSAGE: ";
static char dm_8[] = "DOCMEM(08): Unknown error";
static char * docmem_errors[] = {
  dm_0, dm_1, dm_2, dm_3, dm_4, dm_5, dm_6, dm_7, dm_8, NULL
};

  if (err > DOC_UNKNOWN)
    err = DOC_UNKNOWN;
  return docmem_errors[err];
}



/****************  LOG FILE ROUTINE.  ****************/


/*  Write error to logfile.  */
int DOClog_write(char *label, int err, void _far *ptr)
{
static char nullstr[]="";
FILE *errfp;

  //  If no label, just provide a terminated string.
  if (label == NULL)
    label = nullstr;

  //  Protect against invalid error values and write to the log file.
  if (err > DOC_UNKNOWN)
    err = DOC_UNKNOWN;
  if ((errfp=fopen(DOClogfile, "a")) != NULL) {
    fprintf(errfp, "%-20s %-40s (%Fp)\n", label, DOCerr_string(err), ptr);
    fclose(errfp);
  }

  //  Set the last error and return.
  if (err != DOC_OK && err != DOC_USER)
    DOClasterror = err;
  return err;
}



/****************  DOCMEM ROUTINES.    ****************/


/*  Initialize DocMem.  */
void DOCstart(void)
{
FILE *errfp;

  //  Delete the old log and init the new one
  unlink(DOClogfile);
  if ((errfp=fopen(DOClogfile, "a")) != NULL) {
    fprintf(errfp, "Starting DocMem.\n");
    fclose(errfp);
  }

  //  Init DOChead and DOCtail of list.
  DOChead.next = &DOCtail;
  DOChead.prev = NULL;
  DOChead.magic1 = DOCmagic;
  DOChead.magic2 = DOCmagic;
  DOChead.trueaddr = NULL;
  DOChead.truesize = 0L;

  DOCtail.next = NULL;
  DOCtail.prev = &DOChead;
  DOCtail.magic1 = DOCmagic;
  DOCtail.magic2 = DOCmagic;
  DOCtail.trueaddr = NULL;
  DOCtail.truesize = 0L;
}


/*  Terminate DocMem.  */
int DOCend(void)
{
volatile DocMem _far *temp;
static char label[20];
FILE *errfp;
int i=0;

  //  Count up unfreed allocations.
  temp = DOChead.next;
  while (temp->next) {
    i++;
    temp = temp->next;
  }

  //  If no unfreed allocations, exit with 'no error'.
  if (i == 0) {
    if ((errfp=fopen(DOClogfile, "a")) != NULL) {
      fprintf(errfp, "Ending DocMem.       DOCMEM(--): No 'hanging' allocations");
      fclose(errfp);
    }
    return DOC_OK;
  }

  //  Look through linked list, asserting all in list.
  temp = DOChead.next;
  while (temp->next) {
    DOCfarassert("DOCend assert:",
      (void _far *)((char _huge *)temp->trueaddr+DOCBUFFER));
    temp = temp->next;
  }

  //  Log the unfreed allocations.
  if ((errfp=fopen(DOClogfile, "a")) != NULL) {
    temp = DOChead.next;
    strcpy(label, "Ending Docmem:");
    while (temp->next) {
      fprintf(errfp,
        "%-20s %-40s (%Fp)\n", label, DOCerr_string(DOC_UNFREED),
        (char _huge *)temp->trueaddr+DOCBUFFER);
      label[0] = 0;
      temp = temp->next;
    }
    fclose(errfp);
  }

  //  Free all in list.
  temp = DOChead.next;
  while (temp->next) {
    DOCfarfree((void _far *)((char _huge *)temp->trueaddr+DOCBUFFER));
    temp = DOChead.next;   // back to DOChead.next!!  DOCfarfree moved em!
  }

  //  Set DOClasterror and tell the caller we had some unfreed's.
  DOClasterror = DOC_UNFREED;
  return DOC_UNFREED;
}


/*  Assert a specified pointer.  */
int DOCfarassert(char *label, void _far *pointer)
{
DocMem _far *temp;
char _huge *c;
char _huge *p;
int found=0, i;

  //  Check for NULL pointer (or similar).
  if ((void _huge *)pointer < (void _huge *)DOClowaddr)
    return DOClog_write(label, DOC_LOWADDR, pointer);

  //  See if we find this exact block.
  temp = DOChead.next;
  while (temp->next) {
    if ((char _huge *)temp->trueaddr == (char _huge *)pointer - DOCBUFFER) {
      found = 1;
      break;
    }
    temp = temp->next;
  }

  // If its not an exact allocation base, is it within range of any alloc?
  if (found == 0) {
    temp = DOChead.next;
    p = (char _huge *)pointer - DOCBUFFER;     // adjust addr
    while (temp->next) {
      //  These comparisons must use huge pointers.
      c = (char _huge *)temp->trueaddr + temp->truesize;
      if (p >= (char _huge *)temp->trueaddr  &&  p < c) {
        found = 1;
        break;
      }
      temp = temp->next;
    }
  }

  //  If it isn't within range of ANY of our memory, call it bad.
  if (found == 0)
    return DOClog_write(label, DOC_INVALID, pointer);

  //  Check if the DOC data is still OK.
  if (temp->magic1 != DOCmagic || temp->magic2 != DOCmagic)
    return DOClog_write(label, DOC_CORRUPT, pointer);

  //  Check the bounding bytes for the matching block at the 'under' end.
  //  OPTIMIZE BY USING A SETUP BUFFER AND USE MEMCMP()
  c = temp->trueaddr;
  for (i=0; i<DOCBUFFER; i++)
    if (*c++ != (i%128)+70)
      return DOClog_write(label, DOC_UNDERRUN, pointer);

  // Check the bounding bytes for the matching block at the 'over' end.
  //  OPTIMIZE BY USING A SETUP BUFFER AND USE MEMCMP()
  c += temp->truesize;
  for (i=0; i<DOCBUFFER; i++)
    if (*c++ != (i%128)+70)
      return DOClog_write(label, DOC_OVERRUN, pointer);

  //  All is well!
  return DOC_OK;
}


/*  Perform an allocation.  */
void _far * DOCfarmalloc(unsigned long nbytes)
{
int i;
void _far *temp;
char _huge *c;
DocMem _far *newnode;

  #if 0
  DocMem _far *node;
  printf("entering farmalloc()\n");
  node = &DOChead;
  while (node) {
    printf("addr=%Fp prev=%Fp next=%Fp trueaddr=%Fp truesize=%ld\n",
     node, node->prev, node->next, node->trueaddr, node->truesize);
    node = node->next;
    getch();
  }
  #endif

  //  Allocate memory for our linked list debug info.
  newnode = (DocMem _far *)farmalloc((unsigned long)sizeof(DocMem));
  if (newnode == NULL)
    return (void _far *)NULL;

  //  Allocate the requested memory.
  nbytes += DOCBUFFER2;
  temp = farmalloc(nbytes);
  if (temp == NULL) {
    farfree(newnode);
    return NULL;
  }

  //  Stuff our info.
  DOCtail.prev->next = newnode;
  newnode->prev = DOCtail.prev;
  DOCtail.prev = newnode;
  newnode->next = &DOCtail;
  newnode->magic1 = DOCmagic;
  newnode->magic2 = DOCmagic;
  newnode->truesize = nbytes-DOCBUFFER2;
  newnode->trueaddr = temp;

  //  Stuff our check values at both ends.
  //  OPTIMIZE BY USING A SETUP BUFFER AND USE MEMCPY()
  c = temp;
  for (i=0; i<DOCBUFFER; i++)
    *c++ = (i%128)+70;
  c += newnode->truesize;
  for (i=0; i<DOCBUFFER; i++)
    *c++ = (i%128)+70;

  #if 0
  printf("exiting farmalloc()\n");
  node = &DOChead;
  while (node) {
    printf("addr=%Fp prev=%Fp next=%Fp trueaddr=%Fp truesize=%ld\n",
     node, node->prev, node->next, node->trueaddr, node->truesize);
    node = node->next;
  }
  #endif

  //  Return the working pointer.
  return (void _far *)((char _huge *)newnode->trueaddr + DOCBUFFER);
}


/*  Free an allocation.  */
int DOCfarfree(void _far *block)
{
DocMem _far *temp=DOChead.next;
char _huge *targetblock;

  //  Check for NULL pointer (or similar).
  if ((void _huge *)block < (void _huge *)DOClowaddr) {
    DOClog_write("DOCfarfree:", DOC_LOWADDR, block);
    return DOC_LOWADDR;
  }

  //  Adjust to true address.
  targetblock = (char _huge *)block - DOCBUFFER;

  //  Look for it in the linked list.  If found, assert and remove.
  while (temp->next) {
    if ((char _huge *)temp->trueaddr == targetblock) {
      DOCfarassert("DOCfarfree:", block);
      farfree(temp->trueaddr);
      temp->prev->next = temp->next;
      temp->next->prev = temp->prev;
      farfree(temp);
      return DOC_OK;
    }
    temp = temp->next;
  }

  return DOC_INVALID;
}


