/*
**	Copyright  1994 Novell, Inc.  All rights reserved.
**
**	Permission is granted to the recipient of this work ("you") to use,
**	reproduce and distribute Novell's original publication of the work free
**	of charge provided that you reproduce the work in its entirety and
**	include all Novell copyright notices as they originally appear.
**
**	Novell grants you permission to modify and distribute copies of this
**	work including any portion of this work (a "modified work") provided
**	that you include prominent notification of such modification along with
**	the date of modification on a modified work; distribute or publish a
**	modified work to third parties under the same conditions and granting
**	the same rights as are extended to you by Novell under this under
**	permission notice; and provided that you include a copy of Novell's
**	original publication of the work along with any copy of a modified
**	work.
**
**	NOVELL MAKES NO WARRANTY, REPRESENTATION OR PROMISE THAT THIS WORK OR A
**	MODIFIED WORK WILL SATISFY YOUR REQUIREMENTS OR THAT THIS WORK OR A
**	MODIFIED WORK IS WITHOUT DEFECT OR ERROR.  NOVELL DISCLAIMS AND
**	EXCLUDES ANY AND ALL IMPLIED WARRANTIES OF MERCHANTABILITY, TITLE OR
**	FITNESS FOR A PARTICULAR PURPOSE.
**
**	IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW WILL NOVELL OR ANY OTHER
**	PARTY BE LIABLE FOR DAMAGES INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
**	CONSEQUENTIAL, INDIRECT OR PUNATIVE DAMAGES ARISING OUT OF THE USE OF
**	OR INABLITITY TO USE THE WORK OR A MODIFIED WORK.
**
**	DSOBJECT.CPP - October 1994
**
**	Definition of the DSObject class.
**
**	Author: John Buckle, Asia Pacific Support Centre, Novell Australia
**	==================================================================
**	9 Jan 1995           First release              John Buckle
*/

# include "dsdefs.h"
# include "dsobject.h"
# include "dsvmulti.h"

# include <string.h>

/*
** DSObject::WriteBufferModified	Used to indicate if an object has been
**					modified, and hence whether the NDS
**					needs to be updated.
*/

int DSObject::WriteBufferModified = 0 ;

/*
** NWDSCCODE DSObject::ReadObject(NWPSTR rdn, DSBuffer * buffer)
**
**	Initialise the DSObject by reading the required attributes off
**	the NDS. The attributes to read are listed the object's attribute
**	and value count arrays, which is accessed using the virtual functions
**	AttributeNames() and AttributeCount().
*/

NWDSCCODE DSObject::ReadObject(NWPSTR rdn, DSBuffer * buffer)
{
DSReadObject	dsReadObject(this) ;
NWPSTR *	attrNames = AttributeNames() ;
WORD *		attrCount = AttributeCount() ;

	buffer->InitBuf(DSV_READ) ;

	for (int index = 0 ; attrNames[index] ; index++)
		if (attrCount[index])
			buffer->PutAttrName(attrNames[index]) ;

	return buffer->Read(rdn,1,0,& dsReadObject) ;
} ;

/*
** NWDSCCODE DSObject::WriteObject(
**			NWPSTR		rdn,
**			DSBuffer *	buffer,
**			DSObject *	oldObject = 0)
**
**	Create or modify the NDS copy of the object. If oldObject is null then
**	a new object is created. Otherwise the object is compared with the
**	oldObject and modifications are recorded in the DSBuffer object.
*/

NWDSCCODE DSObject::WriteObject(NWPSTR rdn, DSBuffer * buffer, DSObject * oldObject)
{
	if (oldObject)
		buffer->InitBuf(DSV_MODIFY_ENTRY) ;
	else {
		buffer->InitBuf(DSV_ADD_ENTRY) ;
		buffer->PutAttrName("Object Class",SYN_CLASS_NAME,ClassName()) ;
		}

	WriteBufferModified = 0 ;
	NWDSCCODE status    = ForAll(WriteAll,buffer,oldObject) ;
	if (status || WriteBufferModified == 0) return status ;

	return (oldObject)
		? buffer->ModifyObject(rdn)
		: buffer->AddObject(rdn) ;
} ;

/*
** NWDSCCODE DSObject::CompareObject(DSObject * oldObject)
**
**	Returns 0 if oldObject is the same as this object.
*/

NWDSCCODE DSObject::CompareObject(DSObject * oldObject)
{
	WriteBufferModified = 0 ;

	return ForAll(WriteAll,0,oldObject) || WriteBufferModified ;
} ;

/*
** NWDSCCODE DSObject::WriteNewValue(
**			DSBuffer *	buffer,
**			NWPSTR		attribute,
**			DSValue *	newValue)
**
**	Write a new NDS value to the buffer. This function is executed when
**	a new NDS object is created. If buffer is null then two DSObjects are
**	being compared, hence return 1 to indicate that they are different.
*/

NWDSCCODE DSObject::WriteNewValue(DSBuffer * buffer, NWPSTR attribute, DSValue * newValue)
{
	WriteBufferModified = 1 ;

	if (buffer == 0) return 1 ;

	if (newValue->Syntax == 0) return 0 ;

	if (newValue->type() == DSCPP_MULTI_VALUE){
		DSVMultiValue & newMulti = * (DSVMultiValue *) newValue ;
		buffer->PutAttrName(attribute) ;
		for (DSValueIterator iter(newMulti.List) ; iter() ; iter.next()){
			buffer->PutAttrValue(iter()->Syntax,iter()->data()) ;
			}
		}
	else {
		buffer->PutAttrName(attribute,newValue->Syntax,newValue->data()) ;
		}

	return buffer->status() ;
}

/*
** NWDSCCODE DSObject::WriteSingleValue(
**			DSBuffer *	buffer,
**			NWPSTR		attribute,
**			DSValue *	newValue,
**			DSValue *	oldValue)
**
**	Compare the values and delete the old value and/or add the new value.
**	This function is executed when an NDS object is modified. If buffer is
**	null then two DSObjects are being compared, hence return the state of
**	WriteBufferModified.
*/

NWDSCCODE DSObject::WriteSingleValue(
			DSBuffer *	buffer,
			NWPSTR		attribute,
			DSValue *	newValue,
			DSValue *	oldValue)
{
	if (newValue->Syntax == oldValue->Syntax && * newValue == * oldValue)
		return 0 ;

	if (oldValue->Syntax > 0){
		WriteBufferModified = 1 ;
		if (buffer) buffer->DelAttrValue(attribute,oldValue->Syntax,oldValue->data()) ;
		}
	if (newValue->Syntax > 0){
		WriteBufferModified = 1 ;
		if (buffer) buffer->AddAttrValue(attribute,newValue->Syntax,newValue->data()) ;
		}
	return (buffer) ? buffer->status() : WriteBufferModified ;
}

/*
** NWDSCCODE DSObject::WriteMultiValue(
**			DSBuffer *	buffer,
**			NWPSTR		attribute,
**			DSValue *	newValue,
**			DSValue *	oldValue)
**
**	Compare the two multivalue values and delete the old values and/or
**	add the new values. This function is called when an NDS object is
**	modified. If buffer is null then two DSObjects are being compared,
**	hence return the state of WriteBufferModified.
*/

NWDSCCODE DSObject::WriteMultiValue(
			DSBuffer *	buffer,
			NWPSTR		attribute,
			DSValue *	newValue,
			DSValue *	oldValue)
{
DSVMultiValue & newMulti = * (DSVMultiValue *) newValue ;
DSVMultiValue & oldMulti = * (DSVMultiValue *) oldValue ;

	for (DSValueIterator oldIter(oldMulti.List) ; oldIter() ; oldIter.next()){
		DSValue * value = oldIter() ;
		if ((newMulti == * value) == 0){
			WriteBufferModified = 1 ;
			if (buffer) buffer->DelAttrValue(attribute,value->Syntax,value->data()) ;
			}
		}

	for (DSValueIterator newIter(newMulti.List) ; newIter() ; newIter.next()){
		DSValue * value = newIter() ;
		if ((oldMulti == * value) == 0){
			WriteBufferModified = 1 ;
			if (buffer) buffer->AddAttrValue(attribute,value->Syntax,value->data()) ;
			}
		}

	return (buffer) ? buffer->status() : WriteBufferModified ;
}

/*
** NWDSCCODE DSObject::ForAll(DSForAllFunc func, void * data1, void * data2)
**
**	Execute func() on all attribute values passing the two data parameters.
**	If the function returns a non zero value then the iteration is aborted.
*/

NWDSCCODE DSObject::ForAll(DSForAllFunc func, void * data1, void * data2)
{
NWPSTR *	attrNames = AttributeNames() ;
WORD   *	attrCount = AttributeCount() ;
DSForAllData	data ;
NWDSCCODE	ccode = 0 ;

	data.object = this ;
	data.data1  = data1 ;
	data.data2  = data2 ;

	for (data.attrIndex = 0 ; attrNames[data.attrIndex] ; data.attrIndex++){
		for (data.valueIndex = 0 ; data.valueIndex < attrCount[data.attrIndex] ; data.valueIndex++){
			DSValue * value = FindAttribute(data.attrIndex,data.valueIndex) ;
			if (value == 0) break ;
			if ((ccode = func(value,& data)) != 0) return ccode ;
			}
		}
	return 0 ;
}

/*
** NWDSCCODE DSObject::ReleaseAll(DSValue * value, DSForAllData * data)
**
**	Deallocate memory used by the attribute value. This function is a
**	private static member of DSObject. It is executed from a ForAll()
**	iteration started by calling Release().
*/

NWDSCCODE DSObject::ReleaseAll(DSValue * value, DSForAllData *)
{
	return value->release() ;
}

/*
** NWDSCCODE DSObject::WriteAll(DSValue * newValue, DSForAllData * data)
**
**	Determine if the attribute value has been modified and write the
**	value to the DSBuffer if it has. This is a private static member of
**	DSObject. It is executed from a ForAll() iteration started by
**	calling WriteObject().
*/

NWDSCCODE DSObject::WriteAll(DSValue * newValue, DSForAllData * data)
{
DSBuffer * buffer    = (DSBuffer *) data->data1 ;
DSObject * oldObject = (DSObject *) data->data2 ;
NWPSTR	   attribute = data->object->AttributeNames()[data->attrIndex] ;

	if (oldObject == 0)
		return WriteNewValue(buffer,attribute,newValue) ;

	DSValue * oldValue = oldObject->FindAttribute(data->attrIndex,data->valueIndex) ;

	if (oldValue == 0)
		return DSCPP_MEMORY_ERROR ;

	if (oldValue->type() != newValue->type())
		return DSCPP_WRONG_TYPE ;

	if (oldValue->Syntax == 0 && newValue->Syntax == 0)
		return 0 ;

	if (oldValue->Syntax == 0 && data->valueIndex == 0){
		WriteBufferModified = 1 ;
		if (buffer) buffer->PutChange(DS_ADD_ATTRIBUTE,attribute) ;
		}
	if (newValue->type() != DSCPP_MULTI_VALUE)
		return WriteSingleValue(buffer,attribute,newValue,oldValue) ;

	return WriteMultiValue(buffer,attribute,newValue,oldValue) ;
}

/*
** NWDSCCODE DSObject::DisplayAll(DSValue * value, DSForAllData * data)
**
**	Print a text version of each attribute value. This function is a
**	private static member of DSObject. It is executed from a ForAll()
**	iteration started by calling OStream().
*/

#ifdef DSCPP_IOSTREAM
NWDSCCODE DSObject::DisplayAll(DSValue * value, DSForAllData * data)
{
ostream & s = * (ostream *) data->data1 ;

	if (data->valueIndex == 0)
		s << data->object->AttributeNames()[data->attrIndex] << "::\n" ;

	if (value->type() == DSCPP_MULTI_VALUE){
		DSVMultiValue * multi = (DSVMultiValue *) value ;
		for (DSValueIterator iter(multi->List) ; iter() ; iter.next())
			s << * iter() << endl ;
		}
	else {
		if (value->Syntax) s << * value << endl ;
		}

	return 0 ;
}
#endif DSCPP_IOSTREAM

/*
** NWDSCCODE DSReadObject::SetAttrName(NWPSTR attrName, NWSYNTAX_ID)
**
**	Determine the index of the attribute returned from the NWDSRead()
**	operation. Given that this is a new attribute the ValueIndex is set
**	to zero.
*/

NWDSCCODE DSReadObject::SetAttrName(NWPSTR attrName, NWSYNTAX_ID)
{
NWPSTR*	attributes = Object->AttributeNames() ;
	ValueIndex = 0 ;

	for (AttrIndex = 0 ; attributes[AttrIndex] ; AttrIndex++)
		if (stricmp(attributes[AttrIndex],attrName) == 0) break ;

	return 0 ;
}

/*
** NWDSCCODE DSReadObject::SetAttrValue(NWSYNTAX_ID syntaxID, void * data)
**
**	Find the DSValue object that should be used to store the data from
**	a NWDSRead() operation. A pointer to the value object is obtained
**	by calling DSObject::FindAttribute(). If this returns a null pointer
**	then the data is discarded.
*/

NWDSCCODE DSReadObject::SetAttrValue(NWSYNTAX_ID syntaxID, void * data)
{
DSValue * value = Object->FindAttribute(AttrIndex,ValueIndex++) ;

	if (value) value->assign(syntaxID,data) ;

	return 0 ;
}

