/************************************************************************
 *									*
 *  Copyright (c) 1987							*
 *  by CompuServe Incorporated, Columbus, Ohio				*
 *									*
 *  The information in ths software is subject to change without	*
 *  notice and should not be construed as a commitment by CompuServe	*
 *  Incorporated							*
 *									*
 ************************************************************************/

/**
 * File:  BPLUS.C
 *
 * Facility:  B Plus protocol transport layer
 *
 * Abstract:
 *	transport layer parameter negotiation handler
 *
 * Environment:  None specified
 *
 * Author:  John Pampuch, November 5, 1987
 *
 * Modified by:
 *
 * 1	John Pampuch, January 8, 1987
 *	- Original
 *
 * 2	Randy Raynak, February 17, 1989
 *	- Changed the BS (Block Size) field in the '+' packet to be calculated
 *	  from the PDB->Packet_Size which will allow the caller to set up his
 *	  own packet size.
 **/

#include "comdef.h"
#include "bppacket.h"
#include "bplus.h"

#define S_Buffer		pdb->Send_Buffer
#define R_Buffer		pdb->Read_Buffer

#define Plus_Packet_Size	18
#define BitsPerByte		8
#define LowRange		7	/* where quote mask begins for 0-32 */
#define HiRange			11	/* where quote mask begins for 128+ */

/*
 * the following tables select quoting levels for different characters
 * that require quoting.  These tables are used when the host does not
 * support the quoting mask; therefore, the quoting mask must be searched
 * and as close an approximation needs to be substituted.  The first
 * 32 chars that could be quoted are selected from the first table below;
 * the highest value found for any chars quoted selects the quoting
 * level.  Likewise, for the chars in the range from 0x80 through 0xA0
 * the second table is used.
 *
 * The values in the first two tables correspond to the order of the
 * minimum chars quoted.  The third table maps these chars into the 
 * actual quoting levels.  It will likely be possible to remove this
 * code from the library, as eventually all remote software should
 * support the quoting masks.  
 *
 */

static BYTE Quote_Level_Select_Low [] = 
    {
    1, 3, 3, 0, 3, 0, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    0, 0, 3, 0, 3, 0, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3
    };

static BYTE Quote_Level_Select_Hi [] = 
    {
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
    3, 2, 3, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3
    };

static BYTE Quote_Level_Mapping [] = 
    {
    Quote_Not_NULL,
    Quote_Default,
    Quote_Extended,
    Quote_Full
    };

public UWORD BP_Plus_Respond (pdb)
/*
 * Respond to '+' packet, and setup transport parameters as required
 */
    Protocol_Desc_Block *pdb;
    {
    WORD Status;
    WORD temp_window_size;
    WORD temp_method;
    WORD temp_xport;
    WORD temp_size;
    WORD Desired_Window_Size = 0;
    BYTE Estimated_Quote_Level = 0;
    WORD MaskByte;
    WORD Bit;
    WORD i;
    /* B+ options */

    if (pdb->Use_B_Plus)
    then
	Desired_Window_Size = My_Send_Window_Size;

    S_Buffer[0] = '+';
    S_Buffer[1] = (BYTE) Desired_Window_Size;
    S_Buffer[2] = My_Recv_Window_Size;

    S_Buffer[3] = (unsigned char) (pdb->Packet_Size / 128);

    S_Buffer[4] = My_Check_Method;

    S_Buffer[5] = Quote_Default;		    /* no quoting level */
    S_Buffer[6] = (BYTE) pdb->Use_Transport;

    S_Buffer[15] = My_Download_Resume;
    S_Buffer[16] = My_Upload_Resume;
    S_Buffer[17] = My_File_Information;

/*
 * Based on the contents of array pdb->Mask, build the 8 byte table that
 * is sent to the host to describe masking.  Procedure is to
 * walk through each bit in stream sent to host, and set it if necessary.
 */
    for (i = 0; i < 8; i++) S_Buffer [i + LowRange] = 0;

    for (MaskByte = 0; MaskByte < 4; MaskByte++)
	for (Bit = 0; Bit < BitsPerByte; Bit++)
	    {
	    if (pdb->Mask[MaskByte * BitsPerByte + Bit] & MaskLowRange)
	    then
		S_Buffer [MaskByte + LowRange] |= 0x80 >> Bit;
	    
	    if (pdb->Mask[MaskByte * BitsPerByte + Bit] & MaskHiRange)
	    then
		S_Buffer [MaskByte + HiRange] |= 0x80 >> Bit;
	    }

    /*
     * Setup negotiated parameters.  Some of the must be
     * changed after sending the response packet.  Since
     * a packet could arrive before the ACK to the plus
     * packet, some items must be determined before
     * sending the packet, and set after.  An alternate approach
     * would be to record the remote's plus packet.  Before
     * doing anything though, initialize the fields that
     * were not sent by the remote
     */

    for (i = pdb->R_Buffer_Len; i < Plus_Packet_Size; i++)
	R_Buffer [i] = 0;

    /* 
     * Here, add one to size, since the spec'd value corresponds to
     * the data size, and the packet type is not considered
     * part of the data.
     */

#if 0
    if (R_Buffer [3] < My_Buffer_Size)
#endif
    if (R_Buffer [3] < S_Buffer[3])
	temp_size = (R_Buffer[3] * 128) + 1;	/* use largest host value */
    else
	temp_size = (S_Buffer[3] * 128) + 1;	/* use my value */
#if 0
    if (temp_size == 0)
	    temp_size = 512;
#endif
    temp_window_size = min ((WORD) R_Buffer [2], Desired_Window_Size);

    temp_method = min (R_Buffer [4], My_Check_Method);

    temp_xport = min ( (WORD) R_Buffer [6], pdb->Use_Transport);

    pdb->Valid_To_Resume_Download = min (R_Buffer [15], My_Download_Resume);

    pdb->Valid_To_Resume_Upload = min (R_Buffer [16], My_Upload_Resume);

    pdb->Send_File_Information = min (R_Buffer [17], My_File_Information);

/*
 * Build the new mask value based on 'or-ing' the received
 * mask with what we already have
 */
    if (pdb->R_Buffer_Len >= Plus_Packet_Size)
	for (MaskByte = 0; MaskByte < 4; MaskByte++)
	    for (Bit = 0; Bit < BitsPerByte; Bit++)
		{
		if (R_Buffer [LowRange + MaskByte] & (0x80 >> Bit))
		then
		    pdb->Mask[MaskByte * BitsPerByte + Bit] |= MaskLowRange;

		if (R_Buffer [HiRange + MaskByte] & (0x80 >> Bit))
		then
		    pdb->Mask[MaskByte * BitsPerByte + Bit] |= MaskHiRange;
		}
    else
	{
	/*
	 * At this point we have determined that the host does not
	 * know about the quoting mask.  Therefore, we need to
	 * convert our quoting mask into a suitable quoting level.
	 */
	for (i = 0; i < 32 and Estimated_Quote_Level < 3; i++)
	    {
	    if (pdb->Mask [i] & MaskLowRange)
	    then
		Estimated_Quote_Level =
		max (Quote_Level_Select_Low [i], Estimated_Quote_Level);

	    if (pdb->Mask [i] & MaskHiRange)
	    then
		Estimated_Quote_Level =
		max (Quote_Level_Select_Hi [i], Estimated_Quote_Level);
	    }
	}

    pdb->Quoting_Level = Quote_Full;
    S_Buffer [5] = Quote_Level_Mapping [Estimated_Quote_Level];

    pdb->Actual_Plus = TRUE;

    if ((Status = BP_Send_Packet (Plus_Packet_Size, pdb)) == Success)
    then
	if ((Status = BP_Flush_Pending (pdb)) == Success)
	    {
	    pdb->Actual_Check = temp_method;	/* install correct check */
	    pdb->Actual_Xport = temp_xport;
	    pdb->Packet_Size = temp_size;
	    pdb->Send_Seq_Number = pdb->Read_Seq_Number;
	    pdb->Window_Size = temp_window_size;
	    BP_Alloc_Buffers (pdb);
	    }

    pdb->Quoting_Level = Quote_Mask;

    return Status;

    }
