#include 'GETEXIT.CH'
#include 'INKEY.CH'

/********************************************************************\
 *                                                                  *
 *  Written by Lewis Wood              March 10, 1994               *
 *      Desktop Business Solutions                                  *
 *      CIS #:71441,1031                                            *
 *                                                                  *
 *     clipper time /m /n /w /es2                                   *
 *                                                                  *
 *  Time routines.                                                  *
 *                                                                  *
 *    These routines will manipulate time and accept input          *
 *    for a time variable.                                          *
 *                                                                  *
 *   TimeReader( oGet ) - Accepts input for a time get object.      *
 *                        Adds some validation, and ':' acts as     *
 *                        '.' on numbers.  'A' or 'P' will          *
 *                        advance the user to the 6 position where  *
 *                        the letter will be placed.                *
 *                                                                  *
 *   TimeApplyKey( oGet, nKey ) - Applys the key to the time get.   *
 *                                                                  *
 *   TimeValid( oGet, nPos, lForceValid)                            *
 *           Tells if the time entered as of the current position   *
 *           is valid.  Will force the time to be valid if needed   *
 *           This routine is called by TimeApplyKey(..).            *
 *                                                                  *
 *   TimeValidate( oGet) - Validates the time.  Automatically       *
 *                         called if no postblock is defined.       *
 *                                                                  *
 *   TimeMilitary( cStdTime) - Converts standard time 99:99P to     *
 *   TimeMilitary( nTime   )   military time.  Also accepts a       *
 *                             numeric time representation and      *
 *                             returns a char string in military    *
 *                             format.                              *
 *                                                                  *
 *   TimeStandard( cMilitary) - Converts military time to standard  *
 *   TimeStandard( nTime    )   time.  Also accepts a numeric time  *
 *                              representation and returns a char   *
 *                              string of '99:99A' where 'A' could  *
 *                              be 'P' for pm.                      *
 *                                                                  *
 *   EmptyTime( cTime)        - Determines if the time is an empty  *
 *                              time string.                        *
 *                                                                  *
 *   TimeDecimal( cTime)      - Converts time to a numeric          *
 *                              representation.  Base 10.           *
 *                                                                  *
 *   TimeGet( nRow, nCol, bBlock, cVar, cPicture, cColorSpec)       *
 *                            - Returns a get for a time field.     *
 *                              sets the reader and defaults        *
 *                              the picture to '99:99A'             *
 *                                                                  *
 *   TimeDiff( cStart, cEnd)                                        *
 *                           - Returns the difference of the times  *
 *                                                                  *
\********************************************************************/


// Uncomment _TESTING define to test
//
// define _TESTING_

#define K_UNDO          K_CTRL_U

#ifdef _TESTING_

FUNCTION MAIN()
LOCAL ;
   cTime   := SPACE(6)  ,;
   cTime2  := SPACE(6)  ,;
   GetList := {}        ,;
   oTime

   CLS

   oTime := TimeGet( 5, 5, { |x| IF( x == NIL, cTime, ( cTime := x))},  'cTime')
   AADD( GetList, oTime)

   oTime := TimeGet( 6, 5, { |x| IF( x == NIL, cTime2, ( cTime2 := x))},  'cTime2')
   AADD( GetList, oTime)

   READMODAL( GetList)
   ? cTime
   ? cTime2
   ALTD()

RETURN (NIL)
#endif

FUNCTION TimeGet( nRow, nCol, bBlock, cVar, cPicture, cColorSpec)
LOCAL ;
   oTime

   IF cPicture == NIL
      cPicture := 'XX:XX!'
   END IF
   oTime := GETNEW( nRow, nCol, bBlock, cVar, cPicture, cColorSpec)
   oTime:reader := { |o| TimeReader( o)}

RETURN (oTime) // TimeGet( nRow, nCol, bBlock, cVar, cPicture, cColorSpec)

FUNCTION TimeValidate( oGet)
LOCAL ;
   lHasFocus := oGet:hasFocus ,;
   cTime                      ,;
   nHour                      ,;
   nMin                       ,;
   cAmPm                      ,;
   lValid    := .f.           ,;
   lUpdate   := .f.

   IF lHasFocus
      cTime   := oGet:buffer
   ELSE
      cTime   := oGet:VarGet()
   END IF

   nHour := VAL( SUBSTR( cTime, 1, 2))
   nMin  := VAL( SUBSTR( cTime, 4, 2))

   lValid := nHour < 25 .AND. nMin  < 61

   IF lValid
      IF LEN( cTime) > 5
         cTime := TimeStandard( cTime)
      ELSE
         cTime := TimeMilitary( cTime)
      END IF
      IF lHasFocus
         oGet:buffer := cTime
         oGet:assign()
      ELSE
         oGet:VarPut( cTime)
      END IF
   END IF

RETURN (lValid) // TimeValidate( oGet)



/***
*
*  GetReader()
*
*/
PROCEDURE TimeReader( oGet )

   // Read the GET if the WHEN condition is satisfied
   IF ( GetPreValidate( oGet ) )

      // Activate the GET for reading
      oGet:setFocus()

      WHILE ( oGet:exitState == GE_NOEXIT )

         // Check for initial typeout (no editable positions)
         IF ( oGet:typeOut )
            oGet:exitState := GE_ENTER
         ENDIF

         // Apply keystrokes until exit
         WHILE ( oGet:exitState == GE_NOEXIT )
            TimeApplyKey( oGet, inkey( 0 ) )
         ENDDO

         // Disallow exit if the VALID condition is not satisfied
         IF ( ! TimePostValidate( oGet ) )
            oGet:exitState := GE_NOEXIT
         ENDIF
      ENDDO

      // De-activate the GET
      oGet:killFocus()

   ENDIF

RETURN


// Display coordinates for SCOREBOARD
#define SCORE_ROW      0
#define SCORE_COL      60


/***
*
*  ShowScoreboard()
*
*/
STATIC PROCEDURE ShowScoreboard()

   LOCAL nRow
   LOCAL nCol

   IF ( SET( _SET_SCOREBOARD ) )
      nRow := ROW()
      nCol := COL()

      SETPOS( SCORE_ROW, SCORE_COL )
      DISPOUT( IF( SET( _SET_INSERT ), "Ins", "   " ) )
      SETPOS( nRow, nCol )
   ENDIF

RETURN



/***
*
*  GetPostValidate()
*
*  Test exit condition (VALID clause) for a GET
*
*  NOTE: Bad dates are rejected in such a way as to preserve edit buffer
*
*/
STATIC FUNCTION TimePostValidate( oGet )
LOCAL ;
   lValid

   IF oGet:postblock == NIL
      IF ( lValid := TimeValidate( oGet))
         lValid := GetPostValidate( oGet)
      END IF
   ELSE
      lValid := GetPostValidate( oGet)
   END IF

RETURN ( lValid )

FUNCTION TimeValid( oGet, nPos, lForceValid)
LOCAL ;
   cValue := oGet:buffer   ,;
   cHour, nHour            ,;
   cMin, nMin              ,;
   lValid := .t.

   IF nPos < 3

      cHour := SUBSTR( cValue, 1, nPos)

      lValid := ( nHour := VAL( LTRIM( cHour))) <= 24

   ELSEIF nPos > 3 .AND. nPos < 6

      cMin := SUBSTR( cValue, 4, ( nPos - 4) + 1)

      lValid := ( nMin := VAL( LTRIM( cMin))) <= 60


   END IF

   IF lForceValid # NIL .AND. lForceValid

      IF .NOT. lValid

         IF nPos == 2
            cValue := STUFF( cValue, 1, 2, PADL( LEFT( cValue, 1), 2, ' '))
            oGet:buffer := cValue
            lValid := .t.
         ELSEIF nPos == 5
            cValue := STUFF( cValue, 4, 2, PADL( SUBSTR( cValue, 4, 1), 2, '0'))
            oGet:buffer := cValue
            lValid := .t.
         END IF
         oGet:display()

      ELSE
         IF nPos == 2
            cValue := STUFF( cValue, 1, 2, PADL( ALLTRIM( LEFT( cValue, 2)), 2, ' '))
            oGet:buffer := cValue
            lValid := .t.
         ELSEIF nPos == 5
            cValue := STUFF( cValue, 4, 2, PADL( ALLTRIM( SUBSTR( cValue, 4, 2)), 2, '0'))
            oGet:buffer := cValue
            lValid := .t.
         END IF
         oGet:display()

      END IF

   END IF

RETURN (lValid) //TimeValid( oGet, nPos, lForceValid)




/***
*
*  GetApplyKey()
*
*  Apply a single INKEY() keystroke to a time get
*
*/
PROCEDURE TimeApplyKey( oGet, nKey )
LOCAL ;
   cKey       ,;
   bKeyBlock  ,;
   nTemp      ,;
   lValid     ,;
   cTemp

   lValid := TimeValid( oGet, oGet:pos, .f.)

   // Check for SET KEY first
   IF !( ( bKeyBlock := setkey( nKey ) ) == NIL )
      GetDoSetKey( bKeyBlock, oGet )
      RETURN                           // NOTE
   ENDIF

   DO CASE

   CASE ( nKey == K_UP ) .AND. lValid
      oGet:exitState := GE_UP

   CASE ( nKey == K_SH_TAB ) .AND. lValid
      oGet:exitState := GE_UP

   CASE ( nKey == K_DOWN ) .AND. lValid
      oGet:exitState := GE_DOWN

   CASE ( nKey == K_TAB ) .AND. lValid
      oGet:exitState := GE_DOWN

   CASE ( nKey == K_ENTER ) .AND. lValid
      oGet:exitState := GE_ENTER

   CASE ( nKey == K_ESC )
      IF ( SET( _SET_ESCAPE ) )
         
         oGet:undo()
         oGet:exitState := GE_ESCAPE

      ENDIF

   CASE ( nKey == K_PGUP ) .AND. lValid
      oGet:exitState := GE_WRITE

   CASE ( nKey == K_PGDN ) .AND. lValid
      oGet:exitState := GE_WRITE

   CASE ( nKey == K_CTRL_HOME ) .AND. lValid
      oGet:exitState := GE_TOP


#ifdef CTRL_END_SPECIAL

   // Both ^W and ^End go to the last GET
   CASE ( nKey == K_CTRL_END ) .AND. lValid
      oGet:exitState := GE_BOTTOM

#else

   // Both ^W and ^End terminate the READ (the default)
   CASE ( nKey == K_CTRL_W ) .AND. lValid
      oGet:exitState := GE_WRITE

#endif


   CASE ( nKey == K_INS )
      SET( _SET_INSERT, !SET( _SET_INSERT ) )
      ShowScoreboard()

   CASE ( nKey == K_UNDO )
      oGet:undo()

   CASE ( nKey == K_HOME ) .AND. TimeValid( oGet, oGet:pos, .t.)
      oGet:home()

   CASE ( nKey == K_END ) .AND. TimeValid( oGet, oGet:pos, .t.)
      oGet:end()

   CASE ( nKey == K_RIGHT ) .AND. TimeValid( oGet, oGet:pos, .t.)
      oGet:right()

   CASE ( nKey == K_LEFT ) .AND. TimeValid( oGet, oGet:pos, .t.)
      oGet:left()

   CASE ( nKey == K_BS )
      oGet:backSpace()

   CASE ( nKey == K_DEL )
      oGet:delete()

   CASE ( nKey == K_CTRL_Y )
      oGet:delEnd()

   OTHERWISE

      IF ( nKey >= 48 .AND. nKey <= 57 ) .OR. ; // 0 - 9
         nKey == 32                             // Spacebar

         cKey  := CHR( nKey )
         cTemp := oGet:buffer
         nTemp := oGet:pos

         IF ( SET( _SET_INSERT ) )
            oGet:insert( cKey )
         ELSE
            oGet:overstrike( cKey )
         ENDIF

         IF ( oGet:typeOut )
            IF ( SET( _SET_BELL ) )
               ?? CHR(7)
            ENDIF

            IF ( !SET( _SET_CONFIRM ) )
               oGet:exitState := GE_ENTER
            ENDIF
         ENDIF

         IF .NOT. TimeValid( oGet, nTemp, .f.)
            oGet:buffer := cTemp
            oGet:left()
            oGet:display()
         END IF

      ELSEIF nKey == 58     ; //: this positions the cursor properly
             .AND. TimeValid( oGet, oGet:pos, .t.)

         nTemp := oGet:pos
         IF nTemp < 3  // :
            WHILE ++nTemp <= 3
               oGet:right()
            END WHILE
         ELSEIF nTemp > 3
            WHILE --nTemp >= 3
               oGet:left()
            END WHILE
         END IF

      ELSEIF CHR( nKey) $ 'APap' .AND. TimeValid( oGet, oGet:pos, .t.)

         cKey := UPPER( CHR( nKey ))
         nTemp := oGet:pos
         WHILE nTemp <> 6
            oGet:right()
            nTemp++
         END WHILE

         oGet:overstrike( cKey )

         IF ( oGet:typeOut )
            IF ( SET( _SET_BELL ) )
               ?? CHR(7)
            ENDIF

            IF ( !SET( _SET_CONFIRM ) )
               oGet:exitState := GE_ENTER
            ENDIF
         ENDIF

      ENDIF

   ENDCASE

RETURN //TimeApplyKey( oGet, nKey )



FUNCTION TimeMilitary( cStdTime)
LOCAL ;
   cMilitary  ,;
   nHour      ,;
   nMin       ,;
   cAmPm

   IF VALTYPE( cStdTime) == 'N'
       nHour := INT( cStdTime)
       nMin  := ROUND( ( cStdTime - nHour) * 60, 0)
   ELSE
      nHour := VAL( SUBSTR( cStdTime, 1, 2))
      nMin  := VAL( SUBSTR( cStdTime, 4, 2))
   END IF

   IF LEN( cStdTime) > 5

      cAmPm := SUBSTR( cStdTime, 6, 1)

      IF cAmPm $ 'Pp' .AND. nHour < 12
         nHour += 12
      ELSEIF cAmPm $ 'Aa' .AND. nHour == 12
         nHour := 0
      END IF

   END IF

   IF nHour == 0
      IF nMin == 0

         cMilitary := SPACE(5)

      ELSE
         cMilitary := '  :' + PADL( LTRIM( STR( nMin,  2, 0)), 2, '0')
      END IF
   ELSE
      cMilitary := PADL( LTRIM( STR( nHour, 2, 0)), 2, '0') + ':' + ;
                   PADL( LTRIM( STR( nMin,  2, 0)), 2, '0')
   ENDIF

RETURN (cMilitary) //TimeMilitary( cStdTime)


FUNCTION TimeStandard( cMilitary)
LOCAL ;
   cStandard        ,;
   cAmPm     := 'P' ,;
   nHour            ,;
   nMin

   IF VALTYPE( cMilitary) == 'N'
      nHour := INT( cMilitary)
      nMin  := ROUND( ( cMilitary - nHour) * 60, 0)
      nHour := nHour % 24
      IF nHour > 12
         cAmPm := 'P'
         nHour -= 12
      END IF
   ELSE
      nHour := VAL( SUBSTR( cMilitary, 1, 2)) % 24
      nMin  := VAL( SUBSTR( cMilitary, 4, 2)) % 60
      IF LEN( cMilitary) > 5
         cAmPm := UPPER( SUBSTR( cMilitary, 6, 1))
      END IF
   END IF

   IF nHour > 12

      cAmPm := 'P'
      nHour -= 12

   ELSEIF nHour # 12
      cAmPm := 'A'
   END IF

   IF nHour == 0
      IF nMin == 0

         cStandard := '12:00A'

      ELSE
         cStandard := '12:' + PADL( LTRIM( STR( nMin,  2, 0)), 2, '0') + cAmPm
      END IF
   ELSE
      cStandard := PADL( LTRIM( STR( nHour, 2, 0)), 2, '0') + ':' + ;
                   PADL( LTRIM( STR( nMin,  2, 0)), 2, '0') + ;
                   cAmPm
   ENDIF

RETURN (cStandard) //TimeStandard( cMilitary)


FUNCTION EmptyTime( cTime)

   cTime := LEFT( STRTRAN( cTime, ':', ' '), 5)
   IF VAL( RIGHT( cTime, 2)) == 0
      cTime := LEFT( cTime, 2)
      IF VAL( cTime) == 0
         cTime := ''
      END IF
   END IF

RETURN (EMPTY( cTime)) //EmptyTime( cTime)


FUNCTION TimeDecimal( cTime)
LOCAL ;
   nTime

   cTime := TimeMilitary( cTime)

   nTime := VAL( LEFT( cTime, 2)) + VAL( RIGHT( cTime, 2)) / 60

RETURN ( nTime) //TimeDecimal( cTime)



FUNCTION TimeDiff( cStart, cEnd)
LOCAL ;
   nStart, nEnd  ,;
   nDiff

   nStart := TimeDecimal( cStart)
   nEnd   := TimeDecimal( cEnd  )

   IF nStart > nEnd
      nDiff := ( 24 - nStart) + nEnd
   ELSE
      nDiff := nEnd - nStart
   END IF

RETURN (nDiff) //TimeDiff( cStart, cEnd)


