*=============================================================================*
*
*  PRN_LIB:  Printer Control Function Library
*
*  These functions are part of a tutorial on printer control
*  for the Twin Cities dBASE Compiler User Group.
*
*  Placed in the public domain by Craig Yellick, August 4, 1988,
*  with no restictions on their use but no warranties on their
*  performance.  You can incorporate them freely in your
*  applications as long as you take responsibility for everything.
*  I'd appreciate an ego-stroking credit if you lift these functions
*  verbatim and sell products that use them, but who'd know for sure?
*
*  Author   Craig Yellick
*           Yellick Computing
*           509 Maple Square
*           Wayazata, MN 55391-1036
*           (612) 473-1805
*
*=============================================================================*
*
*  Contents (Alphabetically):
*
*  NextLine()    Print 1..N lines down, with left margin
*  PageHead()    Print a page heading if conditions are right
*  PgHd_Init()   Initialize the page header variables
*  PrintOff()    Turn printer off, send exit string if needed
*  PrintOn()     Turn printer on, send init string if needed
*  Prn_Init()    Initialize printer control variables
*  SameLine()    Print on current line, 0..N columns over
*
*  Various printer control functions:
*
*  Norm()     Bold()     Cond()     Large()     Italic()     UndScr()
*  Norm_On()  Bold_On()  Cond_On()  Large_On()  Italic_On()  UndScr_On()
*  Norm_Off() Bold_Off() Cond_Off() Large_Off() Italic_Off() UndScr_Off()
*  PrnEject()
*
*=============================================================================*
*
*  See tutorial text for usage and implementation notes.
*  See associated programs for sample code, README file has details.
*
*  Functions were written with Clipper, S'87 in mind but should work with
*  any dBASE-syntax compiler that supports user-defined functions and
*  at least one-dimensional arrays.  You could probably write around the
*  use of arrays with some macro substitution if arrays bother you.
*
*=============================================================================*

function NextLine         && Print something 1..N lines down, with left margin

parameters how_many, what

***  Clean up the parameters
*
do case
case pcount() = 0
  how_many= 1
  what= []
case pcount() = 1
  what= []
endcase

@ prow() +how_many, 00     say Norm(space(prn_lmar))    && Left margin

@ prow(), pcol() say what     && Whatever you sent to be printed (optional)

prn_line= prn_line +how_many  && Increment the line counter

return prow()

* end func NextLine
*-----------------------------------------------------------------------------*

function PageHead              && Print a page heading if conditions are right

private did_eject
did_eject= .f.    && Flag whether or not we actually eject

***  It's time to eject if current_line +bottom_margin
*    is beyond the page_length.
*
if (prn_line +prn_bmar) > prn_plen
  did_eject= .t.
  prn_page= prn_page +1
  prn_line= 0
  setprc(0,0)  && Force internal row/col counters to zero

  ***  Don't need to eject if it's the first page
  *
  if prn_first
    prn_first= .f.
    PrintOn()
  else
    PrnEject()
  endif

  ***  Move down beyond the top margin
  *
  NextLine(prn_tmar)

  ***  Loop through the HEADER_ array and print each line
  *
  private i
  for i= 1 to len(header_)
    NextLine(1, header_[i])
  next
  
  ***  New current line is the top_margin plus the length of the header.
  *
  prn_line= prn_tmar +i

endif

return did_eject

* end func PageHead
*-----------------------------------------------------------------------------*

function PgHd_Init                     && Initialize the page header variables

***  Make sure every proc hears about this
*
public prn_first, prn_line, prn_page

***  Init PageHead() variables to default values
*
prn_first= .t.    && We starting are on the first page
prn_page=  0      && Page will be incremented by one in PageHead()
prn_line=  99999  && Force initial page header

return []

* end func PgHd_Init
*-----------------------------------------------------------------------------*

function PrintOff              && Turn printer off, send exit string if needed

SameLine(0, prn_exit)
set device to screen

return []

* end func PrintOff
*-----------------------------------------------------------------------------*

function PrintOn                && Turn printer on, send init string if needed

set device to print
SameLine(0, prn_init)

return []

* end func PrintOn
*-----------------------------------------------------------------------------*

function Prn_Init                      && Initialize printer control variables
parameters  model

***  All procs need to know about these
*
public prn_norm1, prn_cond1, prn_bold1, prn_larg1, prn_ital1, prn_und1
public prn_norm0, prn_cond0, prn_bold0, prn_larg0, prn_ital0, prn_und0
public prn_model, prn_init,  prn_exit,  prn_eject
public prn_lmar,  prn_rmar,  prn_xmar,  prn_tmar
public prn_bmar,  prn_hmar,  prn_fmar,  prn_plen, prn_line, prn_page

***  Store null string to all controls
*
store [] to prn_model, prn_init,  prn_exit,  prn_eject
store [] to prn_norm1, prn_cond1, prn_bold1, prn_larg1, prn_ital1, prn_und1
store [] to prn_norm0, prn_cond0, prn_bold0, prn_larg0, prn_ital0, prn_und0

***  Establish reasonable defaults for important variables
*
prn_model= "MONO-FONT ASCII"
*
prn_eject= chr(12) +chr(13)  && ^L FormFeed + ^M Carriage Return (flush buffer)
prn_cond1= chr(15)           && ^O Condensed-On for most printers
prn_cond0= chr(18)           && ^R Condensed-Off
*
prn_lmar= 0
prn_rmar= 80
prn_xmar= 132
prn_tmar= 3
prn_bmar= 5
prn_hmar= 0
prn_fmar= 0
prn_plen= 66
prn_line= 0
prn_page= 0

***  Load custom values if any are found.
*    Set flag depending on successful load.
*
private is_custom
is_custom= .f.
if pcount() > 0
  if file("&MODEL..MEM")              && Yes, you need TWO periods.
    restore from &MODEL. additive     && The first terminates the macro.
    is_custom= .t.
  endif
else
  if file("PRINTER.MEM")
    restore from PRINTER additive
    is_custom= .t.
  endif
endif

***  Return YES if we loaded a custom set
*    or NO if we are using the defaults.
*
return is_custom

* end func Prn_Init
*-----------------------------------------------------------------------------*

function SameLine                  && Print on current line, 0..N columns over

parameters cols_over, what, with_pict

***  Nothing to print, just move the print head
*
if pcount() < 2
  @ prow(), pcol() +cols_over say []
  what= []

else
  ***  Unformatted data, just print it
  *
  if pcount() < 3
    @ prow(), pcol() +cols_over say what

  ***  Formatted data, print using the template
  *
  else
    @ prow(), pcol() +cols_over say what pict "&with_pict."
  endif
endif

return pcol()

* end func SameLine
*-----------------------------------------------------------------------------*
*
*  Various Printer Control Functions
*
*  Norm()     Bold()     Cond()     Large()     Italic()     UndScr()
*  Norm_On()  Bold_On()  Cond_On()  Large_On()  Italic_On()  UndScr_On()
*  Norm_Off() Bold_Off() Cond_Off() Large_Off() Italic_Off() UndScr_Off()
*  PrnEject()

function  Norm
parameter s
return prn_norm1 +s +prn_norm0

function  Bold
parameter s
return prn_bold1 +s +prn_bold0

function  Cond
parameter s
return prn_cond1 +s +prn_cond0

function  Large
parameter s
return prn_larg1 +s +prn_larg0

function  Italic
parameter s
return prn_ital1 +s +prn_ital0

function  UndScr
parameter s
return prn_und1 +s +prn_und0

function Norm_ON
SameLine(0, prn_norm1)
return []

function Norm_OFF
SameLine(0, prn_norm0)
return []

function Bold_ON
SameLine(0, prn_bold1)
return []

function Bold_OFF
SameLine(0, prn_bold0)
return []

function Cond_ON
SameLine(0, prn_cond1)
return []

function Cond_OFF
SameLine(0, prn_cond0)
return []

function Large_ON
SameLine(0, prn_larg1)
return []

function Large_OFF
SameLine(0, prn_larg0)
return []

function Italic_ON
SameLine(0, prn_ital1)
return []

function Italic_OFF
SameLine(0, prn_ital0)
return []

function UndScr_ON
SameLine(0, prn_und1)
return []

function UndScr_OFF
SameLine(0, prn_und0)
return []

function PrnEject
SameLine(0, prn_eject)
return []

* end of various printer control functions
*-----------------------------------------------------------------------------*
* eof Prn_Lib.Prg
