/**
* NETFN.c      Network Management Routines for use with CLIPPER Compiler.
*              Copyright (c) 1986, 1987: Ross Chappell Associates Inc.
*                                        250 Consumers Road, Suite 1109
*                                        Willowdale, Ontario  M2J 4V6
*                                        (416) 495-1100
*
*              All Rights Reserved.
*
* Entered into the public domain July, 1987 by Ross Chappell Associates Inc.
*
* You may copy these routines and include them with programs which you
* develop provided that you do not remove this copyright notice.
*
*-----------------------------------------------------------------------------
*
* --> These routines use certain proprietary Lattice routines. You much have
*     a copy of LC.LIB (release 3.1 or higher) in order to PLINK a Clipper
*     program using these routines.
*
*     You PLINK86 statement should therefore include:
*
*         LIB CLIPPER, LC  + any other libraries you require.
*
*----------------------------------------------------------------------------
*
* The following routines provide additional network support functions for
* IBM PC/Network BIOS Compatible networks and have been used extensively
* within our own client base using 3Com 3+.
*
* Note: RCA has tested and used all these functions and believes they are
*       error free but does not warrant their fitness for any specific
*       purpose and will not accept responsiblity for any damages incurred
*       through their use (or misuse).
*
*       These routines use specific features of NETBIOS in order to locate
*       and/or update certain network control variables. Used incorrectly
*       these routines may cause a workstation to hang or give erroneous
*       network errors. ** USE AT YOUR OWN RISK **
*
* Questions and/or comments should be sent to NA2642.
*
***
* Functions contained in this module are:
* ---------------------------------------
*
* NETUP..........  Determines if network is up and running.
* DRIVESTAT......  Determines status of a specified drive.
* DEVICEN........  Returns the network id (redirection) of a drive or device.
* SETPRN.........  Sets/Resets network printer redirection.
* SETDRV.........  Sets/Resets network drive   redirection.
* SETRETRY.......  Sets new retry parameters for DOS.
**/

#include "extend.h"
#include "dos.h"
#include "string.h"

/***************************/
/* Standard Definitions... */
/***************************/

#define  DEFAULT        0
#define  FALSE          0
#define  TRUE           1
#define  NULLSTRING     ""

/***************************/
/* Additional declares     */
/***************************/

extern short _OSERR;
/**/
/**
* NETUP()      Is network up?
*
* This function returns a logical (True/False) variable which indicates the
* status of network operations. (Specifically, it checks for the presence
* in memory of the operating systems NETBIOS and a complementary Network
* Manager).
*
* Input.......... None.
* Returns........ .T. if network is up & running.
*                 .F. if not.
*
* Sample usage... IF NETUP()
*           or... x = NETUP()
*
*                 where.... x = <expL>
**/

void NETUP()
{
     if (isnet() && isneton()) _retl(TRUE);
     else                      _retl(FALSE);
     return;
}
/**/
/**
* DRIVSTAT()   What is the status of the specified disk drive.
*
* This function determines the status of the disk drive requested and
* returns a numeric value which the calling program may use to determine
* the disk drives current status. (If no parameter is passed, the status
* of the currently selected drive is returned).
*
* Input........... Letter of disk drive user wishes checked, or no parms
*                  for status of currently selected drive.
*
* Returns......... -2 if invalid parms.
*                  -1 if disk drive does not exist.
*                   0 if disk drive exists and is a local drive.
*                   1 if disk drive exists and is a network drive.
*
* Sample Usage.... x = DRIVSTAT("C")  <== Check status of drive C:
*                  x = DRIVSTAT("F:") <== Check status of drive F:
*                  x = DRIVSTAT()     <== Check status of currently
*                                             selected drive.
*
*                  where... x = <expN> and indicates the specific
*                           status of the requested disk drive.
* WARNINGS...
*
*    Using this function to check for a local drive B will return 0 (exists
*    as a local drive) if a local drive A exists and if drive B has not been
*    redirected to the network, therefore, checks for a local drive B are
*    not reliable.
**/

void DRIVSTAT()
{
     char *uval;
     
     if (PCOUNT == 0) {                        /* Return default drives    */
        _retni(isnetdc(0));                    /* status if none specified */
        return;
        }

     if (PCOUNT == 1 && ISCHAR(1))             /* Check for valid parms    */
        uval = _parc(1);                       /* If ok, pull from stack   */
     else {                                    /* otherwise return FALSE   */
        _retni(-2);
        return;
        }

      _retni(isnetdc(toupper(uval[0]) - '@'));    /* Return drive status  */
      return;
}
/**/
/**
* DEVICEN()    Get current device redirection (network name).
*
* This function does a lookup into the networks device redirection table
* and returns the network name of the device specified. If the device
* is not listed (is not connected or LINKED) then a NULLSTRING value is
* returned.
*
* Input........... Disk Drive id or Device Name (eg LPT1) of the device
*                  requested.
*
* Returns......... A character string which contains the network name
*                  (eg. "\\SERVER1\PRINTER" or "\\KATHYS\USERAREA") of the
*                  device specified.
*
*                  A NULL (empty) string if the device/drive requested is
*                  not redirected (linked) or if the program is not running
*                  on a LAN.
*
*                  A NULL (empty) string if invalid/illegal parms are
*                  passed.
*
* Sample Usage.... x = DEVICEN("LPT1")
*                  x = DEVICEN("F:")
*
*                  where... x = <expC> and contains the NETWORK NAME or the
*                           device/drive specified or a NULL string.
**/

void DEVICEN()
{
     char *uval;
     char lname[128], nname[128];
     int  type, parm, i;
     
     if ((isnet() && isneton()) == 0) {        /* See if we're on a        */
        _retc(NULLSTRING);                     /* network. If not, return  */
        return;                                /* a null string.           */
        }

     if (PCOUNT == 1 && ISCHAR(1))             /* Check for valid parms    */
        uval = _parc(1);                       /* If ok, pull from stack   */
     else {                                    /* Otherwise, return        */
        _retc(NULLSTRING);                     /* a null string.           */
        return;
        }

     _OSERR = 0;                               /* Loop through redirection */
     for (i = 0; _OSERR != 18; i++) {          /* table until we find a    */
         type = pcngdr(i,lname,nname,&parm);   /* match to the local name  */
         if (type < 0) {                       /* supplied by the user.    */
            _retc(NULLSTRING);                 /*                          */
            return;                            /* Return NULLSTRING if we  */
            }                                  /* hit an error, network    */
         if (stricmp(uval,lname) == 0) {       /* name if we find it.      */
            _retc(nname);
            return;
            }
         }
     
     _retc(NULLSTRING);                        /* Can't find. Return a     */
     return;                                   /* null string.             */
}
/**/
/**
* SETPRN()     Set NEW Printer Redirection.   
*
* This function sets a NEW Redirection for the specified printer (LPT1, LPT2
* LPT3, PRN, etc. or removes any existing redirection which might exist.
*
* The purpose of this function is to support printer redirection in
* preparation for a subsequent "SET PRINTER TO LPTx" command so that you
* can connect to a desired network printer from within your Clipper program.
*
* Note: ALWAYS CHECK THE RESULT OF THIS COMMAND BEFORE TRYING TO "SET PRINTER"
*       OR YOUR PROGRAM WILL HANG.
*
* A useful by-product of this command is that DOS will flush your printer
* output so that the print server can begin printing (if it can).
*
* Input.....  1).. Local Name (Printer ID) to for which a redirection is to
*                  be created or reset.
*
*             2).. Network Name (eg: \\SERVER1\PRINTER) to which a new
*                  redirection is desired.
*
*             3).. Password (eg: PASSWORD).
*
* Returns......... -100 if network is not up.
*                  -2   if illegal parameters were passed.
*                  -1   if an error exists (something went wrong).
*                   0   if successful.
*
* Sample Usage....
*             1).. RESULT = SETPRN("LPT1","\\SERVER1\PRINTER","WILLIE")
*                  cancels any current redirection for LPT1 and sets a new
*                  redirection to \\SERVER1\PRINTER. The password is WILLIE.
*
*             2).. RESULT = SETPRN("PRN","\\SERVER6\LASER")
*                  cancels any current redirection for PRN, sets a new
*                  redirection to \\SERVER6\LASER, no password is required.
*
*             3).. RESULT = SETPRN("LPT2")
*                  cancels any current redirection for LPT2. No new redirection
*                  is set.
**/

void SETPRN()
{
     char *uval1, *uval2, *uval3;
     char nname[140], *p;
     int  i;

     if (isnet() && isneton()) ;
     else {                                    /* See if we're on a         */
        _retni(-100);                          /* network.                  */
        return;
        }
     
     if (PCOUNT < 1) {                         /* No parms                  */
        _retni(-2);
        return;
        }

/*----------------------------------------------------------------------------
     We'll do this in two stages... stage #1... get _parc(1) and remove any
     current redirection associated with the device. When that's done we'll
     see if user supplied a second parm. If they did, we'll reconnect to
     that device (if we can).
----------------------------------------------------------------------------*/

     if (ISCHAR(1)) uval1 = _parc(1) ;         /* Get users 1st parm. If    */
     else {                                    /* invalid get out....       */
        _retni(-2);
        return;
        }

     for(i = 0; uval1[i] != '\0'; i++) {       /* Drop any ":" in string.   */
        if (uval1[i] == ':') {                 /* and truncate string       */
           uval1[i] = '\0';                    /* to that location.         */
           break;
           }
        }

     i = pcnrdr(uval1);                        /* Cancel old redirection */
       
/*--------------------------------------------------------------------------
   Now we'll check to see if _parc(2) exists. If it does then the user
   is asking us to redirect the printer specified to the network name
   provided. If _parc(3) exists, we'll also set password.
--------------------------------------------------------------------------*/

     if (PCOUNT < 2) {                         /* Nope.. get out of here  */
        _retni(0);
        return;
        }

     uval2 = _parc(2);                         /* Get 2nd parameter       */

     if (strlen(uval2) > 128) {                /* Reject... string is too */
        _retni(-2);
        return;
        }

     p    = stpcpy(nname,uval2);               /* Copy to nname           */
     p[1] = '\0';                              /* Stick in NULL password. */

     if (PCOUNT == 3) {                        /* Did user supply pword?  */
        uval3 = _parc(3);                      /* If so, edit & then copy */
        if (strlen(uval3) > 8) {               /* it to the end of nname. */
           _retni(-2);
           return;
           }
        p++;                                   /* Skip past /0 and copy   */
        p = stpcpy(p,uval3);                   /* password to there       */
        }
 
     _retni(pcnsdr(3,uval1,nname,0));          /* Redirect printer.       */
     return;
}
/**/
/**
* SETDRV()     Set NEW Disk Drive Redirection.   
*
* This function sets a NEW Redirection for the specified disk drive, or
* removes any existing redirection which might exist.
*
* Input.....  1).. Local Name (Drive ID) to for which a redirection is to
*                  be created or reset.
*
*             2).. Network Name (eg: \\JOHNS\PROGRAMS) to which a new
*                  redirection is desired.
*
*             3).. Password (eg: PASSWORD).
*
* Returns......... -100 if network is not up.
*                  -2   if illegal parameters were passed.
*                  -1   if an error exists (something went wrong).
*                   0   if successful.
*
* Sample Usage....
*             1).. RESULT = SETDRV("F:","\\JOHNS\PROGRAMS","WILLIE")
*                  cancels any current redirection for F: and sets a new
*                  redirection to \\JOHNS\PROGRAMS. The password is WILLIE.
*
*             2).. RESULT = SETDRV("F:","\\JOHNS\PROGRAMS")
*                  cancels any current redirection for F:, sets a new
*                  redirection to \\JOHNS\PROGRAMS, no password is required.
*
*             3).. RESULT = SETDRV("F:")
*                  cancels any current redirection for F:. No new redirection
*                  is set.
*
* WARNING...
*
*     You must include SERVERID within your redirection (unlike 3Com's 3F
*     LINK command, which uses 3Com's Name Service to locate this information
*     for you). Therefore, if you would normally link using 3F LINK using:
*
*           3F LINK F: \\WILLIE\SHARENAME
*
*     and \\WILLIE\SHARENAME is located on SERVER1, then you must create this
*     link using:
*
*           RESULT = SETDRV("F:","\\SERVER1\WILLIE\SHARENAME")
*
*     ---> USE THIS FUNCTION WITH CARE!!!!
*     ---> If you do not provide the correct format for this statement you
*          will corrupt the networks redirector table (at the workstation)
*          and will (usually) cause the workstation to hang.
**/

void SETDRV()
{
     char *uval1, *uval2, *uval3;
     char nname[140], *p;
     int  i;

     if (isnet() && isneton()) ;
     else {                                    /* See if we're on a         */
        _retni(-100);                          /* network.                  */
        return;
        }
     
     if (PCOUNT < 1) {                         /* No parms                  */
        _retni(-2);
        return;
        }

/*----------------------------------------------------------------------------
     We'll do this in two stages... stage #1... get _parc(1) and remove any
     current redirection associated with the device. When that's done we'll
     see if user supplied a second parm. If they did, we'll reconnect to
     that device (if we can).
----------------------------------------------------------------------------*/

     if (ISCHAR(1)) uval1 = _parc(1) ;         /* Get users 1st parm. If    */
     else {                                    /* invalid get out....       */
        _retni(-2);
        return;
        }

     if (strlen(uval1) == 2 && isalpha(uval1[0]) && uval1[1] == ':') ;
     else {
        _retni(-2);                            /* User must supply 2 bytes */
        return;                                /* in format d:             */
        }

     i = pcnrdr(uval1);                        /* Cancel old redirection */
       
/*--------------------------------------------------------------------------
   Ok. Now we'll check to see if _parc(2) exists. If it does then the user
   is asking us to redirect the drive specified to the network name provided.
   If _parc(3) exists, we'll also set password.
--------------------------------------------------------------------------*/

     if (PCOUNT < 2) {                         /* Nope.. get out of here  */
        _retni(0);
        return;
        }

     uval2 = _parc(2);                         /* Get 2nd parameter       */

     if (strlen(uval2) > 128) {                /* Reject... string is too */
        _retni(-2);                            /* long.                   */
        return;
        }

     p    = stpcpy(nname,uval2);               /* Copy to nname           */
     p[1] = '\0';                              /* Stick in dummy password */

     if (PCOUNT == 3) {                        /* Did user supply pword?  */
        uval3 = _parc(3);                      /* If so, edit & then copy */
        if (strlen(uval3) > 8) {               /* it to the end of nname. */
           _retni(-2);
           return;
           }
        p++;                                   /* Skip past /0 and copy   */
        p = stpcpy(p,uval3);                   /* password to there       */
        }
 
     _retni(pcnsdr(4,uval1,nname,0));          /* Redirect drive.         */
     return;
}

/**/
/**
* SETRETRY()     Set NEW Network RETRY Parameters.
*
* This function can be used to setup new DOS retry parameters that control
* the retries when file sharing or locking conflicts occur.
*
* When a conflict occurs, DOS normally retries the operation 3 times, with
* a delay of 1 timing unit between each try.
*
* The <tries> parameter indicates how many times the operation will be
* retried when a conflict occurs. The <delay> parameter indicates how much
* time should elapse between tries. According to the Lattice C manual, MSDOS
* kills time by executing a 64K spin look and so, delay actually indicates
* how many times this spin loop should occur.
*
* Input.....  1).. Tries count.
*             2).. Delay count.
*
* Returns......... -100 if network is not up.
*                  -2   if illegal parameters were passed.
*                  -1   if an error exists (something went wrong).
*                   0   if successful.
*
* Sample Usage....
*             1).. RESULT = SETRETRY(10,3)
*                  which setups for 10 retries with a 3-unit delay between
*                  each.
**/

void SETRETRY()
{

     if (isnet() && isneton()) ;
     else {                                    /* See if we're on a         */
        _retni(-100);                          /* network.                  */
        return;
        }
     
     if (PCOUNT != 2) {                        /* Wrong number of parms     */
        _retni(-2);
        return;
        }

     if (ISNUM(1) && ISNUM(2)) {               /* Make sure there numbers   */
        _retni(settry(_parni(1),_parni(2)));   /* & change retry count      */
        return;
        }
     else {
        _retni(-2);                            /* Otherwise, return error   */
        return;
        }
}
