
#include "error.ch"

#include "FileIO.ch"
#include "ftint86.ch"

#xtranslate NtoS( <n> )  => ltrim( str( <n> ) )

#define CRLF CHR(13) + CHR(10)
#define F_BLOCK 2048


/*
*   Routine: filescan(<pattern>[, <attrib>][, <bBlock>])
*          : 
*   Purpose: Scan files that match the specified file pattern(s)
*          : 
* Arguments: <pattern>  one or more patterns separated by semi-colons
*          : [<attrib>] uses same as found in directory() documentation
*          : [<bBlock>] a code-block that contains a function that will
*          : be passed the current filename as parameter one, and the
*          : second parameter will consist of a path plus the current
*          : filename. Your function can return either numeric 1 or 0.
*          : If 1 is returned then the current filesize is included in
*          : the return value of filescan, and if 0 it is not included.
*          : 
*   Returns: a numeric that represents the total size of files processed
*          : on the outcome of your function within the code-block. If
*          : we are working solely with directories and no files then
*          : the return value will be 0, and if pattern is not a character
*          : string then i tell the error-system we have a BooBoo. Seealso
*          : comments within the function for more details.
*          : 
*  Comments: If you specify "D" in attributes the parent (.) and
*          : current (..) directories are not included in the 
*          : array when a directory is below the root. This code
*          : could be tweaked to allow them (you change the code).
*          : 
*          : Filescan does nothing if you do not send it a it a valid
*          : character string pattern.
*/
function filescan(pattern, attrib, bBlock)
    local oError
    local aFiles_
    local aArray_ := {}
    local aHouse_ := {}
    local nSize   := 0
    local nLen
    local i

    /*
    * KSG
    * if the first parameter is not a character string we
    * must get out of here since the function will fail
    * if say you passed a numeric and the numeric is used
    * in a function that expects a character type. Since
    * we all have different ideas of how to recover i did
    * two versions of recovery. The first is to return
    * to the caller with a numeric -1 which allows your
    * program to continue, but could also cause a failure
    * afterwards. The second method will produce a runtime
    * error which if utilizing the standard error system
    * will drop your operator directly to DOS. When using
    * errornew() i could have used GenCode and done the
    * eval, but wanted to give more info to the user.
    * oError:cargo could be used for writing info to a
    * disk error-logfile.
    */
    if !(valtype(pattern) == "C")
        // less desirable
        // return -1
        // Best method of recovery
        oError:=errornew()
        // Would have been the lazy way to do things.
        // oError:gencode:= EG_ARG
        oError:description:= "Missing pattern"
        oError:severity:= ES_ERROR
        oError:subsystem:="filescan"
        eval(errorblock(),oError)
    endif

    bBlock := if( valtype(bBlock) == "B", bBlock, {|| 1 } )

    /*
    * if there is a valid pattern, then process it
    *       a. convert pattern into an array
    *       b. cycle through each (pattern) element made by StoA()
    *       c. during processing treat directories differently
    * Once processing is completed send the array aFiles_ to
    * aeval() where it passes control to eval() which uses your
    * code-block to allow processing of each entry in the array.
    */
    if !empty(pattern)
        nLen := len(aArray_ := StoA(pattern))
        for i := 1 to nLen
            aFiles_ := directory(aArray_[i],attrib)
            if !empty(attrib) .and. !empty(aFiles_)
                if "D" $ attrib
                    aeval(aFiles_,                                      ;
                        {|a|                                            ;
                            if("D" $ a[5] .and. !(left(a[1],1) == ".") ,;
                               aadd(aHouse_,{a[1],a[2],a[3],a[4],a[5]}),;
                               nil                                      ;
                            );
                        };
                    )
                    aFiles_ := aclone(aHouse_)
                    aHouse_ := {}
                endif
            endif

            /*
            * If the function within the code-block returns
            *   1   then add the filesize to the local nSize
            *   0   then do not add the filesize to the local nSize
            */
            aeval(aFiles_,                                              ;
                {|a|                                                    ;
                    if( EVAL(bBlock,a[1],Pather(a[1],aArray_[i]) ) == 1,;
                        nSize += a[2],;
                        NIL;
                    );
                };
            )

        next
    endif
return nSize

/*
*   Routine: Pather(<cString>,<pattern>) --> C
*          : 
*   Purpose: For adding a path to <cString> if not already included
*          : 
* Arguments: <cString> Filename
*          : <pattern> Possible path to add onto FileName
*          : 
*  Comments: Called from FileScan() above.
*          : 
*/
static function Pather(cString,pattern)
    local cTmp   := ""
    local cPath  := ""
    local cDrive := ft_default()
    local cDir   := curdir()

    if empty(cTmp := substr(pattern,1,rat("\",pattern)))
        if empty(cDir)
            cPath := cDrive + ":\" + cString
        else
            cPath := cDrive + ":\" + cDir + "\" + cString
        endif
    else
        cPath := cTmp + cString
    endif

return upper(cPath)

/*
*   Routine: StoA(<cString>[,<cDelim>]) --> A
*          : 
*   Purpose: converts a delimited string into an array
*          : 
* Arguments: <cString>   possibly a delimited character string
*          : [<cDelim>]  optional delimitor which defaults to ";"
*          : 
*  Comments: if no delimitors then a one element array is returned
*          : If <cString> is not a character string then ByeBye!
*/
function StoA(cString, cDelim)
    local nPtr
    local aArr_ := {}
    
    cDelim  := if(cDelim == NIL, ";", cDelim)
    cString := alltrim(cString)

    while (nPtr := at(cDelim, cString)) != 0
        aadd(aArr_,if(nPtr > 1, rtrim(left(cString, nPtr-1)), ""))
        cString := if(len(cString) > nPtr,ltrim(substr(cString,nPtr +1)),"")
    enddo
    aadd(aArr_, cString)
return (aArr_)

*** <<<EOF>>> ***
