#include "gfm.ch"
#include "getexit.ch"

////////////////////
////////////////////
//
//
// Purpose:
//       (Re)Display gets on a get screen
//
// Syntax:
//       ReadDisplay( [ <aGetList> ], <bTest> )
//
// Formal Arguments: (2)
//       Name        Description
//        
//       aGetList    List of get objects to display. (Defaults to ReadGetList())
//       bTest       Optional test block to evaluate on a get object before
//                   displaying it.
//
// Returns:
//       .T.
//
// Section:
// Source Code
//
// Category:
//       Get System
//
// Examples:
//       ReadDisplay( , {|g| g:type == 'N' } )  // Redisplays all numeric gets
//
// Description:
//       Traverses the specified get list, displaying all gets in the list, or
//       only those that match the specified condition.
//
// Notes:
//       Supplement to GetSysPP.PRG
//
// Include files:
//       gfm.ch
//
// Revisions:
//       01/11/92 16:21:33 1.0   Original version
//       02/10/92 10:40:30 1.1   Updated to accept a conditional code block
//                               argument.
//
// Author:
// John F. Kaster
//
////////////////////
////////////////////
FUNCTION ReadDisplay( aGetList, bTest )
	DEFAULT aGetList TO ReadGetList()
	IF ValType( bTest ) == 'B'
		aEval( aGetList, { |g| IF( Eval( bTest, g ), g:display, NIL ) } )
	ELSE
		aEval( aGetList, { |g| g:display() } )
	ENDIF
RETURN .T.

////////////////////
////////////////////
//
//
// Purpose:
//       Retrieve a get from the get list by name or order
//
// Syntax:
//       GetGet( <cnGetVar> ) -> Get
//
// Formal Arguments: (1)
//       Name        Description
//        
//       cnGetVar    Name or ordinal position of get object to retrieve
//
// Returns:
//       Get object if found, NIL otherwise
//
// Section:
// Source Code
//
// Category:
//       Get System
//
// Examples:
//       GetGet( "FIRST" ):varPut( "JOHN" )
//
// Description:
//       Retrieve a get from the active get list (read layer) either by its
//       name or ordinal position in the get list.
//
// Notes:
//       Supplement to GetSysPP.PRG
//
//       This is an ideal function for stutterers.
//
// Include files:
//       gfm.ch
//
// Revisions:
//       01/11/92 16:24:53 1.0   Original version
//
// Calls:
// GetOrdinal()                  Ordinal position in Get List from a Get's name
//
// Author:
// John F. Kaster
//
// See Also:
//       GetOrdinal()
//
////////////////////
////////////////////
FUNCTION GetGet( cnGetVar )
  LOCAL nGet := GetOrdinal( cnGetVar )
  LOCAL g
  IF ! Empty( nGet )
    g:= ReadGetList()[ nGet ]
  ENDIF
RETURN g

////////////////////
////////////////////
//
//
// Purpose:
//       Override the default movement for the next Get in read
//
// Syntax:
//       GetGo( <cnGetVar> ) -> TRUE
//
// Formal Arguments: (1)
//       Name        Description
//        
//       cnGetVar    Name or ordinal position of get to move to
//
// Returns:
//       TRUE
//
// Section:
// Source Code
//
// Category:
//       Get System
//
// Examples:
//       GetGo( "LAST" )   // Moves to get named "Last" after exiting current
//
// Description:
//       Overrides the default movement method provided in Settle() by posting
//       a value to ReadPos()
//
// Notes:
//       Supplement to GetSysPP.PRG
//
// Include files:
//       gfm.ch
//
// Revisions:
//       01/11/92 16:28:06 1.0   Original version
//
// Calls:
// GetOrdinal()                  Ordinal position in Get List from a Get's name
//
// Author:
// John F. Kaster
//
// See Also:
//       ReadPos()
//
////////////////////
////////////////////
FUNCTION GetGo( cnGetVar )
	ReadPos( GetOrdinal( cnGetVar ) )
RETURN .T.

////////////////////
////////////////////
//
//
// Purpose:
//       Ordinal position in Get List from a Get's name
//
// Syntax:
//       GetOrdinal( <cnGetVar>, [ <nStart> ], [ <nCount> ] ) -> nPosition
//
// Formal Arguments: (3)
//       Name        Description
//        
//       cnGetVar    Name or ordinal position of Get
//       nStart      Ordinal value to start looking for cnGetVar at.  If not
//                   passed, one (1) is the start.
//       nCount      Number of get objects in get array to look for cnGetVar
//                   in.  If not passed, all are looked at.
//
// Returns:
//       Ordinal position of the get if found, 0 otherwise
//
// Section:
// Source Code
//
// Category:
//       Get System
//
// Examples:
//       IF GetOrdinal( "First" ) == 0
//          @ MaxRow(), 0 SAY "'First' is not editable right now."
//       ENDIF
//
// Description:
//       Scans the current read layer's get list in an attempt to locate the
//       current get.
//
// Notes:
//       Supplement to GetSysPP.PRG
//
// Include files:
//       gfm.ch
//
// Revisions:
//       01/11/92 16:31:07 1.0   Original version
//
// Called By:
// GetGet()                   Retrieve a get from the get list by name or order
// GetGo()               Override the default movement for the next Get in read
//
// Author:
// John F. Kaster
//
// See Also:
//       GetGet() GetSub() GetReadVar()
//
////////////////////
////////////////////
FUNCTION GetOrdinal( cnGetVar, nStart, nCount )
	LOCAL nGetPos := 0
	LOCAL aGetList := ReadGetList()
	IF ValType( cnGetVar ) == 'C'
		cnGetVar := Upper( cnGetVar )
		nGetPos := aScan( aGetList, ;
							{ | g | GetReadVar( g ) == cnGetVar }, nStart, nCount )
	ELSE
		nGetPos := IF( cnGetVar<0 .OR. cnGetVar>Len( aGetList ), 0, cnGetVar )
	ENDIF
RETURN nGetPos

#include "setcurs.ch"
////////////////////
////////////////////
//
//
// Purpose:
//       Evaluate and display all gets that have when clauses
//
// Syntax:
//       WhenDisplay( <lReturn> ) -> lReturn
//
// Formal Arguments: (1)
//       Name        Description
//        
//       lReturn     Logical value to return from this function
//
// Returns:
//       The value lReturn contains
//
// Section:
//       Source Code
//
// Category:
//       Get System
//
// Examples:
//       WhenDisplay( InTable() )
//
// Description:
//       Runs through the current Read's GetList, and evaluates all the
//       when clauses for that GetList to refresh the values displayed by
//       the gets.  This routine should ONLY be used for calculated fields.
//       After the value for that get has been changed (supposedly by
//       evaluating the code block), the value is redisplayed.
//
// Notes:
//
//
// Include files:
//       gfm.ch
//
// Revisions:
//       08/07/92 12:31:55 1.0   Original version
//
// Author:
// John F. Kaster
//
////////////////////
////////////////////
FUNCTION WhenDisplay( lReturn )
	LOCAL aGets := ReadGetList()
	LOCAL nGet, nGets := Len( aGets )
	LOCAL nCursor := SetCursor( SC_NONE )
	LOCAL lFocused, oGet, g
	FOR nGet := 1 TO nGets
		IF ValType( ( g := aGets[ nGet ] ):preBlock ) == 'B'
      oGet := GetActive( g )
      IF lFocused := g:hasFocus()
        g:killFocus()
      ENDIF
			Eval( g:preBlock, g )
      IF lFocused
        g:setFocus()
      ELSE
			  g:display()
      ENDIF
      GetActive( oGet )
		ENDIF
	NEXT nGet
	SetCursor( nCursor )
RETURN lReturn

////////////////////
////////////////////
//
//
// Purpose:
// 	Locate a get based on row and column position and give it input focus
//
// Syntax:
// 	GetMoveTo( <nRow>, <nCol>, [<oGet>], [<aGets>] ) --> nFound
//
// Arguments:
//
//     nRow            Row of get to find
//     nCol            Column of get to find
//     oGet            Current get (defaults to GetActive())
//     aGets           Get list (defaults to ReadGetList())
//
// Returns:
// 	Position of found get, or 0
//
// Section:
// 	Source Code
//
// Category:
// 	Get System
//
// Examples:
// 	See GetSysPP.PRG
//
// Description:
// 	Traverses the get list to locate a get based on row and column position.
//		If the row and column position fall within the editing region for a get,
//		the ordinal position of that get in the get list is returned.  The active
//		get is checked first, to optimize movements within the current get.
//
// Notes:
//		If the row and column fall within an editable region for the get,
//		GetPoint() will set the get:pos value to that columnar offset
//
// Include Files:
//
//    gfm.ch
//    getexit.ch
//    setcurs.ch
//
// Calls:
// GetPoint()                                                 GFMGSUPP.PRG   2x
//
// Author:
// John F. Kaster
//
// See Also: GetPoint() GFMGSUPP.PRG
//
////////////////////
////////////////////
FUNCTION GetMoveTo( nRow, nCol, oGet, aGets )
	LOCAL nFound	:= 0
	LOCAL oBox
	LOCAL nGets
	LOCAL nGet

	DEFAULT oGet TO GetActive()
	DEFAULT aGets TO ReadGetList()

	IF oGet # NIL .AND. GetPoint( nRow, nCol, oGet )	// Clicked active get?
		nFound := -1												// Turn off found
	ENDIF

	IF nFound == 0

		nGets := Len( aGets )

		FOR nGet := 1 TO nGets
			IF GetPoint( nRow, nCol, aGets[ nGet ] )
				nFound := nGet
				EXIT
			ENDIF
		NEXT nGet

	ENDIF

	IF nFound == 0 .AND. ( oBox := GFMBox():stackTop() ) # NIL

		 // Check active box for a click on the close dialog
		 IF oBox:lCanClose .AND. oBox:nTop == nRow .AND. oBox:nLeft == nCol

			oGet:exitState := GE_ESCAPE	// Force exit of get

		 ENDIF

	ENDIF

RETURN nFound

////////////////////
////////////////////
//
//
// Purpose:
//   Determine number of columns for a get
//
// Syntax:
// GetCols( <oGet> ) --> nColumns
//
// Arguments:
//
//     oGet		Get object to examine
//
// Returns:
// 	The number of display columns for the get object
//
// Section:
// 	Source Code
//
// Category:
// 	Get System
//
// Examples:
// 	See GetPoint()
//
// Description:
// 	Uses various criteria for determining the number of display columns used
//		by a get object
//
// Notes:
//
//
// Include Files:
//
//    gfm.ch
//    getexit.ch
//    setcurs.ch
//
// Called By:
// GetPoint()                                                 GFMGSUPP.PRG   1x
//
// Author:
// John F. Kaster
//
// See Also: GetPoint() GFMGSUPP.PRG
//
////////////////////
////////////////////
FUNCTION GetCols( oGet )
	LOCAL cType
	LOCAL xVal
	LOCAL nScroll
	LOCAL nCols
	LOCAL cPict

	DEFAULT oGet TO GetActive()
	xVal	:= oGet:varGet()
	cType	:= oGet:type

	DO CASE
	CASE cType == "M"
		nCols := oGet:nRight - oGet:row + 1

	CASE ! Empty( cPict := oGet:picture )			// Picture clause on get
		nScroll := At( "@S", Upper( cPict ) )
		IF nScroll > 0										// Scrolling get
			nCols := Val( SubStr( cPict, nScroll + 2 ) )
		ELSE
			nCols := Len( Trans( xVal, oGet:picture ) )
		ENDIF

	CASE cType == "C"
		nCols := Len( xVal )

	CASE cType == "N"
		nCols := Len( Str( xVal ) )

	CASE cType == "D"
		nCols := Len( Trans( xVal, "@D" ) )

	CASE cType == "L"
		nCols := 1

	ENDCASE

RETURN nCols

////////////////////
////////////////////
//
//
// Purpose:
// 	Determine number of rows for a get object
//
// Syntax:
// 	GetRows( <oGet> ) --> nRows
//
// Arguments:
//
//     oGet	Get object to examine
//
// Returns:
// 	Number of rows occupied by get object
//
// Section:
// 	Source Code
//
// Category:
// 	Get System
//
// Examples:
// 	See GetPoint()
//
// Description:
//		Returns the number of rows needed for displaying the selected Get object.
//		Supports the extension provided by GetMemo by checking the Get:type for
//		"M"emo before returning get:nBottom instead of 1
//
// Notes:
//
//
// Include Files:
//
//    gfm.ch
//    getexit.ch
//    setcurs.ch
//
// Called By:
// GetPoint()                                                 GFMGSUPP.PRG   1x
//
// Author:
// John F. Kaster
//
// See Also: GetPoint() GFMGSUPP.PRG
//
////////////////////
////////////////////
FUNCTION GetRows( oGet )
	LOCAL nRows := 1

	DEFAULT oGet TO GetActive()
	IF oGet:type == "M"
		nRows := oGet:nBottom - oGet:row + 1
	ENDIF

RETURN nRows

////////////////////
////////////////////
//
//
// Purpose:
// 	Determines if a row,col point is inside a get
//
// Syntax:
// 	GetPoint( <nRow>, <nCol>, <oGet> ) --> lInside
//
// Arguments:
//
//     nRow			Row to find
//     nCol			Column to find
//     oGet			Get object
//
// Returns:
// 	.T. if point is inside get object, .F. otherwise
//
// Section:
// 	Source Code
//
// Category:
// 	Get System
//
// Examples:
// 	See GetMoveTo()
//
// Description:
// 	Checks a get object to determine if the passed row and column position
//		fall inside the display area for the get object.  If the point is inside
//		the get, it returns .T.
//
// Notes:
//		If the point is in active get, the get's cursor position is set to the
//		columnar offset of the requested point.  ReadPos() is used to set the
//		ordinal position of the get as the next get to edit in the get	system if
//		the point is not within the active get.  Special support is also provided
//		for GetMemo objects.
//
// Include Files:
//
//    gfm.ch
//    getexit.ch
//    setcurs.ch
//
// Calls:
// GetCols()                                                  GFMGSUPP.PRG   1x
// GetRows()                                                  GFMGSUPP.PRG   1x
//
// Called By:
// GetMoveTo()                                                GFMGSUPP.PRG   2x
//
// Author:
// John F. Kaster
//
// See Also: GetCols() GFMGSUPP.PRG GetRows() GFMGSUPP.PRG GetMoveTo()
//           GFMGSUPP.PRG
//
////////////////////
////////////////////
FUNCTION GetPoint( nRow, nCol, oGet )
	LOCAL lInside
	LOCAL nRight
	LOCAL nBottom

	DEFAULT oGet TO GetActive()
	nRight	:= oGet:col + GetCols( oGet ) - 1
	nBottom	:= oGet:row + GetRows( oGet ) - 1

	IF lInside := InRect( nRow, nCol, oGet:row, oGet:col, nBottom, nRight )

		nCol := nCol - oGet:col + 1
		IF oGet:type == "M"							// Must be using GetMemo object
			oGet:lastRow := nRow - oGet:row + 1
			oGet:lastCol := nCol
		ENDIF
		IF oGet:hasFocus								// Active get?
			oGet:pos		:= nCol						// Move cursor position
			oGet:display()
		ELSE
			ReadGetPos( nCol )						// Set next get's cursor position
		ENDIF

	ENDIF

RETURN lInside
