/*
*  GetsysPP.prg - Modified Standard Clipper 5.2x GET/READ subsystem
*
*  Portions Copyright (c) 1990 - 1994 Computer Associates.  All rights reserved.
*
*  NOTE: compile with /m/n/w
*
	FILE: GETSYSPP.PRG (Get System Plus Plus )

	VER:  5.2c-2
	DATE: 12-16-93 01:27am
	BY:   John Kaster

	Added mouse support using Grumpfish's mouse header file for the definition
	of K_LEFT_DOWN and K_LEFT_DBLCLICK.  Minor modifications were made to
	GetApplyKey(), and ReadGetPos() was added to specify the default cursor
	position in the buffer when a get is activated.  Additional functions
	(such as GetMoveTo()) are in the source code file GFMGSUPP.PRG.

	To enable mouse support, just make sure the manifest constant MOUSE is
	defined when compiling this source file.

	*** END OF LOG FOR VER 5.2c-2

	VER:  5.2c-1
	DATE: 9-9-93
	BY:   John Kaster

	Modified ReadModal() to be compatible with the change to Clipper 5.2c's
	ReadModal() in GetSys.PRG.  The change involved adding the default
	starting position to ReadModal() as its second parameter.

	Added COMMON.CH to the header files, so code could use the DEFAULT ... TO
	command.

	*** END OF LOG FOR VER 5.2c-1

	VER:  5.01-8
	DATE: 3-5-93
	BY:   John Kaster

	Added support for post-validation of all fields via the function
	ReadValid().  Also added the static function FindBadValid()

	Added an additional parameter to ReadModal() to override the default get
	position to start editing.

	*** END OF LOG FOR VER 5.01-8

	VER:  5.01-7
	DATE: 10-2-1991
	BY:   John Kaster

	Fully documented ALL routines in Get System, and made code conform more to
	Nantucket's proposed guidelines.  Removed all persnickety line-by-line
	version change notes and recorded them in revision history in the comment
	header.

	Allowed ReadFormat() to accept NIL as the value to set FORMAT to.

	*** END OF LOG FOR VER 5.01-7

	VER:  5.01-6
	DATE: 10-2-1991
	BY:   John Kaster

	If GetReader() was called directly w/o calling ReadModal() first, the
	static variable KillRead was NIL rather than a logical.  KillRead now
	is initialized to .F.

	Added functions ReadProcName() and ReadProcLine() to provide visibility
	to the routine ReadModal() was called from.  These values are
	contained in aReadInfo[ RI_READNAME ] and aReadInfo[ RI_READLINE ].

	Added GSV_UPDATED to the read stack and removed all references to
	saveUpdated so ReadUpdated() would function correctly.

	Modified ReadBreak() to return the value of Updated (which is returned
	by ReadModal()).

	'Tabbed' the source.  Set your tab size to 3 (in SPE, "tabs 4") to view
	the source in its (close to) original form.  Or set it to whatever you
	like - that's the beauty of tabs.

	*** END OF LOG FOR VER 5.01-6


	VER:  5.01-5
	DATE: 8-1-1991
	BY:   Rob Hannah

	I forgot to update the status of ReadVar() in ReadPos().

	*** END OF LOG FOR VER 5.01-5


	VER:  5.01-4
	DATE: 7-22-1991
	BY:   Rob Hannah

	Further enhancements made to ReadPos() to allow for a cleaner interface
	to routines that actually changed the active get and later wanted to do
	some activity on the new get thru GetActive().

	A new function - ReadBreak() - was added for those (hopefully rare) times
	when you need to BREAK out of a read and want to clean up.  The cleanup
	code at the end of ReadModal() was broken out to do this.  Be sure to call
	ReadBreak() whenever you BREAK.  Otherwise, you could end up with some
	strange stuff.  It isn't necessary to call before the BREAK - nor is it
	needed if you exit the read in any other manner.  Also, if you call
	ReadBreak() do not CLEAR GETS.

	The LOCAL savedGetSysVars (in ReadModal()) was moved into
	aReadInfo[ RI_STATEINFO ] to facilitate cleanup by ReadBreak().

	*** END OF LOG FOR VER 5.01-4


	VER:  5.01-3
	DATE: 7-15-1991
	BY:   Mike Schinkel

	Modifications were made to incorporate ideas from Craig Ogg's
	GETSYS02.PRG from the Spring 1991 Clipper Developer's Conference in Palm
	Spring's, CA.

	This version includes the following additional functions:

		ReadKill()     -  Allow get/set access to the "kill" status.
		ReadUpdated()  -  Allow get/set access to the "updated" status.
		ReadFormat()   -  Allow get/set access to the "format".

	*** END OF LOG FOR VER 5.01-3


	VER:  5.01-2
	DATE: 7-05-1991
	BY:   Rob Hannah

	Modifications were made to ReadGetList() and ReadPos() to work when no
	reads are active.  ReadGetList() will return an empty array and ReadPos()
	returns 0 (and ignores any passed parameters).

	*** END OF LOG FOR VER 5.01-2


	VER:  5.01-1
	DATE: 5-16-1991
	BY:   Mike Schinkel

	MODIFICATION LOG

	This file contains modifications to Nantucket's GETSYS.PRG for the
	purposes of allowing extensions w/o the need to completely rewrite it.
	It provides the following additional functions:


			ReadGetList() - Returns a reference to the current getlist.

			ReadPos()     - Allows get/set access of the current position
								 in the current getlist.


	These two functions make it possible to implement go-to-get routines
	in a VALID or WHEN block, or in a SET KEY procedure.

	In addition, these modifications allow three optional codeblocks to be
	passed to ReadModal() after GetList, in the order listed below. Note
	that each block will be passed the current get object as its only
	parameter when this Get System evaluates it.

			bInKey   - A codeblock that GetReader() can eval() instead of
						  calling InKey(0).  This allows you to process your
						  own keystrokes, or mouse clicks, or whatever.

			bPreGet  - A codeblock that will be evaluated immediately prior to
						  GetReader() is called.  This allows things like
						  get-specific help messages to be displayed.

			bPostGet - A codeblock that will be evaluated immediately after
						  GetReader() is called. This allows things like
						  get-specific help messages to be cleared.  It also
						  allows you to validate all gets upon a users attempt
						  to complete the read.

	The purpose behind these modifcations is to encourage a defacto-standard
	for extensions to Clipper 5.0's Get System, especially among the third
	party.  Thus, I hereby donate these modifications into the PUBLIC
	DOMAIN.  I request, however, that this header remain intact.  In
	addition, if any other modifications are made, I request that they be
	backward-compatible and that they be documented and version-stamped
	with a modification log listed below this one.

	Final note.  Only the ReadModal() and GetReader() functions were
	modified for ver 5.01-1; no other functions were affected.  Also, all
	additions and/or modifications were marked with comments in the following
	form:


			// Begin ver 5.01-1 modification
			*** A modification
			// End ver 5.01-1 modification

			// Begin ver 5.01-1 addition
			*** An addition
			// End ver 5.01-1 addition

	*** END OF LOG FOR VER 5.01-1
*/

#define MOUSE
#ifdef MOUSE
#include "mouse.ch"			// For mouse driver
#endif

#include "Inkey.ch"
#include "Getexit.ch"
#include "Common.ch"			// For DEFAULT ... TO command

#define K_UNDO          	K_CTRL_U


// state variables for active READ
static Format              // Set by __SetFormat() if passed a block
static Updated := .f.      // Retrieved by Updated()
static KillRead := .f.     // Set to TRUE by __KillRead()

static BumpTop             // Internal
static BumpBot             // Internal
static LastExit            // Internal
static LastPos             // Internal
static ActiveGet           // Set/Retrieved by GetActive()
static ReadProcName        // Internal
static ReadProcLine        // Internal


// format of array used to preserve state variables
#define GSV_KILLREAD			1
#define GSV_BUMPTOP			2
#define GSV_BUMPBOT			3
#define GSV_LASTEXIT			4
#define GSV_LASTPOS			5
#define GSV_ACTIVEGET		6
#define GSV_READVAR 			7
#define GSV_READPROCNAME	8
#define GSV_READPROCLINE	9
#define GSV_UPDATED			10

#define GSV_COUNT				10


STATIC aReadInfo:= {}
// format of stack of ReadInfo
#define RI_AGETLIST			1
#define RI_NPOS				2
#define RI_STATEINFO			3
#define RI_READNAME			4
#define RI_READLINE			5

/////////////////////
/////////////////////
//
//	Purpose:
//		Name of procedure for the current read layer
//
//	Syntax:
//		ReadProcName() -> cProcedure
//
//	No arguments specified
//
//	Returns:
//		Name of the procedure for the current read layer
//
//	Examples:
//		? "This READ layer was called from", ReadProcName()
//		?? " on line", ReadProcLine()
//
//	Files:
//
//
//	Description:
//		Provides visibility to the name of the procedure that called
//		ReadModal() for the current read layer.
//
//	Notes:
//		Handy for generic help systems modeled after the Summer '87 help system
//		which was based on the procedure name, procedure line and variable
//		being edited.
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadProcLine()
//
//	Revisions:
//		01/10/92	20:29:54	1.0	John Kaster: Original version
//
/////////////////////
/////////////////////
FUNCTION ReadProcName
	RETURN If( Len( aReadInfo ) == 0, "", ATail( aReadInfo )[ RI_READNAME ] )

/////////////////////
/////////////////////
//
//	Purpose:
//		Line of procedure for the current read layer
//
//	Syntax:
//		ReadProcLine() -> nLine
//
//	No arguments specified
//
//	Returns:
//		Line of the procedure for the current read layer
//
//	Examples:
//		? "This READ layer was called from", ReadProcName()
//		?? " on line", ReadProcLine()
//
//	Files:
//
//
//	Description:
//		Provides visibility to the line of the procedure that called
//		ReadModal() for the current read layer.
//
//	Notes:
//		Handy for generic help systems modeled after the Summer '87 help system
//		which was based on the procedure name, procedure line and variable
//		being edited.
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadProcName()
//
//	Revisions:
//		01/10/92	20:29:54	1.0	John Kaster: Original version
//
/////////////////////
/////////////////////
FUNCTION ReadProcLine
	RETURN If( Len( aReadInfo ) == 0, 0, ATail( aReadInfo )[ RI_READLINE ] )


////////////////////
////////////////////
//
//	Purpose:
//		Return the active GetList
//
//	Syntax:
//		ReadGetList() -> GetList
//
//	No arguments specified
//
//	Returns:
//		The GetList for the current Read layer
//
//	Examples:
//		PROCEDURE ReadDisplay( aGetList )
//		  DEFAULT aGetList TO ReadGetList()
//		  aEval( aGetList, { |g| g:display() } )
//		RETURN
//
//	Files:
//
//
//	Description:
//		Provides visibility to the GetList passed to the current Read layer.
//
//	Notes:
//		Programming in Clipper 5, p. 512
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadPos()
//
//	Revisions:
//		01/10/92	20:35:46	1.0	Mike Schinkel: Original version
//		01/10/92	20:38:51	1.1	Rob Hannah: Returns an empty array when no
//										reads are active.
//
////////////////////
////////////////////
FUNCTION ReadGetList()
	RETURN If( Len( aReadInfo ) == 0, {}, ATail( aReadInfo )[ RI_AGETLIST ] )

////////////////
////////////////
//
//	Purpose:
//		Return or set current position in GetList
//
//	Syntax:
//		ReadPos( [ <nNew> ] ) -> nPosition
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		nNew      	New value for setting Read position
//
//	Returns:
//		The current read position
//
//	Examples:
//		FUNCTION GotoGet( cGet )
//			ReadPos( GetOrdinal( cGet ) )
//		RETURN .T.
//
//	Files:
//
//
//	Description:
//		Returns the ordinal position of the Get object with input focus and
//		optionally specifies the next Get to provide input focus to.
//
//	Notes:
//		Programming in Clipper 5, p. 512
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadList()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		07/01/91	20:39:54	1.0	Mike Schinkel: Original version
//		07/05/92	20:44:02	1.1	Rob Hannah: Return 0 when no read is active
//		08/01/92	15:48:12	1.2	Rob Hannah: Update ReadVar()
//
////////////////
////////////////
FUNCTION ReadPos( nNew )
	LOCAL bGetSet
	LOCAL nOld := 0
	LOCAL nLen, aGetList

	IF Len( aReadInfo ) > 0
		bGetSet := ATail( aReadInfo )[ RI_NPOS ]
		nOld    := Eval( bGetSet )

		IF nNew <> NIL
			aGetList:= ReadGetList()
			DO CASE
			CASE nNew < 1
				nNew:= 1
			CASE nNew > ( nLen:= Len( aGetList ) )
				nNew:= nLen
			ENDCASE
			Eval( bGetSet, nNew )
			aGetList[ nNew ]:exitState:= GE_NOEXIT
			GetActive( aGetList[ nNew ] )
			ReadVar( GetReadVar( aGetList[ nNew ] ) )
		ENDIF
	ENDIF
	RETURN nOld


/////////////////
/////////////////
//
//	Purpose:
//		Provides Get/Set access to the "Kill" status
//
//	Syntax:
//		ReadKill( [ <lNewKill> ] ) -> lOldKill
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		lNewKill  	New value for KillRead
//
//	Returns:
//		Current KillRead setting
//
//	Examples:
//		ReadKill( .T. )	// Kill the current read
//
//	Files:
//
//
//	Description:
//		Provides visibility to the KillRead Get system variable for assigning
//		or retrieving its value.
//
//	Notes:
//		Programming in Clipper 5, p. 513
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadUpdated()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/10/92	20:44:36	1.0	Original version
//
/////////////////
/////////////////
FUNCTION ReadKill( lNewKill )
	LOCAL lOldKill:= KillRead
	IF lNewKill # NIL
		KillRead:= lNewKill
	END
	RETURN lOldKill

////////////////////
////////////////////
//
//	Purpose:
//		Provides Get/Set access to the "Updated" status
//
//	Syntax:
//		ReadUpdated( <lNewUpdated> ) -> lUpdated
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		lNewUpdated	New value for Updated
//
//	Returns:
//		The current setting of Updated
//
//	Examples:
//		See ListPick()
//
//	Files:
//
//
//	Description:
//		Allows the programmer to manually override the Updated status for the
//		current read layer.
//
//	Notes:
//		Programming in Clipper 5, p. 513
//
//		Very valuable for Get utility functions that directly modify a Get
//		without providing manual editing for it.  Updated can then be set to
//		an accurate value.
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ListPick() InTable() InAlias()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/10/92	20:48:09	1.0	Original version
//
////////////////////
////////////////////
FUNCTION ReadUpdated( lNewUpdated )
	LOCAL lOldUpdated:= Updated
	IF lNewUpdated <> NIL
		Updated := lNewUpdated
	END
	RETURN lOldUpdated

///////////////////
///////////////////
//
//	Purpose:
//		Provides Get/Set access to the "Format"
//
//	Syntax:
//		ReadFormat( <bNewFormat> ) -> bFormat
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		bNewFormat	New value for Format
//
//	Returns:
//		Current value of Format
//
//	Examples:
//		ReadFormat( NIL )
//
//	Files:
//
//
//	Description:
//		Sets the "Format" to a new value.
//
//	Notes:
//		Programming in Clipper 5, p. 509, 513
//
//		Support for GetSys02 routines
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadKill() ReadUpdated()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		08/05/91	20:50:59	1.0	Original version
//		01/11/92	15:40:26	1.1	John Kaster: Support setting Format to NIL
//
///////////////////
///////////////////
FUNCTION ReadFormat( bNewFormat )
	LOCAL lOldFormat:= Format
	IF PCOUNT() > 1 .AND. bNewFormat == NIL .OR. ValType( bNewFormat ) == 'B'
		Format:= bNewFormat
	ENDIF
	RETURN lOldFormat


//////////////////
//////////////////
//
//	Purpose:
//		Cleans up and exits out of the current read
//
//	Syntax:
//		ReadBreak() -> lUpdated
//
//	No arguments specified
//
//	Returns:
//		Current Updated status
//
//	Examples:
//		See ReadModal()
//
//	Files:
//
//
//	Description:
//		Pops current read layer off the read stack, and returns the updated
//		status for the freshly popped read.
//
//	Notes:
//		Programming in Clipper 5, p. 513
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadModal()
//
//	Revisions:
//		08/01/91	20:57:15	1.0	Rob Hannah: Original version
//		01/11/92	12:08:53	1.1	John Kaster: Preserved value of Updated
//
//////////////////
//////////////////
FUNCTION ReadBreak()
	LOCAL lSavUpdate := Updated
	RestoreGetSysVars( aTail( aReadInfo )[ RI_STATEINFO ] )	// restore state
	SetPos( MaxRow()-1, 0 )	// S87 compatibility
	ASize( aReadInfo, Len( aReadInfo ) - 1 )
	RETURN Updated := lSavUpdate


//////////////
//////////////
//
//	Purpose:
//		Standard modal READ on an array of GETs.
//
//	Syntax:
//		ReadModal( <GetList>, [ <nPosition> ], [ <bInkey> ], [ <bPreGet> ],
//						[ <bPostGet> ] ) -> lUpdate
//
//	Formal Arguments: (5)
//		Name       	Description
//			
//		GetList   	List of Get objects to edit in full-screen mode
//		nPosition	Default position in the Get List.  (Defaults to 0)
//		bInkey    	Code block for getting key input.  (Defaults to {|g|InKey(0)})
//		bPreGet   	Code block to evaluate before every Get in the Get List.
//						Ignored if not a code block.
//		bPostGet  	Code block to evaluate after every Get in the Get List.
//						Ignored if not a code block.
//
//	Returns:
//		.T. if the value for any Get has been changed, .F. otherwise
//
//	Examples:
//		ReadModal( GetList,, {|g| ShowHelp( g ) }, {|g| RemoveHelp( g ) } )
//
//	Files:
//
//
//	Description:
//		Provides full-screen data entry of an array of Get objects.
//
//	Notes:
//		Programming in Clipper 5, pp. 513 - 515
//
//		This is the function mapped to by the READ command.
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		GetReader()	Settle()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		08/01/91	11:40:25	1.0	Mike Schinkel: Added bInkey, bPreGet, bPostGet
//		03/05/93 15:07:25	1.1	John Kaster: Added nPosition
//		09/09/93 12:15am  1.2	John Kaster: Made nPosition to 2nd parameter for
//										Clipper 5.2c
//
//////////////
//////////////
FUNCTION ReadModal( GetList, nPosition, bInkey, bPreGet, bPostGet )
	LOCAL get, pos

	IF ( ValType( Format ) == "B" )
		Eval( Format )
	ENDIF

	IF ( Empty( GetList ) )
		SetPos( MaxRow()-1, 0 )		// S87 compat.
		RETURN (.f.)					// NOTE
	ENDIF

	DEFAULT bInKey TO {| g | InKey( 0 ) }
	DEFAULT nPosition TO 0

	AAdd( aReadInfo, { GetList, ;
							{|x| IIf( x == NIL, pos, pos:= x ) }, ;
							ClearGetSysVars(), ;
							ProcName(1), ;
							ProcLine(1) } )

	ReadProcName := ProcName(1)	// set these for use in SET KEYs
	ReadProcLine := ProcLine(1)


	pos := Settle( Getlist, nPosition )	// set initial GET to be read

	WHILE ( pos <> 0 )

		// get next GET from list and post it as the active GET
		get := GetList[pos]
		PostActiveGet( get )
		IF ValType( bPreGet ) == "B"
			Eval( bPreGet, get )
		ENDIF

		IF ( ValType( get:reader ) == "B" )
			Eval( get:reader, get, bInkey, @pos )   // use custom reader block
		ELSE
			GetReader( get, bInkey, @pos )
		ENDIF

		IF ValType( bPostGet ) == "B"
			Eval( bPostGet, get )
		ENDIF

		// move to next GET based on exit condition
		pos := Settle( GetList, pos )

	ENDDO

RETURN ReadBreak()

//////////////
//////////////
//
//	Purpose:
//		Modal READ on an array of GETs with enforcement of valid clauses
//
//	Syntax:
//		ReadValid( <GetList>, [ <nPosition> ], [ <bInkey> ], [ <bPreGet> ],
//						[ <bPostGet> ] ) -> lUpdate
//
//	Formal Arguments: (5)
//		Name       	Description
//			
//		GetList   	List of Get objects to edit in full-screen mode
//		nPosition	Default position in the Get List.  (Defaults to 0)
//		bInkey    	Code block for getting key input.  (Defaults to {|g|InKey(0)})
//		bPreGet   	Code block to evaluate before every Get in the Get List.
//						Ignored if not a code block.
//		bPostGet  	Code block to evaluate after every Get in the Get List.
//						Ignored if not a code block.
//
//	Returns:
//		.T. if the value for any Get has been changed, .F. otherwise
//
//	Examples:
//		ReadModal( GetList,, {|g| ShowHelp( g ) }, {|g| RemoveHelp( g ) } )
//
//	Files:
//
//
//	Description:
//		Provides full-screen data entry of an array of Get objects.
//
//	Notes:
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		GetReader()	Settle()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		10-07-93 10:02pm Original. John Kaster - added to GetSysPP.PRG
//
//////////////
//////////////
FUNCTION ReadValid( GetList, nPosition, bInkey, bPreGet, bPostGet )
	LOCAL get, pos, nCheckPos, nLastExit

	IF ( ValType( Format ) == "B" )
		Eval( Format )
	ENDIF

	IF ( Empty( GetList ) )
		SetPos( MaxRow()-1, 0 )		// S87 compat.
		RETURN (.f.)					// NOTE
	ENDIF

	DEFAULT bInKey TO {| g | InKey( 0 ) }
	DEFAULT nPosition TO 0

	AAdd( aReadInfo, { GetList, ;
							{|x| IIf( x == NIL, pos, pos:= x ) }, ;
							ClearGetSysVars(), ;
							ProcName(1), ;
							ProcLine(1) } )

	ReadProcName := ProcName(1)	// set these for use in SET KEYs
	ReadProcLine := ProcLine(1)


	pos := Settle( Getlist, nPosition )	// set initial GET to be read

	WHILE ( pos <> 0 )

		// get next GET from list and post it as the active GET
		get := GetList[pos]
		PostActiveGet( get )
		IF ValType( bPreGet ) == "B"
			Eval( bPreGet, get )
		ENDIF

		IF ( ValType( get:reader ) == "B" )
			Eval( get:reader, get, bInkey, @pos )   // use custom reader block
		ELSE
			GetReader( get, bInkey, @pos )
		ENDIF

		IF ValType( bPostGet ) == "B"
			Eval( bPostGet, get )
		ENDIF

		nCheckPos := pos	// Save get position
		nLastExit := GetList[ pos ]:exitState
		// move to next GET based on exit condition
		pos := Settle( GetList, pos )
		IF pos == 0  .AND. nLastExit # GE_ESCAPE
			pos := FindBadValid( GetList, nCheckPos )
		ENDIF

	ENDDO

	IF nLastExit == GE_ESCAPE
		ReadUpdated(.F.)
	ENDIF

RETURN ReadBreak()

//////////////
//////////////
//
//	Purpose:
//		Find a GET not passing its vaid clause
//
//	Syntax:
//		FindBadValid( <aGetList>, <nStart> ) -> nBadGetPos
//
//	Formal Arguments: (2)
//		Name       	Description
//			
//		aGetList   	List of Get objects to edit in full-screen mode
//		nStart		Starting position in the Get List.  (Defaults to 1)
//
//	Returns:
//		The position of the first invalid get, or 0 if all are okay
//
//	Examples:
//		See ReadValid()
//
//	Files:
//
//
//	Description:
//		Searches the list of gets that have not been edited
//
//	Notes:
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		GetReader()	Settle()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		10-07-93 10:02pm Original version by John Kaster
//
//////////////
//////////////
STATIC FUNCTION FindBadValid( aGetList, nStart )
	LOCAL oGetSave	:= GetActive()
	LOCAL nBad		:= 0
	LOCAL nGets		:= Len( aGetList )
	LOCAL nGet
	LOCAL g

	IF nStart == NIL .OR. nStart == 0
		nStart := 1
	ENDIF

	FOR nGet := nStart TO nGets
		g := aGetList[ nGet ]
		PostActiveGet( g )
		IF ( g:exitState == NIL .OR. g:exitState == GE_NOEXIT ;
				.OR. g:exitState == GE_UP ) .AND. GetPreValidate( g )
			g:setFocus()
			IF ! GetPostValidate( g )	// Is this value bad?
				g:killFocus()
				nBad	:= nGet				// Save bad get position
				EXIT							// Drop out of loop
			ENDIF
			g:killFocus()

		ENDIF

	NEXT nGet

	GetActive( oGetSave )

RETURN nBad


///////////////////
///////////////////
//
//	Purpose:
//		Standard modal read of a single GET.
//
//	Syntax:
//		GetReader( <get>, [<bInKey>], <pos> )
//
//	Formal Arguments: (3)
//		Name       	Description
//			
//		get       	Get object to edit
//		bInKey    	Code block for getting "key" input. Defaults to
//						{| g | InKey( 0 ) }
//		pos       	Position to assign
//
//	Examples:
//		See Ask4Var(), ReadModal()
//
//	Files:
//
//
//	Description:
//		Performs editing of the variable specified in the passed Get object.
//
//	Notes:
//		Programming in Clipper 5, pp. 515 - 516
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		Settle() Ask4Var()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	14:58:42	1.0	Mike Schinkel: Original version
//		12-17-93 05:49pm	1.1	John Kaster: Support for initial cursor position
//
///////////////////
///////////////////
PROCEDURE GetReader( get, bInKey, pos )
	LOCAL nOldPos := pos

	DEFAULT bInKey TO {| g | InKey( 0 ) }	// default input code block

	// read the GET IF the WHEN condition is satisfied
	IF ( GetPreValidate(get) )

		get:SetFocus()					// activate the GET for reading
		get:pos := ReadGetPos()		// Set the get cursor position

		WHILE ( get:exitState == GE_NOEXIT ) .and. nOldPos == pos

			// check for initial typeout (no editable positions)
			IF ( get:typeOut )
				get:exitState := GE_ENTER
			ENDIF

			// apply keystrokes until exit
			WHILE ( get:exitState == GE_NOEXIT ) .and. nOldPos == pos
				GetApplyKey( get, Eval( bInkey, get ) )
			ENDDO

			// disallow exit IF the VALID condition is not satisfied
			IF nOldPos == pos .and. ( !GetPostValidate(get) )
				get:exitState := GE_NOEXIT
			ENDIF

		ENDDO

		get:KillFocus()	// de-activate the GET

	ENDIF

RETURN


/////////////////////
/////////////////////
//
//	Purpose:
//		Apply a single keystroke to a GET.
//
//	Syntax:
//		GetApplyKey( <get>, <key> )
//
//	Formal Arguments: (2)
//		Name       	Description
//			
//		get			Active get object
//		key			Key that was pressed
//
//	Examples:
//		See GetReader()
//
//	Files:
//
//
//	Description:
//		Performs the action specified by the key pressed for editing the
//		passed Get object.
//
//	Notes:
//		Get object must have input focus
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		GetReader() GetMoveTo() ReadGetPos()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	15:03:22	1.0	Original version
//		12-17-93 02:59pm	2.0	Added conditional compilation for mouse support
//
/////////////////////
/////////////////////
PROCEDURE GetApplyKey(get, key)
	LOCAL aMouse
	LOCAL nGet
	LOCAL cKey
	LOCAL bKeyBlock

	// check for SET KEY first
	IF ( (bKeyBlock := SetKey(key)) <> NIL )
		GetDoSetKey(bKeyBlock, get)
		RETURN									// NOTE
	ENDIF

	DO CASE
	CASE ( key == K_UP )
		get:exitState := GE_UP

	CASE ( key == K_SH_TAB )
		get:exitState := GE_UP

	CASE ( key == K_DOWN )
		get:exitState := GE_DOWN

	CASE ( key == K_TAB )
		get:exitState := GE_DOWN

	CASE ( key == K_ENTER )
		get:exitState := GE_ENTER

	CASE ( key == K_ESC )
		IF ( Set(_SET_ESCAPE) )
			get:undo()
			get:exitState := GE_ESCAPE
		ENDIF

#ifdef MOUSE
	CASE ( key == K_LEFT_DOWN .OR. key == K_LEFT_DBLCLICK )
		aMouse	:= GetMousePos()				// Grumpfish's mouse position routine
		nGet		:= GetMoveTo( aMouse[ 1 ], aMouse[ 2 ], get, ReadGetList() )
#endif

	CASE ( key == K_PGUP )
		get:exitState := GE_WRITE

	CASE ( key == K_PGDN )
		get:exitState := GE_WRITE

	CASE ( key == K_CTRL_HOME )
		get:exitState := GE_TOP


#ifdef CTRL_END_SPECIAL

	// both ^W and ^End go to the last GET
	CASE (key == K_CTRL_END)
		get:exitState := GE_BOTTOM

#else

	// both ^W and ^End terminate the READ (the default)
	CASE (key == K_CTRL_W)
		get:exitState := GE_WRITE

#endif


	CASE (key == K_INS)
		Set( _SET_INSERT, !Set(_SET_INSERT) )
		ShowScoreboard()

	CASE (key == K_UNDO)
		get:Undo()

	CASE (key == K_HOME)
		get:Home()

	CASE (key == K_END)
		get:End()

	CASE (key == K_RIGHT)
		get:Right()

	CASE (key == K_LEFT)
		get:Left()

	CASE (key == K_CTRL_RIGHT)
		get:WordRight()

	CASE (key == K_CTRL_LEFT)
		get:WordLeft()

	CASE (key == K_BS)
		get:BackSpace()

	CASE (key == K_DEL)
		get:Delete()

	CASE (key == K_CTRL_T)
		get:DelWordRight()

	CASE (key == K_CTRL_Y)
		get:DelEnd()

	CASE (key == K_CTRL_BS)
		get:DelWordLeft()

	OTHERWISE

		IF (key >= 32 .and. key <= 255)

			cKey := Chr(key)

			IF (get:type == "N" .and. (cKey == "." .or. cKey == ","))
				get:ToDecPos()

			ELSE
				IF ( Set(_SET_INSERT) )
					get:Insert(cKey)
				ELSE
					get:Overstrike(cKey)
				ENDIF

				IF (get:typeOut .and. !Set(_SET_CONFIRM) )
					IF ( Set(_SET_BELL) )
						?? Chr(7)
					ENDIF

					get:exitState := GE_ENTER
				ENDIF

			ENDIF

		ENDIF

	ENDCASE

RETURN

///////////////////////
///////////////////////
//
//	Purpose:
//		Post/Get default starting position inside a get
//
//	Syntax:
//		ReadGetPos( [<nPos>] ) -> nPosition
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		nPos			Position to set cursor in get buffer when it's activated
//
//	Returns:
//		1 if no position override is entered, or the positional override
//
//	Examples:
//		See GetPoint()
//
//	Files:
//
//
//	Description:
//		This routine was added for mouse support, so a mouse click could cause
//		the cursor position to be changed in a get when that get is either the
//		active get or the "to be activated" get.
//
//	Notes:
//
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		GetPoint() GetMoveTo() GetApplyKey()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//		common.ch
//
//	Revisions:
//		12-17-93 02:54pm	Original version
//
///////////////////////
///////////////////////
FUNCTION ReadGetPos( nPos )
	STATIC saPos	:= {}
	LOCAL nRetPos	:= 1					// Default get position
	LOCAL nSize

	DO CASe
	CASE PCOUNT() > 0						// Post a position
		aAdd( saPos, nPos )
		nRetPos	:= nPos
	CASE ( nSize := Len( saPos ) ) > 0
		nRetPos	:= saPos[ nSize ]
		saPos		:= aSize( saPos, --nSize )
	ENDCASE

RETURN nRetPos

///////////////////////
///////////////////////
//
//	Purpose:
//		Verify that a get may be edited with the WHEN clause/preBlock
//
//	Syntax:
//		GetPreValidate( <get> ) -> lEditable
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		get
//
//	Returns:
//		TRUE if the get is editable, FALSE otherwise
//
//	Examples:
//		See GetReader()
//
//	Files:
//
//
//	Description:
//		If a code block is defined for get:preBlock, this routine evaluates
//		that code block, passing the get object as an argument to it.  If the
//		value returned by the evaluation is TRUE, the get is considered
//		editable.  If it is false, editing of the get will not be allowed.
//
//	Notes:
//
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		GetPostValidate() GetReader()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	15:05:17	1.0	Original version
//    10-05-93 00:59am	1.1	John Kaster: doesn't override programmed exit
//										states
///////////////////////
///////////////////////
FUNCTION GetPreValidate( get )
	LOCAL when := .t.

	IF ( get:preBlock <> NIL )
		when := Eval(get:preBlock, get)
		get:Display()
		ShowScoreBoard()
	ENDIF

	IF ( KillRead )
		when := .f.
		get:exitState := GE_ESCAPE			// provokes ReadModal() exit

	ELSEIF ( !when )
													// exitState wasn't reset?
		IF get:exitState == NIL .OR. get:exitState == GE_NOEXIT
			get:exitState := GE_WHEN		// indicates failure
		ENDIF

	ELSE
		get:exitState := GE_NOEXIT			// prepares for editing

	ENDIF

RETURN ( when )



////////////////////////
////////////////////////
//
//	Purpose:
//		Test exit condition with the VALID clause/postBlock for a GET.
//
//	Syntax:
//		GetPostValidate( <get> ) -> lCanExit
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		get       	Get object
//
//	Returns:
//		TRUE if the get editing may be exited, FALSE otherwise
//
//	Examples:
//		See GetReader()
//
//	Files:
//
//
//	Description:
//		Test various data-specific conditions for a get object, and
//		evaluates the get:postBlock (if it is defined).  If the evaluation
//		returns TRUE, the exiting is allowed from the get.  Otherwise, exiting
//		is not allowed.
//
//	Notes:
//		Bad dates are rejected in a way that preserves the edit buffer.
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		GetPreValidate() GetReader()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	15:08:54	1.0	Original version
//
////////////////////////
////////////////////////
FUNCTION GetPostValidate( get )
	LOCAL changed, valid := .t.


	IF ( get:exitState == GE_ESCAPE )
		RETURN (.t.)					// NOTE
	ENDIF

	IF ( get:BadDate() )
		get:Home()
		DateMsg()
		ShowScoreboard()
		RETURN (.f.)					// NOTE
	ENDIF


	// IF editing occurred, assign the new value to the variable
	IF ( get:changed )
		get:Assign()
		Updated := .t.
	ENDIF


	// reform edit buffer, set cursor to home position, redisplay
	get:Reset()


	IF ( get:postBlock <> NIL )	// check VALID condition if specified
		SetPos( get:row, get:col + Len(get:buffer) )	// S87 compat.
		valid := Eval(get:postBlock, get)
		SetPos( get:row, get:col )							// reset compat. pos
		ShowScoreBoard()
		get:UpdateBuffer()

		IF ( KillRead )
			get:exitState := GE_ESCAPE	// provokes ReadModal() exit
			valid := .t.
		ENDIF

	ENDIF

RETURN (valid)

/////////////////////
/////////////////////
//
//	Purpose:
//		Process SET KEY during editing.
//
//	Syntax:
//		GetDoSetKey( <keyBlock>, <get> )
//
//	Formal Arguments: (2)
//		Name       	Description
//			
//		keyBlock  	Set Key block to evaluate
//		get       	Get object
//
//	Examples:
//		See GetApplyKey()
//
//	Files:
//
//
//	Description:
//		Feeds a code block defined by a SET KEY statement the correct
//		parameters for ignoring the Get System routines.
//
//	Notes:
//
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	15:13:05	1.0	Original version
//
/////////////////////
/////////////////////
PROCEDURE GetDoSetKey( keyBlock, get )
	IF ( get:changed )	// if editing has occurred, assign variable
		get:Assign()
		Updated := .t.
	ENDIF

	Eval( keyBlock, ReadProcName, ReadProcLine, ReadVar() )

	ShowScoreboard()
	get:UpdateBuffer()

	IF ( KillRead )
		get:exitState := GE_ESCAPE		// provokes ReadModal() exit
	ENDIF

RETURN



//////////////////////
//////////////////////
//
//	Purpose:
//		Determines ordinal position of next Get to edit
//
//	Syntax:
//		Settle( <GetList>, <pos> ) -> nPosition
//
//	Formal Arguments: (2)
//		Name       	Description
//			
//		GetList   	List of Get objects
//		pos       	Current position
//
//	Returns:
//		Ordinal position in the GetList array of the next get
//
//	Examples:
//		See ReadModal()
//
//	Files:
//
//
//	Description:
//		Determines the new position in the array of Get objects, based on the
//		current position and the get:exitState of the current get.
//
//
//	Notes:
//		A return value of 0 indicates termination of READ
//		exitState of the old Get is transferred to the new Get
//
//		FUNCTION Settle
//		is only visible in its own source file
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadModal()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	15:17:27	1.0	Original version
//
//////////////////////
//////////////////////
STATIC FUNCTION Settle( GetList, pos )
	LOCAL exitState

	IF ( pos == 0 )
		exitState := GE_DOWN
	ELSE
		exitState := GetList[pos]:exitState
		// DEFAULT exitState TO GE_NOEXIT
	ENDIF

	IF ( exitState == GE_ESCAPE .or. exitState == GE_WRITE )
		RETURN ( 0 )					// NOTE
	ENDIF

	IF ( exitState <> GE_WHEN )	// reset state info
		LastPos := pos
		BumpTop := .f.
		BumpBot := .f.

	ELSE						// re-use last exitState, do not disturb state info
		exitState := LastExit
	ENDIF


	DO CASE					// Movements
	CASE ( exitState == GE_UP ) ; pos --

	CASE ( exitState == GE_DOWN ) ; pos ++

	CASE ( exitState == GE_TOP )
		pos := 1
		BumpTop := .T.
		exitState := GE_DOWN

	CASE ( exitState == GE_BOTTOM )
		pos := Len(GetList)
		BumpBot := .T.
		exitState := GE_UP

	CASE ( exitState == GE_ENTER ) ; pos ++

	ENDCASE


	/***
	*	bounce
	*/
	IF ( pos == 0 ) 						// bumped top

		IF ( !ReadExit() .and. !BumpBot )
			BumpTop := .T.
			pos := LastPos
			exitState := GE_DOWN
		ENDIF

	ELSEIF ( pos == Len(GetList) + 1 )		// bumped bottom

		IF ( !ReadExit() .and. exitState <> GE_ENTER .and. !BumpTop )
			BumpBot := .T.
			pos := LastPos
			exitState := GE_UP
		ELSE
			pos := 0
		ENDIF
	ENDIF


	// record exit state
	LastExit := exitState

	IF ( pos <> 0 )
		GetList[pos]:exitState := exitState
	ENDIF

RETURN ( pos )


//////////////////////////////
//////////////////////////////
//
//	Purpose:
//		Set ReadVar() and GetActive() functions to their return values
//
//	Syntax:
//		PostActiveGet( <get> )
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		get
//
//	Examples:
//		See ReadModal()
//
//	Files:
//
//
//	Description:
//		Uses the passed Get object to initialize the return values for the
//		ReadVar() and GetActive() functions.
//
//	Notes:
//
//		PROCEDURE PostActiveGet
//		is only visible in its own source file
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadModal() GetActive() ReadVar()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	15:21:17	1.0	Original version
//
//////////////////////////////
//////////////////////////////
STATIC PROCEDURE PostActiveGet( get )
	GetActive( get )
	ReadVar( GetReadVar( get ) )
	ShowScoreBoard()
RETURN


///////////////////////////////
///////////////////////////////
//
//	Purpose:
//		Save and clear READ state variables
//
//	Syntax:
//		ClearGetSysVars() -> aState
//
//	No arguments specified
//
//	Returns:
//		Array of saved values.
//
//	Examples:
//		See ReadModal()
//
//	Files:
//
//
//	Description:
//
//
//	Notes:
//		'Updated' status is saved and cleared
//
//		FUNCTION ClearGetSysVars
//		is only visible in its own source file
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		RestoreGetSysVars() ReadModal()
//
//	Revisions:
//		01/11/92	15:23:02	1.0	Original version
//
///////////////////////////////
///////////////////////////////
STATIC FUNCTION ClearGetSysVars()
	LOCAL aSaved[ GSV_COUNT ]

	aSaved[ GSV_KILLREAD ] := KillRead
	KillRead := .f.

	aSaved[ GSV_BUMPTOP ] := BumpTop
	BumpTop := .f.

	aSaved[ GSV_BUMPBOT ] := BumpBot
	BumpBot := .f.

	aSaved[ GSV_LASTEXIT ] := LastExit
	LastExit := 0

	aSaved[ GSV_LASTPOS ] := LastPos
	LastPos := 0

	aSaved[ GSV_ACTIVEGET ] := GetActive( NIL )

	aSaved[ GSV_READVAR ] := ReadVar( "" )

	aSaved[ GSV_READPROCNAME ] := ReadProcName
	ReadProcName := ""

	aSaved[ GSV_READPROCLINE ] := ReadProcLine
	ReadProcLine := 0

	aSaved[ GSV_UPDATED ] := Updated

	Updated := .f.

RETURN (aSaved)


/////////////////////////////
/////////////////////////////
//
//	Purpose:
//		Restore READ state variables from array of saved values.
//
//	Syntax:
//		RestoreGetSysVars( <aSaved> )
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		aSaved     	Array of READ state variables
//
//	Examples:
//		See ReadBreak()
//
//	Files:
//
//
//	Description:
//		Restores the READ state variables from the array of saved values for
//		support of nested read layers.
//
//	Notes:
//		'Updated' status is restored
//
//		PROC RestoreGetSysVars
//		is only visible in its own source file
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadBreak()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	15:25:10	1.0	Original version
//		01/11/92	15:27:38	1.1	John Kaster:  Restore 'Updated' status
//
/////////////////////////////
/////////////////////////////
STATIC PROC RestoreGetSysVars(aSaved)

	KillRead			:= aSaved[ GSV_KILLREAD ]

	BumpTop			:= aSaved[ GSV_BUMPTOP ]

	BumpBot			:= aSaved[ GSV_BUMPBOT ]

	LastExit			:= aSaved[ GSV_LASTEXIT ]

	LastPos			:= aSaved[ GSV_LASTPOS ]

	GetActive( aSaved[ GSV_ACTIVEGET ] )

	ReadVar( aSaved[ GSV_READVAR ] )

	ReadProcName 	:= aSaved[ GSV_READPROCNAME ]

	ReadProcLine 	:= aSaved[ GSV_READPROCLINE ]

	Updated			:= aSaved[ GSV_UPDATED ]

RETURN

//////////////////////////
//////////////////////////
//
//	Purpose:
//		Provides READVAR()-compatible value from a GET object
//
//	Syntax:
//		GetReadVar( <get> ) -> cVariable
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		get       	Get object
//
//	Returns:
//		ReadVar()-style name of Get
//
//	Examples:
//		ReadVar( GetReadVar( get ) )
//
//	Files:
//
//
//	Description:
//		Massages the Get:name to provide a ReadVar()-compatible name for the
//		get object.  Array subscripts are also supported.
//
//	Notes:
//
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadVar()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	15:33:00	1.0	Original version
//		01/11/92	16:40:44	1.1	John Kaster:  Made GetReadVar() visible from
//										outside of GetSysPP.PRG
//
//////////////////////////
//////////////////////////
FUNCTION GetReadVar(get)
	LOCAL name := Upper(get:name)
	LOCAL i

	/***
	*	The following code includes subscripts in the name returned by
	*	this function, if the get variable is an array element.
	*
	*	Subscripts are retrieved from the get:subscript instance variable.
	*
	*	NOTE: incompatible with Summer 87
	*/

	IF ( get:subscript <> NIL )
		FOR i := 1 to len(get:subscript)
			name += "[" + ltrim(str(get:subscript[i])) + "]"
		NEXT
	ENDIF

RETURN (name)



/**********************
*
*	system services
*
*/



////////////////////
////////////////////
//
//	Purpose:
//		Assign the 'Format' Get variable
//
//	Syntax:
//		__SetFormat( <b> ) -> NIL
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		b         	Code block to assign to 'Format'
//
//	Returns:
//		NIL
//
//	Examples:
//		SET FORMAT TO SalesScr
//
//	Files:
//
//
//	Description:
//		Sets the 'Format' Get variable for use elsewhere in the get system.
//
//	Notes:
//
//		Usually via the SET FORMAT command.
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadFormat()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	15:50:26	1.0	Original version
//
////////////////////
////////////////////
FUNCTION __SetFormat(b)
	Format := IF ( ValType(b) == "B", b, NIL )
RETURN (NIL)


////////////////////
////////////////////
//
//	Purpose:
//		Sets the Read layer to be cleared
//
//	Syntax:
//		__KillRead()
//
//	No arguments specified
//
//	Examples:
//		__KillRead()
//
//	Files:
//
//
//	Description:
//		Sets the 'KillRead' get variable to TRUE for automatic clearing of
//		the current get system.
//
//	Notes:
//		Called by the CLEAR GETS command.
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadKill()
//
//	Revisions:
//		01/11/92	15:53:29	1.0	Original version
//
////////////////////
////////////////////
PROCEDURE __KillRead()
	KillRead := .t.
RETURN


//////////////////
//////////////////
//
//	Purpose:
//		Get/Set routine for retrieving the active get object
//
//	Syntax:
//		GetActive( [ <g> ] ) -> oCurrentGet
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		g         	Get object
//
//	Returns:
//		Current "active" get object
//
//	Examples:
//		LOCAL oGet := GetActive()
//
//	Files:
//
//
//	Description:
//		Provides access to the currently active get object in the get system.
//		Optionally allows for setting of this value by passing another get
//		object.
//
//	Notes:
//		Programming in Clipper 5, pp. 128, 517, 675
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadVar()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	15:56:15	1.0	Original version
//
//////////////////
//////////////////
FUNCTION GetActive(g)
	LOCAL oldActive := ActiveGet
	IF ( PCount() > 0 )
		ActiveGet := g
	ENDIF
RETURN ( oldActive )


////////////////
////////////////
//
//	Purpose:
//		Updated status of current read
//
//	Syntax:
//		Updated() -> lUpdated
//
//	No arguments specified
//
//	Returns:
//		TRUE if any get in the current Read has been modified, FALSE otherwise
//
//	Examples:
//		IF Updated() .AND. Ask4Var(.T.,"Save changes?")
//			PutValues()
//		ENDIF
//
//	Files:
//
//
//	Description:
//		Indicates whether the current read layer has had any get modified.
//
//	Notes:
//		Provides visibility to the 'Updated' get system variable
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadUpdated()
//
//	Revisions:
//		01/11/92	16:00:45	1.0	Original version
//
////////////////
////////////////
FUNCTION Updated()
RETURN (Updated)


/////////////////
/////////////////
//
//	Purpose:
//		Get/Set the exit handling for the read
//
//	Syntax:
//		ReadExit( <lNew> ) -> lReadExit
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		lNew      	New setting for lReadExit
//
//	Returns:
//		TRUE if read may be exited with arrow keys, FALSE otherwise
//
//	Examples:
//		ReadExit( .T. )	// Allow arrowing out of read screen
//		ReadExit( .F. )	// Force normal exit key to exit read screen
//
//	Files:
//
//
//	Description:
//		Gets or sets the exit handling for the read screen.  Arrow keys may
//		be used to exit the read screen if ReadExit() is passed TRUE.
//
//	Notes:
//
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadInsert()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	16:03:45	1.0	Original version
//
/////////////////
/////////////////
FUNCTION ReadExit(lNew)
RETURN ( Set(_SET_EXIT, lNew) )


///////////////////
///////////////////
//
//	Purpose:
//		Get/Set the insert status for a read screen
//
//	Syntax:
//		ReadInsert( <lNew> ) -> lInsert
//
//	Formal Arguments: (1)
//		Name       	Description
//			
//		lNew      	New setting for ReadInsert()
//
//	Returns:
//		TRUE if ReadInsert() is on, FALSE otherwise.
//
//	Examples:
//		IF ReadInsert()
//			g:insert( c )
//		ELSE
//			g:overStrike( c )
//		ENDIF
//
//	Files:
//
//
//	Description:
//		Returns or sets the insert status for the read screen.
//
//	Notes:
//
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		ReadExit()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	16:06:22	1.0	Original version
//
///////////////////
///////////////////
FUNCTION ReadInsert(lNew)
RETURN ( Set(_SET_INSERT, lNew) )



/**********************************
*
*	wacky compatibility services
*
*/


// display coordinates for SCOREBOARD
#define SCORE_ROW		0
#define SCORE_COL		60


///////////////////////////////
///////////////////////////////
//
//	Purpose:
//		Display the status of the insert key if scoreboard is on
//
//	Syntax:
//		ShowScoreboard()
//
//	No arguments specified
//
//	Examples:
//		ShowScoreboard()
//
//	Files:
//
//
//	Description:
//		Displays the status of the insert key at row 0, column 60 if the
//		scoreboard is turned on.
//
//	Notes:
//
//		PROCEDURE ShowScoreboard
//		is only visible in its own source file
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		GetApplyKey()
//
//	Revisions:
//		01/11/92	16:08:42	1.0	Original version
//		06/12/92	23:13:39	1.1	Modified to indicate the insert status with
//										the cursor block via ShowCursor()
//
///////////////////////////////
///////////////////////////////
STATIC PROCEDURE ShowScoreboard()
	LOCAL nRow, nCol
	IF ( Set(_SET_SCOREBOARD) )
		nRow := Row()
		nCol := Col()

		SetPos(SCORE_ROW, SCORE_COL)
		DispOut( if(Set(_SET_INSERT), "Ins", "   ") )
		SetPos(nRow, nCol)
	ENDIF

	ShowCursor()

RETURN
#include "setcurs.ch"
///////////////////
///////////////////
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	Purpose:
//		Displays the cursor with sensitivity for insert status
//
//	Syntax:
//		ShowCursor() -> nCursor
//
//	Returns:
//		The old cursor setting
//
//	Examples:
//		See ShowScoreboard()
//
//	Files:
//
//
//	Description:
//		Nice user interface touch to indicate the status of insert or typeover
//		mode.
//
//	Notes:
//
//
//	See Also:
//		ShowScoreboard()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//		setcurs.ch
//
//	Revisions:
//		06/12/92	23:16:29	1.0	Original version
//
///////////////////
///////////////////
FUNCTION ShowCursor()
RETURN SetCursor( If( Set( _SET_INSERT ), SC_SPECIAL1, SC_NORMAL ) )

////////////////////////
////////////////////////
//
//	Purpose:
//		Displays "bad date format" message
//
//	Syntax:
//		DateMsg()
//
//	No arguments specified
//
//	Examples:
//		See GetPostValidate()
//
//	Files:
//
//
//	Description:
//		If the scoreboard is on, displays the "bad date format" message.
//
//	Notes:
//
//		PROCEDURE DateMsg
//		is only visible in its own source file
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		GetPostValidate()
//
//	Revisions:
//		01/11/92	16:10:33	1.0	Original version
//
////////////////////////
////////////////////////
STATIC PROCEDURE DateMsg()
	LOCAL nRow, nCol
	IF ( Set(_SET_SCOREBOARD) )
		nRow := Row()
		nCol := Col()

		SetPos(SCORE_ROW, SCORE_COL)
		DispOut("Invalid Date")
		SetPos(nRow, nCol)

		WHILE ( Nextkey() == 0 ) ; ENDDO	// Wait for key input

		SetPos(SCORE_ROW, SCORE_COL)
		DispOut("            ")
		SetPos(nRow, nCol)

	ENDIF

RETURN

///////////////////
///////////////////
//
//	Purpose:
//		Check the value range for an edited get
//
//	Syntax:
//		RangeCheck( <get>, <junk>, <lo>, <hi> ) ->
//
//	Formal Arguments: (4)
//		Name       	Description
//			
//		get       	Get object
//		junk      	Unused (5.00 compatibility)
//		lo        	Minimum value in range
//		hi        	Maximum value in range
//
//	Returns:
//		TRUE if the entered value is in the range, FALSE otherwise
//
//	Examples:
//		See GetPostValidate()
//
//	Files:
//
//
//	Description:
//		Determines whether the entered value for a get is in its allowed
//		range of values.
//
//	Notes:
//		Unused second parameter is for 5.00 compatibility.
//
//	Section:
//		User Interface
//
//	Category:
//		Get System
//
//	See Also:
//		DateMsg()
//
//	Include files:
//		inkey.ch
//		getexit.ch
//
//	Revisions:
//		01/11/92	16:13:12	1.0	Original version
//
///////////////////
///////////////////
FUNCTION RangeCheck(get, junk, lo, hi)
	LOCAL cMsg, nRow, nCol
	LOCAL xValue

	IF ( !get:changed )
		RETURN (.t.)
	ENDIF

	xValue := get:VarGet()

	IF ( xValue >= lo .and. xValue <= hi )
		RETURN (.t.)									// NOTE
	ENDIF

	IF ( Set(_SET_SCOREBOARD) )
		cMsg := "Range: " + Ltrim(Transform(lo, "")) + ;
				" - " + Ltrim(Transform(hi, ""))

		IF ( Len(cMsg) > MaxCol() )
			cMsg := Substr( cMsg, 1, MaxCol() )
		ENDIF

		nRow := Row()
		nCol := Col()

		SetPos( SCORE_ROW, Min(60, MaxCol() - Len(cMsg)) )
		DispOut(cMsg)
		SetPos(nRow, nCol)

		WHILE ( NextKey() == 0 ) ; ENDDO	// Wait for a keypress

		SetPos( SCORE_ROW, Min(60, MaxCol() - Len(cMsg)) )
		DispOut( Space(Len(cMsg)) )
		SetPos(nRow, nCol)

	ENDIF

RETURN (.f.)

// EOF
