/*
   Program    : Network Maintenance Functions
   Author     : Daniel Neil
   Date       : 8 September 1991
   Modified   :
   Depends On :

*/

#include "mdl.ch"
#include "error.ch"

#define ED_NETWORK "Network Error"
#define TIME_OUT    5
#define EC_NETUSE   1001
#define EC_ADDREC   1002
#define EC_RECLOCK  1003
#define EC_FILELOCK 1004

*******************************************************************************
FUNCTION NetUse(cFile,lExcl,nWait,cAlias,lNew,lGenError)
  LOCAL lOpened  := .F.
  LOCAL lRetry,nTimer
  DEFAULT lExcl     TO .F.
  DEFAULT nWait     TO TIME_OUT
  DEFAULT cAlias    TO StripExt(cFile)
  DEFAULT lNew      TO .T.
	DEFAULT lGenError TO .T.
  cFile:=ALLTRIM(cFile)

  IF lNew
    SELECT 0
  ENDIF

  nTimer := nWait
  REPEAT

    IF lExcl
      USE (cFile) EXCLUSIVE ALIAS (cAlias)
    ELSE
      USE (cFile) SHARED ALIAS (cAlias)
    ENDIF
    lOpened:=!NETERR()

    IF !lOpened .AND. nTimer <= 0
			IF lGenError
        GENERATE ERROR ARGUMENTS   {nWait,cFile,lExcl,cAlias}                ;
                       CANRETRY CANDEFAULT                                   ;
                       DESCRIPTION ED_NETWORK                                ;
                       OPERATION   "Opening Table "+                         ;
                                   IF(lExcl,"Exclusive","Shared")            ;
                       SEVERITY    ES_ERROR                                  ;
                       SUBCODE     EC_NETUSE                                 ;
                       SUBSYSTEM   "NETWORK"                                 ;
                  INTO lRetry
			ELSE
				lRetry := .F.
			ENDIF

      IF lRetry
        nTimer := TIME_OUT
      ELSE
        nTimer := -1
      ENDIF
    ELSEIF !lOpened
      TimeWait(1)
      nTimer:=nTimer-1
    ENDIF

  UNTIL lOpened .OR. nTimer < 0


RETURN lOpened

*******************************************************************************
FUNCTION AddRec(nWait)
  LOCAL lAdded:=.F.
  LOCAL lRetry,nTimer
  LOCAL nRecNo:=RECNO()
  LOCAL lDeleted:=SET(_SET_DELETED,.F.)
  DEFAULT nWait TO TIME_OUT

  nTimer := nWait
  REPEAT
    GO TOP
    IF DELETED()
      lAdded:=RLOCK()
    ELSE
      APPEND BLANK
      lAdded:=!NETERR()
    ENDIF

    IF !lAdded .AND. nTimer <= 0
      GENERATE ERROR ARGUMENTS   {nWait}                                    ;
                     CANRETRY CANDEFAULT                                    ;
                     DESCRIPTION ED_NETWORK                                 ;
                     OPERATION   "Appending Record"                         ;
                     SEVERITY    ES_ERROR                                   ;
                     SUBCODE     EC_ADDREC                                  ;
                     SUBSYSTEM   "NETWORK"                                  ;
                INTO lRetry

      IF lRetry
        nTimer := TIME_OUT
      ELSE
        nTimer := -1
        GO nRecNo
      ENDIF
    ELSEIF !lAdded
      TimeWait(.5)
      nTimer:=nTimer-.5
    ENDIF

  UNTIL lAdded .OR. nTimer < 0

  IF lAdded .AND. DELETED()
    RECALL
  ENDIF
  SET(_SET_DELETED,lDeleted)

RETURN lAdded

*******************************************************************************
FUNCTION RecLock(nWait)
  LOCAL lLocked:=.F.
  LOCAL lRetry,nTimer
  DEFAULT nWait TO TIME_OUT

  nTimer := nWait
  REPEAT
    lLocked:=RLOCK()

    IF !lLocked .AND. nTimer <= 0
      GENERATE ERROR ARGUMENTS   {nWait}                                     ;
                     CANRETRY CANDEFAULT                                     ;
                     DESCRIPTION ED_NETWORK                                  ;
                     OPERATION   "Locking Record"                            ;
                     SEVERITY    ES_ERROR                                    ;
                     SUBCODE     EC_RECLOCK                                  ;
                     SUBSYSTEM   "NETWORK"                                   ;
                INTO lRetry

      IF lRetry
        nTimer := TIME_OUT
      ELSE
        nTimer := -1
      ENDIF
    ELSEIF !lLocked
      TimeWait(.5)
      nTimer:=nTimer-.5
    ENDIF
  UNTIL lLocked .OR. nTimer < 0

RETURN lLocked

*******************************************************************************
FUNCTION FileLock(nWait)
  LOCAL lLocked:=.F.
  LOCAL lRetry,nTimer
  DEFAULT nWait TO TIME_OUT

  nTimer := nWait
  REPEAT
    lLocked:=FLOCK()

    IF !lLocked .AND. nTimer <= 0
      GENERATE ERROR ARGUMENTS   {nWait}                                    ;
                     CANRETRY CANDEFAULT                                    ;
                     DESCRIPTION ED_NETWORK                                 ;
                     OPERATION   "Locking File"                             ;
                     SEVERITY    ES_ERROR                                   ;
                     SUBCODE     EC_FILELOCK                                ;
                     SUBSYSTEM   "NETWORK"                                  ;
                INTO lRetry

      IF lRetry
        nTimer := TIME_OUT
      ELSE
        nTimer := -1
      ENDIF
    ELSEIF !lLocked
      TimeWait(.5)
      nTimer:=nTimer-.5
    ENDIF

  UNTIL lLocked .OR. nTimer < 0

RETURN lLocked

*******************************************************************************
FUNCTION UnLock

  UNLOCK

RETURN .T.

*******************************************************************************
STATIC FUNCTION TimeWait(nPeriod)
  LOCAL nTimer

  nTimer:=SECONDS()
  DO WHILE SECONDS()-nTimer < nPeriod
  ENDDO

RETURN .T.

*****************************************************************************
FUNCTION GenError(args,canDefault,canRetry,canSubstitute,cargo,description,;
                  filename,genCode,operation,osCode,severity,subCode,      ;
                  subSystem)
  LOCAL oError:=ErrorNew()

  oError:args          := args
  oError:canDefault    := canDefault
  oError:canRetry      := canRetry
  oError:canSubstitute := canSubstitute
  oError:cargo         := cargo
  oError:description   := description
  oError:filename      := filename
  oError:genCode       := genCode
  oError:operation     := operation
  oError:osCode        := osCode
  oError:severity      := severity
  oError:subCode       := subCode
  oError:subSystem     := subSystem

RETURN EVAL(ERRORBLOCK(),oError)

*****************************************************************************
FUNCTION StripExt(cFile)
	LOCAL nPos:=AT(".",cFile)

RETURN IF(nPos>0,SUBSTR(cFile,1,nPos-1),cFile)
