/***
*
* CLIPWARN.C
*
* Utility to run the Clipper 5.01 compiler and return an error code
* of 2 if a compiler warning is generated.
*
* Usage:  clipwarn <compiler_commands> [/beep] [/log[:logfile]]
*
* Returns: The errorlevel as returned by the compiler unless the
*          compiler returns an errorlevel of 0 and a warning is
*          detected, in which case an errorlevel of 2 is returned.
*
*          /beep              - beep on exit if non-zero errorlevel
*          /log[:<filename>]  - echo all compiler errors and warnings
*                               to <filename>, defaults to file "ERR"
*
* Written by Don Caton - CIS 70007,5454 - 10/17/91
*
* Released to the public domain.
*
*/

#include <process.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <dos.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <sys\types.h>
#include <sys\stat.h>


#define COMPILER "CLIPPER.EXE"    // name of compiler
//#define COMPILER "CLIP50.EXE"   // name I use for 5.01 compiler

#define ERRORLOG "ERR"            // default filename for error log file

struct INT_REGS {                 // structure of arguments passed to
    unsigned int es;              // a function declared as 'interrupt'
    unsigned int ds;
    unsigned int di;
    unsigned int si;
    unsigned int bp;
    unsigned int sp;
    union {
        struct {
            unsigned int bx;
            unsigned int dx;
            unsigned int cx;
            unsigned int ax;
            } words;
        struct {
            unsigned char bl, bh;
            unsigned char dl, dh;
            unsigned char cl, ch;
            unsigned char al, ah;
            } bytes;
        } gpregs;
    unsigned int ip;
    unsigned int cs;
    unsigned int flags;
    };

#define MK_CHARPTR(seg,off) ((char far*)(((unsigned long)(seg)<<16)|(unsigned int)(off)))
#define FALSE  0
#define TRUE   1
#define STDERR 2

static void interrupt far Int21Intercept( struct INT_REGS regs );


static void (interrupt far *oldInt21)();   // pointer to DOS int21 handler

static int anywarnings = FALSE ,           // any warnings? flag
           handle      = -1    ,           // handle for error log file
           errorfile   = FALSE ;           // log errors to file? flag


/*******
*
* main()
*
*/

void main( int argc, char *argv[] )
{
    int exitcode = -1    ,            // exit code returned to DOS
        beeper   = FALSE ,            // beep on non-zero errorlevel? flag
        x                ;            // loop counter

    char logname[80];                 // name for error log file
    char *compiler = COMPILER;        // name of compiler

    if ( argc == 1 ) {
        write( STDERR, "\nClipWarn  v1.0      Written by Don Caton    CIS: 70007,5454\n\n", 64 );
        write( STDERR, "Syntax: clipwarn <compiler_commands> [<options>]\n\n", 50 );
        write( STDERR, "Options: /beep         Beep on exit if non-zero errorlevel\n", 59 );
        write( STDERR, "         /log[:<file>] Write compiler messages to <file> or ERR\n\n", 66 );
        exit(0);
    }

    for ( x = 1; x < argc; x++ ) {
        if( strstr( argv[x], "/beep" ) || strstr( argv[x], "/BEEP" ) ) {
            beeper = TRUE;
            argv[x] = " ";
        }
        else if( strstr( argv[x], "/log" ) || strstr( argv[x], "/LOG" ) ) {
            errorfile = TRUE;
            if( *(argv[x] + 4) == ':' ) {
                if ( *(argv[x] + 5) == 0 ) {
                    printf( "Error: no log file name specified\n" );
                    goto done;
                }
                else
                    strncpy( logname, (argv[x] + 5), 80 );
                }

            else
                strncpy( logname, ERRORLOG, 4 );
            argv[x] = " ";
        }
    }

    if ( errorfile )
       handle = open( logname, O_CREAT | O_TRUNC | O_WRONLY | O_BINARY, S_IWRITE );

    if ( errorfile && handle == -1 ) {
        printf( "Error #%d opening %s\n", errno, logname );
    }
    else {

        oldInt21 = _dos_getvect( 0x21 );
        _dos_setvect( 0x21, Int21Intercept );

        argv[0] = compiler;
        exitcode = spawnvp( P_WAIT, compiler, argv );
        _dos_setvect( 0x21, oldInt21 );

        if ( exitcode != -1 ) {
            if ( anywarnings )
                exitcode = 2;
        }
        else {
            if ( errno == ENOENT ) {
                printf( "Can't find %s\n", compiler );
            }
            else {
                if ( errno == ENOMEM )
                    printf( "Not enough memory to execute %s\n", compiler );
                else
                    printf( "Error #%d trying to spawn %s\n", errno, compiler );
            }
        }

        close( handle );
    }
    done:
    printf( "\nExit code: %u\n", exitcode & 0xFF );
    if ( beeper && exitcode )
       write( STDERR, "\a", 1 );
    exit( exitcode );
}

/*****************
*
* Int21Intercept()
*
*/

static void interrupt far Int21Intercept( struct INT_REGS regs )
{
    /*
    *
    * Traps calls to DOS int 21h, function 40h (Write File or Device)
    *
    * If handle in bx is 1 (STDOUT), the compiler is writing an error
    * or warning message.  Search buffer pointed to by ds:dx for
    * "Warning" and set flag if found.
    *
    */

    if ( regs.gpregs.bytes.ah == 0x40 ) {
        if ( regs.gpregs.words.bx == 1 ) {
            if ( strstr( MK_CHARPTR( regs.ds, regs.gpregs.words.dx ), "Warning" ) )
                anywarnings = TRUE;
            if ( errorfile )
               write( handle, MK_CHARPTR( regs.ds, regs.gpregs.words.dx ), regs.gpregs.words.cx );

        }
    }


    _chain_intr( oldInt21 );
}

// eof: CLIPWARN.C

