/* CLP_FRMT.PRG

   Pretty Formatter and Action Diagammer For Clipper 5.0 Applications

   Author: Charles R. Wilson    CIS: 76307,2227

   Donated to Public Domain

   Parameters:

   scriptname  =>  filename with extenstion of file list in
                   CLP file format.
   switch2
     or        =>  /N = include file line numbers
   switch3         /L = include control block diagram lines

*/

#define N_INDENT  3
#define EOL       chr(13)
#define LF        chr(10)
#define EOF       chr(26)

parameters scriptname,switch2,switch3
public scriptfile,sourcefile,destfile,scriptline := space(1),;
       sourceline := space(1),n_switch := .f.,l_switch := .f.
clear screen
disp_signon()
private i,tempswitch
if pcount() > 1
   for i = 2 to pcount()
      tempswitch := "switch"+strzero(i,1)
      do case
         case upper(&tempswitch) = "/N"
         n_switch := .t.
         case upper(&tempswitch) = "/L"
         l_switch := .t.
      endcase
   next
endif
if scriptname != nil
   ? "Opening Script File: "+scriptname
   scriptfile := open_file(scriptname,.f.)
   if scriptfile != -1
      do while scriptline != EOL .and. lastkey() != 27
         scriptline := script_buff()
         if scriptline != EOL
            ? "Processing Source File: "+scriptline+".PRG"
            sourcefile := open_file(trim(scriptline)+".PRG",.f.)
            if sourcefile != -1
               destfile := proc_path(trim(scriptline))
               if destfile != -1
                  scan_file()
                  close_file(destfile)
               else
                  ? "Error Opening Destination File"
                  return
               endif
               close_file(sourcefile)
            else
               ? "Error Opening Scource File"
               return
            endif
         endif
      enddo
      close_file(scriptfile)
   else
      ? "Error Opening Script File"
      return
   endif
else
   disp_help()
endif
? "Processing complete!"
?
quit


static function proc_path(filename)
/* remove path from destination file */
local reply,buffer := filename,i
i := at("\",buffer)
do while i > 0
   i := at("\",buffer)
   if i > 0
      buffer := substr(buffer,i+1,len(buffer))
   endif
enddo
reply := open_file(trim(buffer)+".FLW",.t.)
return reply

static function open_file(filename,new)
/* low level file open */
local nhandle := -1
local iserror := 0
if new
   if file(filename)
      /* Delete the file if already exists */
      delete file (filename)
   endif
   nhandle := fcreate(filename)
elseif file(filename)
   nhandle := fopen(filename)
endif
iserror := ferror()
if iserror = 0
   fseek(nhandle,0,0)
else
   ? 'Error #:'+strzero(iserror,3)
endif
return nhandle


static function close_file(nhandle)
/* low level file close */
return fclose(nhandle)


static function script_buff
/* read line form script file */
local lineend := .f.
local character := space(1)
local reply := space(0)
do while !lineend
   character := read_chr(scriptfile)
   if len(character) > 0
      if character = chr(13)
         lineend := .t.
      else
         if character <> LF
            reply := reply + character
         endif
      endif
   else
      reply := EOL
      lineend := .t.
   endif
enddo
return reply


static function srce_buff
/* Read line from source file */
local lineend := .f.
local character := space(1)
local reply := space(0)
do while !lineend
   character := read_chr(sourcefile)
   if len(character) > 0
      if character = chr(13)
         lineend := .t.
      else
         if character <> LF
            reply := reply + character
         endif
      endif
   else
      reply := EOL
      lineend := .t.
   endif
enddo
return reply


static function read_chr(nhandle)
/* 128 byte blocks works faster but caused problems */
local reply := space(1)
local cnt := 0
cnt := fread(nhandle,@reply,1)
if cnt = 0
   reply := space(0)
endif
return reply


static function scan_file
/* mainline source file processor */
local cblock,row,col
private indentlevel := iif(l_switch,2,0)
private indent_on := .f.,linecnt := 0,indent_next := .f.,num_on := .t.,;
        indent_amt := 0,control := {},indent_str := space(indentlevel)
sourceline := " "
@ row(),60 say "LINE# "
row := row()
col := col()
do while sourceline != EOL .and. lastkey() != 27
   sourceline := srce_buff()
   cblock := parse_line(sourceline)
   if cblock != nil
      if cblock[1] != nil
         eval(cblock[1],N_INDENT)
      endif
   endif
   linecnt++
   @ row,col say str(linecnt,4)
   if n_switch
      if num_on
         fwrite(destfile,strzero(linecnt,4),4)
      else
         fwrite(destfile,space(4),4)
      endif
      fwrite(destfile,space(N_INDENT),N_INDENT)
   endif
   if indentlevel > 0
      fwrite(destfile,indent_str,indentlevel)
   endif
   if indent_on
      fwrite(destfile,space(indent_amt),indent_amt)
      if num_on
         indent_on := .f.
      endif
   endif
   if num_on
      fwrite(destfile,alltrim(sourceline),len(alltrim(sourceline)))
   else
      fwrite(destfile,trim(sourceline),len(trim(sourceline)))
   endif
   fwrite(destfile,EOL+LF,2)
   if indent_next
      indent_on := .t.
      indent_next := .f.
   endif
   if cblock != nil
      if cblock[2] != nil
         eval(cblock[2],N_INDENT)
      endif
   endif
   if !num_on
      com_off()
   endif
enddo
if n_switch
 fwrite(destfile,space(4),4)
 fwrite(destfile,space(N_INDENT),N_INDENT)
endif
fwrite(destfile,"/* END OF FILE */",21)
fwrite(destfile,EOL+LF,2)
fwrite(destfile,EOF,1)
return nil


static function parse_line(line)
/* Check for control block command */
local reply
local command := "",brace := 0,;
      paren := 0,spchr := 0,i
set exact on
private comm_array := {"IF","ELSE","ELSEI","ENDIF","ENDDO","DO","OTHER","ENDCA",;
                      "END","FOR","NEXT","FUNCT","STATI","RETUR","QUIT","/*"}
private inde_array := {{,{|n| lvlinc(n)}},;                   // IF
                      {{|n| lvldec(n)},{|n| lvlinc(n)}},;     // ELSE
                      {{|n| lvldec(n)},{|n| lvlinc(n)}},;     // ELSEIF
                      {{|n| lvldec(n)},},;                    // ENDIF
                      {{|n| lvldec(n)},},;                    // ENDDO
                      {,{|n| lvlinc(n)}},;                    // DO
                      {{|n| lvldec(n)},{|n| lvlinc(n)}},;     // OTHER
                      {{|n| lvldec(n)},},;                    // ENDCASE
                      {{|n| lvldec(n)},},;                    // END
                      {,{|n| lvlinc(n)}},;                    // FOR
                      {{|n| lvldec(n)},},;                    // NEXT
                      {{|n| chk_static(n)},},;                // FUNCTION
                      {{|n| chk_static(n)},},;                // STATIC
                      {{|n| ret_in()},{|n| ret_out()}},;      // RETURN
                      {{|n| ret_in()},{|n| ret_out()}},;      // QUIT
                      {{|n| com_on()},{|n| com_off()}}}       // COMMENT
command := get_comm(line)
if len(command) > 1
   i := ascan(comm_array,command)
   if i > 0
      reply := inde_array[i]
   endif
endif
if ";" $ line
      indent_next := .t.
   brace := at("{",line)
   paren := at("(",line)
   spchr := at(" ",line)
   if brace > 0 .and. paren > 0
      indent_amt := min(brace,paren) -1
   else
      indent_amt := max(brace,paren) -1
   endif
   if indent_amt < spchr .or. indent_amt > 30
      indent_amt := spchr
   endif
endif
return reply


static function get_comm(line)
/* search for valid command */
local reply := ""
local sarg := upper(substr(ltrim(line),1,5))
local i
i := at(" ",sarg)
if i > 0
   sarg := substr(sarg,1,i-1)
endif
i := ascan(comm_array,sarg)
if i > 0
   reply := comm_array[i]
endif
return reply


static function lvlinc(number)
/* process indentlevel increase request */
indentlevel := indentlevel + number
if l_switch
   indent_str := indent_str + "" + space(number-1)
else
   indent_str := indent_str + space(3)
endif
return nil


static function lvldec(number)
/* process indentlevel decrease request */
indentlevel := indentlevel - number
if indentlevel < 0
   indentlevel := 0
else
   indent_str := substr(indent_str,1,indentlevel)
endif
return nil


static function chk_static(number)
/* check for function declaration */
if "FUNC" $ upper(sourceline)
   indentlevel := iif(l_switch,2,0)
   indent_str := space(indentlevel)
endif
return nil


static function ret_in
/* process return statement */
local i,reply := ""
if l_switch
   if len(indent_str) > 1
      for i = 2 to len(indent_str)
         if substr(indent_str,i,1) = ""
            reply := reply + ""
         else
            reply := reply + ""
         endif
      next
   endif
   indent_str := reply
endif
return nil

static function ret_out
/* recover from return statement */
local i,reply := " "
if l_switch
   if len(indent_str) > 1
      for i = 2 to len(indent_str)
         if substr(indent_str,i,1) = ""
            reply := reply + ""
         else
            reply := reply + " "
         endif
      next
   endif
   indent_str := reply
endif
return nil

static function com_on
/* process comment line start request */
num_on := .f.
com_off()
return nil

static function com_off
/* check if comment lines are complete */
if at('*/',sourceline) > 0
   num_on := .t.
endif
return nil

static function disp_help()
/* help user if no parameters are passed */
? "    SYNTAX:  CLP_FRMT scriptfile.ext [switch] [switch]"
?
? "  SWITCHES:  /N => Include line numbers"
? "             /L => Include diagram lines"
?
inkey(0)
quit
return nil

static function disp_signon()
/* let user know who we are! */
? "CLP_FRMT.PRG"
?
? "Pretty Formatter and Action Diagammer For Clipper 5.0 Applications"
?
? "Author: Charles R. Wilson    CIS: 76307,2227"
?
? "Donated to Public Domain"
?
return nil
