/**************************************************************************/
/* The intent of this example program is to show how to store and         */
/* manipulate data in expanded memory using the C Memory Manager --       */
/* MEMLIB.                                                                */
/*                                                                        */
/* This example shows what developers need to do in order to manipulate   */
/* expanded memory in their own applications by using MEMLIB routines.    */
/* For a more detailed program that uses a doubly linked list data        */
/* structure, see the ROLODEX.C program.                                  */
/*                                                                        */
/* This program follows a seven-step algorithm:                           */
/*   1.  Check the total amount of expanded memory available.             */
/*   2.  Find the largest contiguous block of free expanded memory.       */
/*   3.  Allocate expanded memory for two strings.                        */
/*   4.  Calculate the size of the block allocated for the first string.  */
/*   5.  Print the two strings.                                           */
/*   6.  Manipulate the data by appending one string to the other.        */
/*   7.  Print the new string.                                            */
/**************************************************************************/

#include <stdio.h>
#include <string.h>

/**************************************************************************/
/* The two header files below contain the function prototypes for         */
/* MEMLIB routines and the error messages specific to MEMLIB.  Keep in    */
/* mind that MEMLIB calls EMMLIB functions, so you may get an error code  */
/* relating to EMS.  (EMS codes are explained in Table A-2 in the         */
/* EMS manual.)  You must include these two headers for any application   */
/* that will use MEMLIB.  If you are going to use any EMMLIB calls        */
/* directly, you must also include "emmlib.h."                            */
/**************************************************************************/

#include "memlib.h"   
#include "errors.h"

#define PASSED          0           /* If the MEMLIB call was successful */
                                    /* it will return a zero.            */
#define MAX_STRING_SIZE 255

void sample_abort (unsigned int);  /* If we encounter any errors, we    */
                                   /* call this function to "clean up." */

void main()
{
   unsigned int  status;           /* The status after a MEMLIB call.   */
   char          *string1;         /* Our first data item.              */
   char          *string2;         /* Our second data item.             */
   unsigned int  token1;           /* The identifier to our first       */
                                   /* block of expanded memory.         */
   unsigned int  token2;           /* The identifier to our second      */
                                   /* block of expanded memory.         */
   unsigned int  max_block_size;   /* The size of the largest           */
                                   /* allocatable contiguous block.     */
   unsigned int  size;             /* The size of one of our blocks.    */
   unsigned long max_exp_mem;      /* The amount of expanded mem avail. */
  

   /***********************************************/
   /* See how much total expanded memory we have. */
   /***********************************************/

   status = ememavl (&max_exp_mem);
   if (status != PASSED)
   {
      printf ("Unable to obtain total expanded memory size.\n");
      sample_abort (status);
   }
   else 
      printf ("Total expanded memory available: %lu\n", max_exp_mem);

   /*************************************************/
   /* Get the size of the largest contiguous block. */
   /*************************************************/

   status = ememmax (&max_block_size);
   if (status != PASSED)
   {
      printf ("Unable to obtain the largest contiguous block.\n");
      sample_abort (status);
   }
   else 
      printf ("Largest contiguous block: %u\n\n", max_block_size);

   /*********************************************/
   /* Allocate the memory and check for errors. */
   /*********************************************/

   status = efmalloc (MAX_STRING_SIZE, &token1);
   if (status != PASSED)
   {
      printf ("Unable to allocate first block.\n");
      sample_abort (status);
   }
   else
      printf ("Allocated space for the first block.\n");

   status = efmalloc (MAX_STRING_SIZE, &token2);
   if (status != PASSED)
   {
      printf ("Unable to allocate second block.\n");
      sample_abort (status);
   }
   else
      printf ("Allocated space for the second block.\n\n");

   /********************************/
   /* Check the size of block one. */
   /********************************/

   status = emsize (token1, &size);
   if (status != PASSED)
   {
      printf ("Unable to obtain size for block %u\n", token1);
      sample_abort (status);
   }
   else
      printf ("Size of block %u is %u\n\n", token1, size);

   /***********************************************/
   /* Map in the two blocks, checking for errors. */
   /***********************************************/

   status = set2eptrs (token1, token2, (void**) &string1,(void**) &string2);
   if (status != PASSED)
   {
      printf ("Unable to map in the two blocks.\n");
      sample_abort (status);
   }
   else
      printf ("Blocks mapped in.\n\n");

   printf ("Putting data in expanded memory.\n\n");

   /*********************************************/
   /* Store values in expanded memory using the */ 
   /* pointers given to us from set2eptrs().    */
   /*********************************************/

   strcpy (string1, "What if life is an illusion and nothing exists?\n");
   strcpy (string2, 
           " 'In that case, I definitely overpaid for my carpet.' --Woody Allen");

   printf ("String1: %s\n", string1);
   printf ("String2: %s\n\n", string2);
   
   /**************************************************/
   /* Still using the pointers, manipulate the data. */
   /**************************************************/

   printf ("Manipulating data in expanded memory.\n");

   strcat (string1, string2);

   printf ("String1: %s\n\n", string1);

   /*************************************************************/
   /* Free all expanded memory we've allocated.  If effreeall() */
   /* is unsuccessful, we don't want to call sample_abort()     */
   /* because sample_abort() would just call effreeall() again. */
   /*************************************************************/

   status = effreeall();
   if (status != PASSED)
   {
      printf ("Unable to free memory.\n");
      printf ("ERROR %X\n", status);
      exit (2);
   }
   else
      printf ("Expanded memory freed.\n");
}

/****************************************************************************/
/* Aborts the program with an error message.  Effreeall() frees all         */
/* pages, blocks, and handles associated with this application.  It is      */
/* imperative that you release all the expanded memory you've allocated     */
/* before exiting, so that other applications can use this memory.  If      */
/* you don't, you'll have to reboot to recover the lost memory.  Notice     */
/* also that a call to effreeall() checks the EMM status.  In case that     */
/* call fails,  you'll get an error condtion back to let you know the       */
/* memory was not freed.                                                    */
/*                                                                          */
/* You should always call effreeall() before aborting your program, even    */
/* if you get an error before allocating any memory.  Effreeall() will free */
/* the handles and memory that MEMLIB allocated when it was initialized.    */
/****************************************************************************/
void sample_abort (status)
unsigned int status;
{
   printf ("ERROR %X\n", status);
   status = effreeall();
   if (status != PASSED)
   {
      printf ("ERROR %X from effreeall().\n");
      printf ("Expanded memory not freed.\n");
   }
   exit (1);
}
