Lib.a = "qrytopal"
createlib Lib.a
;==============================================================================
;  Robert Hancock             all rights reserved 1990, 1991.
;  Sirguey-Hancock, Ltd.
;  Compuserve 74020,3446
;
;  May be distributed freely with attribution.
;
;  QueryToPal.u() reads the workspace and translates the query images into
;  PAL code, or, if the workspace is empty, you are prompted for a QuerySave
;  script which is played and then translated.
;
;  Process:    Place a query on the workspace
;              Play QRYTOPAL
;              Enter the name of the script file to contain output
;              Enter the name of the procedure to be generated
;              Watch it run
;
;  Note that this is a programmer's tool and does not contain extensive
;  error checking.
;
; Procedures in this script:
;
; QueryToPAL.u             Main routine
; Input.l                  Gets script and procedure name from user
; ImageInfo.n              Count images and load names of the tables to an array
; ProcBox.u                Write procedure box
; Header.u                 Write PROC Name(formal arguments)
; Ask.u                    Write the query ASK statement
; Translate.u              Translate query row to code
; FmtBSlash.a              Expands \ to \\
; WFile.u                  Write to script file
; Footer.u                 Status line
;==============================================================================

PROC QueryToPAL.u()
   PRIVATE fImgOrd.l,
   imgLim.n,
   qryCur.n,
   rowCur.n,
   cfldCurImg.n,
   bufCmt.a,
   filOut.a,
   rowLim.n,
   procName.a

   ;-----------------------------------------------------------------
   ; Get script and procedure names.
   ;-----------------------------------------------------------------
   IF NOT(Input.l()) THEN
      QUIT
   ENDIF

   ;-----------------------------------------------------------------
   ;  Global variables
   ;-----------------------------------------------------------------
   ARRAY tblName.r[NIMAGES()]     ;names of tables in query
   ARRAY tblVar.r[NIMAGES()]      ;variables that represent these tables

   ;-----------------------------------------------------------------
   ; Scan the workspace for queries
   ; create an array of table & variable names.
   ;-----------------------------------------------------------------
   imgLim.n = ImageInfo.n()

   ;-----------------------------------------------------------------
   ; Write comment box for top of procedure.
   ;-----------------------------------------------------------------
   ProcBox.u(imgLim.n)

   ;-----------------------------------------------------------------
   ; Generate procedure header.
   ;-----------------------------------------------------------------
   Header.u(imgLim.n)

   ;-----------------------------------------------------------------
   ; Loop thru all query images.
   ;-----------------------------------------------------------------
   FOR qryCur.n FROM 1 TO imgLim.n

      ;-----------------------------------------------------------------
      ; Generate code to ask for the table by variable.
      ;-----------------------------------------------------------------
      Ask.u()

      ;-----------------------------------------------------------------
      ; Loop thru each row of the current query image
      ;-----------------------------------------------------------------
      FOR rowCur.n FROM 1 TO rowLim.n
         ;-----------------------------------------------------------------
         ; If more than one row on the query form then insert a comment
         ; box to mark each row.
         ;-----------------------------------------------------------------
         IF (rowCur.n > 1) THEN
            bufCmt.a =  "   ;==============================================\n"
            +"   ; Line "+STRVAL(rowCur.n)+" of "
            +tblVar.r[qryCur.n]+" query image\n"
            +"   ;==============================================\n\n"
            WFile.u(filOut.a, bufCmt.a)
         ENDIF
         ;-----------------------------------------------------------------
         ; Generate PAL code for query row.
         ;-----------------------------------------------------------------
         Translate.u()
      ENDFOR

      ;-----------------------------------------------------------------
      ; Move cursor to home.
      ;-----------------------------------------------------------------
      WFile.u(filOut.a, "   HOME\n")

   ENDFOR

   ;-----------------------------------------------------------------
   ;   Write ENDPROC.
   ;-----------------------------------------------------------------
   WFile.u(filOut.a, "ENDPROC\n")

   ;-----------------------------------------------------------------
   ;   Write WRITELIB and RELEASE PROCS statements.
   ;-----------------------------------------------------------------
   WFile.u(filOut.a, "WRITELIB Lib.a "+procName.a+"\n")
   WFile.u(filOut.a, "RELEASE PROCS "+procName.a+"\n")

   QUIT
ENDPROC
writelib LIB.A QueryToPAL.u
release procs QueryToPAL.u



;=============================================================================

PROC Input.l()
   PRIVATE filQSave.a

   RETVAL = FALSE

   ;-----------------------------------------------------------------
   ;  Enter script name.
   ;-----------------------------------------------------------------
   CLEAR

   filOut.a = ""

   @ 2,0 ?? "Output file name (without .SC): "
   ACCEPT "A64" PICTURE "*!" REQUIRED TO filOut.a

   IF RETVAL = FALSE THEN
      QUIT
   ENDIF

   filOut.a = filOut.a + ".SC"

   RETVAL = FALSE

   ;-----------------------------------------------------------------
   ;  Enter procedure name.
   ;-----------------------------------------------------------------
   @ 4,0 ?? "Procedure name (without '()'): "
   ACCEPT "A64" PICTURE "!*@" REQUIRED TO procName.a

   IF RETVAL = FALSE THEN
      QUIT
   ENDIF

   ;-----------------------------------------------------------------
   ;  If no images are present, it assumes that you have captured
   ;  the query to a QuerySave script.
   ;-----------------------------------------------------------------
   IF (NIMAGES() = 0) THEN
      RETVAL = FALSE

      WHILE NOT RETVAL
         @ 6,0 ?? "Name of QuerySave script (without .SC): "
         ACCEPT "A64" PICTURE "*!" REQUIRED TO filQSave.a

         IF ISFILE(filQSave.a+".SC") THEN
            PLAY filQSave.a
            IF (NIMAGES() = 0) THEN
               BEEP BEEP BEEP
               Footer.u(filQSave.a + " cannot display query images")
               SLEEP 3000
               RETURN FALSE
            ENDIF

         ELSE
            BEEP BEEP BEEP
            Footer.u(filQSave.a + " cannot be found")
            SLEEP 3000
            RETURN FALSE
         ENDIF
      ENDWHILE
   ENDIF

   RETURN TRUE
ENDPROC
writelib LIB.A Input.l
release procs Input.l



;=============================================================================

PROC ImageInfo.n()
   PRIVATE imgCur.n

   ;-----------------------------------------------------------------
   ;  If query is being processed in "ImageOrder" set to True
   ;  else set to False.
   ;-----------------------------------------------------------------
   fImgOrd.l = (QUERYORDER() = "ImageOrder")

   imgCur.n = 0

   ;-----------------------------------------------------------------
   ;  Move to first image.
   ;-----------------------------------------------------------------
   MOVETO 1

   ;-----------------------------------------------------------------
   ;  Loop through all images on the workspace.
   ;-----------------------------------------------------------------
   WHILE IMAGETYPE() = "Query"
      imgCur.n = IMAGENO()
      tblName.r[imgCur.n] = TABLE()
      DOWNIMAGE

      tblName.a = ""
      ;chIllegal.a = "\\\":/|><"
      chIllegal.a = ":/|><"

      ;-----------------------------------------------------------------
      ;  Check for illegal chars.
      ;-----------------------------------------------------------------
      FOR i FROM LEN(tblName.r[imgCur.n]) TO 1 STEP -1
         chInvalid.a = SUBSTR(tblName.r[imgCur.n],I,1)

         IF SEARCH(chInvalid.a, chIllegal.a) > 0 THEN
            BEEP BEEP BEEP
            Footer.u("Illegal characters found in table name.")
            SLEEP 3000
            QUIT
         ENDIF

         tblName.a = chInvalid.a + tblName.a
      ENDFOR

      imgMax.n = imgCur.n

      ;-----------------------------------------------------------------
      ;  Create the table variable.
      ;-----------------------------------------------------------------
      tblVar.r[imgMax.n] = "tbl"+tblName.a+".a"

      ;-----------------------------------------------------------------
      ;  If this is the last image then quit the loop.
      ;-----------------------------------------------------------------
      IF (IMAGENO() = imgMax.n) THEN
         QUITLOOP
      ENDIF
   ENDWHILE

   RETURN (imgMax.n)
ENDPROC
writelib LIB.A ImageInfo.n
release procs ImageInfo.n


;=============================================================================

PROC ProcBox.u(imgLim.n)
   PRIVATE bufOut.a,
           i

   ;-----------------------------------------------------------------
   ;  Top border.
   ;-----------------------------------------------------------------
   bufOut.a = ";" + FILL("=", 78) + "\n"

   WFile.u(filOut.a, bufOut.a)

   ;-----------------------------------------------------------------
   ;  Standard attributes for each procedure.
   ;-----------------------------------------------------------------
   bufOut.a =   ";   Procedure\n;      "+procName.a+"()\n;\n"
               +";   Arguments\n;\n"

   WFile.u(filOut.a, bufOut.a)

   ;-----------------------------------------------------------------
   ;  The actual tables used and the variables assigned to them.
   ;-----------------------------------------------------------------
   WFile.u(filOut.a, ";      Variable Name       Table\n")

   WFile.u(filOut.a, ";      =============       =====================================\n" )

   FOR i FROM 1 TO imgLim.n
      WFile.u(filOut.a, ";      "
      +FORMAT("W20", tblVar.r[I])+FORMAT("W36", tblName.r[I])+"\n" )
   ENDFOR

   bufOut.a =    ";\n"
   +";   Function\n;\n;\n"
   +";   ReturnValues\n;\n;\n"
   +";   Comments\n"
   +";   --------\n"

   WFile.u(filOut.a, bufOut.a)

   ;-----------------------------------------------------------------
   ;  Who generated this script and when.
   ;-----------------------------------------------------------------
   bufOut.a = ";   Generated by user "+UPPER(USERNAME())+" on "
               + STRVAL(TODAY())
               + " at "+STRVAL(TIME())+".\n;\n"

   WFile.u(filOut.a, bufOut.a)

   WFile.u(filOut.a, ";   Script filename "+filOut.a+".\n;\n")

   ;-----------------------------------------------------------------
   ;  Bottom border.
   ;-----------------------------------------------------------------
   WFile.u(filOut.a,";" + FILL("=", 78) + "\n")

ENDPROC
writelib LIB.A ProcBox.u
release procs ProcBox.u



;=============================================================================

PROC Header.u(imgLim.n)
   PRIVATE bufArg.a,
           bufHdr.a

   bufArg.a = ""  ;assign initial values to allow concatenation

   ;-----------------------------------------------------------------
   ;  Loop through the tblVar array to get all variable names to
   ;  use as formal arguments.
   ;-----------------------------------------------------------------
   FOR i FROM 1 TO imgLim.n
      bufArg.a = bufArg.a + tblVar.r[i]
      IF (i < imgLim.n) THEN
         bufArg.a = bufArg.a+", "
      ENDIF
   ENDFOR

   bufHdr.a = "PROC "+ procName.a +"("+bufArg.a+")\n"

   WFile.u(filOut.a, bufHdr.a)

ENDPROC
writelib LIB.A Header.u
release procs Header.u



;=============================================================================

PROC Ask.u()
   PRIVATE bufAsk.a

   MOVETO tblName.r[qryCur.n]

   ;-----------------------------------------------------------------
   ; If you do not wish to pass the table names as formal
   ; arguments, you can enable this code.
   ;-----------------------------------------------------------------
   ;     bufAsk.a = "\n   MENU {Ask}\n\n   IF NOT ISASSIGNED("
   ;      +tblVar.r[qryCur.n]
   ;      +") THEN\n      SELECT \""
   ;      +FmtBSlash.a(tblName.r[qryCur.n])+"\"\n"
   ;      +"   ELSE\n      SELECT "
   ;      +tblVar.r[qryCur.n]
   ;      +"\n   ENDIF\n\n"

   ;-----------------------------------------------------------------
   ;  Place a comment box to separate multiple images.
   ;-----------------------------------------------------------------
   IF (imgLim.n > 1) THEN
      bufAsk.a =  "\n"+
      "   ;==============================================\n"
      +"   ; "+tblVar.r[qryCur.n]+" query image\n"
      +"   ;==============================================\n"
      WFile.u(filOut.a, bufAsk.a)
   ENDIF

   ;-----------------------------------------------------------------
   ;  Write the ASK statement.
   ;-----------------------------------------------------------------
   bufAsk.a = "\n   MENU {Ask} SELECT "
   +FmtBSlash.a(tblVar.r[qryCur.n])+"\n\n"

   WFile.u(filOut.a, bufAsk.a)

   CTRLHOME END                           ;place cursor in last row
   rowLim.n = ROWNO()

   HOME
ENDPROC
writelib LIB.A Ask.u
release procs Ask.u



;=============================================================================

PROC Translate.u()
   PRIVATE bufTrans.a,
           pctDone.a,
           lenWrap.n

   bufTrans.a = ""            ;initial value to allow concatenation

   lenWrap.n = 70             ;length at which to wrap field contents

   CTRLHOME

   ;-----------------------------------------------------------------
   ; Get number of fields in current query image.
   ;-----------------------------------------------------------------
   cfldCurImg.n = NFIELDS(tblName.r[qryCur.n])

   FOR fldCur.n FROM 1 TO NFIELDS(tblName.r[qryCur.n]) + 1
      fldName.a = FIELD()

      ;-----------------------------------------------------------------
      ;  Show percentage completed.
      ;-----------------------------------------------------------------
      pctDone.a = STRVAL(INT(((rowCur.n - 1) * cfldCurImg.n + fldCur.n - 1 )/
                  ((rowLim.n) * cfldCurImg.n)*100))

      Footer.u("Translation of " + tblName.r[qryCur.n] + " is "
               + pctDone.a +"% complete.")

      ;-----------------------------------------------------------------
      ;  Get field name and check mark status
      ;-----------------------------------------------------------------
      IF NOT (ISBLANK([]) AND ISBLANK(CHECKMARKSTATUS())) THEN
         IF (COLNO() > 1) THEN         ;a valid field
            IF fImgOrd.l THEN
               IF (ROWNO() = 1) THEN
                  bufTrans.a = "  RIGHT   WHILE FIELD() <>\""+fldName.a
                  +"\"  ROTATE  ENDWHILE\n"
               ELSE
                  bufTrans.a = "   MOVETO FIELD \""+fldName.a+"\"\n"
               ENDIF
            ELSE
               bufTrans.a = "   MOVETO FIELD \""+fldName.a+"\"\n"
            ENDIF

            WFile.u(filOut.a, bufTrans.a)
         ENDIF

         ;----------------------------------------------------------------
         ;  Write field contents
         ;-----------------------------------------------------------------
         IF NOT ISBLANK([]) THEN
            bufTrans.a = "   ["+fldName.a+"] = \""+FmtBSlash.a([])+"\"\n"
         ELSE
            bufTrans.a = ""
         ENDIF

         ;----------------------------------------------------------
         ;  If bufTrans.a longer than lenWrap.n, cut it up
         ;----------------------------------------------------------
         IF (LEN(bufTrans.a) / lenWrap.n >= 1) THEN
            WHILE (LEN(bufTrans.a) / lenWrap.n >= 1)
               WFile.u(filOut.a, SUBSTR(bufTrans.a, 1, lenWrap.n) +"\"+\n")
               bufTrans.a = "   \"" + SUBSTR(bufTrans.a, lenWrap.n + 1,
                              LEN(bufTrans.a) - lenWrap.n)
            ENDWHILE

            IF (SUBSTR(bufTrans.a, LEN(bufTrans.a), 1) <> "\n") THEN
               IF (SUBSTR(bufTrans.a, LEN(bufTrans.a), 1) <> "\"") AND
                  (SUBSTR(bufTrans.a, LEN(bufTrans.a) - 1, 1) <> "\\") THEN
                  bufTrans.a = bufTrans.a + "\"\n"
               ENDIF
            ENDIF
         ENDIF

         WFile.u(filOut.a, bufTrans.a)
         bufTrans.a = ""
      ENDIF

      ;-----------------------------------------------------------------
      ;  Write check mark status
      ;-----------------------------------------------------------------
      IF NOT ISBLANK(CHECKMARKSTATUS()) THEN
         bufTrans.a = bufTrans.a + "   " + CHECKMARKSTATUS() + "\n\n"
      ENDIF

      WFile.u(filOut.a, bufTrans.a)
      bufTrans.a = ""

      RIGHT
   ENDFOR

   IF (ROWNO() > rowLim.n) THEN
      WFile.u(filOut.a, "   CTRLHOME\n")
   ELSE
      WFile.u(filOut.a, "   CTRLHOME\n   DOWN\n\n")
   ENDIF
ENDPROC
writelib LIB.A Translate.u
release procs Translate.u



;=============================================================================

PROC FmtBSlash.a(bufSrc.a)
   PRIVATE bufDest.a,
           chTest.a,
           i

   bufDest.a = ""    ;assign initial value to allow concatenation

   FOR i FROM 1 TO LEN(bufSrc.a)
      chTest.a = SUBSTR(bufSrc.a, I, 1)
      SWITCH
         CASE chTest.a = "\\" :
            bufDest.a = bufDest.a+"\\\\"
         CASE chTest.a = "\"" :
            bufDest.a = bufDest.a+"\\\""
         OTHERWISE          :
            bufDest.a = bufDest.a+SUBSTR(bufSrc.a, I, 1)
      ENDSWITCH
   ENDFOR

   RETURN bufDest.a
ENDPROC
writelib LIB.A FmtBSlash.a
release procs FmtBSlash.a



;=============================================================================

PROC WFile.u(filWrite.a, bufText.a)

   PRINT FILE filWrite.a bufText.a
ENDPROC
writelib LIB.A WFile.u
release procs WFile.u


;=============================================================================

PROC Footer.u(Footer.a)

   CANVAS OFF
   IF(LEN(Footer.a) > 78) THEN
      Footer.a = SUBSTR(Footer.a, 1, 78)
   ENDIF

   PAINTCANVAS FILL "" ATTRIBUTE 51 24,0,24,79
   STYLE ATTRIBUTE 63   ;white on cyan
   @ 24, 0 ?? FORMAT("w80, ac", Footer.a)
   CANVAS ON
ENDPROC
writelib LIB.A Footer.u
release procs Footer.u
release vars LIB.A
reset
