;********************************************************************
;Procedure       : acceptplus()
;Memory Size     : 13972 bytes
;Description     : This procedure is a more extravagant version of the PAL
;                  ACCEPT command.  The ACCEPT command, of course, can only be
;                  exited by pressing [Esc] or [Enter].  This, of course, limits
;                  the command in terms of designing input screens/forms/popups
;                  because you can't jump between ACCEPT statements/prompts by
;                  pressing the arrow keys.
;
;                  ACCEPTPLUS acts just like the regular ACCEPT command in terms
;                  of basic functionality (ie. key presses).
;
;                  Before calling ACCEPTPLUS, you must initialize two arrays
;                  in your main calling procedure (the one that calls
;                  ACCEPTPLUS).  You can optionally initiate another array for
;                  help and fill popups.  The first array is called COORDINATES
;                  and must contain 3 pieces of information, separated by a comma:
;                  beginning row, beginning column and the field type (Paradox
;                  standard field type).  Example:
;
;                     array coordinates[3]       ;we'll have 3 fields to input
;                     coordinates[1] = "1,1,A5"  ;accept an alpha-5 field
;                     coordinates[1] = "2,1,N"   ;etc...
;                     coordinates[1] = "3,1,D"
;
;                   The 2nd array is called FIELDVALUE and will contain, after
;                   making the call to ACCEPTPLUS, the values the user typed
;                   in.  Example:
;
;                      array fieldvalue[arraysize(coordinates)]
;                                         
;                                         number of fields in COORDINATES[]
;
;                  The 3rd array is called HELPANDFILL, and must contain 6 pieces
;                  of information, separated by a comma: beginning row for popup,
;                  name of popup table, name of field in popup table, beginning
;                  row of popup, beginning column of popup, number of items to
;                  display in popup, heading to display across the top of the
;                  popup window.  If a particular occurance in the HELPANDFILL
;                  is not assigned or is blank, then that field will not have
;                  lookup capability, i.e. a beep will occur when F1 is pressed
;                  on that field and no popup will occur.  Example:
;
;                     array helpandfill[3]       ;we'll have 3 fields to input
;                     helpandfill[2] = "zipcode,zip,5,55,10,Zip Codes"
;                     helpandfill[3] = "states,state,15,40,5,State Codes"
;                       ;[1] will have no popup.
;                       ;[2] sets up a popup lookup menu for table zipcode, field zip,
;                       ;    at row 5, column 55, displaying 10 zips at a time, with
;                       ;    'Zip Codes' displayed across the top of the popup.
;                       ;[3] sets up a popup lookup menu for table states, field state,
;                       ;    at row 15, column 40, displaying 5 states at a time, with
;                       ;    'State Codes' displayed across the top of the popup.
;
;                     Note that the popup will automaticall be cleared when the user
;                     is through with it, so you should place it somewhere in an emply
;                     portion of your screen.
;
;                     Also note that the popup is generated via the POPUP2 scrip from
;                     the Paradox data entry toolkit.  The POPUP2 scrip follows the
;                     ACCEPTPLUS scrip.
;
;                   After these 3 arrays have been declared, it is now up to
;                   you to paint the canvas with you input screen.  *THEN* you
;                   are ready to call ACCEPTPLUS.
;
;                   If you wish to setup default values for certain fields on your
;                   screen, then just set that particular item in FIELDVALUE[] to the
;                   desired default value BEFORE you call ACCEPTPLUS.  It will then
;                   automatically display in that field when you call ACCEPTPLUS.
;                   This is also a handy way to save the users input values between
;                   interations of ACCEPTPLUS.
;
;                   In order to get the values the user typed in, they are
;                   stored in the array FIELDVALUE.  In order to check the
;                   value (if there is one), you must code as follows:
;
;        status = acceptplus()
;        if (status) then
;          if (not isassigned(fieldvalue[1]) or isblank(fieldvalue[1])) then
;            messenger("Field 1 = NOT ASSIGNED",2)
;          else
;            messenger("Field 1 = " + fieldvalue[1],2)
;          endif
;        endif
;
;                   This version of ACCEPTPLUS does no picture-type error
;                   checking.  Perhaps in a later version - it's going to be
;                   a bitch to do!
;Parameters      : none
;Returns Value?  : yes - false if [Esc] was pressed, true if [F2] is pressed
;
;Maintenance Log
;---------------
;
;Date           Author          Description
;----           ------          -----------
;??/??/??       Original        Original Idea
;11/05/91       TB Simpson      Added logic for popup windows, default field
;                               values
;

proc acceptplus()
 private actualfieldlength,  ;holds the length of the actual value in the field
         buffer,             ;while editing a field, this variable stores what the user types in - it is then dumped to a perm var
         colnum,             ;starting column where a field begins
         currentcol,         ;holds current "virtual" column we're in (the offset from variable colnum)
         definedfieldlength, ;holds the length of the field itself as defined by variable fieldspec
         fieldcounter,       ;counter for number of field we're in - use in FOR loops
         fview,              ;true if in field view, false otherwise
         newfield,           ;true if we're going to a new field, false otherwise
         overwrite,          ;true if in overwrite mode in field view, false otherwise
         rownum,             ;starting row where a field begins
         temp                ;temp var for anything that needs it

  cursor off                               ;don't need cursor yet, so hide it
  style reverse                            ;input fields will be like regular ACCEPT input fields
  numcoordinates = arraysize(coordinates)  ;figure out number of fields to prompt
  array actualfieldlength[numcoordinates]  ;set up arrays for proper number of fields
  array definedfieldlength[numcoordinates] ;
  fview = false                            ;initialize
  overwrite = false                        ;
  newfield = true                          ;

  for fieldcounter from 1 to numcoordinates            ;go through and define each field on the screen
    if (not isassigned(fieldvalue[fieldcounter]) or isblank(fieldvalue[fieldcounter])) then
       actualfieldlength[fieldcounter] = 0             ;initialize for use later for when user presses [Enter]
       fieldvalue[fieldcounter] = ""
    else
       actualfieldlength[fieldcounter] = len(fieldvalue[fieldcounter])
   endif
    retval = match(coordinates[fieldcounter],"..,..,..",rownum,colnum,fieldspec) ;figure out start row, col and field type
    switch
      case (substr(fieldspec,1,1) = "A"):
        retval = match(fieldspec,"A..",temp)            ;gives number as a string
        definedfieldlength[fieldcounter] = numval(temp) ;convert the string to a number
      case (fieldspec = "N" or
            fieldspec = "$"):
        definedfieldlength[fieldcounter] = 23
      case (fieldspec = "S"):
        definedfieldlength[fieldcounter] = 5
      case (fieldspec = "D"):
        definedfieldlength[fieldcounter] = 8
    endswitch
    @ numval(rownum),numval(colnum) ?? spaces(definedfieldlength[fieldcounter]) ;place the fields on the screen
    @ numval(rownum),numval(colnum) ?? fieldvalue[fieldcounter]                 ;display default values, if any
  endfor

  fieldcounter = 1                      ;prime counter for first field
  cursor normal                         ;bring back the cursor
  while (true)                          ;loop until [F2] or [Esc]
    if (newfield) then                  ;are we going to a new field?
      newfield = false                  ;reset for future use
      retval = match(coordinates[fieldcounter],"..,..,..",rownum,colnum)  ;get row, column
      rownum = numval(rownum)                                   ;convert to number
      colnum = numval(colnum)                                   ;
      if (isassigned(fieldvalue[fieldcounter])) then            ;this field already has a value
        buffer = fieldvalue[fieldcounter]                       ;place the current contents of the field in the buffer
        currentcol = actualfieldlength[fieldcounter] + 1        ;figure out where cursor should be
        @ rownum,colnum ?? buffer                               ;place the contents of the buffer
      else
        buffer = ""                                             ;new field, so reset buffer
        currentcol = 1                                          ;reset
        @ rownum,colnum                                         ;place cursor
      endif
    else
      @ rownum,colnum + currentcol - 1
    endif
    ch = getchar()                  ;get a character from the keyboard
    switch                          ;what character was pressed?
      case (ch = -60): ;[F2] - exit this routine
        if (fview) then                      ;disallow this key while in field view mode
          beep
        else
          fieldvalue[fieldcounter] = buffer  ;save current field value
          return true                        ;exit acceptplus()
        endif
      case (ch = 27): ;[Esc]
        if (fview) then                      ;disallow this key while in field view mode
          beep
        else
          return false
        endif
      case (ch = -59): ;[F1]
        if (fview) then                      ;disallow this key while in field view mode
          beep
        else
          if (not isassigned(helpandfill[fieldcounter]) or isblank(helpandfill[fieldcounter])) then
             Retval = ""
             beep
          else
             retval = match(helpandfill[fieldcounter],"..,..,..,..,..,..",poptbl,popfld,poprowx,popcolx,popsizex,pophead) ;figure out passed variables for popup
             poprow = numval(poprowx)
             popcol = numval(popcolx)
             popsize = numval(popsizex)
             Menu
             SetPopup2(poptbl,popfld)
             Popup2(poprow, popcol, popsize, 1, pophead,"Press Enter to select item from popup window, or","Press Esc to cancel")
             if not IsBlank(Retval) then
                buffer = Retval
                currentcol = len(buffer)+1
                Style Reverse
                @ rownum,colnum ?? Retval
                @ rownum,colnum + len(buffer)+1
             endif
             Style
             poplast = poprow + popsize + 3
             while (true)
                @ poprow,popcol Clear EOL
                poprow = poprow + 1
                if poprow > poplast then
                   quitloop
                endif
             endwhile
             @ 0,0 clear EOL
             @ 1,0 clear EOL
             Style Reverse
          endif
        endif
      case (ch > 31 and ch < 127): ;all real/valid alpha characters
        if (currentcol = definedfieldlength[fieldcounter] + 1) or  ;is the user in the last column?
           (len(buffer) = definedfieldlength[fieldcounter]) then   ;has the user typed in the max number of chars?
          beep
        else
          @ rownum,colnum + currentcol - 1 ?? chr(ch)                         ;place the char the user typed in
          switch
            case (overwrite):
              temp = substr(buffer,currentcol + 1,len(buffer) - currentcol)   ;get part of string that is after the cursor
              buffer = substr(buffer,1,currentcol - 1)                        ;get part of string before the cursor
              buffer = buffer + chr(ch) + temp                                ;append char to middle of string
            case (fview):
              temp = substr(buffer,currentcol,len(buffer) - currentcol + 1)   ;get part of string that is on *and* after the cursor
              buffer = substr(buffer,1,currentcol - 1)                        ;get part of string before the cursor
              buffer = buffer + chr(ch) + temp                                ;append char to middle of string
              @ rownum,colnum + currentcol ?? temp                            ;place the string from above after the new char
            otherwise:
              buffer = buffer + chr(ch)                                       ;append the char to the string so far
          endswitch
          currentcol = currentcol + 1                                         ;we've advanced 1 column
        endif
      case (ch = 8): ;[Backspace]
        if (currentcol = 1 or fview) then             ;if we're already in the first column or in field view
          beep
        else
          currentcol = currentcol - 1                 ;move cursor back one char
          buffer = substr(buffer,1,len(buffer) - 1)   ;get rid of last char in variable buffer
          @ rownum,colnum + currentcol - 1 ?? " "     ;erase the previous char
        endif
      case (ch = 127): ;[Ctrl][Backspace]
        if (fview) then                                 ;disallow this key while in field view mode
          beep
        else
          @ rownum,colnum ?? spaces(definedfieldlength[fieldcounter])
          fieldvalue[fieldcounter] = ""                 ;empty out the field/buffer
          actualfieldlength[fieldcounter] = 0           ;set length to 0
          newfield = true
        endif
      case (ch = 13 or ;[Enter]
            ch = 9 or  ;[Tab]
            ch = -80): ;[Down]
        if (fview and ch = 13) then                         ;are we in field view mode? only exit if [Enter] is pressed
          cursor normal                                     ;reset the cursor
          fview = false                                     ;reset also
          overwrite = false
          currentcol = len(buffer) + 1                      ;set cursor at end of field/buffer
        else
          if (fview and (ch = 9 or ch = -80)) then          ;can't leave field view by pressing [Down] or [Tab]
            beep
          else
            if (fieldcounter = numcoordinates) then         ;are we on the last field?
              beep
            else
              fieldvalue[fieldcounter] = buffer             ;save contents of field before moving
              actualfieldlength[fieldcounter] = len(buffer) ;save length of actual field contents
              fieldcounter = fieldcounter + 1
              newfield = true
            endif
          endif
        endif
      case (ch = -72 or ;[Up]
            ch = -15):  ;[ShiftTab]
        if (fview) then
          beep
        else
          if (fieldcounter = 1) then
            beep
          else
            fieldvalue[fieldcounter] = buffer                 ;save contents of field before moving
            actualfieldlength[fieldcounter] = len(buffer)     ;save length of actual field contents
            fieldcounter = fieldcounter - 1
            newfield = true
          endif
        endif
      case (ch = -77): ;[Right]
        if (fview) then
          if (currentcol = len(buffer) + 1) then                ;don't allow user to go past the space after the last char
            beep
          else
            currentcol = currentcol + 1                         ;move over to the right one char
          endif
        else
          if (fieldcounter = numcoordinates) then               ;are we on the last field?
            beep
          else
            fieldvalue[fieldcounter] = buffer                   ;save contents of field before moving
            actualfieldlength[fieldcounter] = len(buffer)       ;save length of actual field contents
            fieldcounter = fieldcounter + 1
            newfield = true
          endif
        endif
      case (ch = -75): ;[Left]
        if (fview) then
          if (currentcol = 1) then                            ;don't allow the user to go before the beginning of the field
            beep
          else
            currentcol = currentcol - 1                       ;move back one char
          endif
        else
          if (fieldcounter = 1) then                          ;are we on the first field?
            beep
          else
            fieldvalue[fieldcounter] = buffer                 ;save contents of field before moving
            actualfieldlength[fieldcounter] = len(buffer)     ;save length of actual field contents
            fieldcounter = fieldcounter - 1
            newfield = true
          endif
        endif
      case (ch = -71): ;[Home]
        if (fview) then                                     ;go to beginning of field
          currentcol = 1
        else
          fieldvalue[fieldcounter] = buffer                 ;save contents of field before moving
          actualfieldlength[fieldcounter] = len(buffer)     ;save length of actual field contents
          fieldcounter = 1                                  ;point to first field
          newfield = true
        endif
      case (ch = -79): ;[End]
        if (fview) then                                     ;go to end of field
          currentcol = len(buffer) + 1                      ;set cursor at end of field/buffer
        else
          fieldvalue[fieldcounter] = buffer                 ;save contents of field before moving
          actualfieldlength[fieldcounter] = len(buffer)     ;save length of actual field contents
          fieldcounter = numcoordinates                     ;set to last field
          newfield = true
        endif
      case (ch = 6): ;[Ctrl F] (fieldview)
        fview = true                          ;we're now in field view
        cursor box                            ;box cursor signifies we're in field view mode
      case (ch = -83): ;[Del]
        if (fview) then                                                   ;only allow this key during field view mode
          retval = substr(buffer,currentcol + 1,len(buffer) - currentcol) ;get part of string that is after the cursor
          buffer = substr(buffer,1,currentcol - 1)                        ;get part of string before the cursor
          @ rownum,colnum + currentcol - 1 ?? retval + " "                ;trailing space deletes last char on line
          buffer = buffer + retval                                        ;append the head of the string to the new tail
        else
          beep
        endif
      case (ch = -82): ;[Ins]
        if (fview) then                ;only allow this key during field view mode
          overwrite = not overwrite    ;user may toggle this back and forth
          if (overwrite) then
            cursor normal              ;normal cursor in middle of string signifies overwrite/field view mode
          else
            cursor box                 ;overwrite mode is off
          endif
        else
          beep
        endif
      otherwise:                       ;all other keys, just beep
        beep
    endswitch
  endwhile
endproc


; Copyright (c) 1988, 1989 Borland International.  All Rights Reserved.
;
; General permission to re-distribute all or part of this script is granted,
; provided that this statement, including the above copyright notice, is not
; removed.  You may add your own copyright notice to secure copyright
; protection for new matter that you add to this script, but Borland
; International will not support, nor assume any legal responsibility for,
; material added or changes made to this script.
;
; Revs.:  DCY 12/15/88
; ****************************************************************************
; SetPopup2 initializes variables required by Popup2 from data stored in a
; table.  It requires a table name and a field name from which to read menu
; item information.  Basically, it views and scans the given table, defining
; menu items as elements within an array.  It also determines the widest
; element of the array (not necessarily the width of the field), assigning it
; to another variable also required by Popup2.
;
Proc SetPopup2(PopTbl,Fld)
;  Private;PopTbl,      ;Source table for items of menu
          ;Fld,         ;Source field for items of menu
;  Global ;Item,        ;Array of items of menu
          ;Width        ;Width of widest item

   Array Item[NRecords(PopTbl)]    ;Dimension Item array.  One item per record
   View PopTbl                     ;  in PopTbl.
   MoveTo Field Fld
   Width = 0
   If Search("A",FieldType()) = 0  ;If field is non-alphanumeric, convert it
      Then Scan                    ; to a string value before assigning it
              Item[[#]] = Strval([])
              Width = Max(Len(Item[[#]]),Width)  ;Update max. width
           Endscan
      Else Scan
              Item[[#]] = []
              Width = Max(Len([]),Width)
           Endscan
   Endif

Endproc

; Popup2 displays a similar popup-style menu to that of Popup.  However, it
; does have some enhancements and restrictions compared to Popup.  For
; example, with Popup2, a user can move to a selection by just pressing the
; first letter of a selection.  It there are no selections which begin with
; that first letter below the current menu position, Popup2 sounds a beep.
; You can also instruct Popup2 to highlight a specific menu item when it
; first displays a menu.  Popup2 also allows you to specify a title for the
; menu box as well as a custom top-two line prompt.  Lastly, it includes color
; support (you can define your own color set, see the code below).
;
; Unlike Popup, however, Popup2 stores only one menu item list in its menu
; item array (named Item).  You can either use SetPopup2 to fill the Item
; array from records in a table or you can define and fill the array
; yourself.  If you do so, you must assign the variable Width a value equal
; to the number of characters of the widest item in the Item array.  Because
; Popup2 stores only one menu item list at a time in the Item array, you'll
; need to redimension and reassign Item and Width each time you wish to
; display a different menu.  However, depending upon the size and number of
; different menus you wish to display, the resultant memory savings should
; more than balance the slight performance degradation.
;
; Popup2 dynamically centers the menu box title such that the entire title
; will always be displayed.  Note, however, that for performance Popup2 does
; no special error handling.  Thus you should ensure that the arguments you
; give it are indeed valid, i.e., the canvas coordinates must allow the entire
; menu to fit on the screen, the default item number must be within the legal
; range, etc.

Proc Popup2(R,C,VNum,DefItem,Title,Prompt1,Prompt2)
   Private;R,           ;Row position of upper-left corner of menu box
          ;C,           ;Column position of upper-left corner menu box
          ;VNum,        ;Number of items to be displayed in one menu image
          ;DefItem,     ;Item (number) to show
          ;Title,       ;Title of popup box
          ;Prompt1,     ;First prompt line
          ;Prompt2,     ;Second prompt line
           NItems,      ;Number of items in menu list
           Char,        ;Keycode of last key pressed
           MenuPos,     ;Current (row image) position within menu
           CIndex,      ;Current choice index into Item
           X,           ;Counter variable
           PrmptColr,   ;Color attribute for prompt
           BrdrColr,    ;Color attribute for box border
           ListColr,    ;Color attribute for menu item list
           SlctColr     ;Color attribute for current menu selection
;  Global ;Item,        ;Array of items of menu
          ;Width        ;Width of widest item

   Echo Off                     ;Freeze workspace image
   Cursor Off                   ;Hide blinking cursor
   Canvas Off                   ;Disable immediate printing to canvas

   PrmptColr = SysColor(0)      ;Top two line prompt color
   BrdrColr = SysColor(9)       ;Border color
   ListColr = SysColor(17)      ;Menu list color
   SlctColr = SysColor(18)      ;Current menu selection color

   Style Attribute PrmptColr
   @ 0,0             ;Display prompt information
   ?? Spaces(80)+Prompt2+Spaces(80-Len(Prompt2))
   @ 0,0
   ?? Prompt1

   Width = Max(Len(Title),Width)  ;Expand box width if title is too wide
   NItems = ArraySize(Item)     ;Set number of items in list
   If VNum > NItems
      Then VNum = NItems
   Endif

   Switch
      Case DefItem < VNum :  ;Redraw top screen
         Redraw = 0
         MenuPos = DefItem
      Case DefItem > NItems-VNum :  ;Redraw last screen
         Redraw = NItems-VNum
         MenuPos = DefItem-NItems+VNum
      Otherwise :      ;Redraw intermediate screen
         Redraw = DefItem-1  ;Place item at top of menu
         MenuPos = 1
   Endswitch

                                ;Set default menu settings:
   CIndex = DefItem                     ;First menu item
   LastPos = MenuPos                    ;Last image position is current pos.
   LastIdx = CIndex                     ;Last menu item index

   Style Attribute BrdrColr
   SetMargin C
   @ R,C                        ;Draw menu skeleton and initial image
   ?? "",Fill("",Width+2),""
    ? " ",Format("AC,W"+Strval(Width),Title)," "
    ? "",Fill("",Width+2),""
   For X From 1 To VNum
    ? "",Spaces(Width+2),""
   Endfor
    ?  "",Fill("",Width+2),""

   @ R+3,C+1
   If Redraw = 0                     ;Records above?
      Then ?? " "
      Else ?? ""                    ; Yes- Show items are above
   Endif
   @ R+VNum+2,C+1
   If Redraw+VNum = NItems           ;Records below?
      Then ?? " "
      Else ?? ""                    ; Yes- Show items are below
   Endif

   @ R+2,C+2
   SetMargin C+2
   Style Attribute ListColr
   For X From 1 To VNum
    ? Item[X+Redraw]+Spaces(Width-Len(Item[X+Redraw]))
   Endfor
   SetMargin Off

   Canvas On                    ;Reenable immediate echoing to canvas

   Redraw = -1                       ;Disable menu image redraw

   While True

      Style Attribute SlctColr
      @ MenuPos+R+2,C+2         ;Highlight current selection
      ?? Item[CIndex]
      Style Attribute ListColr

      Char = getchar()

      Switch
         Case Char > 31:                        ;First character search?
            For X From CIndex+1 to NItems       ;Search (down) item array
               If Upper(Substr(Item[X],1,1)) = Upper(Chr(Char))
                  Then QuitLoop                 ;Found a match
               Endif
            Endfor
            If X = NItems+1                     ;Match not found
               Then Beep
               Else If MenuPos+X-CIndex > VNum  ;Is next item already visible?
                       Then Switch              ; No-
                               Case X < VNum :  ;Redraw top screen
                                  Redraw = 0
                                  MenuPos = X
                               Case X > NItems-VNum :  ;Redraw last screen
                                  Redraw = NItems-VNum
                                  MenuPos = X-NItems+VNum
                               Otherwise :      ;Redraw intermediate screen
                                  Redraw = X-1  ;Place item at top of menu
                                  MenuPos = 1
                            Endswitch
                       Else MenuPos = MenuPos+X-CIndex
                            Redraw = -1
                    Endif
                    CIndex = X                  ;Update current item
            Endif
         Case Char = -72 :                      ;Up
            If CIndex = 1                       ;Already at first item?
               Then Beep
               Else If MenuPos > 1              ;Can move within menu image?
                       Then MenuPos = MenuPos-1 ; Yes- Move to previous item
                       Else Redraw = CIndex-2   ; No-  Redraw entire menu
                    Endif
                    CIndex = CIndex-1           ;Update current item
            Endif
         Case Char = -80 :                      ;Down
            If CIndex = NItems                  ;On last item?
               Then Beep
                    Loop
               Else If MenuPos < VNum           ;Can move within menu image?
                       Then MenuPos = MenuPos+1    ; Yes-  Move to next item
                       Else Redraw = CIndex-VNum+1 ; No- Redraw entire menu
                    Endif
                    CIndex=CIndex+1             ;Update current item
            Endif
         Case Char = -71 :                      ;Home
            If MenuPos <> CIndex                ;Already viewing top of menu?
               Then Redraw = 0                  ; No-  Redraw top of menu
            Endif
            MenuPos = 1                         ;Position at first item
            CIndex = 1                          ;Select first item
         Case Char = -79 :                      ;End
            If CIndex+VNum-MenuPos <> NItems    ;Already viewing end of menu?
               Then Redraw = NItems-VNum        ; No-  Redraw end of menu
            Endif
            MenuPos = VNum                      ;Position at bottom of menu
            CIndex = NItems                     ;Select last item
         Case Char = -73 :                      ;PgUp
            If MenuPos = CIndex                 ;Are we within first screen?
               Then Beep                        ; Yes- Disallow PgUp
               Else If CIndex-MenuPos-VNum > 0
                       Then CIndex = CIndex-MenuPos-VNum+1
                       Else CIndex = 1
                    Endif
                    Redraw = CIndex-1           ; No-  Redraw previous page
                    MenuPos = 1                 ;      Position on that item
            Endif
         Case Char = -81 :                      ;PgDn
            If CIndex+VNum-MenuPos = NItems ;Are we within last screen?
               Then Beep                        ; Yes- Disallow PgDn
               Else If NItems-VNum < CIndex+VNum-MenuPos
                       Then CIndex = NItems-VNum+1
                       Else CIndex = CIndex+VNum-MenuPos+1
                    Endif
                    Redraw = CIndex-1           ; No- Redraw next page
                    MenuPos = 1                 ;     Position on that item
            Endif
         Case Char = 13 :                       ;Enter
            Cursor Normal
            Style
            Return Item[CIndex]                 ;Return selection
         Case Char = 27 :                       ;Esc
            Cursor Normal
            Style
            Return ""                           ;Return null selection
         Otherwise:                             ;Illegal key
            Beep
      Endswitch

      If Redraw = -1                            ;Need to redraw entire menu?
         Then @ LastPos+R+2,C+2
              ?? Item[LastIdx]
         Else Canvas Off                        ;Disable immediate canvas echo
              SetMargin C+2
              @ R+2,C+2
              For X from 1 to VNum              ;Redraw entire menu box
                 ? Item[Redraw+X]+Spaces(Width-Len(Item[Redraw+X]))
              Endfor
              SetMargin Off
              @ R+3,C+1
              Style Attribute BrdrColr
              If Redraw = 0                     ;Records above?
                 Then ?? " "
                 Else ?? ""                    ; Yes- Show items are above
              Endif
              @ R+VNum+2,C+1
              If Redraw+VNum = NItems           ;Records below?
                 Then ?? " "
                 Else ?? ""                    ; Yes- Show items are below
              Endif
              Canvas On                         ;Enable immediate canvas echo
              Redraw = -1
      Endif

      LastPos = MenuPos                         ;Update last row position
      LastIdx = CIndex                          ;Update last item index

   Endwhile

Endproc
