program OSTest;

{****************************************************************
  This program contains five sample routines which show you how
  to use the routines from WINCATOS.TPU
  Before running the program, scroll to the MAIN section at the end
  of this file ; specify the name of the catalog you want to access
  and select one of the 5 example routines by uncommenting them 
****************************************************************}

uses WinCatOS, Strings, WinCRT;


{****************************************************************
Print catalog statistics
****************************************************************}
procedure Example0;
begin
  writeln(StrPas(CIB^.CatalogName));
  writeln('Catalog size       : ',CIB^.CatalogSize);
  writeln('Total volumes      : ',CIB^.Total_Volumes);
  writeln('Total files        : ',CIB^.Total_Files);
  writeln('Total used space   : ',CIB^.Total_Used);
  writeln('Total free space   : ',CIB^.Total_Free);
end;


{****************************************************************
This example shows how to access all groups, locations and
owners as defined in the catalog.
****************************************************************}
procedure Example1;
var
I : integer;
begin
  writeln('Groups :');
  Grp := CIB^.Grp_Table;
  for I := 1 to CIB^.Total_Groups do
  begin
    write('  ',StrPas(Grp^.grp));
    Grp := Grp^.link;
  end;
  writeln;
  writeln('Locations :');
  Grp := CIB^.Loc_Table;
  for I := 1 to CIB^.Total_Locations do
  begin
    write('  ',StrPas(Grp^.grp));
    Grp := Grp^.link;
  end;
  writeln;
  writeln('Owners :');
  Grp := CIB^.Own_Table;
  for I := 1 to CIB^.Total_Owners do
  begin
    write('  ',StrPas(Grp^.grp));
    Grp := Grp^.link;
  end;
  writeln;
end;


{****************************************************************
This example shows how to use the SortVolumeTable and SelectVolumes
services.
It will print all volumes belonging to group Windows and with a minimum
free space of 100 KB, sorted by descending free space.
****************************************************************}
procedure Example2;
var
I      : integer;
begin
  { sort volumes by descending free space }
  SortVolumeTable(CIB,VSO_FreeSpaceD);

  { set volume selection criteria }
  StrLCopy(CIB^.SV_Group,'Windows',8);
  CIB^.SV_FreeSpace := 100;

  { select volumes }
  SelectVolumes(CIB,VSL_Select);

  { print selected volumes }
  for I := 1 to CIB^.Total_Volumes do
  begin
    Vol := CIB^.Volume_Table^[I];
    if Vol^.Sel = true then
    begin
      writeln(StrPas(Vol^.VRec.lbl),        { disklabel}
        '   ',StrPas(Vol^.WRec.com),        { comment }
        '   ',Vol^.VRec.fre,                { free space }
        '   ',Vol^.VRec.fil,                { nr. of files }
        '   ',GetDiskType(Vol^.VRec.med));  { disk type }
    end;
  end;

  { print group statistics }
  writeln('Volumes      : ',CIB^.SV_Total_Volumes);
  writeln('Files        : ',CIB^.SV_Total_Files);
  writeln('Used space   : ',CIB^.SV_Total_Used);
  writeln('Free space   : ',CIB^.SV_Total_Free);

  { clear current volume selection }
  SelectVolumes(CIB,VSL_Clear);
end;


{****************************************************************
This example shows how to use the SortFileTable and SelectFiles
services.
It will print all .ZIP files with string WIN in the filename,
in ascending size order.
****************************************************************}
procedure Example3;
var
I : integer;
begin
  { set file selection criteria }
  StrLCopy(CIB^.SF_Filename,'WIN',8);
  {StrLCopy(CIB^.SF_Comment,'DISK',20);}
  StrLCopy(CIB^.SF_Extension,'ZIP',3);

  { select files }
  SelectFiles(CIB,FSL_Select);

  { sort the selected files by size, ascending }
  SortFileTable(CIB,FSO_SizeA);

  { print selected files }
  for I := 1 to CIB^.SF_Total_Files do
  begin
    Fil := CIB^.SF_File_Table^[I];
    Vol := Fil^.VPtr; { get pointer to file's volume }
    Pat := Fil^.PPtr; { get pointer to file's directory }
    writeln(StrPas(Fil^.FRec.nam),
      '   ',ConvDate(Fil^.FRec.dat,DF_ddmmyy),
      '   ',FFormat(Fil^.FRec.siz),
      '   ',StrPas(Vol^.VRec.lbl),
      '   ',StrPas(Pat^.PRec.pat)); 
  end;

  writeln(CIB^.SF_Total_Files,' files found');

  { clear file selection }
  SelectFiles(CIB,FSL_Clear);
end;


{****************************************************************
This example shows how to access and make a selection of all files
in the catalog, without using the SelectFiles service.
****************************************************************}
procedure Example4;
var
I,J      : integer;
begin
  { for all volumes do }
  for I := 1 to CIB^.Total_Volumes do                      
  begin
    { get pointer to volume }
    Vol := CIB^.Volume_Table^[I];
     
    if Vol^.Sel = true then       { we didn't select volumes, so this is always true } 
    begin
      { get pointer to first entry in this volume's file chain }
      Fil := Vol^.FPtr;

      { for all files on this volume do }
      for J := 1 to Vol^.Vrec.fil do
      begin
        { here we can make a selection }
        if StrComp(Fil^.FRec.nam+8,'ZIP') = 0 then       { find specific extension }
        { find files by size :   
        if Fil^.FRec.siz > 5000000 then }                  
        { find specific file : 
        if StrComp(Fil^.FRec.nam,'COMMAND COM') = 0 then} 
        { find files with strig WIN in filename : 
        if StrPos(Fil^.FRec.nam,'WIN') <> nil then }
        { find files with string DISK in comment field :
        if Fil^.Com <> nil then
          if StrPos(StrUpper(StrCopy(work,Fil^.Com)),'DISK') <> nil then }

        begin
          { get pointer to this file's directory entry }
          Pat := Fil^.PPtr;
          { now Vol points to the file's volume 
                Pat points to the file's directory 
                Fil points to the file }
          writeln(StrPas(Fil^.FRec.nam),                   { filename }
            '   ',ConvDate(Fil^.FRec.dat,DF_ddmmyy),       { date modified }
            '   ',ConvTime(Fil^.FRec.tim),                 { time modified }
            '   ',FFormat(Fil^.FRec.siz),                  { size }
            '   ',StrPas(Vol^.VRec.lbl),                   { volume }
            '   ',StrPas(Pat^.PRec.pat));                  { directory }

          { the following code prints the file's description.
            Because the description can be up to 500 characters in length
            we must truncate it when using 'StrPas', which only supports
            strings up to 255 characters in length } 
          {
          if Fil^.com <> nil then
          begin
            StrLCopy(work,Fil^.com,100);
            writeln(StrPas(work));
          end;
          }
        end;

        { point to next entry in file chain }
        Fil := Fil^.Link; 
      end;
    end;
  end;
end;


{****************************************************************
This example shows how to access all directories in the catalog
****************************************************************}
procedure Example5;
var
I,J : integer;
begin
  for I := 1 to CIB^.Total_Volumes do
  begin
    Vol := CIB^.Volume_Table^[I]; 
    if Vol^.Sel = true then        
    begin
      writeln(StrPas(Vol^.VRec.lbl));

     { get pointer to first entry from this volume's directory chain }
      Pat := Vol^.PPtr;

      { for all directories on this volume do }
      for J := 1 to Vol^.Vrec.pat do
      begin
        writeln('--',StrPas(Pat^.PRec.pat),          { directory name }
                '  ',FFormat(Pat^.PRec.fil),         { nr. of files in dir. }
                '  ',FFormat(Pat^.PRec.siz));        { total size of files in dir. }
                 
        {***********************************************************
         you can access all files in this directory like this :
          
        Fil := Pat^.trd;                              
        while Fil <> nil do
        begin
          writeln('-----',StrPas(Fil^.FRec.nam);
          Fil := Fil^.trd;
        end;
        ***********************************************************}

        { point to next entry in directory chain }
        Pat := Pat^.link; 
      end;
    end;
  end;
end;





{****************************************************************
Main 
****************************************************************}
begin
  writeln('Begin');
  { Create Catalog Information Block }
  CIB := CreateCIB;

  { Set name of catalog to open }
  StrCopy(CIB^.CatalogName,'C:\WINDOWS\WINCAT\SAMPLE.CAT');

  { Open, read and close catalog }
  writeln('Loading catalog...');
  ReadCatalog(CIB);

  { Check if open OK }
  if CIB^.Status <> 0 then
  begin
    writeln('Unable to read catalog, RC=',CIB^.Status);
    PurgeCIB(CIB);
    exit;
  end;
  writeln('Catalog loaded');

  { Catalog is now loaded in storage ; let's do some queries }
  
  {Example0;}    { Print catalog statistics }
  {Example1;}    { List groups, locations and owers }
  {Example2;}    { Sort and select volumes }
  Example3;    { Select and sort files }
  {Example4;}    { Access all files in catalog }
  {Example5;}    { Access all directories in catalog }

  { Purge catalog from storage }
  PurgeCIB(CIB);
  writeln('Ready');
end.
