program TestPascalTrig;
(*
  This program illustrates how floating point performance can be improved
by taking advantage of the 80387/i486 (and non-Intel 287 and 387)
trig functions.  The 8087 and 80287 handled these functions in only the
most primitive way, requiring extensive coding to convert the partial
tangent and partial arctangent instructions into sin, cos, and arctan.

  The program consists of several routines to replace the turbo pascal
runtime routines with routines taking advantage of the 387's fsin, fcos,
and fsincos instructions, and the 387's improved implementation of the
fptan and fpatan instructions.

ChangeSystemSinCos patches the tp run time routines to use the fsin and
fcos instructions.  This has been tested on TP5.0 and TP6.0.  The change
gives an improvement of about a factor of 4.

MySin and MyCos are replacement functions for the TP sin and cos functions,
and do not require the TP run time routines to be patched.  They are somewhat
slower than the patched version.

MySinCos is a routine than computes both the sin and the cos of a single
argument.  This is faster than computing the sin and cos independently.

Tan is a function than computes the tangent of an argument using the sin
and cos routines.

Tan387 computes the tangent using the 387 fptan instruction.

Arctan2 computes the arctangent of the quotent of two arguments, setting the
result into the correct quadrant (0 to 2pi) using the TP arctan function.

ArcTan387 computes the arctangent of the quotent of two arguments, setting
the result into the correct quadrant (0 to 2pi) using the 387 fpatan
instruction.

The rest of the program tests the accuracy of the various routines, and
test the speeds of the routines.  The output of the program produced on
my Northgate 486/25 was:

Co-Processor 80387 detected.

fsin vs sin     1.110223E-0016
fcos vs cos     1.110223E-0016
fsincos vs sin  1.110223E-0016
fsincos vs cos  1.110223E-0016
arctan2         8.881784E-0016
tan             4.547474E-0013

        type     ticks   time/op
  TP sin/cos       138    75.824 usec
      TP tan       148    81.319 usec
 new sin/cos        54    29.670 usec
  my sin/cos        67    36.813 usec
    mysincos        40    21.978 usec
     arctan2        79    43.407 usec
   arctan387        37    20.330 usec
      tan387        31    17.033 usec

Joseph R Ahlgren   2218 N Tuckahoe St   Arlington, VA  22205
RBBS  703-241-7980    CompuServe  70461,2340

These routines may be freely distributed provided they are unmodified and
include the above attribution.  I would appreciate any comments or
suggestions.
*)
  const
    loopmax=100;
  procedure ChangeSystemSinCos;
    const
      NotYetTested: boolean = true;
    var
      x,y,z: single;
    begin
      if NotYetTested then
        begin
        NotYetTested:=false;
        if Test8087 >= 3 then
          begin
          x:=1.0;
          y:=sin(x);
          z:=cos(x);
          inline($E8/$00/$00);            { CALL   *+3 }
          inline($5B);                    { POP    BX }
          inline($2E/$C4/$7F/$F3);        { LES    DI,CS:[BX-0D] }
          inline($26/$C7/$05/$D9/$FF);    { MOV    Word Ptr ES:[DI],fcos }
          inline($26/$C6/$45/$02/$CB);    { MOV    Byte Ptr ES:[DI+02],retf }
          inline($2E/$C4/$7F/$E4);        { LES    DI,CS:[BX-1C] }
          inline($26/$C7/$05/$D9/$FE);    { MOV    Word Ptr ES:[DI],fsin }
          inline($26/$C6/$45/$02/$CB);    { MOV    Byte Ptr ES:[DI+02],retf }
          end;
        end;
    end;


  function MySin(x: double): double;
    var
      sin1: double;
    begin
      inline($DD/$86/x);       { fld [x][bp] }
      inline($D9/$FE);         { fsin }
      inline($DD/$9E/sin1);    { fstp [sin1][bp] }
      MySin:=sin1;
    end;
  function MyCos(x: double): double;
    var
      cos1: double;
    begin
      inline($DD/$86/x);       { fld [x][bp] }
      inline($D9/$FF);         { fcos }
      inline($DD/$9E/cos1);    { fstp [cos1][bp] }
      MyCos:=cos1;
    end;
  procedure MySinCos(x: double; var sinx,cosx: double);
    var
      sin2,cos2: double;
    begin
      inline($DD/$86/x);       { fld [x][bp] }
      inline($D9/$FB);         { fsincos }
      inline($DD/$9E/cos2);    { fstp [cos2][bp] }
      inline($DD/$9E/sin2);    { fstp [sin2][bp] }
      sinx:=sin2;
      cosx:=cos2;
    end;
  function Tan(X:double):double;
    begin
      Tan := Sin(X) / Cos(X) ;
    end;
  function Tan387(x: double): double;
    var
      at: double;
    begin
      inline($DD/$86/x);       { fld [x][bp] }
      inline($D9/$F2);         { fptan }
      inline($DD/$9E/at);      { fstp [at][bp] }
      inline($DD/$9E/at);      { fstp [at][bp] }
      Tan387:=at;
    end;
  function ArcTan2(Y,X: double):double;
    var
      AbsTan: double;
    begin
      if X=0.0 then begin
        if Y>=0.0 then ArcTan2:=pi/2 else ArcTan2:=3*pi/2 end
      else begin
        AbsTan := ArcTan(abs(Y/X));
        if X>=0.0 then begin
          if Y>= 0.0 then ArcTan2:=AbsTan else ArcTan2:=2*pi-AbsTan end
         else begin
          if Y>= 0.0 then ArcTan2:=pi-AbsTan else ArcTan2:=pi+AbsTan end
      end;
    end;
  function ArcTan387(y,x: double): double;
    var
      at,temp: double;
    const
      TwoPi: double = 6.28318530717958648;
    begin
      inline($DD/$86/y);       { fld [y][bp] }
      inline($DD/$86/x);       { fld [x][bp] }
      inline($D9/$F3);         { fpatan }
      inline($D9/$E4);         { ftst }
      inline($DF/$E0);         { fstsw ax }
      inline($9E);             { sahf }
      inline($73/$04);         { jnc fff }
      inline($DC/$06/TwoPi);   { fadd [TwoPi] }
      inline($DD/$9E/at);      { fstp [at][bp] }
      ArcTan387:=at;
    end;
  function max(x,y: double): double;
    begin
    if abs(x) > y then
      max:=abs(x)
     else
      max:=y;
    end;
  var
    x,y,z,s,c,s1,s2,c1,c2,xa,xt,xs1,xs2,xc1,xc2,xs3,xc3,pm,pm2,scale: double;
    j,k: word;
    line: string;
    systemtime: longint absolute $0040:$006c;
    t1,t2,t3,t4,t5,t6,t7: longint;
    doubles: array [0..4096] of double;
  begin
    WriteLn;
    if Test8087 > 0 then
      WriteLn('Co-Processor 80',Test8087,'87 detected.')
     else
      WriteLn('No co-processor detected.');
    if Test8087 < 3 then
      begin
      Write('No 80387 detected, system may hang.  Proceed (y/n) ? ');
      ReadLn(line);
      if (length(line) < 1) or (Upcase(line[1]) <> 'Y') then halt;
      end;
    WriteLn;
    WriteLn('operation        maximum error');
    randomize;
    pm:=5*pi;
    pm2:=pm/2;
    xs1:=0;
    xs2:=0;
    xc1:=0;
    xc2:=0;
    xs3:=0;
    xc3:=0;
    xa:=0;
    xt:=0;
    for k:=1 to 5000 do
      begin
      x:=random*pm-pm2;
      doubles[k and 4095]:=x;
      s:=sin(x);
      c:=cos(x);
      s1:=MySin(x);
      c1:=MyCos(x);
      MySinCos(x,s2,c2);
      xs1:=max(abs(s1-s),xs1);
      xs2:=max(abs(s2-s),xs2);
      xc1:=max(abs(c1-c),xc1);
      xc2:=max(abs(c2-c),xc1);
      xc3:=max(abs(c2-c1),xc3);
      xs3:=max(abs(s2-s1),xs3);
      s:=tan(x);
      c:=tan387(x);
      xt:=max(abs(s-c),xt);
      x:=random*10-5;
      y:=random*10-5;
      s:=arctan2(y,x);
      c:=arctan387(y,x);
      xa:=max(abs(s-c),xa);
      end;
    WriteLn('fsin vs sin    ',xs1:15);
    WriteLn('fcos vs cos    ',xc1:15);
    WriteLn('fsincos vs sin ',xs2:15);
    WriteLn('fsincos vs cos ',xc2:15);
    WriteLn('arctan2        ',xa:15);
    WriteLn('tan            ',xt:15);
    WriteLn;
    WriteLn('        type     ticks   time/op');
    scale:=1000/loopmax/18.2;
    t1:=systemtime;
    for k:=1 to loopmax do
     for j:=1 to 1000 do
      begin
      x:=sin(doubles[j]);
      y:=cos(doubles[j]);
      end;
    t2:=systemtime;
    WriteLn('  TP sin/cos',t2-t1:10,(t2-t1)*scale:10:3,' usec');
    t1:=systemtime;
    for k:=1 to loopmax do
     for j:=1 to 1000 do
      begin
      z:=tan(doubles[j]);
      end;
    t2:=systemtime;
    WriteLn('      TP tan',t2-t1:10,(t2-t1)*scale:10:3,' usec');
    ChangeSystemSinCos;
    t1:=systemtime;
    for k:=1 to loopmax do
     for j:=1 to 1000 do
      begin
      x:=sin(doubles[j]);
      y:=cos(doubles[j]);
      end;
    t2:=systemtime;
    WriteLn(' new sin/cos',t2-t1:10,(t2-t1)*scale:10:3,' usec');
    t1:=systemtime;
    for k:=1 to loopmax do
     for j:=1 to 1000 do
      begin
      x:=Mysin(doubles[j]);
      y:=Mycos(doubles[j]);
      end;
    t2:=systemtime;
    WriteLn('  my sin/cos',t2-t1:10,(t2-t1)*scale:10:3,' usec');
    t1:=systemtime;
    for k:=1 to loopmax do
     for j:=1 to 1000 do
      begin
      MySinCos(doubles[j],x,y);
      end;
    t2:=systemtime;
    WriteLn('    mysincos',t2-t1:10,(t2-t1)*scale:10:3,' usec');
    t1:=systemtime;
    for k:=1 to loopmax do
     for j:=1 to 1000 do
      begin
      z:=arctan2(doubles[j],doubles[j+2048]);
      end;
    t2:=systemtime;
    WriteLn('     arctan2',t2-t1:10,(t2-t1)*scale:10:3,' usec');
    t1:=systemtime;
    for k:=1 to loopmax do
     for j:=1 to 1000 do
      begin
      z:=arctan387(doubles[j],doubles[j+2048]);
      end;
    t2:=systemtime;
    WriteLn('   arctan387',t2-t1:10,(t2-t1)*scale:10:3,' usec');
    t1:=systemtime;
    for k:=1 to loopmax do
     for j:=1 to 1000 do
      begin
      z:=tan387(doubles[j]);
      end;
    t2:=systemtime;
    WriteLn('      tan387',t2-t1:10,(t2-t1)*scale:10:3,' usec');
end.