(* ---------------------------------- *)
(* Regs  --  UART Registers           *)
(* Copyright (c) 1994 by Dado Colussi *)
(* All rights reserved                *)
(* ---------------------------------- *)


UNIT Regs;


INTERFACE


USES

  Consts;


CONST

  MAX_PORTS   = 32;     (* 1 - 32 *)
  MAX_IRQS    = 7;      (* 0 - 7 *)

              (* Standard COM[1-4] Addresses *)
  IO_BASE     : ARRAY [1..4] OF Word = ($3F8, $2F8, $3E8, $2E8);


  CLOCK_FREQ  = 1843200;     (* The clock frequency used by UART *)

  RBR         = $00;   (* Receiver Buffer Register (R) *)
  THR         = $00;   (* Transmitter Holding Register (W) *)
  IER         = $01;   (* Interrupt Enable Register (R/W) *)
  IIR         = $02;   (* Interrupt Identification Register (R) *)
  FCR         = $02;   (* FIFO Control Register (W) *)
  LCR         = $03;   (* Line Control Register (R/W) *)
  MCR         = $04;   (* MODEM Control Register (R/W) *)
  LSR         = $05;   (* Line Status Register (R/W) *)
  MSR         = $06;   (* MODEM Status Register (R/W) *)
  SCR         = $07;   (* Scratch Register (R/W) *)
  DLL         = $00;   (* Divisor Latch (LSB) *)
  DLM         = $01;   (* Divisor Latch (MSB) *)


  (* Interrupt Enable Register Bits *)

  IER_ERDAI    = BIT_0;        (* Enable Receive Data Available Interrupt *)
  IER_ETHREI   = BIT_1;        (* Enable Transmitter Holding Register Empty *)
  IER_ERLSI    = BIT_2;        (* Enable Receiver Line Status Interrupt *)
  IER_EMST     = BIT_3;        (* Enable MODEM Status Interrupt *)


  (* Interrupt Indentification Register (Read Only) *)

  IIR_IP       = BIT_0;        (* "0" If Interrupt Pending *)
  IIR_IID0     = BIT_1;        (* Interrupt ID (Bit(0) *)
  IIR_IID1     = BIT_2;        (* Interrupt ID (Bit(1) *)
  IIR_IID2     = BIT_3;        (* Interrupt ID (Bit(2) *)
  IIR_FIFOE0   = BIT_6;        (* FIFOs Enabled (Bit(0) *)
  IIR_FIFOE1   = BIT_7;        (* FIFOs Enabled (Bit(1) *)


  (* Interrupt Types in Priority Order *)

  IIR_NONE     = BIT_0;                   (* No interrupts Enabled *)
  IIR_RLS      = IIR_IID0 + IIR_IID1;     (* Receiver Line Status *)
  IIR_RDA      = IIR_IID1;                (* Received Data Available *)
  IIR_CTI      = IIR_IID1 + IIR_IID2;     (* Character Timeout Indication (FIFO mode only) *)
  IIR_THRE     = IIR_IID0;                (* Transmitter Holding Register Empty *)
  IIR_MS       = 0;                       (* MODEM Status *)


  (* FIFO Control Register (Write Only) *)

  FCR_FIFO_ENABLE          = BIT_0;    (* FIFO Enable *)
  FCR_RCVR_FIFO_RESET      = BIT_1;    (* Receiver FIFO Reset *)
  FCR_XMIT_FIFO_RESET      = BIT_2;    (* Transmitter FIFO Reset *)
  FCR_DMA_MODE_SELECT      = BIT_3;    (* DMA Mode Select *)
  FCR_RCVR_TRIGGER_LSB     = BIT_6;    (* Receiver Trigger (LSB) *)
  FCR_RCVR_TRIGGER_MSB     = BIT_7;    (* Receiver Trigger (MSG) *)


  (* Line Control Register *)

  LCR_WLS0        = BIT_0; (* Word Length Select Bit 0 *)
  LCR_WLS1        = BIT_1; (* Word Length Select Bit 1 *)
  LCR_STB         = BIT_2; (* Number of Stop Bits *)
  LCR_PEN         = BIT_3; (* Parity Enable *)
  LCR_EPS         = BIT_4; (* Even Parity Select *)
  LCR_STICK_PAR   = BIT_5; (* Sticky Parity *)
  LCR_SET_BREAK   = BIT_6; (* Set Break *)
  LCR_DLAB        = BIT_7; (* Divisor Latch Access Bit *)


  (* MODEM Control Register *)

  MCR_DTR          = BIT_0; (* Data Terminal Ready *)
  MCR_RTS          = BIT_1; (* Request To Send *)
  MCR_OUT_1        = BIT_2; (* Out 1 *)
  MCR_OUT_2        = BIT_3; (* Out 2 *)
  MCR_LOOP         = BIT_4; (* Local Loopback for Diagnostics *)


  (* Line Status Register *)

  LSR_DR         = BIT_0; (* Data Ready *)
  LSR_OE         = BIT_1; (* Overrun Error *)
  LSR_PE         = BIT_2; (* Parity Error *)
  LSR_FE         = BIT_3; (* Framing Error *)
  LSR_BI         = BIT_4; (* Break Interrupt *)
  LSR_THRE       = BIT_5; (* Transmitter Holding Register *)
  LSR_TEMT       = BIT_6; (* Transmitter Empty *)
  LSR_ERCVR_FIFO = BIT_7; (* Error in Reveiver FIFO *)


  (* MODEM Status Register *)

  MSR_DCTS        = BIT_0; (* Delta Clear To Send *)
  MSR_DDSR        = BIT_1; (* Delta Data Set Ready *)
  MSR_TERI        = BIT_2; (* Trailing Edge Ring Indicator *)
  MSR_DDCD        = BIT_3; (* Delta Data Carrier Detect *)
  MSR_CTS         = BIT_4; (* Clear To Send *)
  MSR_DSR         = BIT_5; (* Data Set Ready *)
  MSR_RI          = BIT_6; (* Ring Indicator *)
  MSR_DCD         = BIT_7; (* Data Carrier Detect *)


  (* Programmable Interrupt Controller 8259A Registers *)

  ISR             = $20;    (* In Service Register *)
  IMR             = $21;    (* Interrupt Mask Register *)
  SPECIFIC_EOI    = $60;    (* Specific EOI Command *)


IMPLEMENTATION


END. (* Regs *)

