library microscr;

uses
  Strings,
  WinTypes,
  WinProcs;

const
  BorderWidth     = 1;

  ofReserved      = 0;
  ofState         = 2;
  ofBits          = 4;
  ofSize          = 6; { Amount of window extra bytes to use (MBitButton)}

  bsDisabled      = $0001;
  bsFocus         = $0002;
  bsKeyUpDown     = $0004;
  bsKeyDownDown   = $0008;
  bsMouseUpDown   = $0010;
  bsMouseDownDown = $0020;
  bsMouseUp       = $0040;

  btDisableBits   = 100;

  bn_Up           = 6;
  bn_Down         = 7;

  IDTimer         = 899;

  Timer           = 'Timer';
  FirstDelay      = 'FirstDelay';

{$R MICROSCR.RES}

{ GetAppInstance ----------------------------------------------------
     Returns a handle to the current client application
  ------------------------------------------------------------------- }

function GetAppInstance: THandle; near; assembler;
asm
  PUSH  SS
  CALL  GlobalHandle
end;

{ MMicroScrollWinFn ---------------------------------------------------
     Button window procedure
  ------------------------------------------------------------------- }
function MMicroScrollWinFn(hWindow: hWnd; Message: word; wParam: word; lParam: longint): longint; export;

  var
    DC         : hDC;
    BitsNumber : integer;
    Bitmap     : tBitmap;
    Rect       : tRect;
    Pt         : tPoint;
    PS         : tPaintStruct;
    MouseClk   : integer;
    TimerID    : word;

{ Get ---------------------------------------------------------------
     Get a window instance word.
  ------------------------------------------------------------------- }
  function Get(Ofs: Integer): Word;
    begin
      Get := GetWindowWord(HWindow, Ofs);
    end;

{ SetWord -----------------------------------------------------------
     Set the value of a window instance word.
  ------------------------------------------------------------------- }
  procedure SetWord(Ofs: Integer; Val: Word);
    begin
      SetWindowWord(HWindow, Ofs, Val);
    end;

{ State -------------------------------------------------------------
     Get the button's state word.
  ------------------------------------------------------------------- }
  function State: Word;
    begin
      State := Get(ofState);
    end;

{ Bits --------------------------------------------------------------
     Get the bitmap of the button
  ------------------------------------------------------------------- }
  function Bits: Word;
    begin
      Bits := Get(ofBits);
    end;

{ GetState ----------------------------------------------------------
     Get the value of a state bit.
  ------------------------------------------------------------------- }
  function GetState(AState: Word): Boolean;
    begin
      GetState := (State and AState) = AState;
    end;

{ Paint -------------------------------------------------------------
     Paint the button. Called in responce to a WM_PAINT message
     and whenever the button changes state (called by Repaint).
  ------------------------------------------------------------------- }
  procedure Paint(DC: HDC);

    const
      coGray = $00C0C0C0;

    var
      MemDC                 : hDC;
      Oldbitmap             : hBitmap;
      BorderBrush, OldBrush : hBrush;
      LogBrush              : tLogBrush;
      Frame                 : tRect;
      Height, Width         : integer;

    begin
      { Draw border }
      GetClientRect(hWindow, Frame);
      Height := Frame.bottom - Frame.top;
      Width  := Frame.right - Frame.left;
      BorderBrush := GetStockObject(Black_Brush);
      OldBrush := SelectObject(DC, BorderBrush);
      PatBlt(DC, Frame.left, Frame.top, Width, BorderWidth, PatCopy);
      PatBlt(DC, Frame.left, Frame.top, BorderWidth, Height, PatCopy);
      PatBlt(DC, Frame.left, Frame.bottom - BorderWidth, Width, BorderWidth, PatCopy);
      PatBlt(DC, Frame.right - BorderWidth, Frame.top, BorderWidth, Height, PatCopy);
      SelectObject(DC, OldBrush);

      { Draw bitmap }
      MemDC := CreateCompatibleDC(DC);
      OldBitmap := SelectObject(MemDC, Bits);
      GetObject(Bits, Sizeof(Bitmap), @Bitmap);
      if GetState(bsDisabled) then
        begin
          { Gray out the button }
          OldBrush := SelectObject(DC, CreateSolidBrush(coGray));
          PatBlt(DC, BorderWidth, BorderWidth, Bitmap.bmWidth, Bitmap.bmHeight, PatCopy);
          DeleteObject(SelectObject(DC, OldBrush));

          { Draw the bitmap through a checked brush }
          LogBrush.lbStyle := bs_Pattern;
          LogBrush.lbHatch := LoadBitmap(hInstance, MAKEINTRESOURCE(btDisableBits));
          OldBrush := SelectObject(DC, CreateBrushIndirect(LogBrush));
          BitBlt(DC, BorderWidth, BorderWidth, Bitmap.bmWidth, Bitmap.bmHeight, MemDC, 0, 0, $00A803A9{DPSoa});
          DeleteObject(SelectObject(DC, OldBrush));
          DeleteObject(LogBrush.lbHatch);
        end
      else
        BitBlt(DC, BorderWidth, BorderWidth, Bitmap.bmWidth, Bitmap.bmHeight, MemDC, 0, 0, srcCopy);

      InflateRect(Frame, -BorderWidth, -BorderWidth);
      if GetState(bsKeyUpDown) or GetState(bsMouseUpDown) then
        begin
          Dec(Frame.bottom, Height shr 1);
          InvertRect(DC, Frame);
        end;
      if GetState(bsKeyDownDown) or GetState(bsMouseDownDown) then
        begin
          Inc(Frame.top, Height shr 1);
          InvertRect(DC, Frame);
        end;

      SelectObject(MemDC, OldBitmap);
      DeleteDC(MemDC);
    end;

{ Repaint -----------------------------------------------------------
     Repaint the button. Called whenever the button changes
     state.
  ------------------------------------------------------------------- }
  procedure Repaint;

    var
      DC: HDC;

    begin
      DC := GetDC(HWindow);
      Paint(DC);
      ReleaseDC(HWindow, DC);
    end;

{ SetState ----------------------------------------------------------
     Sets the value of state bit. If the word changes value
     the button is repainted.
  ------------------------------------------------------------------- }
  procedure SetState(AState: Word; Enable: Boolean);

    var
      OldState: Word;

    begin
      OldState := State;
      if Enable then
        SetWord(ofState, State or AState)
      else
        SetWord(ofState, State and not AState);
      if State <> OldState then
        Repaint;
    end;

{ InMe --------------------------------------------------------------
     Returns true if given point is in within the border of
     the button
  ------------------------------------------------------------------- }
  function InMe(lPoint: Longint): integer;

    var
      R, HR : tRect;
      Point : tPoint absolute lPoint;

    begin
      InMe := 0;
      GetClientRect(HWindow, R);
      InflateRect(R, -BorderWidth, -BorderWidth);
      if not PtInRect(R, Point) then Exit;
      HR := R;
      Dec(HR.bottom, (R.bottom - R.top) shr 1);
      if PtInRect(HR, Point) then
        InMe := 1
      else
        InMe := -1;
    end;

{ ButtonPressed -----------------------------------------------------
     Called when the button is pressed either the keyboard or
     by the mouse.
  ------------------------------------------------------------------- }
  procedure ButtonPressed(UpDown: integer);

    var
      Notify: word;

    begin
      if UpDown = 1 then
        Notify := bn_Up
      else
        Notify := bn_Down;
      SendMessage(GetParent(HWindow), wm_Command, GetDlgCtrlID(hWindow), MakeLong(hWindow, Notify));
    end;

{ LoadBits ----------------------------------------------------------
     Load the bitmap for the button or the "NO BITMAP" version
     if it does not exist.
  ------------------------------------------------------------------- }
  procedure LoadBits(Wrd: Word; MapNumber: Word);

    var
      MapBits: HBitmap;

    begin
      MapBits := LoadBitmap(hInstance, PChar(MapNumber));
        if MapBits = 0 then
          MapBits := LoadBitmap(GetAppInstance, PChar(MapNumber));
        if MapBits = 0 then
          MapBits := LoadBitmap(hInstance, PChar(MapNumber));
      SetWord(Wrd, MapBits);
    end;

  begin
    MMicroScrollWinFn := 0;
    case Message of

      wm_Create:
        begin
          { Detect EGA monitor }
          DC := GetDC(0);
          if (GetSystemMetrics(sm_CYScreen) < 480) or (GetDeviceCaps(DC, numColors) < 16) then
            BitsNumber := 899
          else
            BitsNumber := 899;
          ReleaseDC(0, DC);

          { Load bitmaps from resource }
          LoadBits(ofBits, BitsNumber);
          GetObject(Bits, SizeOf(Bitmap), @Bitmap);
          GetWindowRect(HWindow, Rect);
          Pt.X := Rect.Left;
          Pt.Y := Rect.Top;
          ScreenToClient(pCreateStruct(lParam)^.hwndParent, Pt);
          MoveWindow(HWindow, Pt.X, Pt.Y, Bitmap.bmWidth + BorderWidth * 2,
                     Bitmap.bmHeight + BorderWidth * 2, False);

          { Initialize button state }
          with pCreateStruct(lParam)^ do
          begin
            if Style and ws_Disabled <> 0 then
              SetState(bsDisabled, True);
          end;

          SetProp(hWindow, Timer, 0);
          SetProp(hWindow, FirstDelay, 0);
        end;

      wm_NCDestroy:
        begin
          { Destroy all saved bitmaps before the button is destroyed }
          RemoveProp(hWindow, Timer);
          RemoveProp(hWindow, FirstDelay);
          MMicroScrollWinFn := DefWindowProc(HWindow, Message, wParam, lParam);
          DeleteObject(Bits);
        end;

      wm_Paint:
        begin
          BeginPaint(HWindow, PS);
          Paint(PS.hDC);
          EndPaint(HWindow, PS);
        end;

      wm_EraseBkGnd:
        begin
        { Squelch the painting of the background to eliminate flicker }
        end;

      wm_Enable:
        SetState(bsDisabled, wParam = 0);

      wm_SetFocus:
        SetState(bsFocus, True);

      wm_KillFocus:
        SetState(bsFocus, False);

      wm_KeyDown:
        begin
          if (wParam = vk_Up) or (wParam = vk_Left) then
            begin
              SetState(bsKeyUpDown, True);
              ButtonPressed(1);
            end;
          if (wParam = vk_Down) or (wParam = vk_Right) then
          begin
            SetState(bsKeyDownDown, True);
            ButtonPressed(-1);
          end;
        end;

      wm_KeyUp:
        if GetState(bsKeyDownDown) or GetState(bsKeyUpDown) then
          SetState(bsKeyUpDown + bsKeyDownDown, False);

      wm_LButtonDown:
        begin
          MouseClk := InMe(lParam);
          if (MouseClk <> 0) then
            begin
              if GetFocus <> HWindow then
                SetFocus(HWindow);
              if MouseClk > 0 then
                SetState(bsMouseUpDown, True);
              if MouseClk < 0 then
                SetState(bsMouseDownDown, True);
              ButtonPressed(MouseClk);
              SetCapture(hWindow);
              SetState(bsMouseUp, True);
              SetProp(hWindow, Timer, SetTimer(hWindow, IDTimer, 300, nil));
              SetProp(hWindow, FirstDelay, 1);
            end;
        end;

      wm_LButtonUp:
        if GetState(bsMouseUp) then
          begin
            TimerID := GetProp(hWindow, Timer);
            if TimerID <> 0 then
               KillTimer(hWindow, TimerID);
            SetProp(hWindow, Timer, 0);
            ReleaseCapture;
            SetState(bsMouseUp + bsMouseUpDown + bsMouseDownDown, False);
          end;

      wm_MouseMove:
        begin
          if GetState(bsMouseUp) then
            if InMe(lParam) = 0 then
              begin
                ReleaseCapture;
                SetState(bsMouseUp + bsMouseUpDown + bsMouseDownDown, False);
                TimerID := GetProp(hWindow, Timer);
                if TimerID <> 0 then
                  KillTimer(hWindow, TimerID);
                SetProp(hWindow, Timer, 0);
              end;
        end;

      wm_Timer:
          if GetState(bsMouseUp) then
            begin
              if GetState(bsMouseUpDown) then
                ButtonPressed(1)
              else
                ButtonPressed(-1);
              if GetProp(hWindow, FirstDelay) <> 0 then
                begin
                  SetProp(hWindow, FirstDelay, 0);
                  TimerID := GetProp(hWindow, Timer);
                  if TimerID <> 0 then
                    KillTimer(hWindow, TimerID);
                  SetProp(hWindow, Timer, SetTimer(hWindow, IDTimer, 20, nil));
                end;
            end;

      wm_GetDlgCode:
        MMicroScrollWinFn := dlgc_WantArrows;

      else
        MMicroScrollWinFn := DefWindowProc(HWindow, Message, wParam, lParam);
    end;
  end;

var
  MSClass: tWndClass;

exports
  MMicroScrollWinFn;

(*----------------------------------------------------------------------------*)
begin
  with MSClass do
  begin
    lpszClassName := 'MScroll';
    hCursor       := LoadCursor(0, idc_Arrow);
    lpszMenuName  := nil;
    style         := cs_HRedraw or cs_VRedraw or cs_GlobalClass;
    lpfnWndProc   := tFarProc(@MMicroScrollWinFn);
    hInstance     := System.hInstance;
    hIcon         := 0;
    cbWndExtra    := ofSize;
    cbClsExtra    := 0;
    hbrBackground := 0;
  end;
  RegisterClass(MSClass);
end.
