{$N+}
UNIT TrigStuff;

INTERFACE

CONST   maxfloatray = 6551;                             { needed for POLY }
TYPE    floatray = ARRAY[0..maxfloatray] OF DOUBLE;     { needed for POLY }

FUNCTION TAN(x: DOUBLE):DOUBLE;
{ returns the tangent of x }

PROCEDURE SINCOS(x:DOUBLE; VAR y,z:DOUBLE);
{ returns y = SIN(x) and z = COS(x) }

FUNCTION ArcTangent(x,y: DOUBLE):DOUBLE;
{ returns arctan(y/x) in radians between zero and 2 pi }

FUNCTION ArcCOS(x: DOUBLE):DOUBLE;
{ returns the inverse cosine of x }

FUNCTION ArcSIN(x: DOUBLE):DOUBLE;
{ returns the inverse sine of x }

FUNCTION Expo(x,y: DOUBLE):DOUBLE;
{ exponentiation: x^y }

FUNCTION Two2TheX(x:DOUBLE):DOUBLE;
{ exponentiation: 2^x }

FUNCTION Ten2TheX(x:DOUBLE):DOUBLE;
{ exponentiation: 10^x }

FUNCTION LOG(x: DOUBLE):DOUBLE;
{ returns the logarithm to base 10 of x }

FUNCTION CEIL(x:DOUBLE):DOUBLE;
{ returns smallest integer larger than x }

FUNCTION FLOOR(x:DOUBLE):DOUBLE;
{ returns largest integer smaller than x }

FUNCTION FMOD(x,y:DOUBLE):DOUBLE;
{ returns f such that x := a*y + f where a = integer & f in [0,y) }

FUNCTION FREXP(x:DOUBLE; VAR n:INTEGER):DOUBLE;
{ returns m for which x = m*2^n with m in [0.5,1) }

FUNCTION HYPOT(x,y:DOUBLE):DOUBLE;
{ returns hypotenuse = SQRT(SQR(x)=SQR(y)) }

FUNCTION LDEXP(x:DOUBLE; i:INTEGER):DOUBLE;
{ returns x*2^i }

FUNCTION MODF(x:DOUBLE; VAR i:DOUBLE):DOUBLE;
{ returns fractional part of x; gives integer part of x in i }

FUNCTION POLY(x:DOUBLE; n:INTEGER; VAR degree):DOUBLE;
{ returns y = c0 + c1*x + c2*x^2 + ... + cn*x^n; n = polynomial order }

FUNCTION SINH(x:DOUBLE):DOUBLE;
{ returns hyperbolic sine }

FUNCTION COSH(x:DOUBLE):DOUBLE;
{ returns hyperbolic cosine }

FUNCTION TANH(x:DOUBLE):DOUBLE;
{ returns hyperbolic tangent }

FUNCTION ASINH(x:DOUBLE):DOUBLE;
{ returns inverse hyperbolic sine }

FUNCTION ACOSH(x:DOUBLE):DOUBLE;
{ returns inverse hyperbolic cosine }

FUNCTION ATANH(x:DOUBLE):DOUBLE;
{ returns inverse hyperbolic tangent }

IMPLEMENTATION


  FUNCTION TAN(x:DOUBLE):DOUBLE;
  CONST   piM2 = 6.283185308;

  VAR     cosx: DOUBLE;
  
    BEGIN
      TAN := SIN(x) / COS(x);
    END {TAN};

  
  PROCEDURE SINCOS(x:DOUBLE; VAR y,z:DOUBLE);
  
    BEGIN
      y := SIN(x);
      z := COS(x);
    END {SINCOS};
      

  FUNCTION ArcTangent;
  VAR  a: DOUBLE;
  
    BEGIN
      IF x <> 0 THEN
        BEGIN
          a := ARCTAN(ABS(y/x));
          IF x > 0 THEN
            IF y >= 0 THEN ArcTangent := a       { first quadrant }
            ELSE           ArcTangent := 2*pi-a  { fourth quadrant }
          ELSE { x < 0 }
            IF y >= 0 THEN ArcTangent := pi - a  { second quadrant }
            ELSE           ArcTangent := pi + a  { third quadrant }
        END
      ELSE { x = 0 }
        IF y = 0 THEN ArcTangent := 0.0
        ELSE
          IF y > 0  THEN ArcTangent := pi/2
          ELSE           ArcTangent := 3*pi/2;
    END {ArcTangent};
      
  
  FUNCTION ArcCOS;
  VAR   result: DOUBLE;

  BEGIN
    IF x = 0 THEN
      result := pi/2
    ELSE
      IF x = 1 THEN
        result := 0
      ELSE
        IF x = -1 THEN
          result := pi
        ELSE
          result := ArcTangent(x/SQRT(1 - SQR(x)),1);
    ArcCOS := result;
  END {ArcCOS};


   
 FUNCTION ArcSIN;
 
   BEGIN
     IF x = 0 THEN
       ArcSIN := 0
     ELSE
       IF x = 1 THEN
         ArcSIN := pi/2
       ELSE
         IF x = -1 THEN
           ArcSIN := - pi/2
         ELSE
           ArcSIN := ARCTAN(x/SQRT(1-SQR(x)));
   END {ArcSIN};
   
 
 FUNCTION Expo;
 
   BEGIN
     IF x > 0 THEN
       Expo := EXP(y*LN(x))
     ELSE
       Expo := 0;
   END {Expo};
   
 
  FUNCTION Two2TheX(x:DOUBLE):DOUBLE;
    
    BEGIN
      Two2TheX := Expo(2,x);
    END {Two2TheX};
  
  
  FUNCTION Ten2TheX(x:DOUBLE):DOUBLE;
    
    BEGIN
      Ten2TheX := Expo(10,x);
    END {Ten2TheX};

  
 FUNCTION LOG;
 CONST LN10 = 2.302585093;
 
   BEGIN
     LOG := LN(x) / LN10
   END {LOG};
   
   
  FUNCTION CEIL(x:DOUBLE):DOUBLE;

    BEGIN
      IF x > 0 THEN CEIL := TRUNC(x) + 1 ELSE CEIL := TRUNC(x);
    END {CEIL};

  
  FUNCTION FLOOR(x:DOUBLE):DOUBLE;
  VAR  y: DOUBLE;

    BEGIN
      IF x >= 0 THEN FLOOR := TRUNC(x) ELSE BEGIN
        y := ROUND(x);
        IF ROUND(x) > x THEN FLOOR := y - 1 ELSE FLOOR := y;
      END {FLOOR};
    END;


  FUNCTION FMOD(x,y:DOUBLE):DOUBLE;
  VAR  a,f: DOUBLE;

    BEGIN
      a := TRUNC(x/y);
      f := x - a*y;
      FMOD := f;
    END {FMOD};


  FUNCTION FREXP(x:DOUBLE; VAR n:INTEGER):DOUBLE;
  CONST   ln2 = 0.6931471805599453094172;{ natural logarithm of 2  }
  VAR     m: DOUBLE;

   BEGIN
     { n is ROUND TO +INF(ln(x)/ln2) }
     IF x <> 0 THEN BEGIN
       n := TRUNC(ln(ABS(x))/ln2);
       IF ABS(x) >= 1 THEN INC(n);
       m := EXP(ln(ABS(x))-n*ln2);
       IF x < 0 THEN m := -m;
     END ELSE BEGIN
       m := 0; n := 0;
     END;
     FREXP := m;
   END {FREXP};


  FUNCTION HYPOT(x,y:DOUBLE):DOUBLE;

   BEGIN
     HYPOT := SQRT(SQR(x) + SQR(y));
   END {HYPOT};
  

  FUNCTION LDEXP(x:DOUBLE; i:INTEGER):DOUBLE;
  
    BEGIN
      LDEXP := x*EXP(i*LN(2));
    END {LDEXP};
  

  FUNCTION MODF(x:DOUBLE; VAR i:DOUBLE):DOUBLE;

    BEGIN
      i := TRUNC(x);
      MODF := x - i;
    END {MODF};
 

  FUNCTION POLY(x:DOUBLE; n:INTEGER; VAR degree):DOUBLE;
  VAR  i:INTEGER;
       y: DOUBLE;

    BEGIN
      y := 0;
      FOR i := n DOWNTO 0 DO  y := y*x + floatray(degree)[i];
      POLY := y;
    END {POLY};


  FUNCTION SINH(x:DOUBLE):DOUBLE;
  VAR temp: DOUBLE;

    BEGIN
      temp := EXP(x);
      SINH := 0.5*(temp-1/temp);
    END {SINH};

  FUNCTION COSH(x:DOUBLE):DOUBLE;
  VAR temp: DOUBLE;

    BEGIN
      temp := EXP(x);
      COSH := 0.5*(temp+1/temp);
    END {COSH};

  FUNCTION TANH(x:DOUBLE):DOUBLE;
  VAR temp: DOUBLE;

    BEGIN
      temp := EXP(2*x);
      TANH := (temp-1)/(temp+1);
    END {TANH};

  FUNCTION ASINH(x:DOUBLE):DOUBLE;

    BEGIN
      ASINH := LN(x + SQRT(SQR(x)+1));
    END {ASINH};

  FUNCTION ACOSH(x:DOUBLE):DOUBLE;

    BEGIN
      ACOSH := LN(x + SQRT(SQR(x)-1));
    END {ACOSH};

  FUNCTION ATANH(x:DOUBLE):DOUBLE;

    BEGIN
      ATANH := 0.5 * LN((1+x)/(1-x));
    END {ATANH};

END.