{+------------------------------------------------------------
 | Unit FlexEdit
 |
 | Version: 1.0  Last modified: 10/06/94, 17:56:25
 | Author : P. Below
 | Project: FlexEdit
 | Description:
 |   This Unit provides a TEdit descendant, that is able to change
 |	 its window styles on the fly. It does this by saving position,
 |	 Z-order, font and content of the edit control window, destroying
 |	 the old window and creating a new one with the new styles. The
 |	 saved items are then set into the new edit window.
 |	 Methods are provided to change alignment, casing and readonly 
 |   styles. The latter is treated separately, because it can be
 |	 changed by sending a message. The other styles need recreation
 |	 of the window.
 |	 The new edit control will always have a border, if SetAlign or
 |	 SetCase are used to change the style. This is due to the fact
 |	 that Windows removes the WS_BORDER style from the edits window
 |	 style, so it is impossible to find out if an existing edit control
 |	 has a border or not.
 |	 If SetNewStyle is used to change the style it is the responsibility
 |	 of the calling party to include WS_BORDER in the passed styles.
 +------------------------------------------------------------}
Unit Flexedit;

Interface

Uses WinTypes, Win31, Objects, ODialogs, OWindows;

Type
	PConstructionInfo = ^TConstructionInfo;
	TConstructionInfo = Record
	  oldAttr : TWindowAttr;
		newstyle: LongInt;
		textlen : Integer;
		oldtext : PChar;
		oldfont : THandle;
		prevControl: HWnd;
		focusSelf: Boolean;
	End;

  PFlexEdit = ^TFlexEdit;
  TFlexEdit = Object( TEdit )
	  cInfo : PConstructionInfo;
		Constructor Init(AParent: PWindowsObject; AnId: Integer; 
		                 ATitle: PChar; X,Y,W,H, ATextLen: Integer; 
		                 Multiline : Boolean);
    Constructor InitResource(AParent: PWindowsObject; ResourceID: Word; 
														 ATextLen: Word);
    Procedure SetupWindow; virtual;
		Procedure UpdateAttr;
		Function  GetStyle: LongInt;
    Function  SetNewStyle( style: LongInt ): Boolean; virtual;
		Function  SetAlign( newAlign: LongInt ): Boolean;
		Function  SetCase( newCase : LongInt ): Boolean;
		Procedure SetReadOnly( on: Boolean );
		Function  GetConstructionInfo: Boolean;
		Procedure FreeConstructionInfo;
		Procedure WMSetFocus( Var msg: TMessage );
			virtual wm_first+wm_setfocus;
	End;

Implementation

Uses WinProcs;

(* The Init constructor just calls the inherited method and then
   sets the cinfo field to nil. *)
Constructor TFlexEdit.Init(AParent: PWindowsObject; AnId: Integer; 
                 ATitle: PChar; X,Y,W,H, ATextLen: Integer; 
                 Multiline : Boolean);
  Begin
  	inherited Init( AParent, AnID, ATitle, X, Y, W, H, ATextLen,
  	                Multiline );
    cInfo := Nil;
  End; { TFlexEdit.Init }

(* The InitResource constructor just calls the inherited method and then
   sets the cinfo field to nil. *)
Constructor TFlexEdit.InitResource(AParent: PWindowsObject; 
                 ResourceID: Word; ATextLen: Word);
  Begin
  	inherited InitResource( AParent, ResourceID, ATextLen );
		cInfo := Nil;
  End; { TFlexEdit.InitResource }

{************************************************************
 * Procedure TFlexEdit.SetupWindow;
 *
 * Parameters:
 *	none
 * Description:
 *	After calling the inherited method, this method checks if the
 *	cinfo field is <> nil. If it is, the method has been called
 *	because a new edit window has just been created. It then 
 *	proceeds to restore all the saved info from the old edit window,
 *	which has already been destroyed. This includes the edit text,
 *	the Z-order, font (if not the default), and focus, if it was
 *	on the old edit control. Size and position have already been
 *	taken care of by Create.
 *
 *	The method will also send an em_setreadonly message to the control
 *	if it has the es_readonly style.
 *
 * CAVEAT:
 *  If you derive an object type from TFlexEdit and override this
 *	SetupWindow method, do not assume that your SetupWindow will 
 *	only be called once! It will be called every time the edit 
 *	styles are changed (unless via SetReadOnly)!
 * Error Conditions:
 *	none
 *
 *Created: 10/06/94 18:06:03 by P. Below
 ************************************************************}
Procedure TFlexEdit.SetupWindow;
  Begin
  	inherited SetupWindow;
		If Assigned( cinfo ) Then 
		  With cInfo^ Do Begin
			  If textLen > 1 Then
				  SetText( oldText )
				Else
			    SetText( Nil );
				If prevControl = 0 Then
				  prevControl := HWND_BOTTOM;
				SetWindowPos( HWindow, prevControl, 0,0,0,0,
				              SWP_NOSIZE or SWP_NOMOVE );
				If oldFont <> 0 Then
				  SendMessage( HWindow, WM_SETFONT, oldFont, 0 );
				If focusSelf Then
				  PostMessage( Parent^.HWindow, WM_NEXTDLGCTL,
				               Hwindow, 1 );
				If ( newStyle and es_readonly ) <> 0 Then
					Attr.Style := Attr.Style or es_readonly;
		  End; { With }
			If ( Attr.style and es_readonly ) <> 0 Then 
			  SendMessage( HWindow, em_setreadonly, Ord( true ), 0 );
  End; (* TFlexEdit.SetupWindow *)

{************************************************************
 * Procedure TFlexEdit.UpdateAttr;
 *
 * Parameters:
 *	none
 * Description:
 *	This method updates the Attr record from the Window, in
 *	case position and size have been changed or the window
 *	has been created from a resource ( in which case Attr.Style
 *  will not reflect the actual style).
 * Error Conditions:
 *	none
 *
 *Created: 10/06/94 18:13:24 by P. Below
 ************************************************************}
Procedure TFlexEdit.UpdateAttr;
	var
	  WndRect: TRect;
  Begin
  	Attr.Style := ( Attr.Style and WS_BORDER ) or 
  	              GetWindowLong( HWindow, GWL_STYLE );
		Attr.ExStyle := GetWindowLong( HWindow, GWL_EXSTYLE );

		(* following code is a downtuned version of TWindow.UpdateWindowRect,
		   which is defined in OWindows but perversely as private! *)
    GetWindowRect(HWindow, WndRect);
    Attr.W := WndRect.right - WndRect.left;
    Attr.H := WndRect.bottom - WndRect.top;
    ScreenToClient(Parent^.HWindow, PPoint(@WndRect)^);
    Attr.X := WndRect.left;
    Attr.Y := WndRect.top;
  End; (* TFlexEdit.UpdateAttr *)

{************************************************************
 * Function  TFlexEdit.GetStyle: LongInt;
 *
 * Parameters:
 *	none
 * Returns:
 *	the current window style word
 * Description:
 *	Updates the Attr record and then returns the style. Due to
 *	an idiosyncrasy of the windows edit control, this style will
 *	not contain the WS_BORDER style, even if the window has a 
 *	border!
 * Error Conditions:
 *	none
 *
 *Created: 10/06/94 18:15:28 by P. Below
 ************************************************************}
Function  TFlexEdit.GetStyle: LongInt;
  Begin
  	UpdateAttr;
  	GetStyle := Attr.Style;
  End; (* TFlexEdit.GetStyle *)

{************************************************************
 * Function  TFlexEdit.SetNewStyle
 *
 * Parameters:
 *	style: new style to use for the edit control
 * Returns:
 *	true if successful, false if no memory could be allocated
 *	for the info that has to be saved.
 * Description:
 *	This method first calls GetConstructionInfo to save all the
 *	info on the current edit control we need to restore, then
 *	destroys the old edit window and creates a new one. SetupWindow
 *	gets called during the Create processing and restores the saved
 *	info into the new edit window. The stored info is finally 
 *	released again.
 * Error Conditions:
 *	may run out of memory
 *
 *Created: 10/06/94 18:17:13 by P. Below
 ************************************************************}
Function  TFlexEdit.SetNewStyle( style: LongInt ): Boolean; 
  Begin
  	If GetConstructionInfo Then Begin
		  SetNewStyle := True;
			If Attr.Style <> style Then Begin
				Destroy;
				HWindow := 0;
				cinfo^.newStyle := style or WS_VISIBLE;;
				Attr.Style := style;
				SetFlags( wb_FromResource, false );
				SetNewStyle := Create;
				FreeConstructionInfo;
			End; { If }
  	End { If }
		Else
		  SetNewStyle := False;
  End; (* TFlexEdit.SetNewStyle *)

{************************************************************
 * Function  TFlexEdit.SetAlign
 *
 * Parameters:
 *	newAlign: new alignment style to set. This style must be
 *	          one of es_left, es_center or es_right.
 * Returns:
 *	true if successful, false if failed due to out of memory
 * Description:
 *	Builds a new style for the edit from the old one and the
 *	passed parameter and calls SetNewStyle with this new style,
 *	which always includes WS_BORDER.
 * Error Conditions:
 *	SetNewStyle may run out of memory.
 *
 *Created: 10/06/94 18:21:20 by P. Below
 ************************************************************}
Function  TFlexEdit.SetAlign( newAlign: LongInt ): Boolean;
  Const
  	es_align = es_right or es_center;
  Var
  	oldStyle: LongInt;
  Begin
  	oldStyle := GetStyle;

		(* limit parameter to the align styles and make sure only
		   one is specified *)
		newAlign := newAlign and es_align;
		If newAlign = es_align Then
		  newAlign := es_center;

		(* check if the requested style is different from the current 
		   style. Take action if yes, else return true to indicate no error*)
		If ( oldStyle and es_align ) <> newAlign Then
		  SetAlign := SetNewStyle(( oldStyle and not es_align ) or newAlign 
		                or WS_BORDER )
		Else
		  SetAlign := True;
  End; (* TFlexEdit.SetAlign *)

{************************************************************
 * Function  TFlexEdit.SetCase
 *
 * Parameters:
 *	newAlign: new casing style to set. This style must be
 *	          one of es_uppercase, es_lowercase or 0.
 * Returns:
 *	true if successful, false if failed due to out of memory
 * Description:
 *	Builds a new style for the edit from the old one and the
 *	passed parameter and calls SetNewStyle with this new style,
 *	which always includes WS_BORDER.
 * Error Conditions:
 *	SetNewStyle may run out of memory.
 *
 *Created: 10/06/94 18:21:20 by P. Below
 ************************************************************}
Function  TFlexEdit.SetCase( newCase : LongInt ): Boolean;
  Const
  	es_case = es_uppercase or es_lowercase;
  Var
  	oldStyle: LongInt;
  Begin
  	oldStyle := GetStyle;

		(* limit parameter to the case styles and make sure only
		   one is specified *)
		newCase := newCase and es_case;
		If newCase = es_case Then
		  newCase := es_uppercase;

		(* check if the requested style is different from the current 
		   style. Take action if yes, else return true to indicate no error*)
		If ( oldStyle and es_case ) <> newCase Then
		  SetCase := SetNewStyle(( oldStyle and not es_case ) or newCase 
		                         or WS_BORDER )
		Else
		  SetCase := True;
  End; (* TFlexEdit.SetCase *)

{************************************************************
 * Procedure TFlexEdit.SetReadOnly
 *
 * Parameters:
 *	on: wether to set (true) or clear (false) the read-only state.
 * Description:
 *	This method uses the em_setreadonly message to change the
 *	edit to read-only and back. Due to the WM_SETFOCUS processing,
 *	a read-only edit cannot get the focus.
 * Error Conditions:
 *	none
 *
 *Created: 10/06/94 18:25:12 by P. Below
 ************************************************************}
Procedure TFlexEdit.SetReadOnly( on: Boolean );
  Begin
		If (( GetStyle and es_readonly ) <> 0) and not on Then
		  If on Then Begin
			  SendMessage( HWindow, em_setreadonly, Ord( true ), 0 );
				Attr.Style := Attr.Style or es_readonly;
		  End { If }
		  Else Begin
			  SendMessage( HWindow, em_setreadonly, Ord( false ), 0 );
				Attr.Style := Attr.Style and not es_readonly;
		  End; { Else }
  End; (* TFlexEdit.SetReadOnly *)

{************************************************************
 * Function  TFlexEdit.GetConstructionInfo
 *
 * Parameters:
 *	none
 * Returns:
 *	true if successful, false if out of memory
 * Description:
 *	Allocates memory for a TConstructionInfo record and stores
 *	the pointer in the cinfo field. The record is then filled
 *	with info on the current edit window.
 * Error Conditions:
 *	may run out of memory
 *
 *Created: 10/06/94 18:28:11 by P. Below
 ************************************************************}
Function  TFlexEdit.GetConstructionInfo: Boolean;
  Begin
  	New( cInfo );
		GetConstructionInfo := False;
		If Assigned( cInfo ) Then 
		  With cInfo^ Do Begin
		    textLen := GetTextLen+1;
			  GetMem( oldText, textLen );
				If Assigned( oldText ) Then Begin
				  GetText( oldText, textLen );
					UpdateAttr;
					oldAttr := Attr;
					newStyle := 0;
					oldfont := LoWord( SendMessage( Hwindow, WM_GETFONT, 0, 0 ));
					prevControl := GetWindow( HWindow, GW_HWNDPREV );
					focusSelf := GetFocus = HWindow;
					GetConstructionInfo := true;
				End { If }
				Else
				  Dispose( cInfo );
			End; { With }
  End; (* TFlexEdit.GetConstructionInfo *)

{************************************************************
 * Procedure TFlexEdit.FreeConstructionInfo;
 *
 * Parameters:
 *	none
 * Description:
 *	releases the memory pointed to by the cinfo field.
 * Error Conditions:
 *	none
 *
 *Created: 10/06/94 18:30:22 by P. Below
 ************************************************************}
Procedure TFlexEdit.FreeConstructionInfo;
  Begin
  	If Assigned( cInfo ) Then Begin
		  If Assigned( cinfo^.oldText ) Then
			  FreeMem( cinfo^.oldText, textlen );
			Dispose( cinfo );
			cinfo := Nil;
  	End; { If }
  End; (* TFlexEdit.FreeConstructionInfo *)

{************************************************************
 * Procedure TFlexEdit.WMSetFocus
 *
 * Parameters:
 *	msg: Standard message record
 * Returns:
 *	msg.result = 0 if message processed
 * Description:
 *	This message handles the WM_SETFOCUS message. It will
 *	prevent the edit from getting the focus, if it is read-only.
 * Error Conditions:
 *	none
 *
 *Created: 10/06/94 18:33:47 by P. Below
 ************************************************************}
Procedure TFlexEdit.WMSetFocus( Var msg: TMessage );
  Begin
  	If ( Attr.Style and es_readonly ) <> 0 Then Begin
		  PostMessage( Parent^.HWindow, WM_NEXTDLGCTL, 0, 0 );
			msg.result := 0;
  	End { If }
		Else
		  DefWndProc( msg );
  End; (* TFlexEdit.WMSetFocus *)

End.
