{ This unit implements a TPhoneColl, which is a collection of TPhone
  objects. It has methods which allow one to view the collection, one record
  at a time, and insert, edit or delete records. A TPhone is simply a record
  containing a name and phone number, along with appropriate Load and Store
  methods. TPhone can easily be modified to accomodate additional data.      }

{$X+}
unit PhoneDlg;

interface

uses
  Objects,Drivers,Views,Dialogs,App;

const
  cmAdd    = 104;  { Add a new TPhone }
  cmEdit   = 105;  { Edit the current TPhone }
  cmRemove = 106;  { Remove the current TPhone }
  cmNext   = 107;  { View next TPhone }
  cmPrev   = 108;  { View previous TPhone }

type
  NameStr = String[40];
  NumberStr = String[12];

  PPhone = ^TPhone;
  TPhone = object (TObject)
    Name: NameStr;
    Number: NumberStr;
    constructor Init (AName: NameStr; ANumber: NumberStr);
    constructor Load (var S: TStream);
    procedure Store (var S: TStream);
    end;

{ TPhoneColl is implemented as a descendant of TCollection. It may be more
  appropriate to implement it as a descendant of a TPhoneCollection, so that
  the TPhones can be sorted in alphabetical order by name, for example. The
  Show method opens up a dialog box which allows viewing and editing of the
  TPhones.                                                                   }

  PPhoneColl = ^TPhoneColl;
  TPhoneColl = object (TCollection)
    function Show: Word;
    end;

{ TViewDialog is a descendant of TDialog which is used to display the TPhone
  information, and allow for editing. PhoneColl points to the associated
  TPhoneColl object. Selected keeps track of which TPhone in the TPhoneColl
  is currently selected (and being viewed). SelName and SelNumber display
  the fields of the currently selected TPhone.                               }

  PViewDialog = ^TViewDialog;
  TViewDialog = object (TDialog)
    PhoneColl: PPhoneColl;
    Selected: Word;
    SelName,SelNumber: PStaticText;
    constructor Init (APhoneColl: PPhoneColl);
    procedure HandleEvent (var Event: TEvent); virtual;
    end;

{ The RegisterPhone procedure takes care of registering the newly defined
  object types so that they can be written to or read from a stream. Only
  those object types which are actually expected to be stored are
  registered.                                                                }

procedure RegisterPhone;

implementation

{ TPhone methods }

constructor TPhone.Init (AName: NameStr; ANumber: NumberStr);

begin
TObject.Init;
Name := AName;
Number := ANumber;
end;

constructor TPhone.Load (var S: TStream);

begin
S.Read (Name,SizeOf (NameStr));
S.Read (Number,SizeOf (NumberStr));
end;

procedure TPhone.Store (var S: TStream);

begin
S.Write (Name,SizeOf (NameStr));
S.Write (Number,SizeOf (NumberStr));
end;

{ TPhoneColl methods }

{ TPhoneColl.Show ExecViews a TViewDialog, and returns the result of that
  ExecView as its own result.                                                }

function TPhoneColl.Show: Word;

begin
Show := DeskTop^.ExecView (New (PViewDialog,Init (@Self)));
end;

{ ModifyRecord instantiates a dialog box which is used for adding a new
  TPhone record or editing an existing one. In the case of adding a record,
  the calling routine passes empty strings as the values of Name and Number
  in the Phone parameter; upon return, Phone contains the new values of Name
  and Number. In the case of editing, the calling routine passes the
  existing values of Name and Number in the Phone parameter, and they are
  replaced by the new values upon return. ModifyRecord returns a value equal
  to the result of ExecViewing the dialog; if the dialog was cancelled by
  the user, the Phone parameter is returned unaltered.                       }

function ModifyRecord (Phone: PPhone; Title: TTitleStr): Word;

var
  R: TRect;
  D: PDialog;
  N,P: PInputLine;

begin
R.Assign (27,11,73,21);
D := New (PDialog,Init (R,Title + ' a record'));
R.Assign (2,2,44,3);
N := New (PInputLine,Init (R,40));
N^.SetData (Phone^.Name);
D^.Insert (N);
R.Assign (2,1,44,2);
D^.Insert (New (PLabel,Init (R,'~N~ame',N)));
R.Assign (2,5,16,6);
P := New (PInputLine,Init (R,12));
P^.SetData (Phone^.Number);
D^.Insert (P);
R.Assign (2,4,44,5);
D^.Insert (New (PLabel,Init (R,'~P~hone',N)));
R.Assign (5,7,15,9);
D^.Insert (New (PButton,Init (R,'~O~K',cmOK,bfDefault)));
R.Assign (20,7,30,9);
D^.Insert (New (PButton,Init (R,'Cancel',cmCancel,bfNormal)));
D^.SelectNext (False);
ModifyRecord := DeskTop^.ExecView (D);
N^.GetData (Phone^.Name);
P^.GetData (Phone^.Number);
Dispose (D,Done);
end;

{ TViewDialog methods }

{ TViewDialog.Init is a basic dialog box constructor; nothing fancy here.    }

constructor TViewDialog.Init (APhoneColl: PPhoneColl);

var
  R: TRect;

begin
R.Assign (10,5,70,16);
TDialog.Init (R,'Phone List');
PhoneColl := APhoneColl;
Selected := 0;
R.Assign (2,2,42,3);
if PhoneColl^.Count > 0 then SelName := New (PStaticText,Init (R,
  PPhone (PhoneColl^.At (Selected))^.Name))
else SelName := New (PStaticText,Init (R,''));
Insert (SelName);
R.Assign (2,4,14,5);
if PhoneColl^.Count > 0 then SelNumber := New (PStaticText,Init (R,
  PPhone (PhoneColl^.At (Selected))^.Number))
else SelNumber := New (PStaticText,Init (R,''));
Insert (SelNumber);
R.Assign (48,1,58,3);
Insert (New (PButton,Init (R,'~N~ext',cmNext,bfNormal)));
R.Assign (48,4,58,6);
Insert (New (PButton,Init (R,'~P~rev',cmPrev,bfNormal)));
R.Assign (2,8,12,10);
Insert (New (PButton,Init (R,'~A~dd',cmAdd,bfNormal)));
R.Assign (13,8,23,10);
Insert (New (PButton,Init (R,'~E~dit',cmEdit,bfNormal)));
R.Assign (24,8,34,10);
Insert (New (PButton,Init (R,'~R~emove',cmRemove,bfNormal)));
R.Assign (37,8,47,10);
Insert (New (PButton,Init (R,'~S~ave',cmOK,bfDefault)));
R.Assign (48,8,58,10);
Insert (New (PButton,Init (R,'Cancel',cmCancel,bfNormal)));
SelectNext (False);
end;

{ TViewDialog.HandleEvent takes care of the special commands (cmAdd,cmEdit,
  cmRemove, cmNext, and cmPrev) used by TViewDialog. It also updates the
  Selected field and the text in SelName and SelNumber to display the
  currently selected TPhone. cmPrev and cmNext are enabled and disabled as
  appropriate, depending on the number of TPhones in the collection and
  which one is currently selected. cmEdit and cmRemove are disabled if the
  TPhoneColl is empty.                                                       }

procedure TViewDialog.HandleEvent (var Event: TEvent);

var
  P: PPhone;

begin
TDialog.HandleEvent (Event);
if Event.What = evCommand then
  begin
  case Event.Command of
    cmAdd: begin
      P := New (PPhone,Init ('',''));
      if ModifyRecord (P,'Add') <> cmCancel then PhoneColl^.Insert (P)
      else Dispose (P,Done);
      Selected := PhoneColl^.Count - 1;
      end;
    cmEdit: begin
{
      P := New (PPhone,Init (PPhone (PhoneColl^.At (Selected))^.Name,
        PPhone (PhoneColl^.At (Selected))^.Number));
      if ModifyRecord (PPhone (PhoneColl^.At (Selected)),'Edit') <> cmCancel then
        begin
        PPhone (PhoneColl^.At (Selected))^.Name := P^.Name;
        PPhone (PhoneColl^.At (Selected))^.Number := P^.Number;
        end;
      Dispose (P,Done);
}
      ModifyRecord (PPhone (PhoneColl^.At (Selected)),'Edit');
      end;
    cmRemove: begin
      PhoneColl^.AtDelete (Selected);
      if (Selected >= PhoneColl^.Count) and (Selected > 0) then
        Dec (Selected);
      end;
    cmNext: if Selected < PhoneColl^.Count - 1 then Inc (Selected);
    cmPrev: if Selected > 0 then Dec (Selected);
    end;
  DisposeStr (SelName^.Text);
  DisposeStr (SelNumber^.Text);
  if PhoneColl^.Count > 0 then
    begin
    SelName^.Text := NewStr (PPhone (PhoneColl^.At (Selected))^.Name);
    SelNumber^.Text := NewStr (PPhone (PhoneColl^.At (Selected))^.Number);
    end
  else begin
    SelName^.Text := NewStr ('');
    SelNumber^.Text := NewStr ('');
    end;
  Redraw;
  end;
if PhoneColl^.Count >= 1 then EnableCommands ([cmRemove,cmEdit])
else DisableCommands ([cmRemove,cmEdit]);
if Selected <= 0 then DisableCommands ([cmPrev])
else EnableCommands ([cmPrev]);
if Selected >= PhoneColl^.Count - 1 then DisableCommands ([cmNext])
else EnableCommands ([cmNext]);
end;

{ stream registration records }

const
  srPhone     = 10001;
  srPhoneColl = 10002;

  RPhone: TStreamRec = (
    ObjType: srPhone;
    VMTLink: Ofs (TypeOf (TPhone)^);
    Load: @TPhone.Load;
    Store: @TPhone.Store
  );

  RPhoneColl: TStreamRec = (
    ObjType: srPhoneColl;
    VMTLink: Ofs (TypeOf (TPhoneColl)^);
    Load: @TPhoneColl.Load;
    Store: @TPhoneColl.Store
  );

procedure RegisterPhone;

begin
RegisterType (RPhone);
RegisterType (RPhoneColl);
end;

end.
