{ Test shell for Async unit }

{ DEFINE Test}

PROGRAM TTY ;

uses
  Dos,
  Crt,
  BPlus,
  BPStatus,
  Async4,
  Timers;

Type
  ESC_State_Type = (No_ESC, ESC_Seen, ANSI_ESC);

CONST
  PAR_Path = '\TTY.PAR';       { Name and path to the parameter file }

VAR
  c             : char ;
  TestPort      : INTEGER ;
  TestRate      : aBpsRate ;
  TestParity    : aParitySetting ;
  TestWordLen   : byte ;
  TestStopBits  : byte ;
  CurrRate      : aBpsRate ;
  CurrParity    : aParitySetting ;
  CurrWordLen   : byte ;
  CurrStopBits  : byte ;
  DelayCount    : INTEGER ;
  YorN          : CHAR ;
  State         : (MenuMode, TermMode, Exitting) ;
  Open          : BOOLEAN ;
  ESC_State     : ESC_State_Type;
  ESC_String_Len: Word;
  ESC_String    : array [0..255] of Char;
  SaveX, SaveY  : Word;



PROCEDURE ClosePort ;

BEGIN { ClosePort }
  WRITELN ( 'Closing async' ) ;
  Async_Close ;   { reset the interrupt system, etc. }
  Open := FALSE
  END { ClosePort } ;

PROCEDURE OpenPort ;

BEGIN { OpenPort }
  IF NOT Async_Open ( TestPort,
                     TestRate,
                     TestParity,
                     TestWordLen,
                     TestStopBits )
  then
    begin
      WRITELN ('**ERROR: Async_Open failed') ;
      Open := FALSE
    END
  ELSE
    Open := TRUE;

  WRITELN ( 'Parameters set to:' ) ;

  WriteLn ('Port: ', Ord (TestPort));

  WRITE ( '  ' ) ;
  CASE TestRate OF
    bps110  : WRITE ( '110'  ) ;
    bps150  : WRITE ( '150'  ) ;
    bps300  : WRITE ( '300'  ) ;
    bps600  : WRITE ( '600'  ) ;
    bps1200 : WRITE ( '1200' ) ;
    bps1800 : write ('1800' ) ;
    bps2400 : WRITE ( '2400' ) ;
    bps4800 : WRITE ( '4800' ) ;
    bps9600 : WRITE ( '9600' )
  END ; { case }

  WRITELN ( ' bps' ) ;
  WRITELN ( TestWordLen:3, ' data bits' ) ;
  WRITELN ( TestStopBits:3, ' stop bits' ) ;
  WRITE ( '  ' ) ;
  CASE TestParity OF
    NoParity   : WRITE ( 'No' ) ;
    OddParity  : WRITE ( 'Odd' ) ;
    EvenParity : WRITE ( 'Even' )
  END ; { case }

  WRITELN ( ' parity' )

  END { OpenPort } ;



PROCEDURE WriteParamFile;
var
  ParamFile : text;

begin
  Assign (ParamFile, PAR_Path);
  Rewrite (ParamFile);

  WriteLn (ParamFile, TestPort);
  WriteLn (ParamFile, Ord (TestRate));
  WriteLn (ParamFile, Ord (TestParity));
  WriteLn (ParamFile, TestWordLen);
  WriteLn (ParamFile, TestStopBits);
  WriteLn (ParamFile, Ord (BP_Auto_Resume));
  WriteLn (ParamFile, 0);                    { Dummy! }

  Close (ParamFile);
end;


PROCEDURE SetParams ;

VAR
  Parity    : CHAR ;
  Rate      : word ;
  GoodPorts : aSetOfPorts ;
  NewUartBase : word ;
  NewIrq      : byte ;

BEGIN { SetParams }
    If Open
    then ClosePort;

    REPEAT
      WRITE ( 'Port (1=com1, 2=com2' ) ;
      IF Async_ComputerType = DG1 THEN
        WRITE ( ', 3=INternal modem' ) ;
      WRITE ( ')? ' ) ;
      READLN ( TestPort ) ;

      Async_AvailablePorts ( GoodPorts ) ;

      IF NOT (TestPort IN GoodPorts) THEN BEGIN
        WRITE ( '  Enter uart base address (in DECIMAL): ' ) ;
        READLN ( NewUartBase ) ;
        WRITE ( '  Enter irq: ' ) ;
        READLN ( NewIrq ) ;
        IF Async_DefinePort ( TestPort, NewUartBase, NewIrq ) THEN
          Async_AvailablePorts ( GoodPorts )
        ELSE
          WRITELN ( '*** Error defining port number ', TestPort, ' ***' )
        END
      UNTIL TestPort IN GoodPorts;

   WRITE ( 'Baud? ' ) ;
   READLN ( Rate ) ;
   TestRate := Async_MapBpsRate (Rate);
   WRITE ( 'Word length (7, 8)? ' ) ;
   READLN ( TestWordLen ) ;
   WRITE ( 'Stop bits (1, 2)? ' ) ;
   READLN ( TestStopBits ) ;
   WRITE ( 'Parity (O, E, N)? ' ) ;
   READLN ( Parity ) ;
   CASE upcase ( Parity ) OF
     'O' : TestParity := OddParity ;
     'E' : TestParity := EvenParity ;
     'N' : TestParity := NoParity
   END ;

   Write ('Automatic Download Resume is ');
   if BP_Auto_Resume
   then WriteLn ('enabled.')
   else WriteLn ('disabled.');
   Write ('    Do you wish to toggle it? ');
   ReadLn (Parity);

   if UpCase (Parity) = 'Y'
   then BP_Auto_Resume := not BP_Auto_Resume;


   Write ('Save parameters? ');
   ReadLn (YorN);

   if UpCase (YorN) = 'Y'
   then WriteParamFile;

   OpenPort;
END { SetParams } ;

{ Read the existing Parameter file, TTY.PAR }
{ If not found, call SetParams }
{$I-}   { Disable I/O failure }
procedure ReadParamFile;
var
  ParamFile : text;
  ParamPort      : Integer ;
  ParamRate      : Integer ;
  ParamParity    : Integer ;
  ParamWordLen   : byte ;
  ParamStopBits  : byte ;
  ParamResume    : byte;
  ParamDummy     : byte;

begin
  Assign (ParamFile, PAR_Path);
  Reset (ParamFile);

  if IoResult = 0
  then
    begin
      ReadLn (ParamFile, ParamPort);
      ReadLn (ParamFile, ParamRate);
      ReadLn (ParamFile, ParamParity);
      ReadLn (ParamFile, ParamWordLen);
      ReadLn (ParamFile, ParamStopBits);
      ReadLn (ParamFile, ParamResume);
      ReadLn (ParamFile, ParamDummy);

      if IoResult = 0
      then
        begin
          Close (ParamFile);
          TestPort := ParamPort;
          TestRate := aBpsRate (ParamRate);
          TestParity := aParitySetting (ParamParity);
          TestWordLen := ParamWordLen;
          TestStopBits := ParamStopBits;

          if ParamResume <> 0
          then BP_Auto_Resume := true
          else BP_Auto_Resume := false;

          OpenPort;
        end
      else
        begin
          Close (ParamFile);
          SetParams;
        end
    end
  else
    SetParams;
end;

{$I+}


{ Process ANSI Escape Sequence }
Procedure Do_ANSI (c : char);
var
  String_Index : Word;
  x, y, z : Integer;

  Function Get_Num : Integer;
  var
    Num : Integer;

  begin
    Num := 0;

    While (String_Index < ESC_String_Len) do
      begin
        Inc (String_Index);
        if ESC_String [String_Index] in ['0'..'9']
        then Num := Num*10 + (Ord (ESC_String [String_Index]) - Ord ('0'))
        else
          begin
            Get_Num := Num;
            exit;
          end;
      end;

    Get_Num := Num;
  end;

  Function Get_Num_Min_1 : Integer;
  var
    tmp : Integer;
  begin
    tmp := Get_Num;
    if tmp = 0
    then tmp := 1;
    Get_Num_Min_1 := tmp;
  end;

begin
  ESC_State := No_ESC;
  String_Index := 0;
{ShowIt (c);}
  Case C of
    'H', 'f' :
         { CUP	cursor position	ESC[y;xH Sets cursor position. }
         { HVP	cursor position	ESC[y;xf Same as CUP; not recommended. }
      begin
        y := Get_Num_Min_1;
        x := Get_Num_Min_1;

        if x > 79
        then x := 80;

        if y > 25
        then y := 25;

        GoToXY (x, y);
      end;

    'A' : { CUU	cursor up ESC[nA n = # of lines to move }
      begin
        x := Get_Num_Min_1;
        y := WhereY - x;
        If (y > 1)
        then GoToXY (WhereX, y);
      end;

    'B' : { CUD	cursor down ESC[nB }
      begin
        x := Get_Num_Min_1;
        y := WhereY + x;
        if (y < 25)
        then GoToXY (WhereX, y);
      end;

    'C' : { CUF	cursor forward ESC[nC n = # of columns to move }
      begin
        x := Get_Num_Min_1;
        x := WhereX + x;
        If (x > 80)
        then x := 80;
        GoToXY (x, WhereY);
      end;

    'D' : { CUB cursor backward ESC[nD }
      begin
        x := Get_Num_Min_1;
        x := WhereX - x;
        If (x < 1)
        then x := 1;
        GoToXY (x, WhereY);
      end;

    's' : { SCP Save Cursor Position ESC[s }
      begin
        SaveX := WhereX;
        SaveY := WhereY;
      end;

    'u' : { RCP Restore Cursor Position ESC[u }
      begin
        GoToXY (SaveX, SaveY);
      end;

    'K' : { EK Erase in Line    ESC[K  Clears to end of line. }
      ClrEol;

    'J' : { ED Erase in Display ESC[2J Clears screen. }
      begin
        x := Get_Num;
        if x = 2
        then
          begin
            ClrScr;
            ST_Display_Time_of_Day;
          end
        else if x = 0
        then           { Evidently ESC[0J clears to end of screen }
          begin
          x := WhereX;
          y := WhereY;
          ClrEol;
          for z := WhereY + 1 to 25 do
            begin
              GoToXY (1, z);
              ClrEol;
            end;
          GoToXY (x, y);
          end;
      end;
    end;
end;


PROCEDURE TermTest ;

  PROCEDURE Help ( ExitKey : string ) ;

  BEGIN { Help }
    WRITELN ;
    WRITELN ( '*** ', ExitKey, ' to exit ***' ) ;
    WRITELN
    END { Help } ;

  PROCEDURE Quit ;
  VAR
    Seconds : LongInt;

  BEGIN { Quit }
    Seconds := ElapsedSeconds (2);
    WRITELN ;
    WRITELN ('=== End of TTY Emulation ===');
    {$IFDEF Test}
      WRITELN ('Max Buffer Used = ', Async_MaxBufferUsed);
      {$ENDIF}
    WRITELN ;
    State := MenuMode;
    WriteLn ('Total time used:  ', ST_Time_String (Seconds));
    WriteLn ('Total bytes read: ', PortRead : 6, ' (',
             (PortRead div Seconds), ' CPS)');
    WriteLn ('Total bytes sent: ', PortSent : 6, ' (',
             (PortSent div Seconds), ' CPS)');
    END { Quit } ;

BEGIN { TermTest }

      {$IFDEF Test}
      WRITE ( 'Delay (milliseconds)? ' ) ;
      READLN ( DelayCount ) ;
      {$ENDIF}
    WRITELN ('TTY Emulation begins now...');
    WRITELN ('Press <F10> to terminate...');

    ESC_State := No_ESC;    { Not processing an ESC-sequence }
    State := TermMode ;

    REPEAT
      if Async_Buffer_Check ( c )
      then
        begin
        c := chr ( ord (c) and $7f ) ;

        Case ESC_State of
          No_ESC :
            begin
              CASE c OF
                #000 : ;  { strip incoming nulls }
                #005 :
                  begin
                    Async_Change ( TestRate, NoParity, 8, TestStopBits ) ;
                    BP_Term_ENQ;          { Do the Enquire Process }
                    Async_Change ( TestRate, TestParity, TestWordLen, TestStopBits ) ;
                  end;
                #016 :
                  begin
                    Async_Change ( TestRate, NoParity, 8, TestStopBits ) ;
                    BP_DLE_Seen;	    { Do B Plus Process      }
                    Async_Change ( TestRate, TestParity, TestWordLen, TestStopBits ) ;
                  end;
                #009 : begin              { Horizontal Tab         }
                         write (' ');     { Write at least one space }
                         while (WhereX - 1) mod 8 <> 0 do write (' ');
                       end;
                #010 : WriteLN;
                #012 : begin
                         ClrScr ; { clear screen on a form feed }
                         ST_Display_Time_of_Day;
                       end;
                #013 : GoToXY (1, WhereY);
                #027 : ESC_State := ESC_Seen;
                ELSE
                  WRITE ( c )  { else write incoming char to the screen }
                END { case }
              end; { No_ESC}

          ESC_Seen :
            begin
              Case c of
                '[' :     { Begin ANSI Escape Sequence }
                  begin
                    ESC_State := ANSI_ESC;
                    ESC_String_Len := 0;
                  end;
                'I' :     { Send <ESC><I> response }
                  begin

                    BP_Term_ESC_I ('#IBX,CA,SS7o,PB');
                    ESC_State := No_ESC;
                  end;
                else ESC_State := No_ESC;
                end;
            end;   { ESC_Seen }

          ANSI_ESC :      { Collect Parameters }
            begin
              if (c >= '@') and (c <= '~')
              then Do_ANSI (c)
              else
                begin
                  Inc (ESC_String_Len);
                  ESC_String [ESC_String_Len] := c;
                end;
            end;
          end;    { Case ESC_State }
        END ; { If }

      IF KeyPressed THEN BEGIN
        c := ReadKey ;
        IF (c = #0) THEN  { handle IBM Extended Ascii codes } BEGIN
          c := ReadKey ;  { get the rest of the extended code }
          CASE c OF
            #59 : {f1 } Help ( 'F10' ) ;
            #60 : {f2 } Help ( 'F10' ) ;
            #61 : {f3 } Help ( 'F10' ) ;
            #62 : {f4 } Help ( 'F10' ) ;
            #63 : {f5 } Help ( 'F10' ) ;
            #64 : {f6 } Help ( 'F10' ) ;
            #65 : {f7 } Help ( 'F10' ) ;
            #66 : {f8 } Help ( 'F10' ) ;
            #67 : {f9 } Help ( 'F10' ) ;
            #68 : {f10} Quit ;
            END ; { case }
          END
        ELSE
          Async_Send ( c )
        END
    {$IFDEF Test}
      ELSE
        delay ( DelayCount )
      {$ENDIF}
      UNTIL State = MenuMode
  END { TermTest } ;


PROCEDURE EnablePort ;

BEGIN { EnablePort }
  WRITE ( '  Enable: P (ort or D (TR? ' ) ;
  REPEAT
    c := upcase ( ReadKey )
    UNTIL c IN ['P', 'D'] ;
  WRITELN ( c ) ;
  IF c = 'P' THEN BEGIN
    WRITE ( '    Enable Port: via B (IOS or D (irect? ' ) ;
    REPEAT
      c := upcase ( ReadKey )
      UNTIL c IN ['B', 'D'] ;
    WRITELN ( c ) ;
    IF c = 'B' THEN BEGIN
      (*IF Async_dg1_enableport ( _async_Port, _dg1_IntOrExt ) THEN*)
        (*WRITELN ( '      Port enabled via BIOS' )*)
      END
    ELSE BEGIN
      writeln ( '*** NOT IMPLEMENTED YET ***' )
      END
    END
  (*|||
  ELSE BEGIN
    _async_dtr ( _async_Port, TRUE ) ;
    WRITELN ( '    DTR asserted' )
    END
  |||*)
  END { EnablePort } ;


PROCEDURE DisablePort ;

BEGIN { DisablePort }
  WRITE ( '  Disable: P (ort or D (TR? ' ) ;
  REPEAT
    c := upcase ( ReadKey )
    UNTIL c IN ['P', 'D'] ;
  WRITELN ( c ) ;
  IF c = 'P' THEN BEGIN
    WRITE ( '    Disable Port: via B (IOS or D (irect? ' ) ;
    REPEAT
      c := upcase ( ReadKey )
      UNTIL c IN ['B', 'D'] ;
    WRITELN ( c ) ;
    IF c = 'B' THEN BEGIN
      (*_dg1_disableport ( _async_Port, _dg1_IntOrExt ) ;*)
      (*WRITELN ( '      Port disabled via BIOS' )*)
      END
    ELSE BEGIN
      writeln ( '*** NOT IMPLEMENTED YET ***' )
      END
    END
  (*|||
  ELSE BEGIN
    _async_dtr ( _async_Port, FALSE ) ;
    WRITELN ( '    DTR cleared' )
    END
  |||*)
  END { DisablePort } ;



BEGIN { TtyDG }
  TextColor (LightGray);
  ClrScr ;
  Window (1, 2, 80, 24);   { Use 24 line screen }
  WRITELN ( '* TTY: Test driver for Async & BPlus units' ) ;
  WRITELN (
    '* Using Async  version ', Async4.UnitVersion, ' (', Async4.UnitVerDate, ')');
  WRITELN (
    '* Using BProto version ', BPlus.UnitVersion,  ' (', BPlus.UnitVerDate, ')');

  Open          := false ;
  DelayCount    := 1 ;
  TestPort      := 1 ;
  TestRate      := bps1200 ;
  TestWordLen   := 7 ;
  TestStopBits  := 1 ;
  TestParity    := EvenParity ;
  BP_Auto_Resume := true;             { Let BPLUS do resume downloads }

  BP_Quote_This (-1);                 { Clear the Special Quote Table  }
  BP_Quote_This ($01);  {^A}          { Set up as if we're using a     }
  BP_Quote_This ($81);  {^A + $80 }   { MicroCom MNP 2400 baud modem   }
  BP_Quote_This ($91);  {^Q + $80 }   { Actually, these would probably }
  BP_Quote_This ($93);  {^S + $80 }   { be preserved in TTY.PAR        }
                                      { but I'm lazy at the moment.    }
  ReadParamFile;
  ResetTimer (2);
  PortFlowControl := true;            { We want receive XON/XOFF to work }

{$IfNdef VER40}
  Async4.UpDateTimeOfDay := BPStatus.ST_Display_Time_of_Day;
{$EndIf}

  TermTest;

  REPEAT
    State := MenuMode ;
    WRITE ( 'S (et/change params, T (erminal, or Q (uit ' ) ;
    REPEAT
      c := upcase ( ReadKey ) ;
      UNTIL c IN ['S', 'T', 'Q'] ;
    WRITELN ( c ) ;
    CASE c OF
      'S' : SetParams ;
      'O' : OpenPort ;
      'T' : TermTest ;
      'E' : EnablePort ;
      'D' : DisablePort ;
      'C' : ClosePort ;
      'Q' : State := Exitting
      END ; { CASE }
    UNTIL State = Exitting ;
  IF Open THEN BEGIN
    WRITELN ( 'Closing async' ) ;
    Async_Close
    END
  END.

