{
Turbo Pascal ANSI Drivers
Version 1.12
Copyright (c) 1990 by Not So Serious Software

Original concept by Ian Silver
Design and implementation by Kevin Dean

Kevin Dean
Fairview Mall P.O. Box 55074
1800 Sheppard Avenue East
Willowdale, Ontario
CANADA    M2J 5B9
CompuServe ID: 76336,3114
}


{$I-,F-,S-,R-}
unit ANSI;


interface


uses
  DOS;


type
  IOFunc =				{ I/O function }
    function(var F : Text) : integer;

  IOFunctions =				{ I/O functions attached to ANSI driver }
    record
    NextOpen, NextInOut, NextFlush, NextClose : pointer
    end;


const
  { Special ASCII characters }
  NUL =
    #0;
  BRK =
    #3;
  BEL =
    #7;
  BS =
    #8;
  TAB =
    #9;
  LF =
    #10;
  FF =
    #12;
  CR =
    #13;
  EOF_ =
    #26;
  ESC =
    #27;

  { Color table }
  Black =
    0;
  Blue =
    1;
  Green =
    2;
  Cyan =
    3;
  Red =
    4;
  Magenta =
    5;
  Brown =
    6;
  LightGray =
    7;
  DarkGray =
    8;
  LightBlue =
    9;
  LightGreen =
    10;
  LightCyan =
    11;
  LightRed =
    12;
  LightMagenta =
    13;
  Yellow =
    14;
  White =
    15;
  Blink =
    128;

  On =				{ ANSI controls enabled }
    true;
  Off =				{ ANSI controls disabled }
    false;

  WrapAround : boolean =	{ True if cursor wrap around is supported }
    true;
  MaxX : byte =			{ Maximum number of columns on "screen" (0 = no maximum) }
    80;
  MaxY : byte =			{ Maximum number of rows on "screen" (0 = no maximum) }
    25;
  TabLength : byte =		{ Length between tab stops }
    8;

  TextAttr : byte =		{ Current text attributes }
    LightGray;

  CheckEOF : boolean =		{ Check for end of file on input }
    false;
  CheckBreak : boolean =	{ Check for Ctrl-Break }
    true;


var
  ANSIFile : Text;		{ ANSI control file }


function Redirected : boolean;
procedure ANSIStatus(Status : boolean);
procedure AssignANSI(var F : Text; IOChain : IOFunctions);
procedure ClrEol;
procedure ClrScr;
procedure Delay(MS : word);
procedure GotoXY(X, Y : byte);
function WhereX : byte;
function WhereY : byte;
procedure HighVideo;
procedure LowVideo;
procedure NormVideo;
procedure TextBackground(Color : byte);
procedure TextColor(Color : byte);


implementation


const
  OldTextAttr : byte =		{ Old text attributes }
    0;

  { Video attributes }
  Foreground =
    $07;
  BlueFgnd =
    $01;
  GreenFgnd =
    $02;
  RedFgnd =
    $04;
  BoldOn =
    $08;
  Background =
    $70;
  BlueBkgnd =
    $10;
  GreenBkgnd =
    $20;
  RedBkgnd =
    $40;
  BlinkOn =
    $80;

  ANSIOn : boolean =		{ ANSI control status }
    On;

var
  DelayCount : word;		{ Delay count used by Delay procedure }
  CursorX, CursorY : byte;	{ Current cursor position }


{***}
{ Check for redirection of standard input or output }
function Redirected : boolean;

const
  IOCTL =	{ IOCTL function }
    $44;
  GetDevInfo =	{ Get device information }
    $00;
  StdinNum =	{ Standard input file number }
    $00;
  StdoutNum =	{ Standard output file number }
    $01;
  IsDev =	{ Device flag }
    $80;
  IsConsole =	{ Console device active }
    $10;
  IsCot =	{ Console output device }
    $02;
  IsCin =	{ Console input device }
    $01;

var
  Regs : registers;		{ Registers for MSDOS call }
  _Redirected : boolean;	{ True if input or output have been redirected }

begin
{ Check for input redirection }
Regs.AH := IOCTL;
Regs.AL := GetDevInfo;
Regs.BX := StdinNum;
MSDOS(Regs);

_Redirected := Regs.DX and (IsDev or IsConsole or IsCin) <> IsDev or IsConsole or IsCin;
if not _Redirected then
  { Check for output redirection }
  begin
  Regs.AH := IOCTL;
  Regs.AL := GetDevInfo;
  Regs.BX := StdoutNum;
  MSDOS(Regs);

  _Redirected := Regs.DX and (IsDev or IsConsole or IsCot) <> IsDev or IsConsole or IsCot
  end;

Redirected := _Redirected
end;


{***}
{ Enable or disable ANSI control status }
procedure ANSIStatus(Status : boolean);

begin
ANSIOn := Status
end;


{***}
{ Write ANSI color escape sequence }
procedure WriteColor;

const
  TranslateColor : array [Black .. LightGray] of byte =		{ ANSI color table }
    (
    30, 34, 32, 36, 31, 35, 33, 37
    );

begin
OldTextAttr := TextAttr;

Write(ANSIFile, ESC, '[0');

if TextAttr and BoldOn = BoldOn then
  Write(ANSIFile, ';1');

if TextAttr and BlinkOn = BlinkOn then
  Write(ANSIFile, ';5');

Write(ANSIFile, ';', TranslateColor[TextAttr and Foreground]);
Write(ANSIFile, ';', TranslateColor[(TextAttr and Background) shr 4] + 10, 'm')
end;


{$F+}

{***}
{ Chain to next input function }
function ANSIInput(var F : Text) : integer;

begin
with TextRec(F) do
  if @IOFunctions(UserData).NextInOut <> nil then
    ANSIInput := IOFunc(IOFunctions(UserData).NextInOut)(F)
  else
    { Device read fault }
    ANSIInput := 161
end;


{***}
{ Translate text, modify cursor position, and chain to next output function }
function ANSIOutput(var F : Text) : integer;

var
  I : byte;		{ Buffer index }
  Result : integer;	{ I/O result }

begin
with TextRec(F) do
  if @IOFunctions(UserData).NextInOut <> nil then
    begin
    Result := 0;

    { Change color if necessary }
    if TextAttr <> OldTextAttr then
      begin
      WriteColor;
      Result := IOResult
      end;

    { Modify cursor position if output is not to ANSI device }
    if @F <> @ANSIFile then
      begin
      if BufPos <> 0 then
	{ Analyze buffer for control characters and modify cursor position }
	for I := 0 to BufPos - 1 do
	  case BufPtr^[I] of
	    BEL:
	      { Ignore }
	      ;

	    BS:
	      { Move back one character }
	      if CursorX <> 1 then
		Dec(CursorX);

	    TAB:
	      { Move to next tab stop (every TabLength characters) }
	      begin
	      CursorX := (CursorX + TabLength - 1) div TabLength * TabLength + 1;
	      if (MaxX <> 0) and (CursorX > MaxX) then
		CursorX := 1
	      end;

	    LF:
	      { Move to next line or scroll if necessary }
	      if (MaxY = 0) or (CursorY < MaxY) then
		Inc(CursorY);

	    FF:
	      { Move to top of screen }
	      begin
	      CursorX := 1;
	      CursorY := 1
	      end;

	    CR:
	      { Move to beginning of line }
	      CursorX := 1;

	    else
	      begin
	      Inc(CursorX);

	      { Check for end of line }
	      if (MaxX <> 0) and (CursorX > MaxX) then
		if WrapAround then
		  { Wrap around to next line }
		  begin
		  CursorX := 1;
		  if (MaxY = 0) or (CursorY < MaxY) then
		    Inc(CursorY)
		  end
		else
		  { Wrap around not supported, back up to end of this column }
		  CursorX := MaxX
	      end
	    end;

      if Result = 0 then
	{ Write text }
	ANSIOutput := IOFunc(IOFunctions(UserData).NextInOut)(F)
      else
	ANSIOutput := Result
      end
    else
      if ANSIOn then
	if Result = 0 then
	  { Write text }
	  ANSIOutput := IOFunc(IOFunctions(UserData).NextInOut)(F)
	else
	  ANSIOutput := Result
      else
	{ Ignore ANSI control sequence }
	ANSIOutput := Result
    end
  else
    { Device write fault }
    ANSIOutput := 160
end;


{***}
{ Flush ANSI file, chain to next flush function }
function ANSIFlush(var F : Text) : integer;

begin
with TextRec(F) do
  if @IOFunctions(UserData).NextFlush <> nil then
    ANSIFlush := IOFunc(IOFunctions(UserData).NextFlush)(F)
  else
    ANSIFlush := 0
end;


{***}
{ Open ANSI file, chain to next open function }
function ANSIOpen(var F : Text) : integer;

begin
with TextRec(F) do
  begin
  if Mode = fmInOut then
    Mode := fmOutput;

  if Mode = fmInput then
    InOutFunc := @ANSIInput
  else
    InOutFunc := @ANSIOutput;

  if @IOFunctions(UserData).NextOpen <> nil then
    ANSIOpen := IOFunc(IOFunctions(UserData).NextOpen)(F)
  else
    ANSIOpen := 0
  end
end;


{***}
{ Close ANSI file, chain to next close function }
function ANSIClose(var F : Text) : integer;

begin
with TextRec(F) do
  begin
  if @IOFunctions(UserData).NextClose <> nil then
    ANSIClose := IOFunc(IOFunctions(UserData).NextClose)(F)
  else
    ANSIClose := 0;

  Mode := fmClosed
  end
end;

{$F-}


{***}
{ Assign ANSI terminal control to a file }
procedure AssignANSI(var F : Text; IOChain : IOFunctions);

begin
with TextRec(F) do
  begin
  Mode := fmClosed;
  BufSize := sizeof(Buffer);
  BufPtr := @Buffer;
  OpenFunc := @ANSIOpen;
  InOutFunc := nil;
  FlushFunc := @ANSIFlush;
  CloseFunc := @ANSIClose;
  IOFunctions(UserData) := IOChain;
  Name[0] := #0
  end
end;


{***}
{ Clear to end of line }
procedure ClrEol;

begin
if TextAttr <> OldTextAttr then
  WriteColor;

Write(ANSIFile, ESC, '[K')
end;


{***}
{ Clear screen }
procedure ClrScr;

begin
if TextAttr <> OldTextAttr then
  WriteColor;

Write(ANSIFile, ESC, '[2J');
GotoXY(1, 1)
end;


var
  InterruptCount : byte;	{ Interrupt call counter }


{***}
{ Decrement InterruptCount on each successive call }
procedure CountInterrupts(Flags, CS, IP, AX, BX, CX, DX, SI, DI, DS, ES, BP : word);
interrupt;

begin
Dec(InterruptCount)
end;


{***}
{
Wait for InterruptCount to become 0 (becomes 0 on next clock tick).  Count the
number of cycles required to do this.

This code cannot be placed in the timer interrupt itself, since the BIOS code
that calls the timer interrupt will not call it again until the first call has
returned.
}
procedure SetDelayCount;

begin
inline
(
{ Initialize CX }

$B9/$FF/$FF/			{ MOV	CX, -1 }

{ Do this loop until timer interrupt called again and sets InterruptCount to 0 }

				{ WaitForEnd: }
$80/$3E/InterruptCount/$00/	{ CMP	byte ptr [InterruptCount], 0 }
$74/$02/			{ JE	SaveCount }
$E2/$F7/			{ LOOP	WaitForEnd }

{ Save CX in DelayCount }

				{ SaveCount: }
$F7/$D9/			{ NEG	CX }
$89/$0E/DelayCount		{ MOV	[DelayCount], CX }
);

{ Normalize to 1 millisecond }
DelayCount := (DelayCount + 54) div 55
end;


{***}
{ Implement delay counter with busy wait loop, simulate comparison to random memory location for accurate simulation }
procedure Delay(MS : word);

var
  DelayTime : longint;	{ Number of cycles to delay }

begin
{ Determine delay time }
DelayTime := longint(DelayCount) * longint(MS);

inline
(
{ Cycle through high word of delay }

$8B/$9E/DelayTime + 2/		{ MOV	BX, high word [DelayTime] }

				{ NextBX: }
$83/$FB/$00/			{ CMP	BX, 0 }
$74/$0E/			{ JZ	LowWord }

$33/$C9/			{ XOR	CX, CX }

{ Wait loop (high word) }

				{ AgainH: }
$80/$3E/$34/$12/$00/		{ CMP	byte ptr [1234], 0 }
$74/$00/			{ JE	LoopH }
				{ LoopH: }
$E2/$F7/			{ LOOP	AgainH }

$4B/				{ DEC	BX }
$EB/$F0/			{ JMP	NextBX }

{ Cycle through low word of delay }

$8B/$8E/DelayTime/		{ MOV	CX, low word [DelayTime] }
$83/$F9/$00/			{ CMP	CX, 0 }
$74/$09/			{ JZ	EndDelay }

{ Wait loop (low word) }

				{ AgainL: }
$80/$3E/$34/$12/$00/		{ CMP	byte ptr [1234], 0 }
$74/$00/			{ JE	LoopL }
				{ LoopL: }
$E2/$F9				{ LOOP	AgainL }
				{ EndDelay: }
)
end;


{***}
procedure InitDelay;

const
  TimerInt =	{ Timer interrupt }
    $1C;

var
  OldTimer : pointer;	{ Pointer to old timer interrupt }

begin
{ Save old timer interrupt }
GetIntVec(TimerInt, OldTimer);

{ Initialize InterruptCount and wait for first interrupt (timer edge) }
InterruptCount := 2;
SetIntVec(TimerInt, @CountInterrupts);
while InterruptCount <> 1 do;

SetDelayCount;

{ Restore old timer interrupt }
SetIntVec(TimerInt, OldTimer)
end;


{***}
procedure GotoXY(X, Y : byte);

begin
if ((MaxX = 0) or (X > 0) and (X <= MaxX)) and ((MaxY = 0) or (Y > 0) and (Y <= MaxY)) then
  begin
  Write(ANSIFile, ESC, '[', Y, ';', X, 'H');
  CursorX := X;
  CursorY := Y
  end
end;


{***}
{ Return X position of cursor }
function WhereX : byte;

begin
WhereX := CursorX
end;


{***}
{ Return Y position of cursor }
function WhereY : byte;

begin
WhereY := CursorY
end;


{***}
procedure HighVideo;

begin
TextAttr := TextAttr or BoldOn
end;


{***}
procedure LowVideo;

begin
TextAttr := TextAttr and not BoldOn
end;


{***}
procedure NormVideo;

begin
TextAttr := LightGray
end;


{***}
procedure TextBackground(Color : byte);

begin
TextAttr := (TextAttr and not Background) or ((Color and Foreground) shl 4)
end;


{***}
procedure TextColor(Color : byte);

begin
TextAttr := (TextAttr and not (Foreground or BoldOn)) or (Color and (Foreground or BoldOn))
end;


{***}
begin
InitDelay
end.