;       Lowlevel.asm - Call an interrupt service

;       This program is designed to permit calling any
;       interrupt from within dBASE or a similar
;       environment that lacks its own interface.

;       It expects at least one parameter, a string of
;       characters containing:

;               the number of the interrupt called
;               the names, and contents at the call
;                  to the interrupt if material, of
;                  all registers needed to pass
;                  values to the interrupt call or
;                  to return them.

;       The interrupt number should be expressed in
;       ASCII hexadecimal and followed by a comma.  On
;       return, the interrupt number will be replaced
;       with NC (no carry) or CF (carry flag set).  The
;       interrupt number must be first in the argument.

;       Registers may be named in the argument string
;       in any order.  Eight-bit or 16-bit registers
;       may be specified, but not 32-bit 80386
;       registers.  Registers should be named as "AX",
;       "AL", etc. and followed by their values at the
;       call (if material) in hexadecimal.  "=" signs
;       are encouraged for readability but not
;       required.  All register names and values should
;       be separated from the following register name
;       by a comma.  Attempts to specify registers CS,
;       IP, SS or SP will cause an error.  The flags
;       register may be named as "FX" in order to
;       access its values after the call, but any
;       values declared for it to hold before the call
;       will be ignored.
 
;       Enough additional spaces must be attached to
;       the argument to allow expansion of each
;       designated register to include an equal sign
;       and the number of digits, two or four, of its
;       value, since dBASE IV will truncate the
;       argument to its original length and will not
;       have access to values returned in its
;       extension.

;       If the argument cannot be parsed, it is
;       returned with a question mark following the
;       character that caused the error.

;       If using dBASE IV, string-type arguments and
;       buffers may be passed in one of two ways.
;       Inclusion in the first argument of a register
;       pair separated by a colon and followed by an
;       optional equal sign and a number from 2 through
;       the number of arguments passed instructs the
;       program to call the interrupt with the register
;       pair containing the address of the argument of
;       that number, and to return the contents of
;       memory pointed to by that register pair to the
;       argument. That is, to open a file called
;       "\dBASE\myfile", the call might be:

;               Arg1="21,AX=3D00,DS:DX=2"
;               Arg2="\dBASE\myfile"
;               CALL Lowlevel WITH Arg1,Arg2

;       On return, Arg1 might be "NC,AX=000A,DS:DX=?"
;       where the "NC" indicates no carry, showing
;       successful opening of the file, AX contains the
;       file handle, 000A hex or 10 decimal, and ? is
;       garbage. Arg2 will be garbage too, the contents
;       of memory pointed to by DS:DX after the call.

;       Alternatively, since some arguments contain
;       nulls which cannot be contained in dBASE
;       strings or can exceed 254 bytes, they can be
;       passed as buffers using the usual syntax.  One
;       way to create a buffer at a known location in
;       dBASE IV is to DECLARE an array of dimensions
;       such that rows*columns >=(bytes required)/56,
;       find its address using Where.bin, and pass the
;       address in the appropriate registers.  Such
;       registers and their values should be named
;       separately, not using colons.

;       Those DOS and BIOS calls that return an address
;       or buffer of which the address is unknown at
;       the call (such as the current DTA area or the
;       ROM address of a character set) should be
;       handled by placing the appropriate registers to
;       receive the address separately in the first
;       argument.  This program will return the address
;       and Peek.bin can be used to extract the data.

;       This routine may be called from programs other
;       than dBASE IV, provided the dBASE IV convention
;       of having ES:DI point to a table of far
;       pointers to the arguments and CX holding the
;       argument count is followed.

;       This program must be assembled and linked, then
;       converted to a binary image file by EXE2BIN.
;       It cannot be converted to a .COM file, and it
;       cannot be adapted to work in protected mode
;       because the code must modify itself to insert
;       the interrupt number into the "int 0"
;       instruction.

;------------------------------------------------------

;                       Code and Data

                .Model tiny
                code segment
                assume cs:code

Lowlevel        proc far        ; FAR return required
                jmp start

argtable        dd 7 dup (?)
buffdone        dw ?
argchars	dw ?
argstart	dw ?
argcount        dw ?
regnames        db 'ALAHBLBHCLCHDLDH'
                db 'SI  DI  BP  DS  ES  FX  '
regbytes	equ ($-regnames)/2 
regbuff         equ $
regax           dw ?
regbx           dw ?
regcx           dw ?
regdx           dw ?
regsi           dw ?
regdi           dw ?
regbp           dw ?
regds           dw ?
reges           dw ?
regfx           dw ?

linebuff        db 'Copyright (C) 1990, Jay Parsons'
                db 256-($-linebuff) dup (0)
buffend         equ $
buffwds         equ 256/2

start:          mov ax,es       ; point to arg pointers
                mov ds,ax       ; with ds:si
                mov si,di
                mov cs:argcount,cx
                mov ax,cs
                mov es,ax       ; and to our table
                mov di,offset cs:argtable  ; with es:di
                sal cx,1        ; 2 words per argument
                rep movsw       ; move them
                mov ds,ax       ; and restore ds

                assume ds:code

                les di,argtable ; load pointer to arg 1
                call countchar
                mov argchars,cx ; save its length
                mov bx,cx       ; twice
		mov ax,cs
		mov es,ax
		mov cx,buffwds
		mov di,offset linebuff
		xor ax,ax
                rep stosw       ; clear argument buffer
		mov di,offset regbuff
                mov cx,regbytes/2
                rep stosw       ; and register buffer
                mov di,offset buffend
                mov buffdone,offset buffend
                sub di,bx       ; set up for move here
                mov cx,bx       ; character count
                mov bx,di       ; new start of argument
                lds si,argtable ; point to argument 1
                rep movsb       ; move it to buffer end
                mov si,bx       ; it will now be source
		mov ax,cs
		mov ds,ax
                mov argstart,si ; for parsing and move
                mov di,offset linebuff ; to front
                call caps       ; get first real char
                mov ah,al
                call caps       ; and second
                cmp al,','      ; just a comma?
                jnz lowlev2
                mov al,'0'      ; then substitute 0
                xchg ah,al      ; and put front-words
lowlev2:        xchg ah,al      ; put back-words
                stosw           ; save int #
                mov al,','      ; and a comma
                stosb           ; save it
                mov si,argstart
                call hextobin   ; convert #
                cmp bx,100h     ; can't exceed FFh
		jae erret
                mov byte ptr intno,bl ; patch code
                call parse      ; then parse argument
                jc erret        ; quit if bad syntax
                mov buffdone,di ; mark end of argument
                mov byte ptr [di],0 ; also with ASCIIZ
                mov si,offset linebuff+3 ; point past #
                call fill       ; fill registers
                call irupt      ; call the interrupt
		mov si,offset linebuff
                call extract    ; get the return values
		mov si,offset linebuff

allret:         mov cx,argchars ; get count
                les di,argtable ; point to arg 1
                rep movsb       ; replace from buffer
                ret             ; all done

erret:          mov byte ptr [si],'?' ; mark the error
                mov si,argstart ; point to the string
		jmp short allret		
Lowlevel        endp

;       Parse - move through the string at ds:si
;       capitalizing register names, adding equal signs
;       and zeroes to hold returned results, and
;       copying it to the buffer at es:di.

Parse		proc near
parsenext:      call caps       ; get first letter
                cmp si,buffdone ; check for done
		jae parsedone		

;       test letters of register name

parse0:		cmp al,','
                jz parsenext    ; toss out commas
                call regcheck
                jnz parserr     ; if not, an error

;	save word for register, what's next

parse3:         stosw           ; save register name
                call caps       ; get next non-space
                cmp si,buffdone
		jae parse5
                cmp al,':'      ; a register pair?
                jz parse7       ; then jump
                cmp al,','      ; or comma
		jz parse5
                cmp al,'='      ; equals is OK
                jz parse4
                mov al,'='      ; else save =
                dec si          ; and put back value
parse4:         stosb           ; save the equal sign
                call getnsize
                call numget     ; get number if any
                jnc parse9

;       return code (here to keep jumps short)

parserr:	stc
parsedone:	ret

;	nothing or a comma	

parse5:         mov al,'='      ; add an equals
                stosb
                call getnsize   ; add how many zeroes?
                xor cx,cx       ; have no digits yet
                call zerofill   ; add them
                jmp short parse9

;       found a pair

parse7:         cmp byte ptr [di-1],'S' ; was #1 seg?
                jnz parserr     ; if not, error
                stosb           ; save the colon
                call caps       ; get first char
                call regcheck
                jnz parserr
                cmp ah,'S'
                jz parserr      ; can't be segreg
                stosw           ; save #2
                mov al,'='
                stosb           ; and an '='
                call caps       ; get arg # or =
                cmp al,'='      ; equal sign?
                jnz noeq        ; if not, keep it
                call caps       ; else get the next
noeq:           sub al,30h      ; convert to digit
                cbw             ; and extend
                cmp ax,argcount ; was there such arg #?
                ja parserr      ; above is error
                cmp ax,2
                jb parserr      ; as is one or less
                add al,30h      ; convert back to ASCII
                stosb           ; and save it

;       if done, fine.  Else save comma and continue

parse9:         call caps
		cmp si,buffdone
		jae parsedone
                dec si          ; put it back
                mov al,','      ; save the comma
                stosb
                jmp parsenext   ; and go on

Parse           endp

;       Fill - having prettied up the string, look
;       for values and move them to regbuff.

Fill		proc near
fill1:		cmp si,buffdone
		jae fillret
		lodsw
		call findreg	
		lodsb
                cmp al,':'
                jnz fillnum     ; if not colon, =

fillarg:        push di         ; save segment place
                lodsw
                call findreg    ; and offset place
                inc si          ; skip =
                lodsb           ; get next character
                sub al,30h      ; should be a digit
                cbw             ; make it a word
                sal ax,1
                sal ax,1        ; *4 bytes per pointer
                mov bx,offset argtable-4 ; arg 1 is 0
                add bx,ax       ; now arg # pointer
                mov ax,word ptr [bx] ; get arg offset
                stosw           ; save it in regbuff
                pop di
                mov ax,word ptr [bx+2] ; and segment
                jmp short fill3

fillnum:        mov cx,si       ; save this
                call hextobin   ; convert value to bin
		mov ax,bx
		mov dx,si		
                sub dx,cx       ; how many bytes/reg?
                cmp dx,5        ; 5 means there are 2
                jz fill3        ; so jump
                stosb           ; save one
		jmp short fill1			

fill3:          stosw           ; or two
                jmp short fill1 ; and carry on

fillret:        ret
FIll		endp

;       Irupt - set up registers for the interrupt from
;       regbuff, execute the interrupt, and put
;       register values returned into regbuff.
;       Destroys ax,bx,cx,dx,si and di.

Irupt		proc near
                push bp
                push es
                push ds
                mov ax,word ptr regax
		mov bx,word ptr regbx
		mov cx,word ptr regcx
		mov dx,word ptr regdx
		mov si,word ptr regsi
		mov di,word ptr regdi
		mov bp,word ptr regbp
                mov es,word ptr reges
                mov ds,word ptr regds
intno           equ $+1         ; the interrupt number
                int 0           ; will replace the 0
                mov word ptr cs:regds,ds
                pop ds
                mov word ptr regax,ax
                pushf
		pop ax
		mov word ptr regfx,ax
                mov word ptr regbx,bx
		mov word ptr regcx,cx
		mov word ptr regdx,dx
		mov word ptr regsi,si
		mov word ptr regdi,di		
		mov word ptr regbp,bp
                mov word ptr reges,es
                pop es
                pop bp
                ret
Irupt		endp
	
;       Extract - extract values from register buffer
;       and place them in parameter string, after the
;       carry status.

Extract		proc near
                jc gotcarry
                mov word ptr [si],'CN' ; save 'NC'
                jmp short extr0
gotcarry:       mov word ptr [si],'YC' ; or 'CY'
extr0:          add si,3        ; bump past comma

extr1:          cmp si,buffdone
		jae extret
		lodsw
                cmp byte ptr [si],':' ; got a double?
                jz extr5

                call findreg    
extr2:          inc si          ; bump past '='
                cmp ah,'L'      ; short register?
                jz extr3
                cmp ah,'H'      ; H is short too
                jnz extr4

;	8-bit register

extr3:          mov al,byte ptr [di]
		mov cx,2
		call bintohex
		jmp short extr1

;	16-bit register

extr4:          mov ax,word ptr [di]
		mov cx,4
		call bintohex
		jmp short extr1

;       register pair
                     
extr5:          add si,3
                lodsb
                sub al,30h      ; convert arg #
                cbw
                sal ax,1
                sal ax,1
                mov bx,offset argtable-4
                add bx,ax       ; point bx to argptr
                push si         ; save our place
                mov ax,word ptr [si-7] ; segreg name
                call findreg
                push di         ; return segment addr
                mov ax,word ptr [si-4] ; now offset
                call findreg
                mov si,di       ; return offset addr
                mov di,word ptr [bx]
                mov es,word ptr [bx+2] ; es:di to arg
                call countchar  ; characters to copy
                mov di,word ptr [bx]
                mov es,word ptr [bx+2] ; arg again
                pop ds          ; segment of results
                rep movsw       ; move result to arg
                mov ax,cs
                mov ds,ax       ; restore segment
                pop si          ; and place pointer
extr6:          call caps       ; save terminator
                stosb
                cmp al,0        ; if not done,
                jnz extr1       ; continue

extret:		ret
Extract		endp

;       Bintohex - Translate number in al or ax into
;       ASCII hexadecimal of the number of bytes in cx,
;       store at es/ds:si, adjust si past it and the
;       comma if any.

Bintohex	proc near
                mov dx,ax       ; copy number
                mov bx,cx       ; count
                mov di,si       ; and location
                mov al,'0'      ; fill with zeroes
		rep stosb
		mov cx,bx
                dec di          ; point to last one
                std             ; go backwards
binloop:	cmp dx,0
		jz bin3
                mov al,dl       ; get least part
		and al,0Fh		
                add al,30h      ; convert to ASCII
		cmp al,'9'
		jbe bin2
                add al,7        ; correct for letters
bin2:           stosb           ; save it
		shr dx,1
		shr dx,1
		shr dx,1
                shr dx,1        ; shift in next digit
		loop binloop
bin3:		add si,bx
                inc si          ; bump past any comma
		cld
		ret

Bintohex	endp

;       Hextobin - ASCII hex to binary conversion
;       routine. Converts a number expressed in ASCII
;       hexadecimal at DS:SI to binary value in BX.
;       Does not check for overflow but returns value
;       mod 65,536. Ignores spaces, treats any other
;       non-digit as terminator.  Cannot handle
;       negative numbers. Destroys AX,BX.  On return,
;       SI points past terminator.

Hextobin	proc near
		push cx
                mov cl,4        ; shift count
                xor bx,bx       ; initialize number

htob_next:      lodsb           ; get a character
                cmp al,' '      ; space?
                jz htob_next    ; ignore spaces
                sub al,'0'      ; else make it hex
                jb htob_done    ; done if less than 0
                cmp al,9        ; a digit?
                jbe htob_num    ; jump for that
                and al,1Fh      ; capitalize letters
                sub al,7        ; make 'A'=0ah, etc.
                cmp al,0ah      ; was it A-F?
                jb htob_done    ; done if not
		cmp al,0fh		
		ja htob_done
		
htob_num:       sal bx,cl       ; shift old val left 4
                add bl,al       ; add the new
		jmp short htob_next
	
htob_done:	pop cx
		ret
Hextobin	endp

;       Countchar - count characters in an ASCIIZ
;       string at ES:DI

Countchar	proc near	
		xor al,al			
                mov cx,-1
		repne scasb
                not cx          ; make cl the count
countret:	ret
Countchar	endp

;       Getnsize - Find number of digits needed for the
;       register whose second letter is in ah.  Return
;       number in bx and cx.

Getnsize	proc near
                mov cx,2        ; assume two hex digits
                cmp ah,'H'      ; if 'H' register
		jz getn1
                cmp ah,'L'      ; or 'L'
		jz getn1
                mov cx,4        ; otherwise, four
getn1:          mov bx,cx       ; save twice
		ret
Getnsize	endp

;       Numget - get a number (if any) and format it
;       into buffer with correct number of hex digits,
;       adding leading zeroes as needed.

Numget		proc near
		xor cx,cx
                dec si          ; back up

numstart:       inc si          ; next digit
                cmp byte ptr [si],' ' ; strip spaces
                jz numstart
                cmp byte ptr [si],'=' ; and =
                jz numstart
                     
numnext:        lodsb
                cmp al,'0'
                jb numgend      ; test for 0-9,A-F
                cmp al,'9'
		jbe num3
                and al,5fh      ; capitalize letters
                cmp al,'F'              
		ja numgend
		cmp al,'A'
		jb numgend		 
num3:           stosb           ; save any good char
                inc cx          ; one less to get
                jmp short numnext ; and loop
numgend:        call zerofill
numerr:		ret
Numget		endp

;       Zerofill - check for two or four "digits", as
;       required, found.  Too many is an error, just
;       return.  If too few, move any forward last-to-
;       first by the number of digits short and pad
;       with zeroes.  On entry, bx is desired number
;       of digits, cx is number found.

Zerofill	proc near	
                sub bx,cx       ; digits short
                jbe zerret      ; too many = carry
                mov ax,cx       ; # digits to move
                mov dx,si       ; rest of string
                jcxz zer1       ; none found=no move
                dec di          ; last char done
		mov si,di
                add di,bx       ; move them this far
                std             ; backwards
		rep movsb
                cld             ; clear direction
                sub di,bx       ; space cleared
		inc di
zer1:           mov cx,bx       ; the count again
                mov bx,ax       ; and digits moved
                mov al,'0'      ; load a zero
                rep stosb       ; fill with zeroes
                add di,bx       ; point past digits
                mov si,dx       ; restore
zerret:		ret	
Zerofill	endp

;       Regcheck - get the second letter of presumed
;       register, capitalize it, check for valid
;       register, return zero if so.

Regcheck        proc near
                mov ah,al       ; save letter
                call caps       ; get second letter
                xchg ah,al      ; switch
                push di
                call findreg    ; is it valid reg?
                pop di
                ret
Regcheck        endp

;       Findreg - look in ds/es:regnames table for word
;       in ax.  If found, point di to its spot in
;       regbuff and return with zero set, else nonzero.
;       Since the "X" registers are not in the table,
;       calculate for them.  Destroys cx,dx.

Findreg		proc near
                mov di,offset regnames          
		mov cx,regbytes
                mov dx,cx
                repne scasw  ; find reg in regnames
                jnz findx    ; jump if not there
                sub dx,cx
                mov di,offset regbuff ; regbuff spot
                add di,dx
		dec di			
                xor dx,dx    ; zero means found
                jmp short findxret

findx:		push ax
		mov di,offset regnames
                sub al,'A'   ; calculate offset
                jb finderr
		cbw
                sal ax,1     ; 2 bytes each
		mov di,offset regbuff
                add di,ax    ; offset we want
		pop ax
		cmp al,'D'
                ja findxret
                cmp ah,'X'   ; nonzero if not X reg
findxret:       ret

finderr:        pop ax
                ret

Findreg         endp

;       Caps - get next non-space char and capitalize
;       it if a letter

Caps		proc near
caploop:	lodsb
                cmp al,0        ; at end?
		jz capsdone			
                cmp al,' '      ; ignore spaces
		jz caploop
		cmp al,'a'
		jb capsdone
                and al,5Fh      ; capitalize if letter
capsdone:	ret
Caps		endp

code            ends    
                end

