/*    DBF2PRG.C                                                             */
/*    Version 1.0     May 5, 1992                                           */
/*    Programmer:     Bowen Moursund                                        */

/*  Note: Thanks to Jay Parsons for the DBF structures and some of the      */
/*        routines used. See his DBDOC.ZIP.                                 */
/****************************************************************************/

#include <dos.h>
#include <string.h>
#include <fcntl.h>
#include <io.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <errno.h>

/****************************************************************************/
/*                          definitions                                     */
/****************************************************************************/

/****                      ****  values  ****                            ****/

#define FALSE          0
#define TRUE           1
#define HEADLEN       32            /* sizeof ( mainhead or thisfield )     */
#define MEMOLEN       10            /* length of memo field in .dbf         */
#define DBFTYPE        3            /* value of bits 0 and 1 if .dbf        */
#define EOH         0x0D            /* end-of-header marker in .dbf file    */

/****                     ****  data types  ****                         ****/

typedef unsigned long ulong;        /* just an abbreviation                 */
typedef unsigned char byte;

/*  first 32 bytes of a .dbf file   */

typedef struct
{
    byte                dbf:2,      /*  both 1 for dBASE III or IV .dbf     */
                           :1,
                     db4dbt:1,      /*  1 if a dBASE IV-type .dbt exists    */
                           :3,
                     anydbt:1;      /*  1 if any .dbt exists                */
    char filedate[3];               /*  date, YYMMDD, binary. YY=year-1900  */
    ulong records;                  /*  records in the file                 */
    unsigned headerlen;             /*  bytes in the header                 */
    unsigned recordlen;             /*  bytes in a record                   */
    int rsrvd1;
    char incompleteflag;            /*  01 if incomplete, else 00           */
    char encryptflag;               /*  01 if encrypted, else 00            */
    char rsrvd2[12];                /*  for LAN use                         */
    char mdxflag;                   /*  01 if production .mdx, else 00      */
    char rsrvd3[3];
} dbfheader;

/*  field descriptor of a .dbf file         */

typedef struct
{
    char fname[11];                 /*  field name, in capitals, null filled*/
    char ftype;                     /*  field type, C, D, F, L, M or N      */
    char rsrvd4[4];                 /*  used in memvars, not in files.      */
    byte flength;                   /*  field length                        */
    byte decimals;                  /*  number of decimal places            */
    int rsrvd5;
    char workarea;
    char rsrvd6[10];
    char mdxfield;                  /* 01 if tag field in production .mdx   */
} descriptor;

/****              ****  function prototypes  ****                       ****/

void prg_header(FILE *prgfile, char *out_name, char *recreate_name);

void dbf_doc( FILE *dbffile, FILE *prgfile, dbfheader *buf, char *recreate_name );

void rest_of_prg( FILE *dbffile, FILE *prgfile, char *out_name, char *recreate_name, char *outpath);

FILE *dbfopen( char *filespec, dbfheader *buf );


void main( int argc, char *argv[] )
{
    char outpath[_MAX_PATH], fname[_MAX_FNAME], recreate_name[13],
         out_name[13], ext[_MAX_EXT], drive[_MAX_DRIVE], dir[_MAX_DIR];
    int outpathspec;
    FILE *dbffile, *prgfile;
    dbfheader mainhead;
    struct find_t find;

    outpathspec = FALSE;
    fprintf(stdout,"\nDBF2PRG.EXE version 1.0 05/05/1992 * Bowen Moursund  ");
    switch(argc)
    {
    case 2:
        break;
    case 3:
        outpathspec = TRUE;
        break;
    default:
        fprintf(stdout,"\nA utility to produce dBase IV code that re-creates DBFs.");
        fprintf(stdout,"\nSyntax: DBF2PRG <DBF filespec> <PRG filespec>" );
        fprintf(stdout,"\nThe output file name defaults to <DBF name>.PRG");
        exit(0);
    }

    /* determine file name to re-create */
    _splitpath( argv[1], drive, dir, fname, ext );
    _makepath( recreate_name, "", "", fname, ext);

    /* determine output file path */
    if(outpathspec)
    {
        _splitpath( argv[2], drive, dir, fname, ext );
        _makepath( outpath, drive, dir, fname, ext );
        _makepath( out_name, "", "", fname, ext );
    }
    else
        _makepath( outpath, "", "", fname, ".PRG" );
        _makepath( out_name, "", "", fname, ".PRG" );

    strupr( outpath );
    strupr( out_name );
    strupr( recreate_name );

    if ( ( dbffile = dbfopen( argv[1], &mainhead ) ) == NULL )
    {
        fprintf( stdout, "\nOops. Some sort of problem reading %s.", argv[1] );
        fprintf( stdout, "\nPerhaps %s doesn't exist? Hmmm?", argv[1] );
        fprintf( stdout, "\nOr maybe the DBF is corrupt, or perhaps it's an OS error." );
        exit(1);
    }

    if( !_dos_findfirst( outpath, 0xffff, &find ) )
    {
        fprintf(stdout,"\nFile %s already exists. Overwrite? ",outpath);
        if( toupper(getch()) != 'Y' )
        {
            fclose( dbffile );
            exit(0);
        }
    }

    if ( ( prgfile = fopen( outpath, "w+" ) ) == NULL )
    {
        perror( outpath );
        exit(1);
    }

    prg_header( prgfile, out_name, recreate_name );
    dbf_doc( dbffile, prgfile, &mainhead, recreate_name );
    rest_of_prg( dbffile, prgfile, out_name, recreate_name, outpath);
    exit(0);

}

void prg_header( FILE *prgfile, char *out_name, char *recreate_name )
{
    fprintf( prgfile, "* %s\n", out_name );
    fprintf( prgfile, "* Generated by DBF2PRG.EXE version 1.0 05/05/1992\n");
    fprintf( prgfile, "* This procedure will recreate the file %s\n*\n", recreate_name );
}

void dbf_doc( FILE *dbffile, FILE *prgfile, dbfheader *buf, char *recreate_name )
{
    char message[80], tempbuff[80];
    descriptor thisfield;
    int field, fields, err;

    fprintf( prgfile, "* Structure of %s\n", recreate_name );
    sprintf( message, "* %d records, last changed ",  ( int ) buf->records );
    sprintf ( tempbuff, "%02d/", buf->filedate[1] );
    strcat( message, tempbuff );
    sprintf ( tempbuff, "%02d/", buf->filedate[2] );
    strcat( message, tempbuff );
    sprintf ( tempbuff, "%d", buf->filedate[0] + 1900 );
    strcat( message, tempbuff );
    fprintf( prgfile, "%s\n", message );

    fprintf( prgfile, "* Field    Type  Len  Dec\n" );

/*  do it field by field.  We are at byte 32 of file  */

    fields = ( buf->headerlen - 33 ) / 32;
    for ( field = 1; field <= fields; field++ )
    {
        err = fread( &thisfield, HEADLEN, 1, dbffile ) - 1;
        if ( err )
            break;
        sprintf( message, "%-11s", thisfield.fname );
        strcat( message, " " );
        *tempbuff = thisfield.ftype;
        *( tempbuff + 1 ) = '\0';
        strcat( message, tempbuff );
        sprintf( tempbuff, "  %3d   %2d", thisfield.flength, thisfield.decimals );
        strcat( message, tempbuff );
        fprintf( prgfile, "* %s\n", message );
    }
    fprintf( prgfile, "\n");
}

void rest_of_prg( FILE *dbffile, FILE *prgfile, char *out_name, char *recreate_name, char *outpath)
{
    char out_buf[16];
    unsigned int line_length, string_length, buf_length;
    byte currentbyte;
    rewind( dbffile );
    fprintf( prgfile, "SET PRINTER TO FILE %s\n", recreate_name );
    fputs( "SET PRINTER ON\n", prgfile );

    fputs( "??? \"", prgfile );
    line_length = 5;
    string_length = 0;
    /* process source file byte by byte */
    while ( fread( &currentbyte, 1, 1, dbffile) )  /* 0 means EOF */
    {
        /* convert current byte to DB4 print redirection decimal construct */
        buf_length = sprintf( out_buf, "{%d}", currentbyte );
        fputs(out_buf,prgfile);
        line_length = line_length + buf_length;
        string_length = string_length + buf_length;
        if (string_length > 245)
        {
            fputs("\"\n??? \"",prgfile);
            line_length = 5;
            string_length = 0;
        }
        else
        {
            if (line_length > 72)
            {
                fputs("\"+;\n\"",prgfile);
                line_length = 1;
            }
        }
    }
    fclose( dbffile );
    fputs( "\"\n", prgfile );
    fputs( "SET PRINTER TO\n", prgfile );
    fputs( "SET PRINTER OFF\n", prgfile );
    fputs( "RETURN\n\n", prgfile );
    fprintf( prgfile, "* End of file %s", outpath );
    fclose( prgfile );
}

/****************************************************************************/
/*                              dbfopen                                     */
/*      Routine to open .dbf file.                                          */
/*  Parameters:                                                             */
/*      char *filespec  -- pointer to spec of file to open                  */
/*      dbfheader *buf  -- pointer to dbfheader structure                   */
/*  Returns:                                                                */
/*      Pointer to C file control structure (to NULL if error)              */
/*  Side effects:                                                           */
/*      Opens file and moves pointer to byte 32; fills buffer at buf with   */
/*  first 32 bytes of file.  Closes file on error.                          */
/****************************************************************************/

FILE *dbfopen( char *filespec, dbfheader *buf )
{
    FILE *file;
    char *endmark;

    if ( ( file = fopen( filespec, "rb" ) ) == NULL )
    return NULL;                            /* can't open .dbf              */

/* read the first 32 bytes into buffer      */

    if ( fread( buf, HEADLEN, 1, file ) != 1 )
    {
        fclose( file );
        return NULL;                        /* can't read 32 bytes          */
    }

/* check first byte to be sure .dbf type                                    */

    if ( buf->dbf != DBFTYPE )
    {
        fclose( file );
        return NULL;                        /* not a .dbf file              */
    }

/* check last byte of header                */

    if ( fseek( file, buf->headerlen - 1, SEEK_SET ) != 0 )
    {
        fclose( file );
        return NULL;                        /* header corrupted             */
    }
    if ( fread( endmark, 1, 1, file ) != 1 )
    {
        fclose( file );
        return NULL;                        /* can't read end of header     */
    }
    if ( *endmark != EOH )
    {
        fclose( file );
        return NULL;                        /* no 0Dh at end of header      */
    }
    fseek( file, HEADLEN, SEEK_SET );
    return file;
}

