/* Read dBASE II and III .DBF files from your C program!
 * This program does not create or write to the .DBF files,
 * just reads them so that you can use the data in your program.
 *
 * This source code written for the Microsoft C v4.0 compiler.
 * Compile with the /Zp (structure packing) flag and link with a 
 * stack size of 10000 or so.
 * Like this:
 *      msc /Zp cdb;
 *      link cdb, /STACK:10000;
 *
 * This code is thrown out in public domain in the hope that it
 * saves someone some time.  I don't offer any guarantees other than
 * it does what it is supposed to (which is more than most commercial
 * products will do).
 * 
 * So how about returning the favor?  I am looking for the same 
 * type of source code to read Lotus .WKS, .WK1, and Symphony files.
 * If this dBASE function is of use to you, please keep a lookout for
 * similar source for the Lotus files and leave me a note on Compuserve,
 * Mikes C Board (619)722-8724, or voice phone (503)232-4955.
 * Thanks.
 *
 * Bert Sperling
 * Fast Forward, Inc.
 */


#include "string.h"
#include "stdio.h"
#include "stdlib.h"
#include "math.h"
#include "dos.h"
#include "malloc.h"
#include "conio.h"
#include "fcntl.h"
#include "io.h"


/************ defines and structures for dBASE III files ***************/

#define MAX3FIELD        128     /* maximum fields per record */
#define MAX3RECLEN       4000    /* maximum record length in bytes */
#define MAXFLDLEN        254

/*DBASE header structure*/
struct db3head
        {
        char    db;             /* file flag */ 
                          /* 03h=no .DBT file    83h=.DBT file present */
        char    year;           /* \                            */
        char    month;          /*  >  date of last update      */
        char    day;            /* /                            */
        long    quan;           /* number of records in file    */
        int     headlen;        /* length of header structure   */
        int     reclen;         /* number of bytes in record    */
        char    misc[20];       /* other stuff                  */
        };

/*DBASE field structure*/
struct db3field
        {
        char name[11];          /*field name*/
        char type;           /* C = Character
                                N = Numeric
                                L = Logical (YyNnTtFf)
                                M = Memo (10 digits representing a .DBT
                                    block number)
                                D = Date (8 digits in YYYYMMDD format) */
        long fdaddr;            /*  field data address 
                                    (32 bit number set by dBASE in memory) */
        char length;            /* field length */
        char decimal;           /* decimal positions if numeric */
        char other[13];         /* reserved by dBASE */
/*      CR (0Dh)                   used as a field terminator */
        };

struct db3info
        {
        struct  db3head info;            /*file header information*/
        int     fieldnum;               /*number of fields per record*/
        struct  db3field field[MAX3FIELD];         /*field pointer*/
        } db3;




/************ defines and structures for dBASE II files ***************/

#define MAX2FIELD         32     /* maximum fields per record */
#define MAX2RECLEN      1000     /* maximum record length in bytes */

/*DBASE header structure*/
struct db2head {
        char    db;             /* flag: 02h = dBASE II file */
    unsigned    quan;           /* number of records in file */
        char    month;          /* \                         */
        char    day;            /*  >  date of last update   */
        char    year;           /* /                         */
        int     reclen;         /* number of bytes in record */
         };

/*DBASE field structure*/
struct db2field
        {
        char name[11];          /*field name*/
        char type;           /* C = Character
                                N = Numeric
                                L = Logical (YyNnTtFf) */
        char length;            /* field length */
        int  fdaddr;            /*  field data address 
                                    (16 bit number set by dBASE in memory) */
        char decimal;           /* decimal positions if numeric */
        };

struct dbinfo
        {
        struct  db2head info;            /*file header information*/
        int     fieldnum;               /*number of fields per record*/
        struct  db2field field[MAX2FIELD];         /*field pointer*/
        } db2;

FILE *fileptr;
#define SPACE       32



/****************************************************************************/
main (argc,argv)
    int argc;           /*number of arguments*/
    char *argv[];       /*argument string pointers*/
{
int loop,count;                 /*general purpose integer variables*/
char dbfname[41];                     /*file path name*/
char filext[5];
char filetype;
long start, end;


/* atritrary start and end record numbers to display */
/* put in your own */
start = 1L;
end   = 1000L;

strcpy (dbfname, argv[1]);

/*check for .DBF extension*/
strextr (filext, dbfname, strlen(dbfname)-4, 4);
if (strcmpi (filext, ".DBF") != 0) {
    printf("\7Filename MUST have .DBF extension\n");
    exit(1);
    }
 
/*open .DBF  file*/
if ((fileptr = fopen (dbfname, "rb")) == NULL) {
   printf("\7Cannot open %s\n",dbfname);
   exit(1);
   }

fscanf (fileptr, "%c", &filetype);

if (((filetype & 0xf) != 3) && (filetype != 2)) {
   printf("\7Not a dBASE .DBF file\n");
   exit(1);
   }

/* reset file pointer to beginning of the file */
fseek (fileptr, 0L, SEEK_SET);


/* this routine has the two db2 and db3 functions complete in themselves
 * to increase the speed in searching the database.
 * With a little work, functions common to both file types can be 
 * devised, decreasing the amount of code required, but perhaps increasing
 * the speed, which may or may not be that important.
 */
if (filetype == 2) getdb2 (dbfname, start, end);
    else getdb3 (dbfname, start, end);

fclose (fileptr);

}



/*************************************************************************/
getdb2 (dbfname, start, end)
    char *dbfname;
    long start, end;
{
int loop,count;                 /*general purpose integer variables*/
char recpart [MAXFLDLEN];
long rec, recnum;               /*record numbers*/
int recpos;                     /*record character position pointer*/
long offset, filepos;           /*file position pointer*/
char record [MAX2RECLEN];        /*record buffer*/
char typestr[12];               /* string for type of field */

/* read database information */
/* I know...good programming would suggest that the fread function be
 * written 'fread (&(db2.info), 1, sizeof(db2.info), fileptr)', and
 * that is just the way I did it, at first.
 * However, Microsoft C aligns all structures on word boundaries so
 * that it would always return 10 bytes instead of the proper 8.
 * So I decided to give up and just hard-code the proper number of bytes.
 * Be sure to compile with the /Zp option to pack the structures and
 * get around the word boundary problem.
 * The /Zp option may negate the need to eliminate the sizeof() function,
 * but I haven't taken the time to check it out.
 */
fread (&db2.info, 1, 8, fileptr);

/*read & count field descriptors*/
count=0;
fields:
        fread (&(db2.field[count]), 1, 16, fileptr);
        if (db2.field[count].name[0] != 0xd) 
                {               /*0D signals end of header & start of data*/
                count++;        /*field counter*/
                goto fields;
                }
db2.fieldnum = count;

printf ("\nStructure for database:  %s", dbfname);
printf ("\nType of .dbf file     :  dBASE II");
printf ("\nNumber of data records:  %d", db2.info.quan);
printf ("\nDate of last update   :  %hi/%hi/%hi", db2.info.month, db2.info.day, db2.info.year);
printf ("\n\nField  Field Name  Type       Width    Dec");

/*print field titles*/
for (loop=0; loop < db2.fieldnum; loop++) {
    switch (db2.field[loop].type) {
        case 'C': strcpy (typestr, "Character");
                  break;
        case 'N': strcpy (typestr, "Numeric");
                  break;
        case 'L': strcpy (typestr, "Logical");
                  break;
        }
    printf ("\n%5d  %-11s %-10s %5hi    %3hi", loop+1, db2.field[loop].name, typestr, db2.field[loop].length, db2.field[loop].decimal);
    }
printf ("\n** Total **                   %5d", db2.info.reclen);


start=(start) ? (start) : 1;    /* if start=0, make it 1 */
end = (end <= db2.info.quan) ? (end) : (db2.info.quan);
                                /* if end > max record, end=max record */

/*list records*/
for (recnum=start; recnum<=end; recnum++) {
    /*position file pointer*/
    filepos = 521L + (recnum-1)*(db2.info.reclen);

    if (fseek (fileptr, filepos, SEEK_SET)) {
       printf("\7Seek error.\n");
       exit(1);
       }

    /*read a record's length of bytes and check for errors*/
    if (fread (record, 1, db2.info.reclen, fileptr) != db2.info.reclen) {
        printf("\7Record read error.\n");
        exit(1);
        }
 
    recpos=1;  

/* the first byte in the record indicates whether the record has been
 * deleted or not.  The deleted record has an asterisk '*'as the first
 * byte, whereas the good record has a blank space.
 * The space of the deleted records is reclaimed when the database is 
 * repacked.
 */
    if (record[0] == SPACE) {
        printf("\n\nRecord #%ld) ",recnum);  
        for (count=0; count<db2.fieldnum; count++) {
/* This is the function that does all the work.
 * I am using the strncpy function from the Microsoft library because
 * it is so fast.  It doesn't check to see that the number of bytes and
 * the position are correct for the string, it should all be OK because
 * dBASE will align them correctly.
 */
            strncpy (recpart, record+recpos, db2.field[count].length);

/* You may want to add a function that terminates the string at the
* beginning of the trailing blanks.
*/
            recpart[db2.field[count].length] = NULL;
            printf("\n%s",recpart);  
            recpos=recpos+db2.field[count].length;
            }
        printf("\n"); 
        }
    }
}



/*********************************************************************/
getdb3 (dbfname, start, end)
    char *dbfname;
    long start, end;
{    
int loop,count;                 /*general purpose integer variables*/
char recpart [MAXFLDLEN];
long rec, recnum;               /*record numbers*/
int recpos;                     /*record character position pointer*/
long offset, filepos;           /*file position pointer*/
char record [MAX3RECLEN];        /*record buffer*/
char typestr[12];               /* string for type of field */

/* read database information */
fread (&(db3.info), 1, 32, fileptr);

/* read & count field descriptors */
count=0;
fields:
        fread (&(db3.field[count]), 1, 32, fileptr);
        if (db3.field[count].name[0] != 0xd) 
                {               /*0D signals end of header & start of data*/
                count++;        /*field counter*/
                goto fields;
                }
db3.fieldnum = count;


printf ("\nStructure for database:  %s", dbfname);
printf ("\nType of .dbf file     :  dBASE III");
printf ("\nNumber of data records:  %ld", db3.info.quan);
printf ("\nDate of last update   :  %hi/%hi/%hi", db3.info.month, db3.info.day, db3.info.year);
printf ("\n\nField  Field Name  Type       Width    Dec");

/*print field titles*/
for (loop=0; loop < db3.fieldnum; loop++) {
    switch (db3.field[loop].type) {
        case 'C': strcpy (typestr, "Character");
                  break;
        case 'N': strcpy (typestr, "Numeric");
                  break;
        case 'L': strcpy (typestr, "Logical");
                  break;
        case 'M': strcpy (typestr, "Memo");
                  break;
        case 'D': strcpy (typestr, "Date");
                  break;
        }
    printf ("\n%5d  %-11s %-10s %5hi    %3hi", loop+1, db3.field[loop].name, typestr, db3.field[loop].length, db3.field[loop].decimal);
    }
printf ("\n** Total **                   %5d", db3.info.reclen);
printf ("\n");


start=(start) ? (start) : 1;    /* if start=0, make it 1 */
end = (end <= db3.info.quan) ? (end) : (db3.info.quan);
                                /* if end > max record, end=max record */

/*list records*/
offset = 34 + ((db3.fieldnum) * 32);

for (recnum=start; recnum<=end; recnum++) {
    /*position file pointer*/
    filepos = offset + (recnum-1)*(db3.info.reclen);

    if (fseek (fileptr, filepos, SEEK_SET)) {
       printf("\7Seek error.\n");
       exit(1);
       }

    /*read a record's length of bytes and check for errors*/
    if (fread (record, 1, db3.info.reclen, fileptr) != db3.info.reclen) {
        printf("\7Record read error.\n");
        exit(1);
        }
 
    recpos=1;  

    if (record[0] == SPACE) {
        printf("\n\nRecord #%ld) ",recnum);
        for (count=0; count<db3.fieldnum; count++) {
            strncpy (recpart, record+recpos, db3.field[count].length);
            recpart[db3.field[count].length] = NULL;
            printf("\n%s",recpart);
            recpos=recpos+db3.field[count].length;
            }
/*        printf("\n");  */
        }
    }
}



/*************************************************************************
strextr  (string extract)
*dest = pointer to destination                        
*source = pointer to source                             
start = start of string to extract  (0 = 1st char.)
num = number of characters to extract             
*************************************************************************/
int strextr (dest, source, start, num)
    char *dest, *source;
    int start, num;
{
int i,j;
j=i=0;
while (*source) {
    if (i==start) break;
    ++i;
    ++source;
    }
if (i<start) return 0;
while (*source) {
    *dest++ = *source++;
    ++j;
    if (j==num) break;
    }
*dest='\0';
return j;
}
                                                                                                                          