/* Volume Label Class
   By Tom Astin 73407,3427
   Copyright 1991
   see VOLLAB.H for brief docs
*/

#include <errno.h>
#include "vollab.h"

#define min(a,b)    (((a) < (b)) ? (a) : (b))

void VolLabel::SetupXFCB()
{
	memset(&xf,0,sizeof(xf));
	xf.xfcb_flag=0xFF;
	xf.xfcb_attr=8;
}

void VolLabel::CustomDos(unsigned char ah, void far *data)
  // class custom dos caller method
{
	REGS rg;
	SREGS sr;

	rg.h.ah=ah;
	rg.x.dx=FP_OFF(data);
	sr.ds=FP_SEG(data);
	res=intdosx(&rg,&rg,&sr);
	bDosErr=rg.x.cflag != 0;
}

unsigned short VolLabel::SectorSize(int nDrive)
{
	REGS rg;

	rg.h.ah=0x1C;
	rg.h.dl=nDrive;
	res=intdos(&rg,&rg);
	bDosErr = rg.x.cflag != 0;
	return (bDosErr ? 0 : rg.x.cx);
}

char *VolLabel::GetLabel(int nDrive)
  // get current volume label
{
	char far *save_dta;

	SetupXFCB();
	xf.xfcb_fcb.fcb_drive=nDrive;
	memset(xf.xfcb_fcb.fcb_name,'?',11);
	save_dta=getdta();
	setdta((char far *) &buf);
	CustomDos(0x11,(void far *) &xf); 	// fcb find first
	setdta(save_dta);
	if (!(char)res) {  					// AL=0xFF on fail
		pxf=(xfcb *) &buf;
		pxf->xfcb_fcb.fcb_name[11]='\0';
		return pxf->xfcb_fcb.fcb_name;
	}
	else
		return 0;
}

int VolLabel::SetLabel(int nDrive, char *szLabel)
  // set new label
{
	char *pOldv;

	if (pOldv=GetLabel(nDrive)) {  // does one exist?
		memset(&fcbSpec,0,sizeof(fcbSpec));  // yes
		memmove(fcbSpec.oldn,pOldv,11);
		memset(fcbSpec.newn,' ',11);
		memmove(fcbSpec.newn,szLabel,min(11,strlen(szLabel)));
		fcbSpec.flg=0xFF;
		fcbSpec.attr=8;
		fcbSpec.dr=nDrive;
		CustomDos(0x17,(void far *) &fcbSpec);  // fcb rename
		return (!char(res));					// AL=0xFF on fail
	}
	else { 					// no, doesn't exist
		SetupXFCB();
		xf.xfcb_fcb.fcb_drive=nDrive;
		memset(&xf.xfcb_fcb.fcb_name,' ',11);	// fcb create file
		memmove(&xf.xfcb_fcb.fcb_name,szLabel,min(11,strlen(szLabel)));
		CustomDos(0x16,(void far *) &xf);		// AL=0xFF on fail
		return (!(char)res);
	}
}

int VolLabel::Remove(int nDrive)
  // remove current label, only Version 3 and higher recommended by Duncan
{
	if (_osmajor<=2) // don't delete under 2.2
		return 0;

	SetupXFCB();
	xf.xfcb_fcb.fcb_drive=nDrive;
	memset(&xf.xfcb_fcb.fcb_name,'?',11);
	CustomDos(0x13,(void far *) &xf);   		// fcb delete file
	return (!(char) res);						// AL=0xFF on fail
}

float VolLabel::MediaSize(unsigned char nDrive)
	// media size of nDrive.  Cur=0, A=1, B=2, etc.
{
	REGS rg;
	double sectors;
	double bytesize;

	rg.h.ah=0x36;
	rg.h.dl=nDrive;
	intdos(&rg,&rg);
	sectors = (double) rg.x.dx * rg.x.ax;  // clus per drive * sec per clus
	bytesize = (double) rg.x.cx * sectors; // bytes per sec * sec per drive
	return (float) (bytesize / 1000000); // return megabytes
}

int VolLabel::myabsread(int drive, int nsects, long lsect, void far *buffer)
	// myabsread - drive=A=0, B=1, etc.
{
	REGS rg;
	SREGS sr;
	int errorAX;
	strParmBlock pb;

	if (MediaSize(drive+1) >= 32.0F) { // media size uses Cur=0, A=1, B=2, etc
		pb.nSector=lsect;
		pb.nToRead=nsects;
		pb.pBuffer=buffer;
		rg.h.al=drive;
		rg.x.cx=-1;
		rg.x.bx=FP_OFF((void far *) &pb);
		sr.ds=FP_SEG((void far *) &pb);
		int86x(0x25,&rg,&rg,&sr); // set doserrno
		errno=_doserrno; // ... absread() sets errno
		bDosErr = ((rg.x.cflag == 0) ? 0 : 1); // set bool 1=Err, 0=Success
		return (bDosErr ? -1 : 0); // return -1=error or 0=success
	}
	else
		return absread(drive,nsects,lsect,buffer);
}

int VolLabel::myabswrite(int drive, int nsects, long lsect, void far *buffer)
	// myabsread - drive=A=0, B=1, etc.
{
	REGS rg;
	SREGS sr;
	int errorAX;
	strParmBlock pb;

	if (MediaSize(drive+1) >= 32.0F) { // media size uses Cur=0, A=1, B=2, etc
		pb.nSector=lsect;
		pb.nToRead=nsects;
		pb.pBuffer=buffer;
		rg.h.al=drive;
		rg.x.cx=-1;
		rg.x.bx=FP_OFF((void far *) &pb);
		sr.ds=FP_SEG((void far *) &pb);
		int86x(0x26,&rg,&rg,&sr); // set doserrno
		errno=_doserrno; // ... absread() sets errno
		bDosErr = ((rg.x.cflag == 0) ? 0 : 1); // set bool 1=Err, 0=Success
		return (bDosErr ? -1 : 0); // return -1=error or 0=success
	}
	else
		return abswrite(drive,nsects,lsect,buffer);
}

unsigned long VolLabel::GetSerial(unsigned char nDrive)
  // return unsigned long serial number from boot record
  // nDrive Cur=0, A=1, B=2, etc.
{
	char *buf;
	unsigned short ss=SectorSize(nDrive);
	unsigned long VolID=0;

	if ((!ss | bDosErr) ||
	   ((buf=new char[ss+2]) == 0)) // sector size + extra for safety
		return 0;
	nDrive=(nDrive ? --nDrive : GetDisk()); // convert nDrive A=0, B=1

	if (myabsread(nDrive,1,0,buf) == 0)
		VolID=*((unsigned long *) &buf[0x27]); // vol id offset

	delete buf;
	return VolID;
}

int VolLabel::SetSerial(unsigned char nDrive, unsigned long nSerial)
  // set the serial # by modifying disk boot record
{
	char *buf;
	unsigned short ss=SectorSize(nDrive);
	int bOk=0;

	if ((!ss | bDosErr) ||
	   ((buf=new char[ss+2]) == 0)) // sector size + extra for safety
		return 0;
	nDrive=(nDrive ? --nDrive : GetDisk());
	if (myabsread(nDrive,1,0,buf) == 0) {
		*((unsigned long *) &buf[0x27])=nSerial; // vol id offset
		if (myabswrite(nDrive,1,0,buf) == 0)
			bOk=1;
	}

	delete buf;
	return bOk;
}

unsigned char VolLabel::GetDisk()
{
	asm mov ah,0x19		// get current disk
	asm int 0x21
	return _AL;
}
