#include "mat.h"

/* --------------  Vec2f library --------------- */

/**********************************************************************
  Creates a new vector (= point).
***********************************************************************/
Vec2f *Vec2fNew()
{
   Vec2f *Vec;

   if ((Vec = (Vec2f *) malloc (sizeof(Vec2f))) == NULL) {
      printf("Memory allocation problems\n");
      exit(0);
   }

   Vec->x= Vec->y = 0;

   return (Vec);
}


/**********************************************************************
  Creates a new vector (= point), with x = X, y = Y.
***********************************************************************/
Vec2f *Vec2fCreate (float X, float Y)
{
   Vec2f *Vec;
   int i;

   Vec = Vec2fNew ();
   Vec->x = X;
   Vec->y = Y;

   return (Vec);
}

/**********************************************************************
  Copy v2 to v1.
***********************************************************************/
void Vec2fCopy (Vec2f *v1, Vec2f *v2)
{

  v1->x = v2->x;
  v1->y = v2->y;
}


/**********************************************************************
  Set X of v to val.
***********************************************************************/
Vec2f *Vec2fSetX(Vec2f *v, float val)
{
  v->x = val;
  return (v);
}

/**********************************************************************
  Set Y of v to val.
***********************************************************************/
Vec2f *Vec2fSetY(Vec2f *v, float val)
{
  v->y = val;
  return (v);
}



/**********************************************************************
  Return the sum of 2 vectors v1 & v2 in result.
***********************************************************************/
Vec2f *Vec2fAdd(Vec2f *v1, Vec2f *v2, Vec2f *result)
{
  result->x = v1->x + v2->x;
  result->y = v1->y + v2->y;
  return (result);
}

/**********************************************************************
  Return the vector factorization of v1 & v2.
***********************************************************************/
float Vec2fMul(Vec2f *v1, Vec2f *v2)
{
  return (v1->x * v2->x + v1->y * v2->y);
}

/**********************************************************************
  Return v, with each coord, multiplied by s.
***********************************************************************/
Vec2f *Vec2fMulScalar(float s, Vec2f *v, Vec2f *result)
{
  result->x = v->x * s;
  result->y = v->y * s;
  return (result);
}

/**********************************************************************
  Return v - normalized.
***********************************************************************/
Vec2f *Vec2fNormal (Vec2f *v, Vec2f *result)
{
  float len = Vec2fLen (v);
  if (fabs(len) > Epsilon) {
    result->x = v->x / len;
    result->y = v->y / len;
  }
  else {
    result->x = 0   ;
    result->y = 0   ;
  }

  return (result);
}


/**********************************************************************
  Return the result of substracting v2 from v1 in result.
***********************************************************************/
Vec2f *Vec2fSub(Vec2f *v1, Vec2f *v2, Vec2f *result)
{
  result->x = v1->x - v2->x;
  result->y = v1->y - v2->y;
  return (result);
}

/**********************************************************************
  Return the length of v.
***********************************************************************/
float Vec2fLen(Vec2f *v)
{
   return (sqrt(v->x * v->x + v->y * v->y));
}


/**********************************************************************
  Returns the negative of v.
***********************************************************************/
Vec2f *Vec2fNeg(Vec2f *v, Vec2f *Result)
{
  Result->x = -v->x;
  Result->y = -v->y;
  return (v);
}


/* --------------  Vec3f library --------------- */

/**********************************************************************
  Creates a new vector (= point).
***********************************************************************/
Vec3f *Vec3fNew()
{
   Vec3f *Vec;

   if ((Vec = (Vec3f *) malloc (sizeof(Vec3f))) == NULL) {
      printf("Memory allocation problems\n");
      exit(0);
   }

   Vec->x= Vec->y = Vec->z = 0;

   return (Vec);
}


/**********************************************************************
  Creates a new vector (= point), with x = X, y = Y.
***********************************************************************/
Vec3f *Vec3fCreate (float X, float Y, float Z)
{
   Vec3f *Vec;
   int i;

   Vec = Vec3fNew ();
   Vec->x = X;
   Vec->y = Y;
   Vec->z = Z;
   return (Vec);
}

/**********************************************************************
  Copy v2 to v1.
***********************************************************************/
void Vec3fCopy (Vec3f *v1, Vec3f *v2)
{

  v1->x = v2->x;
  v1->y = v2->y;
  v1->z = v2->z;
}


/**********************************************************************
  Set X of v to val.
***********************************************************************/
Vec3f *Vec3fSetX(Vec3f *v, float val)
{
  v->x = val;
  return (v);
}

/**********************************************************************
  Set Y of v to val.
***********************************************************************/
Vec3f *Vec3fSetY(Vec3f *v, float val)
{
  v->y = val;
  return (v);
}

/**********************************************************************
  Set Z of v to val.
***********************************************************************/
Vec3f *Vec3fSetZ(Vec3f *v, float val)
{
  v->z = val;
  return (v);
}


/**********************************************************************
  Return the sum of 2 vectors v1 & v2 in result.
***********************************************************************/
Vec3f *Vec3fAdd(Vec3f *v1, Vec3f *v2, Vec3f *result)
{
  result->x = v1->x + v2->x;
  result->y = v1->y + v2->y;
  result->z = v1->z + v2->z;
  return (result);
}

/**********************************************************************
  Return the vector factorization of v1 & v2.
***********************************************************************/
float Vec3fMul(Vec3f *v1, Vec3f *v2)
{
  return (v1->x * v2->x + v1->y * v2->y + v1->z * v2->z);
}

/**********************************************************************
  Return v, with each coord, multiplied by s.
***********************************************************************/
Vec3f *Vec3fMulScalar(float s, Vec3f *v, Vec3f *result)
{
  result->x = v->x * s;
  result->y = v->y * s;
  result->z = v->z * s;
  return (result);
}

/**********************************************************************
  Return v - normalized.
***********************************************************************/
Vec3f *Vec3fNormal (Vec3f *v, Vec3f *result)
{
  float len = Vec3fLen (v);
  if (fabs(len) > Epsilon) {
    result->x = v->x / len;
    result->y = v->y / len;
    result->z = v->z / len;
  }
  else {
    result->x = 0   ;
    result->y = 0   ;
    result->z = 0   ;
  }

  return (result);
}


/**********************************************************************
  Return the result of substracting v2 from v1 in result.
***********************************************************************/
Vec3f *Vec3fSub(Vec3f *v1, Vec3f *v2, Vec3f *result)
{
  result->x = v1->x - v2->x;
  result->y = v1->y - v2->y;
  result->z = v1->z - v2->z;
  return (result);
}

/**********************************************************************
  Return the length of v.
***********************************************************************/
float Vec3fLen(Vec3f *v)
{
   return (sqrt(v->x * v->x + v->y * v->y + v->z * v->z));
}


/**********************************************************************
  Returns the negative of v.
***********************************************************************/
Vec3f *Vec3fNeg(Vec3f *v, Vec3f *Result)
{
  Result->x = -v->x;
  Result->y = -v->y;
  Result->z = -v->z;
  return (v);
}




/* ----------------------- Matrix library -------------------------- */

/**********************************************************************
  Creates a new matrix which is rows x cols by size, and all elements 
 are zeros.
***********************************************************************/
Matrix *MatNew(rows, cols)
{
   Matrix *Mat;
   int i, j;

   if ((Mat = (Matrix *) malloc (sizeof(Matrix))) == NULL) {
      printf("Memory allocation problems\n");
      exit(0);
   }
   if ((Mat->elems = (list) malloc (rows * cols * sizeof(float))) == NULL) {
      printf("Memory allocation problems\n");
      exit(0);
   }

   MatRows(Mat) = rows;
   MatCols(Mat) = cols;

   for (i = 0; i < rows; i++)
      for (j = 0; j < cols; j++)
         MatSet(Mat, i, j, 0.0);

   return (Mat);

}

/**********************************************************************
  Creates a new matrix which is rows x cols by size, and has elements
 as its elements.
***********************************************************************/
Matrix *MatCreate (int rows, int cols, float *elements)
{
   struct Matrix *Mat;
   int i;

   Mat = MatNew (rows, cols);

   for (i = 0; i < rows * cols; i++) {
      Mat->elems[i] = elements[i];
   }
   return (Mat);
}

/**********************************************************************
  Free matrix.
***********************************************************************/
void MatFree(Matrix *Mat)
{
   free (Mat->elems);
   free (Mat);
}


/**********************************************************************
 Returns the result of Mat1 * Mat2 in MatRet. (Result can be Mat1 or Mat2)
***********************************************************************/
Matrix *MatMul (Matrix *Mat1, Matrix *Mat2, Matrix *MatRet)
{
   int i, j, k;
   float e;
   Matrix *Temp = MatNew(MatRows(Mat1), MatCols(Mat2));

   if (MatNull(Mat1) || MatNull(Mat2)) {
      exit(0);
      }

   if (Mat1->cols != Mat2->rows) {
      printf("Matrixs do not match!\n");
      exit(0);
   }

   for (i = 0; i < MatRows(Mat1); i++) {
      for (j = 0; j < MatCols(Mat1); j++) {
         e = 0;
         for (k = 0; k < MatCols(Mat1); k++) {
            e += (MatGet(Mat1, i, k) * MatGet(Mat2, k, j));
         }
         MatSet (Temp, i, j, e);
      }
   }

   MatCopy (MatRet, Temp);
   MatFree (Temp);
   return (MatRet);
}


/**********************************************************************
  Gets a matrix, s = scalar and a pointer for result, and returns the
 result - s * Mat (a matrix).
***********************************************************************/

Matrix *MatScalarMul(float s, Matrix *Mat, Matrix *MatRet)
{
   int i, j;
   int r = MatRows(Mat), c = MatCols(Mat);
   list l1 = MatElems(Mat);
   list l2 = MatElems(MatRet);

   if (MatNull(Mat) || MatNull(MatRet)) {
      exit(0);
      }

   for (i = 0; i < r * c; i++) {
      l2[i] = l1[i] * s;
      }

   return (MatRet);
}


/**********************************************************************
 Prints the matrix Mat
***********************************************************************/

void MatPrint (Matrix *Mat)
{
   int r, c;

   if (MatNull(Mat)) {
      printf ("A NULL matrix.\n");
      return;
      }

   for (r = 0; r < MatRows(Mat); r++) {
     for (c = 0; c < MatCols(Mat); c++) {
       printf("%.3f  ",MatGet(Mat, r, c));
     }
     printf("\n");
   }
}


/**********************************************************************
 Returns 0 if Mat != Null (Illegal matrix), !=0, else
***********************************************************************/
int MatNull(Matrix *Mat)
{
   return (Mat == NULL);
}

/**********************************************************************
 Returns the determinante of Mat
***********************************************************************/
float MatDet(Matrix *Mat)
{
   int sign = 1;
   int r = MatRows(Mat), c = MatCols(Mat);
   list l = MatElems(Mat);
   float result = 0.0;
   int i;

   if (r != c) {
      printf("# of rows in matrix != # of cols. Exiting.\n");
      exit(0);
      }

   if (r == 2) {
      return (l[0] * l[3] - l[1] * l[2]);
      }

   /* Determinanta developement from the 1st line  */
   for (i = 0; i < c; i++) {
      result += sign * MatGet(Mat, 0, i) * Minor (Mat, 0, i);
      sign = -sign;
      }
   return (result);
}

/**********************************************************************
  Gets a matrix Mat and r = row, c = column, and returns the
 determinante of the minor Mat(r,c)
***********************************************************************/
float Minor(Matrix *Mat, int r, int c)
{
   int i1, i2, n1, n2;
   float det;
   int rows = MatRows(Mat), cols = MatCols(Mat);
   Matrix *minor;

   if (rows != cols) {
      printf("# of rows in matrix != # of cols. Exiting.\n");
      exit(0);
      }

   minor = MatNew(rows-1, cols-1);
   n1 = 0;

   for (i1 = 0; i1 < rows; i1++) {
      n2 = 0;
      if (i1 != r) {
         for (i2 = 0; i2 < cols; i2++) {
            if (i2 != c) {
               MatSet(minor, n1, n2++, MatGet(Mat, i1, i2));
               }
            }
         n1++;
         }
      }
   det = MatDet(minor);
   MatFree (minor);
   return(det);
}


/**********************************************************************
  Gets a matrix and a pointer for result, and returns the result -
 which is the inverted input matrix
***********************************************************************/
Matrix *MatInv(Matrix *Mat, Matrix *MatRet)
{
   int i, j;
   int sign = 1;

   int rows = MatRows(Mat), cols = MatCols(Mat);

   if (rows != cols) {
      printf("# of rows in matrix != # of cols. Exiting.\n");
      exit(0);
      }

   for (i = 0; i < rows; i++) {
      for (j = 0; j < cols; j++) {
         MatSet(MatRet, j, i, Minor (Mat, i, j) * sign);
         sign = -sign;
     }
      sign = (i%2) ? 1 : -1;   /* Fix the sign - every 2nd line, sign = -1 */
   }
   return (MatScalarMul((1 / MatDet(Mat)), MatRet, MatRet));
}


/**********************************************************************
  Creates a new matrix - with: rows = Mat->cols and cols = Mat->cols
 and makes it the transposed matrix of Mat. Remember to free it,
 when you are through with it.
***********************************************************************/
Matrix *MatTra(Matrix *Mat)
{
   int i, j;
   Matrix *Tran;

   int rows = MatRows(Mat), cols = MatCols(Mat);

   Tran = MatNew(cols, rows);

   for (i = 0; i < rows; i++) {
      for (j = 0; j < cols; j++) {
	  MatSet(Tran, j, i, MatGet (Mat, i, j));
         }
      }
   return (Tran);
}

/**********************************************************************/
/* Copy Mat2 into Mat1  */
void MatCopy (Matrix *Mat1, Matrix *Mat2)
{
  int i, j;

  for (i = 0; i < MatRows(Mat1); i++) {
    for (j = 0; j < MatCols(Mat1); j++) {
      MatSet (Mat1, i, j, MatGet (Mat2, i, j));
    }
  }
}

/**********************************************************************/
/* Copy Mat into Vec3  */
void Mat2Vec3fCopy (Matrix *Mat, Vec3f *Vec3)
{
  Vec3fSetX (Vec3, MatGet (Mat, 0, 0));
  Vec3fSetY (Vec3, MatGet (Mat, 0, 1));
  Vec3fSetZ (Vec3, MatGet (Mat, 0, 2));
}


/**********************************************************************/
/* Copy Vec3 into Mat  */
void Vec3f2MatCopy (Vec3f *Vec3, Matrix *Mat)
{
  MatSet (Mat, 0, 0, Vec3fGetX (Vec3));
  MatSet (Mat, 0, 1, Vec3fGetY (Vec3));
  MatSet (Mat, 0, 2, Vec3fGetZ (Vec3));
}

