//---------------------------------------------
// MEMVIEW.CPP - A memory-watcher for Windows
// 5/1/92
// by Bert Whetstone
//---------------------------------------------

#include <stdlib.h>
#include <stdio.h>
#include <owl.h>
#include <button.h>
#include <listbox.h>

#define APPNAME		"MEMVIEW"
#define CM_ONTOP			101
#define CM_ABOUT			102
#define CM_QUIT			103
#define CM_BYTES			104
#define CM_KB				105
#define CM_MB				106
#define CM_BLOCK			107
#define CM_LISTOLD		108
#define CM_LISTDIFF		109
#define CM_AUTOSCROLL	110
#define ID_CLEAR_BUTTON	150
#define ID_OLDMEM_LIST	200


char	*div_str[]={"Bytes","KB","MB"};
char	*type_str[]={"Free","Largest Free Block"};
DWORD	div_num[]={1,1024,1048576L};


//----------------------------------------------------
//						CLASS DECLARATIONS
//----------------------------------------------------
_CLASSDEF(MemApp)
class MemApp : public TApplication
{
public:
	MemApp(LPSTR AName,HANDLE hInstance,HANDLE hPrevInstance,LPSTR lpCmdLine,int nCmdShow)
			 : TApplication(AName,hInstance,hPrevInstance,lpCmdLine,nCmdShow)
	{
	};

	virtual void InitMainWindow();
};


_CLASSDEF(MemWindow)
class MemWindow : public TWindow
{
public:
	char			disp;
	char			about_flag,block_flag;
	char			listold_flag,listdiff_flag,scroll_flag;
	UINT			checked;
	DWORD			old_freemem;
	HMENU			menu_h;
	HICON			my_icon;
	PTButton		clear_button;
	PTListBox	oldmem_list;

	MemWindow(PTWindowsObject AParent);
	~MemWindow();
	virtual void GetWindowClass(WNDCLASS&);
	virtual LPSTR GetClassName();
	virtual void SetupWindow();

private:
	void Display(int format);
	void KillZeros(char *buff);
	void ShowFreeMem(int show);
	void ToggleMenuCheck(int command);
	void WMSize(RTMessage Msg);
	void WMTimer()=[WM_FIRST+WM_TIMER];
	void About()=			[CM_FIRST+CM_ABOUT];
	void AutoScroll()=	[CM_FIRST+CM_AUTOSCROLL];
	void ClearButton()=	[ID_FIRST+ID_CLEAR_BUTTON];
	void DisplayB()=		[CM_FIRST+CM_BYTES];
	void DisplayK()=		[CM_FIRST+CM_KB];
	void DisplayM()=		[CM_FIRST+CM_MB];
	void DisplayBlock()=	[CM_FIRST+CM_BLOCK];
	void ListDiff()=		[CM_FIRST+CM_LISTDIFF];
	void ListOld()=		[CM_FIRST+CM_LISTOLD];
	void OnTop()=			[CM_FIRST+CM_ONTOP];
	void Quit()=			[CM_FIRST+CM_QUIT];
};



//----------------------------------------------------------
//					MEMWINDOW FUNCTION DEFINITIONS
//----------------------------------------------------------
MemWindow::MemWindow(PTWindowsObject AParent):TWindow(AParent,"MemView")
{
	disp=0;
	old_freemem=GetFreeSpace(NULL);
	listold_flag=TRUE;
	listdiff_flag=TRUE;
	scroll_flag=TRUE;
	about_flag=FALSE;
	block_flag=FALSE;
	checked=CM_BYTES;
	AssignMenu(APPNAME);
	Attr.W=250;
	Attr.H=27;

	clear_button=new TButton(this,ID_CLEAR_BUTTON,"&Clear List",0,0,200,20,FALSE);
	oldmem_list=new TListBox(this,ID_OLDMEM_LIST,0,20,200,200);
	oldmem_list->Attr.Style|=LBS_NOINTEGRALHEIGHT;
	oldmem_list->Attr.Style&=~LBS_SORT;
}


void MemWindow::SetupWindow()
{
	TWindow::SetupWindow();
	menu_h=GetMenu(HWindow);
	if(!SetTimer(HWindow,0,500,NULL))
	{
		MessageBox(NULL,"No TIMERS available!\nFree up resources and try again.",
						"WHOA DUDE!",MB_ICONEXCLAMATION+MB_OK);
		CloseWindow();
	}

	OnTop();
	ShowFreeMem(1);
}


void MemWindow::~MemWindow()
{
	KillTimer(HWindow,0);
	DestroyIcon(my_icon);
	delete oldmem_list;
}


LPSTR MemWindow::GetClassName()
{
	return(APPNAME);
}


void MemWindow::GetWindowClass(WNDCLASS& window_class)
{
	TWindow::GetWindowClass(window_class);
	window_class.hIcon=LoadIcon(GetApplication()->hInstance,APPNAME);
	my_icon=window_class.hIcon;
}


void MemWindow::Display(int format)
{
	disp=format;
	CheckMenuItem(menu_h,checked,MF_UNCHECKED);
	CheckMenuItem(menu_h,CM_BYTES+disp,MF_CHECKED);
	checked=CM_BYTES+disp;
	ShowFreeMem(1);
}


void MemWindow::KillZeros(char *buff)
{
int	t;

	t=lstrlen(buff)-1;
	if(buff[t]=='0')
	{
		buff[t]=0;
		if(buff[t-1]=='0')
			buff[t-2]=0;
	}
}


void MemWindow::ShowFreeMem(int show)
{
char	title_buff[50];
char	diff_buff[50];
char	list_buff[100];
char	junk[15];
int	t;
float	m;
DWORD	x;

	if(!block_flag)
		m=(float)(x=GetFreeSpace(NULL))/div_num[disp];
	else
		m=(float)(x=GlobalCompact(NULL))/div_num[disp];

	if(show || (x!=old_freemem))
	{
		sprintf(junk,"%.2f",m);
		KillZeros(junk);
		sprintf(title_buff,"%s %s ",junk,div_str[disp]);

		if(listold_flag || listdiff_flag)
		{
			sprintf(list_buff,"%s  ",div_str[disp]);

			if(listold_flag)
				lstrcat(list_buff,junk);

			if(listdiff_flag)
			{
				sprintf(diff_buff,"  %+.2f",m-(float)old_freemem/div_num[disp]);
				KillZeros(diff_buff);
				lstrcat(list_buff,diff_buff);
			}

			t=oldmem_list->AddString(list_buff);
			if(scroll_flag)
				oldmem_list->SetSelIndex(t);
		}

		lstrcat(title_buff,type_str[block_flag]);
		SetWindowText(HWindow,title_buff);
		old_freemem=x;
	}
}


void MemWindow::ToggleMenuCheck(int command)
{
	if(GetMenuState(menu_h,command,MF_BYCOMMAND) & MF_CHECKED)
		CheckMenuItem(menu_h,command,MF_UNCHECKED);
	else
		CheckMenuItem(menu_h,command,MF_CHECKED);

	ShowFreeMem(1);
}


void MemWindow::WMSize(RTMessage Msg)
{
	MoveWindow(clear_button->HWindow,0,0,Msg.LP.Lo,20,TRUE);
	MoveWindow(oldmem_list->HWindow,0,20,Msg.LP.Lo,Msg.LP.Hi-20,TRUE);
}


void MemWindow::WMTimer()
{
	ShowFreeMem(0);
}


void MemWindow::About()
{
	GetModule()->ExecDialog(new TDialog(this,"MEMVIEW"));
}


void MemWindow::AutoScroll()
{
	ToggleMenuCheck(CM_AUTOSCROLL);
	scroll_flag^=1;
}


void MemWindow::ClearButton()
{
	oldmem_list->ClearList();
}


void MemWindow::DisplayBlock()
{
	ToggleMenuCheck(CM_BLOCK);
	block_flag^=1;
}


void MemWindow::DisplayB()
{
	Display(0);
}


void MemWindow::DisplayK()
{
	Display(1);
}


void MemWindow::DisplayM()
{
	Display(2);
}


void MemWindow::ListDiff()
{
	ToggleMenuCheck(CM_LISTDIFF);
	listdiff_flag^=1;
}


void MemWindow::ListOld()
{
	ToggleMenuCheck(CM_LISTOLD);
	listold_flag^=1;
}


void MemWindow::OnTop()
{
HWND	position;

	ToggleMenuCheck(CM_ONTOP);
	if(GetMenuState(menu_h,CM_ONTOP,MF_BYCOMMAND) & MF_CHECKED)
		position=HWND_TOPMOST;
	else
		position=HWND_NOTOPMOST;

	SetWindowPos(HWindow,position,0,0,0,0,SWP_NOMOVE | SWP_NOSIZE);
}


void MemWindow::Quit()
{
	CloseWindow();
}



//--------------------------------------------------------
//					MEMAPP FUNCTION DEFINITIONS
//--------------------------------------------------------
void MemApp::InitMainWindow()
{
	MainWindow=new MemWindow(NULL);
}



//-------------------------------------------------------
//						PROGRAM STARTS HERE
//-------------------------------------------------------
int PASCAL WinMain(HANDLE hInstance,HANDLE hPrevInstance,LPSTR lpCmdLine,int nCmdShow)
{
	MemApp mem_viewer(APPNAME,hInstance,hPrevInstance,lpCmdLine,nCmdShow);
	if(!mem_viewer.hPrevInstance)
		mem_viewer.Run();

	return mem_viewer.Status;
}
