/*********************************************************************/
/*********************************************************************/
/*        IPX/SPX PROTOCOL APPLICATIONS PROGRAMMING INTERFACE        */
/*********************************************************************/
/*********************************************************************/

#ifndef IPXUSER
#define IPXUSER

#include <time.h>

/* ********************************************************************** */
/* ********************************************************************** */

/* ================================================================= */
/* ================================================================= */
/* constants */
/* ================================================================= */
/* ================================================================= */

/* ------------- */
/* MISCELLANEOUS */
/* ------------- */

#define BYTE            unsigned char

#define TRUE            1
#define SEGSIZE         65536L
#define ESCAPE_KEY      0x1B
#define RSHIFT_KEY_MASK 0x01
#define LSHIFT_KEY_MASK 0x02
#define CTRL_KEY_MASK   0x04
#define ALT_KEY_MASK    0x08
#define KEYB_FLAGS      0x00400017
#define FILE_PROBLEM    -1

#define SEQLEN          1
#define MINIPXDATALEN   512
#define MAXIPXDATALEN   4096
#define ACKSTR          "ACK"
#define FIRST_SEQ_NUM   1

#define IPX_TIMEOUT     1   // # secs to wait before retry
#define IPX_TRIES       5   // # times to try RX/TX before quitting

/* ------------------------------------- */
/* IPX/SPX COMMANDS, CODES, & PARAMETERS */
/* ------------------------------------- */

/* packet types */
#define UNKNOWN_PACKET_TYPE             0x00
#define ROUTING_INFO_PACKET_TYPE        0x01
#define ECHO_PACKET_TYPE                0x02
#define ERROR_PACKET_TYPE               0x03
#define IPX_PACKET_TYPE                 0x04
#define SPX_PACKET_TYPE                 0x05

/* ipx presence detection */
#define IPX_PRESENT                     0xFF
#define IPX_MULTIPLEX                   0x7A00
#define DOS_MULTIPLEX_VECTOR            0x2F

/* ipx API commands */
#define SOCKET_OPEN                     0x0000
#define SOCKET_CLOSE                    0x0001
#define GET_LOCAL_TARGET                0x0002
#define SEND_PACKET                     0x0003
#define LISTEN_FOR_PACKET               0x0004
#define SCHEDULE_EVENT                  0x0005
#define CANCEL_EVENT                    0x0006
#define GET_INTERVAL_MARKER             0x0008
#define GET_INTERNETWORK_ADDR           0x0009
#define RELINQUISH_CONTROL              0x000A
#define DISCONNECT_FROM_TARGET          0x000B

/* ---------------- */
/* input parameters */
/* ---------------- */

/* socket parameters */
#define SOCKET_DYNAMIC_ASSIGN           0x00
#define SOCKET_SHORT_LIVED              0x00
#define SOCKET_LONG_LIVED               0xFF

/* ---------------- */
/* IPX return codes */
/* ---------------- */

/* generic */
#define IPX_SUCCESS                     0x00

#define EVENT_CANCELED                  0xFC
#define FRAGMENT_SIZE_ERROR             0xFD

/* IPXSocketOpen() */
#define SOCKET_TABLE_FULL               0xFE
#define SOCKET_ALREADY_OPEN             0xFF

/* IPXGetInternetworkAddress() */
#define DEST_NODE_PATH_NOT_FND          0xFA

/* IPXListenForPacket() */
#define ECB_SOCKET_NOT_OPEN             0xFF

/* IPXSendPacket() */
#define DESTINATION_NOT_FOUND           0xFE
#define NETWORK_FAILURE                 0xFF

/* IPXCancelEvent() */
#define CANNOT_CANCEL_ECB               0xF9
#define ECB_NOT_IN_USE                  0xFF

/* ------------------------------- */
/* "Insulation Layer" Return Codes */
/* ------------------------------- */

#define GOOD_IPX_RETURN                 0x00

#define USER_TERMINATED                 0xBB
#define IPX_NOT_PRESENT                 0xDD
#define BAD_MEMORY_ALLOC                0xEE

#define BAD_LOCAL_TARGET                0x10
#define BAD_SOCKET_PARAM                0x11
#define BAD_SOCKET_OPEN                 0x12

#define BAD_TX_TRY                      0x20
#define BAD_RETRIES                     0x21
#define BAD_RX_SEQ_NUM                  0x22
#define BAD_RX_SRCE_ADDR                0x23

#define BAD_FILE_OPEN                   0x30
#define BAD_FILE_WRITE                  0x31
#define BAD_FILE_READ                   0x32

#define BAD_RX_BFR_LEN                  0x40

/* ================================================================= */
/* ================================================================= */
/* macros */
/* ================================================================= */
/* ================================================================= */

#define HILOSWAP( x )       ((x << 8) + (x >> 8))
#define CTRLHIT             ( *(BYTE far *)KEYB_FLAGS & CTRL_KEY_MASK )
#define ALTHIT              ( *(BYTE far *)KEYB_FLAGS & ALT_KEY_MASK )
#define RSHIFTHIT           ( *(BYTE far *)KEYB_FLAGS & RSHIFT_KEY_MASK )
#define LSHIFTHIT           ( *(BYTE far *)KEYB_FLAGS & LSHIFT_KEY_MASK )
#define USERBREAK           ( RSHIFTHIT && LSHIFTHIT && ALTHIT )
#define SAFEACOPY( x, y )   x[0] = '\0', strncat( x, y, sizeof(x)-1 )

/* ================================================================= */
/* ================================================================= */
/* structures                                                        */
/* ================================================================= */
/* ================================================================= */

/* -------------- */
/* sub-structures */
/* -------------- */

typedef struct
{
BYTE   Network[4];
BYTE   Node[6];
} INTNETADDR;

typedef struct
{
INTNETADDR  INA;
BYTE        ImmediateAddr[6];
} LOCALTARGET;

typedef struct
{
INTNETADDR  INA;
BYTE        Socket[2];
} NETADDR;

typedef struct
{
void far  *BufAddr;
unsigned  Len;
} ECBFRAGMENT;

/* ----------------- */
/* IPX PACKET HEADER */
/* ----------------- */

typedef struct
{
unsigned  Chksum;
unsigned  Len;             /* hi/lo */
BYTE      TransportCtrl;
BYTE      PacketType;
NETADDR   DestAddr;
NETADDR   SrceAddr;
} IPXHEADER;

/* ------------------- */
/* EVENT CONTROL BLOCK */
/* ------------------- */

typedef struct
{
void far     *LinkAddr;
void far     (*ESR)(void);
BYTE         InUse;
BYTE         CompletionCode;
BYTE         Socket[2];
BYTE         IPXWorkspace[4];
BYTE         DriverWorkspace[12];
BYTE         ImmediateAddr[6];
unsigned     FragmentCnt;
ECBFRAGMENT  Fragment[2];
} EVENTCONTROLBLOCK;

/* ================================================================= */
/* ================================================================= */
/* function prototypes                                               */
/* ================================================================= */
/* ================================================================= */

/* --------------------------- */
/* low-level IPX API functions */
/* --------------------------- */

unsigned IPXInstalled( void );
unsigned IPXSocketOpen( BYTE *socket, BYTE longevity );
void     IPXSocketClose( BYTE *socket );
void     IPXSendPacket( EVENTCONTROLBLOCK *ecb );
void     IPXListenForPacket( EVENTCONTROLBLOCK *ecb );
void     IPXGetInternetworkAddress( INTNETADDR *ina );
void     IPXRelinquishControl( void );
void     IPXDisconnectFromTarget( NETADDR *naddr );
unsigned IPXCancelEvent( EVENTCONTROLBLOCK *ecb );
void     IPXScheduleEvent( EVENTCONTROLBLOCK *ecb, unsigned delay );
unsigned IPXGetIntervalMarker( void );
unsigned IPXGetLocalTarget( LOCALTARGET *lt );

/* -------------------------- */
/* Internal Utility functions */
/* -------------------------- */

int IPXTimeOutRT( EVENTCONTROLBLOCK *ecb,
                  time_t delay,
                  void (*RTfunc)(void) );
int  xatoxb( BYTE *d, char *s );
void xantoxb( BYTE *d, char *s, unsigned len );
void ShowECB( EVENTCONTROLBLOCK *ecb );
void ShowIPXPacket( IPXHEADER *ipxhdr );

/* ---------------------------------------------- */
/* High-Level Application IPX Interface functions */
/* ---------------------------------------------- */

unsigned IPXReceiveRT( long *BfrLen,
                       BYTE *FileOrBfr,
                       char *SocketStr,
                       void (*RTfunc)(void) );

unsigned IPXSendRT( long     *BfrLen,
                    BYTE     *FileOrBfr,
                    char     *SocketStr,
                    NETADDR  *Destination,
                    void     (*RTfunc)(void) );

/* ================================================================= */
/* ================================================================= */
/* globals                                                           */
/* ================================================================= */
/* ================================================================= */

/*
NOTE: These globals default to the constants listed within RKIPX.H,
      but can be changed by the application that employs them.
*/

extern unsigned IPXerrno;    // latest IPX function result code
extern time_t   IPXto;       // # secs to wait for IPX command completion
extern unsigned IPXtries;    // # times to try RX/TX in IPXSend/Receive()
extern unsigned IPXdatalen;  // size of data buffer in IPX packet

/* ********************************************************************** */
/* ********************************************************************** */

#endif

/************************************************************************/
/* EOF  EOF  EOF  EOF  EOF  EOF  EOF  EOF  EOF  EOF  EOF  EOF  EOF  EOF */
/************************************************************************/