//****************************************************************************
// File: file.c
//
// Purpose: file manipulation routines
//
// Functions:
//    HGLOBAL ReadDIBFile(LPCSTR szFile)
//    HGLOBAL LoadBM2(LPCSTR szFile, PALETTEENTRY *pe1, PALETTEENTRY *pe2)
//    HGLOBAL LoadBMP(LPCSTR szFile, PALETTEENTRY *pe16)
//    BOOL GetFileName (LPSTR szFName, WORD wIDString)
//    BOOL WriteDIBFile (LPCSTR szFile, HGLOBAL hdib)
//    BOOL FAR PASCAL InfoDlgProc (HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
//    BOOL FAR PASCAL OptionsDlgProc (HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
//    BOOL FAR PASCAL FileOpenHookProc (HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
//
//  Several of these were borrowed or modified from DIBView.
//
// Development Team:
//            Jeff Saathoff
//
// Written by Microsoft Product Support Services, Windows Developer Support
// COPYRIGHT:
//
//   (C) Copyright Microsoft Corp. 1993.  All rights reserved.
//
//   You have a royalty-free right to use, modify, reproduce and
//   distribute the Sample Files (and/or any modified version) in
//   any way you find useful, provided that you agree that
//   Microsoft has no warranty obligations or liability for any
//   Sample Application Files which are modified.
//
//****************************************************************************

#include <windows.h>
#include <commdlg.h>
#include <dlgs.h>
#include <stdlib.h>
#include <io.h>
#include <direct.h>
#include "global.h"
#include "file.h"
#include "dib.h"
#include "errors.h"
#include "palfade.h"

// Local function prototypes
int CheckIfFileExists (char * szFName);
int GetDibInfo (char * szFName, INFOSTRUCT * info);

//***********************************************************************
// Function: ReadDIBFile
//
// Purpose: Reads a DIB file from disk into a buffer
//
// Parameters:
//    szFile  == name of DIB file to read
//
// Returns: handle to buffer containing the DIB or NULL if error
//
// Comments:
//
// History: Date      Author        Reason
//
//***********************************************************************

HGLOBAL ReadDIBFile (LPCSTR szFile)
{
  BITMAPFILEHEADER  bmfHeader;
  DWORD             dwSize;
  HGLOBAL           hdib;
  LPSTR             lpdib;
  OFSTRUCT          of;
  int               fh;

  if (!szFile)
    return NULL;

  if ((fh = OpenFile(szFile, &of, OF_READ)) == -1)
    return NULL;

  // get length of DIB in bytes for use when reading

  dwSize = _filelength(fh);

  // Go read the DIB file header and check if it's valid.

  if ((_lread(fh, (LPSTR) &bmfHeader, sizeof(bmfHeader)) != sizeof(bmfHeader))
        || (bmfHeader.bfType != DIB_HEADER_MARKER))
    {
    _lclose(fh);
    DIBError(ERR_NOT_DIB);
    return NULL;
    }     

  // Allocate memory for DIB

  dwSize -= sizeof(BITMAPFILEHEADER);
  hdib = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT, dwSize);

  if (!hdib)
    {
    _lclose(fh);
    DIBError (ERR_MEMORY);
    return NULL;
    }

  lpdib = GlobalLock (hdib);

  // Read the bits.

  if (_hread (fh, lpdib, dwSize) != (LONG) dwSize)
    {
    _lclose(fh);
    GlobalUnlock (hdib);
    GlobalFree   (hdib);
    DIBError (ERR_READ);
    return NULL;
    }

  GlobalUnlock (hdib);
  _lclose(fh);
  return hdib;
}
	
//***********************************************************************
// Function: LoadBM2
//
// Purpose: Reads a "dual" DIB and extracts the two 16-color color tables
//
// Parameters:
//    szFile  == name of DIB file to read
//    pe1     == address of buffer for first color table
//    pe2     == address of buffer for second color table
//
// Returns: handle to buffer containing the BM2 or NULL if error
//
// Comments:
//  Assumes the buffers pointed to by pe1 and pe2 are large enough to hold
//  16 PALETTEENTRY's each.  After extracting the color tables, calls
//  CreatePalArray to create the 256-color palette and puts indices
//  for DIB_PAL_COLORS into the BM2's color table.
//
// History: Date      Author        Reason
//          4/15/92   JMS           Created
//***********************************************************************

HGLOBAL LoadBM2(LPCSTR szFile, PALETTEENTRY *pe1, PALETTEENTRY *pe2)
{
  HGLOBAL             hBM2;
  LPBITMAPINFOHEADER  lpbi;
  LPRGBQUAD           lpRGB;
  LPWORD              lpPalIndex;
  PALETTEENTRY       *pe;
  WORD                i;

  if (!(szFile && pe1 && pe2))
    return NULL;

  if (!(hBM2 = ReadDIBFile(szFile)))
    return NULL;

  lpbi = (LPBITMAPINFOHEADER) GlobalLock(hBM2);

  // Make sure there is room for DIB_PAL_COLORS indices
  if (PaletteSize((LPSTR) lpbi) < 256*sizeof(WORD))
    {
    GlobalUnlock(hBM2);
    GlobalFree(hBM2);
    return NULL;
    }

  if (lpbi->biClrImportant != 15)
    lpbi->biClrImportant = 16;

  // Load the 16-color palettes from the color table
  lpRGB = (LPRGBQUAD) (lpbi + 1);

  for (i=0, pe=pe1; i< (WORD) lpbi->biClrImportant; i++, pe++, lpRGB++)
  {
    pe->peRed   = lpRGB->rgbRed;
    pe->peGreen = lpRGB->rgbGreen;
    pe->peBlue  = lpRGB->rgbBlue;
    pe->peFlags = PC_RESERVED;
  }

  for (i=0, pe=pe2; i< (WORD) lpbi->biClrImportant; i++, pe++, lpRGB++)
  {
    pe->peRed   = lpRGB->rgbRed;
    pe->peGreen = lpRGB->rgbGreen;
    pe->peBlue  = lpRGB->rgbBlue;
    pe->peFlags = PC_RESERVED;
  }

  // Create the palette and color array.

  if(!(hPal = CreatePalArray(pe1, pe2, (int) lpbi->biClrImportant)))
    {
    GlobalUnlock(hBM2);
    GlobalFree(hBM2);
    DIBError(ERR_CREATEPAL);
    return NULL;
    }

  // Reload the color table with DIB_PAL_COLORS indices and we are done

  lpPalIndex = (LPWORD) (lpbi + 1);

  for (i=0; i<256; i++, lpPalIndex++)
    *lpPalIndex = i;

  GlobalUnlock(hBM2);

  return hBM2;
}

//***********************************************************************
// Function: LoadBM2
//
// Purpose: Reads a standard DIB, converts it to 4-bpp, BI_RGB and 
//          extracts the color table.
//
// Parameters:
//    szFile  == name of DIB file to read
//    pe16    == address of buffer for color table
//
// Returns: handle to buffer containing the DIB or NULL if error
//
// Comments:
//  Assumes the buffer pointed to by pe16 is large enough to hold
//  16 PALETTEENTRY's.  After extracting the color tables, puts 
//  indices for DIB_PAL_COLORS into the DIB's color table.  In the
//  process, finds the darkest and lightest colors and puts them
//  first and last, respectively, in the color table.
//
// History: Date      Author        Reason
//          4/15/92   JMS           Created
//***********************************************************************

HGLOBAL LoadBMP(LPCSTR szFile, PALETTEENTRY *pe16)
{
  HGLOBAL             hdib;
  LPBITMAPINFOHEADER  lpbi;
  LPRGBQUAD           lpRGB;
  LPWORD              lpPalIndex;
  PALETTEENTRY        peTemp;
  DWORD               dwMax=0L, dwMin=0x30000L, dwTemp;
  WORD                i, wFirst=0, wLast=15;

  if (!szFile)
    return NULL;

  if (!(hdib = ReadDIBFile(szFile)))
    return NULL;

  if (!(lpbi = (LPBITMAPINFOHEADER) GlobalLock(hdib)))
    {
    GlobalFree(hdib);
    return NULL;
    }

  // Check the DIB format and render it as 4-bit & BI_RGB if necessary
  if ((lpbi->biBitCount != 4) || (lpbi->biCompression != BI_RGB))
    {
    HGLOBAL h;

    GlobalUnlock(hdib);

    if (lpbi->biBitCount > 4)
      if (MessageBox(GetFocus(), "This bitmap must be converted to a 16 color bitmap.\r\nDo you wish to continue?",
          NULL, MB_ICONEXCLAMATION | MB_YESNO) == IDNO)
        {
        GlobalFree(hdib);
        return NULL;
        }

    if (h = RenderDIBFormat(hdib, 4, BI_RGB))
      {
      hdib = h;
      if (!(lpbi = (LPBITMAPINFOHEADER) GlobalLock(hdib)))
        {
        GlobalFree(hdib);
        return NULL;
        }
      }
    else
      {
      GlobalFree(hdib);
      return NULL;
      }
    }

  // Load the 16-color palette from the color table
  // and find the lightest and darkest colors in the color table
  lpRGB = (LPRGBQUAD) (lpbi + 1);

  // should check for less than 16 colors but oh well
  for (i=0; i<16; i++, lpRGB++)
    {
    pe16[i].peRed   = lpRGB->rgbRed;
    pe16[i].peGreen = lpRGB->rgbGreen;
    pe16[i].peBlue  = lpRGB->rgbBlue;
    pe16[i].peFlags = PC_RESERVED;

    dwTemp =   ((DWORD) pe16[i].peRed  * pe16[i].peRed)
             + ((DWORD) pe16[i].peGreen* pe16[i].peGreen)
             + ((DWORD) pe16[i].peBlue * pe16[i].peBlue);

    if (dwTemp > dwMax)
      {
      dwMax = dwTemp;
      wLast = i;
      }
  
    if (dwTemp < dwMin)
      {
      dwMin = dwTemp;
      wFirst = i;
      }
    }

  // Load color table with DIB_PAL_COLORS indices

  lpPalIndex = (LPWORD) (lpbi + 1);
  for (i=0; i<16; i++, lpPalIndex++)
    *lpPalIndex = i;

  // Switch _PAL_ indices and palette entries so that the 
  // darkest color is first and the lightest is last.
  // This stuff is only necessary when we'll be using all 16
  // colors, but we don't know that here, so just do always.

  lpPalIndex = (LPWORD) (lpbi + 1);

  if ((i = lpPalIndex[0]) != wFirst)
    {
    lpPalIndex[0] = wFirst;
    lpPalIndex[wFirst] = i;

    peTemp = pe16[0];
    pe16[0] = pe16[wFirst];
    pe16[wFirst] = peTemp;
    }

  if ((i = lpPalIndex[15]) != wLast)
    {
    lpPalIndex[15] = wLast;
    lpPalIndex[wLast] = i;

    peTemp = pe16[15];
    pe16[15] = pe16[wLast];
    pe16[wLast] = peTemp;
    }

  GlobalUnlock(hdib);

  return hdib;
}

//***********************************************************************
// Function: GetFileName
//
// Purpose: Prompts user for a filename through the use of a Windows 3.1
//          FileOpen common dialog box.
//
// Parameters:
//    szFile    == address of buffer for file name
//    wIDString == identifies whether to get an 'open' or 'save' file name
//
// Returns: TRUE if a filename is selected.
//          FALSE if no filename is selected.
//
// Comments:
//
// History: Date      Author        Reason
//          7/27/92   JMS           Created
//***********************************************************************

BOOL GetFileName (LPSTR szFName, WORD wIDString)
{
  OPENFILENAME  of;
  DWORD         flags;
  BOOL          bRet;
  static char   szTitle[30];          // Dialog Box title
  static char   szFile[_MAX_PATH];    // File name
  static char   szFileTitle[13];      // Title
  static char  *szFilter[] = {"Dual Bitmap",
                              "*.bm2",
                              "" };

  // Initialize the OPENFILENAME members

  if (wIDString == IDS_OPEN)          // GetOpenFileName
    {
    flags = OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
    }
  else if (wIDString == IDS_SAVEAS)   // GetSaveFileName
    {
    flags = OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT;
    }
  else
    return FALSE;

  LoadString (ghInst, wIDString, szTitle, sizeof (szTitle));
  szFile[0] = 0;

  of.lStructSize       = sizeof(OPENFILENAME);
  of.hwndOwner         = GetFocus();
  of.hInstance         = ghInst;
  of.lpstrFilter       = szFilter[0];
  of.lpstrCustomFilter = NULL;
  of.nMaxCustFilter    = 0L;
  of.nFilterIndex      = 1L;
  of.lpstrFile         = szFile;
  of.nMaxFile          = sizeof(szFile);
  of.lpstrFileTitle    = szFileTitle;
  of.nMaxFileTitle     = sizeof(szFileTitle);
  of.lpstrInitialDir   = szDirName;
  of.lpstrTitle        = szTitle;
  of.Flags             = flags;
  of.nFileOffset       = 0;
  of.nFileExtension    = 0;
  of.lpstrDefExt       = NULL;
  of.lpfnHook          = NULL;
  of.lpTemplateName    = NULL;

   // Call the Get(Open/Save)Filename function

  if (wIDString == IDS_OPEN)
    {
    if (GetOpenFileName (&of))
      {
      lstrcpy (szFName, of.lpstrFile);
      bRet = TRUE;
      }
    else
      bRet = FALSE;
    }
  else
    {
    if (GetSaveFileName (&of))
      {
      lstrcpy (szFName, of.lpstrFile);
      bRet = TRUE;
      }
    else
      bRet = FALSE;
    }

  // Save current working directory for next call to GetFileName
  _getcwd (szDirName, sizeof (szDirName));

  return bRet;
}

//***********************************************************************
// Function: WriteDIBFile
//
// Purpose: Writes a DIB to a file.
//
// Parameters:
//    szFile    == name of file
//    hdib      == buffer containing DIB in CF_DIB format
//
// Returns: TRUE if successful, FALSE otherwise
//
// Comments:
//
// History: Date      Author        Reason
//
//***********************************************************************

BOOL WriteDIBFile (LPCSTR szFile, HGLOBAL hdib)
{
  BITMAPFILEHEADER    hdr;
  LPBITMAPINFOHEADER  lpbi;
  int                 fh;
  OFSTRUCT            of;
  DWORD               dwSize;

  if (!(szFile && hdib))
    return FALSE;

  if ((fh = OpenFile(szFile, &of, OF_CREATE|OF_READWRITE)) == -1)
    return FALSE;

  lpbi = (LPBITMAPINFOHEADER) GlobalLock(hdib);
  dwSize = GlobalSize(hdib);

  /* Fill in the fields of the file header */
  hdr.bfType      = DIB_HEADER_MARKER;
  hdr.bfSize      = dwSize + sizeof(BITMAPFILEHEADER);
  hdr.bfReserved1 = 0;
  hdr.bfReserved2 = 0;
  hdr.bfOffBits   = (DWORD)sizeof(BITMAPFILEHEADER) + lpbi->biSize
                                    + PaletteSize((LPSTR)lpbi);

  /* Write the file header */
  if (_lwrite(fh, (LPSTR) &hdr, sizeof(BITMAPFILEHEADER)) != sizeof(BITMAPFILEHEADER))
    {
    GlobalUnlock(hdib);
    _lclose(fh);
    return FALSE;
    }

  /* Write the DIB header and the bits */
  if (_hwrite(fh, (HPBYTE) lpbi, dwSize) != (LONG) dwSize)
    {
    GlobalUnlock(hdib);
    _lclose(fh);
    return FALSE;
    }

  GlobalUnlock (hdib);
  _lclose (fh);
  return TRUE;
}

//***********************************************************************
// Function: InfoDlgProc
//
// Purpose: Window procedure for the Dib Information dialog box.
//
// Parameters: standard window procedure parameters
//
// Returns: TRUE if the message was processed, FALSE otherwise
//
// Comments: Borrowed from DIBView
//
// History: Date      Author        Reason
//          6/27/91                 Created
//***********************************************************************

BOOL FAR PASCAL InfoDlgProc (HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
  switch (msg)
    {
    case WM_INITDIALOG:
      {
      char szBuffer[20];
      INFOSTRUCT *pInfo = (INFOSTRUCT *) lParam;

      // Set the strings into the dialog box.

      SetDlgItemText(hDlg, IDD_NAME, (LPSTR) pInfo->szName);
      SetDlgItemText(hDlg, IDD_FORMAT, (LPSTR) pInfo->szType);
      wvsprintf ((LPSTR)szBuffer, (LPSTR) "%lu", (LPSTR) &pInfo->cbWidth);
      SetDlgItemText(hDlg, IDD_WIDTH, (LPSTR) szBuffer);
      wvsprintf ((LPSTR)szBuffer, (LPSTR) "%lu", (LPSTR) &pInfo->cbHeight);
      SetDlgItemText(hDlg, IDD_HEIGHT, (LPSTR) szBuffer);
      wvsprintf ((LPSTR)szBuffer, (LPSTR) "%lu", (LPSTR) &pInfo->cbColors);
      SetDlgItemText(hDlg, IDD_COLORS, (LPSTR) szBuffer);

      if (pInfo->szCompress[0] == 0)
        ShowWindow(GetDlgItem (hDlg, IDD_COMPHEAD), SW_HIDE);
      else
        SetDlgItemText(hDlg, IDD_COMPRESS, (LPSTR) pInfo->szCompress);

      return TRUE;
      }

    case WM_COMMAND:
      if (wParam == IDOK || wParam == IDCANCEL)
        {
        EndDialog(hDlg, TRUE);
        return TRUE;
        }
      break;
    }
  return FALSE;
}

//***********************************************************************
// Function: FileOpenHookProc
//
// Purpose: Hook procedure for FileOpen common dialog box.
//
// Parameters: standard window procedure parameters
//
// Returns: TRUE if the message was processed, FALSE otherwise
//
// Comments: Borrowed from DIBView
//
// History: Date      Author        Reason
//          6/27/91                 Created
//          8/6/92    JMS           Added options dialog stuff
//***********************************************************************

BOOL FAR PASCAL FileOpenHookProc (HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
  static LPARAM lpOptions;

  switch (msg)
    {
    case WM_INITDIALOG:
      lpOptions = ((LPOPENFILENAME) lParam)->lCustData;
      break;

    case WM_COMMAND:
      switch (wParam)
        {
        case IDOK:
          _getcwd (szDirName, sizeof (szDirName));
          break;

        case IDD_INFO:
          {
          DLGPROC lpDlg;
          INFOSTRUCT info;
          char szFName[_MAX_PATH];

          // User has selected the Dib Information button.  Query the
          // text in the edit control, check if it is a valid file, get
          // the file statistics, and then display the dialog box.

          GetDlgItemText (hDlg, edt1, (LPSTR) szFName, sizeof (szFName));

          if (!CheckIfFileExists(szFName))
            {
            DIBError (ERR_OPEN);
            SetFocus (GetDlgItem(hDlg, edt1));
            return FALSE;
            }

          if (!GetDibInfo (szFName, &info))
            {
            DIBError (ERR_READ);
            return FALSE;
            }

          lpDlg = (DLGPROC) MakeProcInstance((FARPROC)InfoDlgProc, ghInst);
          DialogBoxParam(ghInst, "INFO", hDlg, lpDlg, (LPARAM)(LPINFOSTRUCT) &info);
          FreeProcInstance ((FARPROC) lpDlg);
          return TRUE;
          }

        case IDD_OPTIONS:
          {
          DLGPROC lpDlg = (DLGPROC) MakeProcInstance((FARPROC)OptionsDlgProc, ghInst);
          DialogBoxParam(ghInst, "OPTIONS", hDlg, lpDlg, lpOptions);
          FreeProcInstance ((FARPROC) lpDlg);
          return TRUE;
          }
        }
      break;
    }    
  return FALSE;
}

/***********************************************************************

  Function: CheckIfFileExists (char *)

   Purpose: Checks to see if the user selected file exists.

   Returns: TRUE if the file is opened.
            FALSE if the file does not exists.

  Comments: After checking the file, the file is immediately closed again.

   History: Date      Reason

            6/27/91   Created

*************************************************************************/

int CheckIfFileExists (char *szFName)
{
  int      hFile;
  OFSTRUCT ofstruct;

  if (szFName[0] == 0)
    return FALSE;

  hFile = OpenFile(szFName, &ofstruct, OF_EXIST);

  if (hFile > 0)
    return TRUE;
  else
    return FALSE;
}


/*************************************************************************

  Function: GetDibInfo (char *, INFOSTRUCT *)

   Purpose: Retrieves the INFOSTRUCT specifications about the selected
            file when the "Dib Information..." button is selected.

   Returns: TRUE if successful
            FALSE if an error occurs.

  Comments: This routine will handle both Windows and PM bitmaps.
            If an PM bitmap is found, NULL is returned for the
            compression type.

   History: Date      Reason

            6/27/91   Created

*************************************************************************/

int GetDibInfo (char * szFName, INFOSTRUCT * info)
{
  BITMAPFILEHEADER   bmfHeader;
  BITMAPINFOHEADER   DIBHeader;
  DWORD              dwHeaderSize;
  int                hFile;
  OFSTRUCT           ofstruct;
  BITMAPCOREHEADER   bmCore;
  long               lFilePos;
  char               szBuffer[20];


  if (!szFName[0])
    return NULL;

  // fill in filename into structure.

  lstrcpy ((LPSTR) info->szName, (LPSTR) szFName);

  hFile=OpenFile((LPSTR) szFName, &ofstruct, OF_READ | OF_SHARE_DENY_WRITE);

  if (hFile == 0)
    return NULL;

  // read the BITMAPFILEHEADER structure and check for BM marker

  if ((_lread (hFile, (LPSTR) &bmfHeader, sizeof (bmfHeader)) != sizeof (bmfHeader)) ||
       (bmfHeader.bfType != DIB_HEADER_MARKER))
    {
    _lclose(hFile);
    DIBError (ERR_NOT_DIB);
    return NULL;
    }

  // Get the current file pointer position, and then read the next
  // DWORD.  This DWORD is the size of the following structure which
  // will determine if it is a PM DIB or Windows DIB.

  if (((lFilePos = _tell (hFile)) == -1) ||
      (_lread (hFile, (LPSTR) &dwHeaderSize, sizeof (dwHeaderSize))
          != sizeof (dwHeaderSize)))
    {
    _lclose(hFile);
    return NULL;
    }

  // Back the file pointer up a DWORD so that we can read the information
  // into the correct data structure.

  _lseek (hFile, lFilePos, SEEK_SET);


  if (dwHeaderSize == sizeof (BITMAPCOREHEADER))      // PM dib
    {
    _lread (hFile, (LPSTR) &bmCore, sizeof (bmCore));

    LoadString (ghInst, IDS_PMBMP, szBuffer, sizeof(szBuffer));
    lstrcpy ((LPSTR)info->szType, (LPSTR) szBuffer);

    info->cbWidth  = bmCore.bcWidth;
    info->cbHeight = bmCore.bcHeight;
    info->cbColors = (DWORD)1L << (bmCore.bcBitCount);
    szBuffer[0]=0;
    lstrcpy ((LPSTR) info->szCompress, (LPSTR) szBuffer);

    }
  else if (dwHeaderSize == sizeof (BITMAPINFOHEADER))  // windows dib
    {
    _lread (hFile, (LPSTR) &DIBHeader, sizeof (DIBHeader));

    LoadString (ghInst, IDS_WINBMP, szBuffer, sizeof(szBuffer));
    lstrcpy ((LPSTR)info->szType, (LPSTR) szBuffer);

    info->cbWidth  = DIBHeader.biWidth;
    info->cbHeight = DIBHeader.biHeight;
    info->cbColors = (DWORD)1L << DIBHeader.biBitCount;

    switch (DIBHeader.biCompression)
      {
      case BI_RGB:
        LoadString (ghInst, IDS_RGB, szBuffer, sizeof(szBuffer));
        break;

      case BI_RLE4:
        LoadString (ghInst, IDS_RLE4, szBuffer, sizeof(szBuffer));
        break;

      case BI_RLE8:
        LoadString (ghInst, IDS_RLE8, szBuffer, sizeof(szBuffer));
        break;

      default:
        szBuffer[0]=0;
      }

    lstrcpy ((LPSTR) info->szCompress, (LPSTR) szBuffer);
    }
  else
    {
    DIBError (ERR_NOT_DIB);
    _lclose(hFile);
    return NULL;
    }

  _lclose(hFile);
  return 1;
}
