//****************************************************************************
// File: mainwnd.c
//
// Purpose: Main overlapped window's message handler
//
// Functions:
//    MainWndProc() - message handler for main overlapped window
//    DoCommands() - called by MainWndProc to handle all WM_COMMAND messages
//
// Development Team:
//
//
// Written by Microsoft Product Support Services, Windows Developer Support
// COPYRIGHT:
//
//   (C) Copyright Microsoft Corp. 1993.  All rights reserved.
//
//   You have a royalty-free right to use, modify, reproduce and
//   distribute the Sample Files (and/or any modified version) in
//   any way you find useful, provided that you agree that
//   Microsoft has no warranty obligations or liability for any
//   Sample Application Files which are modified.
//
//****************************************************************************

#include <windows.h>
#include <commdlg.h>
#include <stdlib.h>
#include <string.h>
#include <direct.h>
#include "global.h"
#include "mainwnd.h"
#include "file.h"
#include "about.h"
#include "palfade.h"
#include "dib.h"
#include "errors.h"
#include "options.h"

// Global variables local to this file
static HBITMAP  hbm;
static char     szAppName[64];
static char     szFileName[_MAX_PATH];
static int      nCurrentStep;
static int      nTotalSteps;
static OPTIONS  Options;
static BOOL     bUpdateIni;

// Local function prototypes
static LONG DoCommands (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);
static VOID Cleanup(HWND hWnd);
static VOID SizeWindowToDIB(HWND hWnd, HGLOBAL hdib);

//****************************************************************************
// Function: MainWndProc
//
// Purpose: Message handler for main overlapped window.
//
// Parameters:
//    hWnd    == Handle to _this_ window.
//    message == Message to process.
//    wParam  == WORD parameter -- depends on message
//    lParam  == LONG parameter -- depends on message
//
// Returns: Depends on message.
//
// Comments:
//
// History:  Date       Author        Reason
//           2/10/92    JMS           Created
//****************************************************************************

LONG FAR PASCAL MainWndProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
  switch (msg)
    {
    // Dispatch WM_COMMAND messages to our command handler, DoCommands().
    case WM_COMMAND:
      return DoCommands(hWnd, msg, wParam, lParam);

    case WM_CREATE:
      // Initialize global variables
      hDIB          = NULL;
      hPal          = NULL;
      hbm           = NULL;
      bUpdateIni    = FALSE;

      // Get initial or default options
      ReadOptions(&Options);

      nCurrentStep  = 0;
      nTotalSteps   = Options.nSteps;

      LoadString(ghInst, IDS_APPNAME, szAppName, sizeof(szAppName));
      _getcwd (szDirName, sizeof (szDirName));

      // Check the appropriate Animate menu item.
      CheckMenuItem(GetMenu(hWnd), IDM_0 + nTotalSteps, MF_BYCOMMAND | MF_CHECKED);

      // No need for scroll bar until we have a bitmap.
      SetScrollRange(hWnd, SB_VERT, 0, 0, FALSE);

      break;

    case WM_PAINT:
      {
      PAINTSTRUCT ps;
      BITMAP      bm;
      HDC         hMemDC;
      HBITMAP     hbmOld;
      HPALETTE    hpalOld;

      BeginPaint(hWnd, &ps);

      if (!(hbm && hPal))     // Do we have a bitmap?
        {
        EndPaint(hWnd, &ps);
        break;
        }

      hpalOld = SelectPalette(ps.hdc, hPal, FALSE);
      RealizePalette(ps.hdc);

      hMemDC = CreateCompatibleDC(ps.hdc);
      hbmOld = SelectObject(hMemDC, hbm);
      GetObject(hbm, sizeof(BITMAP), (LPSTR) &bm);

      BitBlt(ps.hdc,
        0, 0,
        bm.bmWidth, bm.bmHeight,
        hMemDC,
        0, 0,
        SRCCOPY);

      SelectObject(hMemDC, hbmOld);
      SelectPalette(ps.hdc, hpalOld, TRUE);
      DeleteDC(hMemDC);
      EndPaint(hWnd, &ps);
      break;
      }

    case WM_PALETTECHANGED:
      // If we caused the palette change, just break and do nothing, 
      // otherwise, fall through to WM_QUERYNEWPALETTE case.

      if ((HWND)wParam == hWnd)
        break;

    case WM_QUERYNEWPALETTE:
      if (hbm && hPal)
        {
        int i;
        HDC hDC;
        HPALETTE hpalOld;
        LPBITMAPINFOHEADER lpbi;

        hDC = GetDC(hWnd);

        lpbi = (LPBITMAPINFOHEADER) GlobalLock(hDIB);
        if ((msg==WM_QUERYNEWPALETTE) && (lpbi->biClrImportant != 15))
          {
          UnrealizeObject(hPal);
          UsePalette(hDC, TRUE);
          }
        GlobalUnlock(hDIB);

        hpalOld = SelectPalette(hDC, hPal, FALSE);
        i = RealizePalette(hDC);
        SelectPalette(hDC, hpalOld, TRUE);
        ReleaseDC(hWnd, hDC);

        // If the palette realization changed, force a redraw
        if (i)
          InvalidateRect (hWnd, NULL, TRUE);

        return i;
        }
      else
        return 0;
      break;

    case WM_ACTIVATEAPP:
      // If we're activating, we'll get a WM_QUERYNEWPALETTE so do nothing here.
      if (wParam)
        return 1;

      // If we're de-activating, we may need to give back the static colors.
      if (hbm && hPal)
        {
        HDC hDC = GetDC(hWnd);

        if (GetSystemPaletteUse(hDC) == SYSPAL_NOSTATIC)
          {
          UnrealizeObject(hPal);
          UsePalette(hDC, FALSE);
          }

        ReleaseDC(hWnd, hDC);
        }
      return 0;

    case WM_VSCROLL:
      switch(wParam)
        {
        case SB_LINEDOWN:
        case SB_PAGEDOWN:
          // Animate forward
          return DoCommands(hWnd, WM_COMMAND, IDM_STEPFORWARD, 0L);

        case SB_LINEUP:
        case SB_PAGEUP:
          // Animate backward
          return DoCommands(hWnd, WM_COMMAND, IDM_STEPBACKWARD, 0L);
 
        case SB_THUMBTRACK:
        case SB_THUMBPOSITION:
          // Get current thumb position and update window accordingly.
          nCurrentStep = LOWORD(lParam);
          SetScrollPos(hWnd, SB_VERT, nCurrentStep, TRUE);
          FadePalette(hWnd, nCurrentStep, nTotalSteps);
          return NULL;
 
        default:
          return NULL;
        }
      break;

    // See if we have an unsaved file before closing the app.
    // Note that the IDM_CLOSE handler will clean up any stray objects.

    case WM_CLOSE:
      if (DoCommands(hWnd, WM_COMMAND, IDM_CLOSE, 0L))
        DestroyWindow(hWnd);
      break;

    case WM_QUERYENDSESSION:
      return DoCommands(hWnd, WM_COMMAND, IDM_CLOSE, 0L);

    // On WM_DESTROY, terminate this app by posting a WM_QUIT message.

    case WM_DESTROY:
      if (Options.wFlags&OPTIONSCHANGED)
        WriteOptions(&Options);
      PostQuitMessage(0);
      break;

    // We didn't handle, pass to DefWindowProc.

    default:
      return DefWindowProc(hWnd, msg, wParam, lParam);
    }
  return (NULL);
}

//****************************************************************************
// Function: DoCommands
//
// Purpose: Called by MainWndProc() to handle all WM_COMMAND messages.
//
// Parameters:
//    hWnd    == Handle to _this_ window.
//    message == Message to process.
//    wParam  == WORD parameter -- depends on message
//    lParam  == LONG parameter -- depends on message
//
// Returns : Depends on message.
//
// Comments:
//
// History:  Date       Author        Reason
//           2/10/92    JMS           Created
//****************************************************************************

LONG DoCommands (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
 static BOOL bStop = FALSE;

  switch (wParam)
  {
    case IDM_NEW:

      // Close current file and clean up if necessary

      if (hDIB && !DoCommands(hWnd, WM_COMMAND, IDM_CLOSE, 0L))
        break;

      {
      OPENFILENAME of;
      HANDLE  hdib1, hdib2;
      FARHOOK fpDlgFunc;
      int     i;
      static  char  szTitle[30];         // Dialog Box title
      static  char  szTemplate[20];      // Dialog Box template
      static  char  szFile[_MAX_PATH];   // File name
      static  char  szFileTitle[13];     // Title
      static  char *szFilter[] = {"Windows Bitmap",
                                  "*.bmp;*.dib;*.rle",
                                  "" };

      // Initialize the OPENFILENAME members

      szFile[0] = 0;
      LoadString (ghInst, IDS_OPENFIRST, szTitle, sizeof (szTitle));
      LoadString (ghInst, IDS_FILEOPEN, szTemplate, sizeof (szTemplate));
      fpDlgFunc = (FARHOOK) MakeProcInstance((FARPROC)FileOpenHookProc, ghInst);

      of.lStructSize        = sizeof(OPENFILENAME);
      of.hwndOwner          = hWnd;
      of.hInstance          = ghInst;
      of.lpstrFilter        = szFilter[0];
      of.lpstrCustomFilter  = NULL;
      of.nMaxCustFilter     = 0L;
      of.nFilterIndex       = 1L;
      of.lpstrFile          = szFile;
      of.nMaxFile           = sizeof(szFile);
      of.lpstrFileTitle     = szFileTitle;
      of.nMaxFileTitle      = sizeof(szFileTitle);
      of.lpstrInitialDir    = szDirName;
      of.lpstrTitle         = szTitle;
      of.Flags              = OFN_ENABLEHOOK | OFN_ENABLETEMPLATE
                                | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
      of.nFileOffset        = 0;
      of.nFileExtension     = 0;
      of.lpstrDefExt        = NULL;
      of.lCustData          = (LPARAM)(LPOPTIONS)&Options;
      of.lpfnHook           = fpDlgFunc;
      of.lpTemplateName     = szTemplate;

      // Get first file name

      if (!GetOpenFileName(&of))
        break;

      // Read first bitmap
      if (hdib1 = LoadBMP(of.lpstrFile, pe16Color1))
        {
        // Get second file name
        LoadString (ghInst, IDS_OPENSECOND, szTitle, sizeof (szTitle));
        if (GetOpenFileName(&of))
          {
          HCURSOR hcur;

          // Put up hourglass
          hcur = SetCursor(LoadCursor(NULL, IDC_WAIT));

          // Read second bitmap
          if (hdib2 = LoadBMP(of.lpstrFile, pe16Color2))
            {
            // Create the dual bitmap and initialize the fade step array
            if (hDIB = CreateBM2(&hdib1, pe16Color1, &hdib2, pe16Color2, &Options))
              CalculateFadeStep(pe16Color1, pe16Color2, nTotalSteps);

            GlobalFree(hdib2);      // Don't need hdib2 anymore.
            }
          else
            DIBError(ERR_READ);

          SetCursor(hcur);          // Restore cursor
          }
        GlobalFree(hdib1);          // Don't need hdib1 anymore.
        }
      else
        DIBError(ERR_READ);

      // Free hook procedure thunk
      FreeProcInstance((FARPROC) fpDlgFunc);

      if (!(hPal && hDIB))  // Sanity check. If these don't exist, bail out now.
        {
        Cleanup(hWnd);
        break;
        }

      nCurrentStep = 0;

      // Create device-dependent bitmap for display
      if (!(hbm = BM2ToBitmap(hDIB, hPal)))
        {
        DIBError(ERR_CREATEDDB);
        Cleanup(hWnd);
        break;
        }

      // Resize window to fit bitmap and enable scroll bar and menu items

      SizeWindowToDIB(hWnd, hDIB);

      SetScrollRange(hWnd, SB_VERT, 0, nTotalSteps, FALSE);
      EnableMenuItem(GetMenu(hWnd), IDM_SAVE, MF_BYCOMMAND | MF_ENABLED);
      EnableMenuItem(GetMenu(hWnd), IDM_CLOSE, MF_BYCOMMAND | MF_ENABLED);
      for (i=IDM_ANIMATEFIRST; i<=IDM_ANIMATELAST; i++)
        EnableMenuItem(GetMenu(hWnd), i, MF_BYCOMMAND | MF_ENABLED);

      break;
      }

    case IDM_OPEN:

      // Close current file and clean up if necessary

      if (hDIB && !DoCommands(hWnd, WM_COMMAND, IDM_CLOSE, 0L))
        break;

      // prompt for filename

      if (!GetFileName(szFileName, IDS_OPEN))
        break;

      // Read the BM2 file and initialize the fade step array
      if (hDIB = LoadBM2(szFileName, pe16Color1, pe16Color2))
        CalculateFadeStep(pe16Color1, pe16Color2, nTotalSteps);
      else
        {
        Cleanup(hWnd);
        break;
        }

      nCurrentStep = 0;

      // Create bitmap for display
      if (!(hbm = BM2ToBitmap(hDIB, hPal)))
        {
        DIBError(ERR_CREATEDDB);
        Cleanup(hWnd);
        break;
        }

      SetWindowText(hWnd, szFileName);
      SizeWindowToDIB(hWnd, hDIB);      // This causes a repaint

      // Turn on scroll bar.
      SetScrollRange(hWnd, SB_VERT, 0, nTotalSteps, FALSE);

      // Since we have a bitmap, enable those menu items that
      // are only valid with a bitmap.
      EnableMenuItem(GetMenu(hWnd), IDM_SAVE, MF_BYCOMMAND | MF_ENABLED);
      EnableMenuItem(GetMenu(hWnd), IDM_CLOSE, MF_BYCOMMAND | MF_ENABLED);
      {
      int i;

      for (i=IDM_ANIMATEFIRST; i<=IDM_ANIMATELAST; i++)
        EnableMenuItem(GetMenu(hWnd), i, MF_BYCOMMAND | MF_ENABLED);
      }

      break;

    case IDM_SAVE:
      if (!hDIB)
        return TRUE;      // nothing to save so return success

      // Get the FileName to save under.

      if (GetFileName (szFileName, IDS_SAVEAS))
        {
        int i;
        LPRGBQUAD lpRGB;
        LPBITMAPINFOHEADER lpbi = (LPBITMAPINFOHEADER) GlobalLock(hDIB);

        SetCursor(LoadCursor(NULL, IDC_WAIT));
        SetWindowText(hWnd, szFileName);

        // Restore the color tables and write the file

        lpRGB = (LPRGBQUAD) (lpbi + 1);

        for (i=0; i<(int)lpbi->biClrImportant; i++, lpRGB++)
          {
          lpRGB->rgbRed   = pe16Color1[i].peRed;
          lpRGB->rgbBlue  = pe16Color1[i].peBlue;
          lpRGB->rgbGreen = pe16Color1[i].peGreen;
          }

        for (i=0; i<(int)lpbi->biClrImportant; i++, lpRGB++)
          {
          lpRGB->rgbRed   = pe16Color2[i].peRed;
          lpRGB->rgbBlue  = pe16Color2[i].peBlue;
          lpRGB->rgbGreen = pe16Color2[i].peGreen;
          }

        if (!WriteDIBFile (szFileName, hDIB))
          DIBError (ERR_WRITEDIB);

        // Clean up and return.

        SetCursor(LoadCursor(NULL, IDC_ARROW));
        return TRUE;
        }
      else
        return FALSE;   // didn't get a file name
      break;
   
    case IDM_CLOSE:
      if (!hDIB)            // Nothing to save so return success
      	return TRUE;
   
      if (szFileName[0])    // Already have file name (assume saved)
        {
        Cleanup(hWnd);
        InvalidateRect(hWnd, NULL, TRUE);
        return TRUE;
        }
   
      // Have file but no file name, prompt to save
      switch (MessageBox(hWnd, "Save current file?", 
                szAppName, MB_ICONEXCLAMATION|MB_YESNOCANCEL))
        {
        case IDYES:
          if (!DoCommands(hWnd, WM_COMMAND, IDM_SAVE, 0L))
            return FALSE;
          // else FALL TRHOUGH!

        case IDNO:
          Cleanup(hWnd);
          InvalidateRect(hWnd, NULL, TRUE);
          return TRUE;
   
        default:            // e.g. IDCANCEL
          return FALSE;
        }
      break;
   
    case IDM_8:
    case IDM_16:
    case IDM_32:
    case IDM_64:
      CheckMenuItem(GetMenu(hWnd), IDM_0 + nTotalSteps, MF_BYCOMMAND | MF_UNCHECKED);
      CheckMenuItem(GetMenu(hWnd), wParam, MF_BYCOMMAND | MF_CHECKED);
      nTotalSteps = Options.nSteps = wParam - IDM_0;
      nCurrentStep = 0;
      if(hbm)
        {
        FadePalette(hWnd, 0, 1);  // Show first image
        CalculateFadeStep(pe16Color1, pe16Color2, nTotalSteps);
        SetScrollRange(hWnd, SB_VERT, 0, nTotalSteps, FALSE);
        SetScrollPos(hWnd, SB_VERT, 0, TRUE);
        }

        // Set high bit of Options.wFlags to indicate that we will need to 
        // save the options on exit.
        Options.wFlags |= OPTIONSCHANGED;
      break;

    case IDM_CUSTOM:
      {
      DLGPROC lpDlg;
      int nRet;

      lpDlg = (DLGPROC) MakeProcInstance((FARPROC)StepDlgProc, ghInst);
      nRet = DialogBoxParam(ghInst, "STEPSIZE", hWnd, lpDlg, (LONG) nTotalSteps);
      FreeProcInstance ((FARPROC) lpDlg);

      if (nRet > 0)
        {
        CheckMenuItem(GetMenu(hWnd), IDM_0 + nTotalSteps, MF_BYCOMMAND | MF_UNCHECKED);
        nTotalSteps = Options.nSteps = nRet;
        nCurrentStep = 0;
        if(hbm)
          {
          FadePalette(hWnd, 0, 1);  // Show first image
          CalculateFadeStep(pe16Color1, pe16Color2, nTotalSteps);
          SetScrollRange(hWnd, SB_VERT, 0, nTotalSteps, FALSE);
          SetScrollPos(hWnd, SB_VERT, 0, TRUE);
          }

        // Set high bit of Options.wFlags to indicate that we will need to 
        // save the options on exit.
        Options.wFlags |= OPTIONSCHANGED;
        }

      break;
      }

    case IDM_STEPFORWARD:
      // Animate one step forward

      if (nCurrentStep < nTotalSteps)
        {
        FadePalette(hWnd, ++nCurrentStep, nTotalSteps);
        SetScrollPos(hWnd, SB_VERT, nCurrentStep, TRUE);
        }

      break;

    case IDM_STEPBACKWARD:
      // Animate one step backward

      if (nCurrentStep > 0)
        {
        FadePalette(hWnd, --nCurrentStep, nTotalSteps);
        SetScrollPos(hWnd, SB_VERT, nCurrentStep, TRUE);
        }

      break;

    case IDM_GOFORWARD:
      // Animate all the way forward.

      while (nCurrentStep < nTotalSteps)
        {
        FadePalette(hWnd, ++nCurrentStep, nTotalSteps);
        SetScrollPos(hWnd, SB_VERT, nCurrentStep, TRUE);
        }

      break;

    case IDM_GOBACKWARD:
      // Animate all the way backward.

      while (nCurrentStep > 0)
        {
        FadePalette(hWnd, --nCurrentStep, nTotalSteps);
        SetScrollPos(hWnd, SB_VERT, nCurrentStep, TRUE);
        }

      break;

	
        // User picked File.Exit, check to see if we have an unsaved 
        // file before terminating the app.  Note that the IDM_CLOSE 
        // handler will clean up any stray objects.

    case IDM_EXIT:
      if (DoCommands(hWnd, WM_COMMAND, IDM_CLOSE, 0L))
        DestroyWindow(hWnd);
      break;


        // Put up the About box.

    case IDM_ABOUT:
      {
      DLGPROC lpDlgProc;

      lpDlgProc = (DLGPROC) MakeProcInstance((FARPROC)AboutDlg, ghInst);

      DialogBox(ghInst,           // current instance
              AboutBoxName,       // resource to use 
              hWnd,               // parent handle   
              lpDlgProc);         // About() instance address

      FreeProcInstance((FARPROC) lpDlgProc);
      }
      break;


        // Must be some system command -- pass it on to the default
        //  window procedure.

    default:
      return DefWindowProc(hWnd, msg, wParam, lParam);
  }

  return (NULL);
}

//****************************************************************************
// Function: Cleanup
//
// Purpose: Deletes objects, frees memory and disables some menu items.
//          Called when closing the current file or to cleanup after errors.
//
// Parameters:
//    hWnd    == Handle to the window.
//
// Returns : No Return
//
// Comments:
//
// History:  Date       Author        Reason
//           2/10/92    JMS           Created
//****************************************************************************

VOID Cleanup(HWND hWnd)
{
	if (hWnd)
		{
    int i;
		HDC hDC = GetDC(hWnd);

    // Give back static colors if necessary.
    if (GetSystemPaletteUse(hDC) == SYSPAL_NOSTATIC)
      UsePalette(hDC, FALSE);

		ReleaseDC(hWnd, hDC);

    SetWindowText(hWnd, szAppName);
		SetScrollRange(hWnd, SB_VERT, 0, 0, FALSE);
		EnableMenuItem(GetMenu(hWnd), IDM_SAVE, MF_BYCOMMAND | MF_GRAYED);
		EnableMenuItem(GetMenu(hWnd), IDM_CLOSE, MF_BYCOMMAND | MF_GRAYED);
    for (i=IDM_ANIMATEFIRST; i<=IDM_ANIMATELAST; i++)
      EnableMenuItem(GetMenu(hWnd), i, MF_BYCOMMAND | MF_GRAYED);
		}

	if (hbm)
		{
		DeleteObject(hbm);
		hbm = NULL;
		}
			
	if (hPal)
		{
		DeleteObject(hPal);
		hPal = NULL;
		}

	if (hDIB)
		{
		GlobalFree(hDIB);
		hDIB = NULL;
		}

	szFileName[0] = 0;
}

//****************************************************************************
// Function: SizeWindowToDIB
//
// Purpose: Resizes the window to fit around the bitmap (DIB).
//
// Parameters:
//    hWnd    == Handle to the window.
//    hdib    == Handle to the DIB.
//
// Returns : No Return
//
// Comments:
//
// History:  Date       Author        Reason
//           2/10/92    JMS           Created
//****************************************************************************

VOID SizeWindowToDIB(HWND hWnd, HGLOBAL hdib)
{
  static int cx=0, cy=0, sx=0, sy=0;
  LPBITMAPINFOHEADER lpbi = (LPBITMAPINFOHEADER) GlobalLock(hdib);

  if (!cx)  
	  cx = 2*GetSystemMetrics(SM_CXFRAME)
         + GetSystemMetrics(SM_CXVSCROLL) - 1;

  if (!cy)
    cy = GetSystemMetrics(SM_CYCAPTION)
         + GetSystemMetrics(SM_CYMENU)
         + 2*GetSystemMetrics(SM_CYFRAME);

  if (!sx || !sy)
    {
    HDC hdc = GetDC(hWnd);
    sx = GetDeviceCaps(hdc, HORZRES);
    sy = GetDeviceCaps(hdc, VERTRES);
    ReleaseDC(hWnd, hdc);
    }

  SetWindowPos( hWnd, NULL, 0, 0,
        min(max((int) lpbi->biWidth  + cx, 200), sx),
        min((int) lpbi->biHeight + cy, sy),
        SWP_NOMOVE|SWP_NOZORDER );

  GlobalUnlock(hdib);
}
